/**
 * @swagger
 * tags:
 *   - name: Queue
 *     description: 큐 모니터링 및 관리 관련 API
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     QueueStatus:
 *       type: object
 *       properties:
 *         waiting:
 *           type: integer
 *           description: 대기 중인 작업 수
 *         active:
 *           type: integer
 *           description: 실행 중인 작업 수
 *         completed:
 *           type: integer
 *           description: 완료된 작업 수
 *         failed:
 *           type: integer
 *           description: 실패한 작업 수
 *         delayed:
 *           type: integer
 *           description: 지연된 작업 수
 *     
 *     QueueStatusResponse:
 *       type: object
 *       properties:
 *         success:
 *           type: boolean
 *         data:
 *           type: object
 *           additionalProperties:
 *             $ref: '#/components/schemas/QueueStatus'
 *         timestamp:
 *           type: string
 *           format: date-time
 *     
 *     Job:
 *       type: object
 *       properties:
 *         id:
 *           type: string
 *           description: 작업 ID
 *         data:
 *           type: object
 *           description: 작업 데이터
 *         timestamp:
 *           type: integer
 *           description: 작업 생성 시간 (Unix timestamp)
 *         processedOn:
 *           type: integer
 *           description: 작업 처리 시작 시간 (Unix timestamp)
 *         finishedOn:
 *           type: integer
 *           description: 작업 완료 시간 (Unix timestamp)
 *         attemptsMade:
 *           type: integer
 *           description: 시도 횟수
 *         failedReason:
 *           type: string
 *           description: 실패 사유
 *         stacktrace:
 *           type: array
 *           items:
 *             type: string
 *           description: 오류 스택 트레이스
 *     
 *     FailedJobsResponse:
 *       type: object
 *       properties:
 *         success:
 *           type: boolean
 *         data:
 *           type: object
 *           properties:
 *             queueName:
 *               type: string
 *               description: 큐 이름
 *             failedJobs:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Job'
 *               description: 실패한 작업 목록
 *             count:
 *               type: integer
 *               description: 실패한 작업 수
 *             timestamp:
 *               type: string
 *               format: date-time
 *     
 *     JobsResponse:
 *       type: object
 *       properties:
 *         success:
 *           type: boolean
 *         data:
 *           type: object
 *           properties:
 *             queueName:
 *               type: string
 *               description: 큐 이름
 *             status:
 *               type: string
 *               description: 조회한 작업 상태
 *             jobs:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Job'
 *               description: 작업 목록
 *             count:
 *               type: integer
 *               description: 작업 수
 *             timestamp:
 *               type: string
 *               format: date-time
 *     
 *     RetryJobResponse:
 *       type: object
 *       properties:
 *         success:
 *           type: boolean
 *         data:
 *           type: object
 *           properties:
 *             queueName:
 *               type: string
 *               description: 큐 이름
 *             jobId:
 *               type: string
 *               description: 재시도된 작업 ID
 *             message:
 *               type: string
 *               description: 재시도 결과 메시지
 *             timestamp:
 *               type: string
 *               format: date-time
 */

/**
 * @swagger
 * /api/v1/queues/status:
 *   get:
 *     summary: 모든 큐의 상태 조회
 *     description: 시스템의 모든 큐(report-generation-queue, chat-request-queue, api-request-queue 등)의 현재 상태를 조회합니다.
 *     tags: [Queue]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: 큐 상태 조회 성공
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/QueueStatusResponse'
 *             example:
 *               success: true
 *               data:
 *                 report-generation-queue:
 *                   waiting: 0
 *                   active: 1
 *                   completed: 5
 *                   failed: 1
 *                   delayed: 0
 *                 chat-request-queue:
 *                   waiting: 2
 *                   active: 0
 *                   completed: 10
 *                   failed: 0
 *                   delayed: 0
 *               timestamp: "2025-07-03T06:30:00.000Z"
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       500:
 *         description: 서버 오류
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/queues/{queueName}/failed:
 *   get:
 *     summary: 특정 큐의 실패한 작업 목록 조회
 *     description: 지정된 큐에서 실패한 작업들의 상세 정보를 조회합니다.
 *     tags: [Queue]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: queueName
 *         required: true
 *         schema:
 *           type: string
 *         description: 큐 이름 report-generation-queue, chat-request-queue, api-request-queue
 *         example: report-generation-queue
 *     responses:
 *       200:
 *         description: 실패한 작업 목록 조회 성공
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/FailedJobsResponse'
 *             example:
 *               success: true
 *               data:
 *                 queueName: "report-generation-queue"
 *                 failedJobs:
 *                   - id: "1"
 *                     data:
 *                       userId: "user123"
 *                       task: "문서 생성 작업"
 *                     timestamp: 1735883088209
 *                     processedOn: 1735883088300
 *                     finishedOn: 1735883088400
 *                     attemptsMade: 3
 *                     failedReason: "메모리 부족으로 인한 처리 실패"
 *                     stacktrace: ["Error: Out of memory", "at generateReport()"]
 *                 count: 1
 *                 timestamp: "2025-07-03T06:30:00.000Z"
 *       404:
 *         description: 큐를 찾을 수 없음
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       500:
 *         description: 서버 오류
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/queues/{queueName}/jobs:
 *   get:
 *     summary: 특정 큐의 작업 목록 조회
 *     description: 지정된 큐의 특정 상태 작업들을 조회합니다.
 *     tags: [Queue]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: queueName
 *         required: true
 *         schema:
 *           type: string
 *         description: 큐 이름 report-generation-queue, chat-request-queue, api-request-queue
 *         example: report-generation-queue
 *       - in: query
 *         name: status
 *         schema:
 *           type: string
 *           enum: [waiting, active, completed, failed, delayed]
 *           default: active
 *         description: 조회할 작업 상태
 *     responses:
 *       200:
 *         description: 작업 목록 조회 성공
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/JobsResponse'
 *             example:
 *               success: true
 *               data:
 *                 queueName: "report-generation-queue"
 *                 status: "active"
 *                 jobs:
 *                   - id: "2"
 *                     data:
 *                       userId: "user456"
 *                       task: "보고서 생성 중"
 *                     timestamp: 1735883200000
 *                     processedOn: 1735883210000
 *                 count: 1
 *                 timestamp: "2025-07-03T06:30:00.000Z"
 *       404:
 *         description: 큐를 찾을 수 없음
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       500:
 *         description: 서버 오류
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/queues/{queueName}/jobs/{jobId}/retry:
 *   post:
 *     summary: 실패한 작업 재시도
 *     description: 지정된 큐의 실패한 작업을 재시도합니다.
 *     tags: [Queue]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: queueName
 *         required: true
 *         schema:
 *           type: string
 *         description: 큐 이름 report-generation-queue, chat-request-queue, api-request-queue
 *         example: report-generation-queue
 *       - in: path
 *         name: jobId
 *         required: true
 *         schema:
 *           type: string
 *         description: 재시도할 작업 ID
 *         example: "1"
 *     responses:
 *       200:
 *         description: 작업 재시도 성공
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/RetryJobResponse'
 *             example:
 *               success: true
 *               data:
 *                 queueName: "report-generation-queue"
 *                 jobId: "1"
 *                 message: "작업이 성공적으로 재시도되었습니다."
 *                 timestamp: "2025-07-03T06:30:00.000Z"
 *       400:
 *         description: 잘못된 요청 (실패 상태가 아닌 작업 등)
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       404:
 *         description: 큐 또는 작업을 찾을 수 없음
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       500:
 *         description: 서버 오류
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/queues/{queueName}/failed/retry-all:
 *   post:
 *     summary: 모든 실패한 작업 재시도
 *     description: 지정된 큐의 모든 실패한 작업을 재시도합니다.
 *     tags: [Queue]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: queueName
 *         required: true
 *         schema:
 *           type: string
 *         description: 큐 이름 report-generation-queue, chat-request-queue, api-request-queue
 *         example: report-generation-queue
 *     responses:
 *       200:
 *         description: 모든 작업 재시도 성공
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: true
 *                 data:
 *                   type: object
 *                   properties:
 *                     queueName:
 *                       type: string
 *                       description: 큐 이름
 *                     retriedCount:
 *                       type: integer
 *                       description: 재시도된 작업 수
 *                     message:
 *                       type: string
 *                       description: 재시도 결과 메시지
 *                     timestamp:
 *                       type: string
 *                       format: date-time
 *       404:
 *         description: 큐를 찾을 수 없음
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       500:
 *         description: 서버 오류
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

export const queueSchemas = {
    QueueStatus: {
        type: 'object',
        properties: {
            waiting: {
                type: 'integer',
                description: '대기 중인 작업 수'
            },
            active: {
                type: 'integer',
                description: '실행 중인 작업 수'
            },
            completed: {
                type: 'integer',
                description: '완료된 작업 수'
            },
            failed: {
                type: 'integer',
                description: '실패한 작업 수'
            },
            delayed: {
                type: 'integer',
                description: '지연된 작업 수'
            }
        }
    },
    Job: {
        type: 'object',
        properties: {
            id: {
                type: 'string',
                description: '작업 ID'
            },
            data: {
                type: 'object',
                description: '작업 데이터'
            },
            timestamp: {
                type: 'integer',
                description: '작업 생성 시간 (Unix timestamp)'
            },
            processedOn: {
                type: 'integer',
                description: '작업 처리 시작 시간 (Unix timestamp)'
            },
            finishedOn: {
                type: 'integer',
                description: '작업 완료 시간 (Unix timestamp)'
            },
            attemptsMade: {
                type: 'integer',
                description: '시도 횟수'
            },
            failedReason: {
                type: 'string',
                description: '실패 사유'
            },
            stacktrace: {
                type: 'array',
                items: {
                    type: 'string'
                },
                description: '오류 스택 트레이스'
            }
        }
    }
}; 