/**
 * @swagger
 * tags:
 *   - name: RAG
 *     description: RAG(Retrieval-Augmented Generation) 관련 API
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     RAGSearchRequest:
 *       type: object
 *       required:
 *         - query
 *       properties:
 *         query:
 *           type: string
 *           description: 검색할 텍스트
 *         username:
 *           type: string
 *           description: 사용자명 (권장 방식, 특정 사용자의 문서만 검색할 경우)
 *         userId:
 *           type: integer
 *           description: 사용자 ID (정수형, 하위 호환성용)
 *     RAGSearchResponse:
 *       type: object
 *       properties:
 *         success:
 *           type: boolean
 *         data:
 *           type: object
 *           properties:
 *             documents:
 *               type: array
 *               items:
 *                 type: object
 *                 properties:
 *                   text:
 *                     type: string
 *                   metadata:
 *                     type: object
 *             timestamp:
 *               type: string
 *               format: date-time
 *     RAGUploadRequest:
 *       type: object
 *       properties:
 *         file:
 *           type: string
 *           format: binary
 *           description: 업로드할 파일
 *         user_id:
 *           type: string
 *           description: 사용자 ID
 *         rag_mode:
 *           type: string
 *           enum: [fast, normal, rich, intelligence]
 *           description: RAG 처리 모드 (fast=빠른처리, normal=일반, rich=이미지설명포함, intelligence=Chandra AI 분석)
 *         rag_pdf_backend:
 *           type: string
 *           enum: [pdfplumber, pymupdf, camelot]
 *           description: PDF 파싱 백엔드
 *         rag_process_text_only:
 *           type: string
 *           enum: ['yes', 'no']
 *           description: 텍스트만 처리 여부
 *         rag_generate_image_embeddings:
 *           type: string
 *           enum: ['yes', 'no']
 *           description: 이미지 임베딩 생성 여부
 *         rag_enable_graphrag:
 *           type: string
 *           enum: ['yes', 'no']
 *           description: GraphRAG 활성화 여부
 *         use_enhanced_embedding:
 *           type: string
 *           enum: ['yes', 'no']
 *           description: 향상된 임베딩(예상질문 생성) 사용 여부
 *         project_name:
 *           type: string
 *           description: 프로젝트명 (문서 그룹화용)
 *     RAGDocument:
 *       type: object
 *       properties:
 *         id:
 *           type: string
 *           description: 문서 ID
 *         text:
 *           type: string
 *           description: 문서 내용
 *         metadata:
 *           type: object
 *           description: 문서 메타데이터
 *     RAGListResponse:
 *       type: object
 *       properties:
 *         success:
 *           type: boolean
 *           description: 요청 성공 여부
 *         data:
 *           type: object
 *           properties:
 *             name:
 *               type: string
 *               description: 컬렉션 이름
 *             total_chunks:
 *               type: integer
 *               description: 전체 청크 수
 *             unique_files:
 *               type: integer
 *               description: 고유 파일 수
 *             metadata:
 *               type: object
 *               description: 컬렉션 메타데이터
 *             documents:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/RAGDocument'
 *               description: 문서 목록
 *     RAGPathResponse:
 *       type: object
 *       properties:
 *         success:
 *           type: boolean
 *           description: 요청 성공 여부
 *         data:
 *           type: object
 *           properties:
 *             path:
 *               type: string
 *               description: 설정된 문서 경로
 *             message:
 *               type: string
 *               description: 상태 메시지
 *             timestamp:
 *               type: string
 *               format: date-time
 *     RAGQueueStatus:
 *       type: object
 *       properties:
 *         total_documents:
 *           type: integer
 *           description: 전체 문서 수
 *         remaining:
 *           type: integer
 *           description: 남은 처리 문서 수
 *         processing:
 *           type: integer
 *           description: 현재 처리 중인 문서 수
 *         completed:
 *           type: integer
 *           description: 완료된 문서 수
 *         failed:
 *           type: integer
 *           description: 실패한 문서 수
 *         current_batch:
 *           type: integer
 *           description: 현재 배치 번호
 *         batch_size:
 *           type: integer
 *           description: 배치 크기
 *         progress_percentage:
 *           type: number
 *           format: float
 *           description: 진행률 (%)
 *         timestamp:
 *           type: string
 *           format: date-time
 *           description: 상태 확인 시간
 *     RAGInitStatusResponse:
 *       type: object
 *       properties:
 *         status:
 *           type: string
 *           enum: [success, error]
 *           description: 요청 상태
 *         queue_status:
 *           $ref: '#/components/schemas/RAGQueueStatus'
 *         failed_documents:
 *           type: array
 *           items:
 *             type: string
 *           description: 처리 실패한 문서 목록
 *     RAGTotalTimeResponse:
 *       type: object
 *       properties:
 *         status:
 *           type: string
 *           enum: [success, error]
 *           description: 응답 상태
 *         start_time:
 *           type: string
 *           format: date-time
 *           description: 초기화 시작 시간
 *         end_time:
 *           type: string
 *           format: date-time
 *           nullable: true
 *           description: 초기화 완료 시간 (완료되지 않은 경우 null)
 *         total_time:
 *           type: object
 *           nullable: true
 *           properties:
 *             seconds:
 *               type: number
 *               description: 총 소요 시간 (초)
 *             formatted:
 *               type: string
 *               description: 포맷된 시간 문자열 (HH:MM:SS)
 *             hours:
 *               type: number
 *               description: 총 소요 시간 (시간)
 *             minutes:
 *               type: number
 *               description: 총 소요 시간 (분)
 *         is_completed:
 *           type: boolean
 *           description: 초기화 완료 여부
 *         current_status:
 *           type: string
 *           enum: [success, processing, pending, partial_success, error]
 *           description: 현재 큐 상태
 *         queue_info:
 *           type: object
 *           nullable: true
 *           properties:
 *             total_documents:
 *               type: integer
 *               description: 전체 문서 수
 *             remaining:
 *               type: integer
 *               description: 남은 문서 수
 *             processing:
 *               type: integer
 *               description: 처리 중인 문서 수
 *             completed:
 *               type: integer
 *               description: 완료된 문서 수
 *             failed:
 *               type: integer
 *               description: 실패한 문서 수
 *             progress_percentage:
 *               type: number
 *               description: 진행률 (%)
 *             current_batch:
 *               type: integer
 *               description: 현재 처리 중인 배치 번호
 *             total_batches:
 *               type: integer
 *               description: 전체 배치 수
 *             batch_size:
 *               type: integer
 *               description: 배치 크기
 *             batch_progress:
 *               type: string
 *               description: 배치 진행 상태 (현재/전체)
 *         timestamp:
 *           type: string
 *           format: date-time
 *           description: 응답 생성 시간
 *     RAGChunk:
 *       type: object
 *       properties:
 *         index:
 *           type: integer
 *           description: 청크 인덱스(순서)
 *         content:
 *           type: string
 *           description: 청크 텍스트 내용
 *         metadata:
 *           type: object
 *           description: 청크 메타데이터(파일 경로, chunk_index 등)
 *     RAGChunkListResponse:
 *       type: object
 *       properties:
 *         success:
 *           type: boolean
 *           description: 요청 성공 여부
 *         chunks:
 *           type: array
 *           items:
 *             $ref: '#/components/schemas/RAGChunk'
 *           description: 청크 리스트
 */

/**
 * @swagger
 * /api/v1/rag/search:
 *   post:
 *     summary: RAG 검색 (비동기)
 *     description: 문서 데이터베이스에서 관련 문서를 검색합니다. 작업 ID를 반환하며, 상태 확인 엔드포인트를 통해 결과를 확인할 수 있습니다.
 *     tags: [RAG]
 *     security:
 *       - apiKeyAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/RAGSearchRequest'
 *           example:
 *             query: "AI에이전트"
 *             username: "admin"
 *     responses:
 *       200:
 *         description: 작업이 성공적으로 큐에 등록됨
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     jobId:
 *                       type: string
 *                       description: 작업 ID
 *                     message:
 *                       type: string
 *                       description: 상태 메시지
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/rag/search/sync:
 *   post:
 *     summary: RAG 검색 (동기)
 *     description: 문서 데이터베이스에서 관련 문서를 검색하고 결과를 즉시 반환합니다.
 *     tags: [RAG]
 *     security:
 *       - apiKeyAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/RAGSearchRequest'
 *           example:
 *             query: "AI에이전트"
 *             username: "admin"
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     query:
 *                       type: string
 *                       description: 검색 쿼리
 *                     documents:
 *                       type: array
 *                       items:
 *                         type: object
 *                         properties:
 *                           text:
 *                             type: string
 *                             description: 문서 내용
 *                           metadata:
 *                             type: object
 *                             description: 문서 메타데이터
 *                     total:
 *                       type: integer
 *                       description: 검색된 문서 수
 *                     timestamp:
 *                       type: string
 *                       format: date-time
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/rag/upload:
 *   post:
 *     summary: 문서 업로드
 *     description: RAG 시스템에 새로운 문서를 업로드합니다.
 *     tags: [RAG]
 *     security:
 *       - apiKeyAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         multipart/form-data:
 *           schema:
 *             $ref: '#/components/schemas/RAGUploadRequest'
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 message:
 *                   type: string
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/rag/init:
 *   post:
 *     summary: RAG 시스템 초기화
 *     description: RAG 시스템을 초기화하고 문서를 다시 인덱싱합니다.
 *     tags: [RAG]
 *     security:
 *       - apiKeyAuth: []
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 message:
 *                   type: string
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */


/**
 * @swagger
 * /api/v1/rag/init-status:
 *   get:
 *     operationId: getRAGInitStatus
 *     summary: RAG 초기화 상태 확인
 *     description: RAG 시스템 초기화 작업의 현재 진행 상태를 확인합니다.
 *     tags: [RAG]
 *     security:
 *       - apiKeyAuth: []
 *     responses:
 *       200:
 *         description: 초기화 상태 정보
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/RAGInitStatusResponse'
 *             example:
 *               status: "success"
 *               queue_status:
 *                 total_documents: 7
 *                 remaining: 0
 *                 processing: 0
 *                 completed: 7
 *                 failed: 0
 *                 current_batch: 1
 *                 batch_size: 32
 *                 progress_percentage: 100.0
 *                 timestamp: "2025-05-17T18:29:49.147705"
 *               failed_documents: []
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       500:
 *         description: 서버 오류
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/rag/path:
 *   post:
 *     operationId: setRAGPath
 *     summary: RAG 문서 경로 설정
 *     description: RAG 시스템의 문서 경로를 설정합니다.
 *     tags: [RAG]
 *     security:
 *       - apiKeyAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - path
 *             properties:
 *               path:
 *                 type: string
 *                 description: 설정할 문서 경로
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/RAGPathResponse'
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       500:
 *         description: 서버 오류
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/rag/list:
 *   get:
 *     operationId: listRAGDocuments
 *     summary: RAG 문서 목록
 *     description: RAG 시스템에 저장된 문서 목록을 조회합니다. 사용자 ID를 지정하면 해당 사용자의 문서만 조회합니다.
 *     tags: [RAG]
 *     security:
 *       - apiKeyAuth: []
 *     parameters:
 *       - in: query
 *         name: userId
 *         schema:
 *           type: string
 *         description: 특정 사용자의 문서만 조회하고 싶을 때 사용자 ID를 지정합니다. 지정하지 않으면 모든 문서를 조회합니다.
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/RAGListResponse'
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       500:
 *         description: 서버 오류
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/rag/add:
 *   post:
 *     operationId: addRAGDocument
 *     summary: RAG 문서 추가
 *     description: RAG 시스템에 새로운 문서를 추가합니다.
 *     tags: [RAG]
 *     security:
 *       - apiKeyAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - path
 *             properties:
 *               path:
 *                 type: string
 *                 description: 추가할 문서의 경로
 *               user_id:
 *                 type: string
 *                 description: 사용자 ID (기본값 'all')
 *           example:
 *             path: "/path/to/document.txt"
 *             user_id: ""
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     status:
 *                       type: string
 *                       example: "success"
 *                     message:
 *                       type: string
 *                       example: "Successfully processed documents"
 *                     files_count:
 *                       type: integer
 *                       description: 처리된 파일 수
 *                     original_path:
 *                       type: string
 *                       description: 원본 파일 경로
 *                     rag_docs_path:
 *                       type: string
 *                       description: RAG 문서 저장 경로
 *                     user_id:
 *                       type: string
 *                       description: 사용자 ID
 *                     summary:
 *                       type: object
 *                       description: 처리 결과 요약
 *       400:
 *         description: 잘못된 요청
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/rag/cleanup:
 *   post:
 *     operationId: cleanupRAGFiles
 *     summary: RAG 파일 정리
 *     description: PostgreSQL에서 실제로 존재하지 않는 파일들의 데이터를 정리합니다.
 *     tags: [RAG]
 *     security:
 *       - apiKeyAuth: []
 *     requestBody:
 *       required: false
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               user_id:
 *                 type: string
 *                 description: 특정 사용자의 파일만 정리하고 싶을 때 사용자 ID를 지정합니다. 지정하지 않으면 모든 파일을 정리합니다.
 *           example:
 *             user_id: "user123"
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     status:
 *                       type: string
 *                       example: "success"
 *                     message:
 *                       type: string
 *                       example: "100개의 청크가 정리되었습니다."
 *                     deleted_chunks:
 *                       type: integer
 *                       description: 삭제된 청크 수
 *                     deleted_files:
 *                       type: integer
 *                       description: 삭제된 파일 수
 *                     missing_files:
 *                       type: array
 *                       items:
 *                         type: string
 *                       description: 삭제된 파일 경로 목록
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       500:
 *         description: 서버 오류
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/rag/status/{jobId}:
 *   get:
 *     operationId: getRAGStatus
 *     summary: RAG 검색 작업 상태 확인
 *     description: RAG 검색 요청의 처리 상태를 확인합니다.
 *     tags: [RAG]
 *     security:
 *       - apiKeyAuth: []
 *     parameters:
 *       - in: path
 *         name: jobId
 *         required: true
 *         schema:
 *           type: string
 *         description: 작업 ID
 *     responses:
 *       200:
 *         description: 작업 상태 정보
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     jobId:
 *                       type: string
 *                       description: 작업 ID
 *                     state:
 *                       type: string
 *                       enum: [completed, failed, active, waiting, delayed]
 *                       description: 작업 상태
 *                     documents:
 *                       type: array
 *                       description: 검색된 문서 목록 (완료 상태인 경우)
 *                       items:
 *                         type: object
 *                         properties:
 *                           text:
 *                             type: string
 *                             description: 문서 내용
 *                           metadata:
 *                             type: object
 *                             description: 문서 메타데이터
 *                     message:
 *                       type: string
 *                       description: 상태 메시지
 *                     timestamp:
 *                       type: string
 *                       format: date-time
 *       404:
 *         description: 작업을 찾을 수 없음
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: object
 *                   properties:
 *                     message:
 *                       type: string
 *                       example: '해당 작업을 찾을 수 없습니다.'
 */

/**
 * @swagger
 * /api/v1/rag/delete:
 *   post:
 *     operationId: deleteRAGDocument
 *     summary: RAG 문서/폴더 삭제
 *     description: RAG 시스템에서 문서나 폴더를 삭제합니다.
 *     tags: [RAG]
 *     security:
 *       - apiKeyAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - file_paths
 *             properties:
 *               file_paths:
 *                 oneOf:
 *                   - type: string
 *                   - type: array
 *                     items:
 *                       type: string
 *                 description: 삭제할 파일이나 폴더의 경로 (단일 경로 또는 경로 배열)
 *               user_id:
 *                 type: string
 *                 description: 사용자 ID (기본값 'all')
 *           example:
 *             file_paths: "/path/to/document.txt"
 *             user_id: ""
 *     responses:
 *       200:
 *         description: 성공적인 응답
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     success:
 *                       type: boolean
 *                     user_id:
 *                       type: string
 *                       description: 사용자 ID
 *                     results:
 *                       type: array
 *                       items:
 *                         type: object
 *                         properties:
 *                           path:
 *                             type: string
 *                             description: 삭제된 파일/폴더 경로
 *                           success:
 *                             type: boolean
 *                             description: 삭제 성공 여부
 *                           chunks_deleted:
 *                             type: integer
 *                             description: 삭제된 청크 수
 *                     error:
 *                       type: string
 *                       nullable: true
 *                       description: 오류 메시지 (실패 시)
 *                     timestamp:
 *                       type: string
 *                       format: date-time
 *                       description: 작업 완료 시간
 *       400:
 *         description: 잘못된 요청
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/rag/chunks:
 *   get:
 *     operationId: getRAGChunks
 *     summary: 문서별 청크 리스트 조회
 *     description: 지정한 파일명에 해당하는 문서의 청크(분할된 텍스트) 리스트를 반환합니다.
 *     tags: [RAG]
 *     security:
 *       - apiKeyAuth: []
 *     parameters:
 *       - in: query
 *         name: filename
 *         schema:
 *           type: string
 *         required: true
 *         description: 청크를 조회할 문서 파일명
 *     responses:
 *       200:
 *         description: 청크 리스트 반환
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/RAGChunkListResponse'
 *             example:
 *               success: true
 *               chunks:
 *                 - index: 0
 *                   content: "청크 내용 1"
 *                   metadata: { source: "/path/to/file.pdf", chunk_index: 0 }
 *                 - index: 1
 *                   content: "청크 내용 2"
 *                   metadata: { source: "/path/to/file.pdf", chunk_index: 1 }
 *       400:
 *         description: 잘못된 요청(파라미터 누락 등)
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       500:
 *         description: 서버 오류
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/rag/download:
 *   get:
 *     operationId: downloadRAGFile
 *     summary: RAG 파일 다운로드
 *     description: RAG 시스템에 저장된 문서 파일을 다운로드합니다. 출처 문서를 직접 다운로드하거나 뷰어에서 열 때 사용할 수 있습니다.
 *     tags: [RAG]
 *     security:
 *       - apiKeyAuth: []
 *     parameters:
 *       - in: query
 *         name: filename
 *         schema:
 *           type: string
 *         required: true
 *         description: 다운로드할 파일명 (상대 경로 또는 절대 경로). 절대 경로인 경우 RAG_DOCS_PATH 내에 있어야 함.
 *         example: "manual.pdf"
 *     responses:
 *       200:
 *         description: 파일 다운로드 성공
 *         content:
 *           application/octet-stream:
 *             schema:
 *               type: string
 *               format: binary
 *           application/pdf:
 *             schema:
 *               type: string
 *               format: binary
 *           application/vnd.openxmlformats-officedocument.wordprocessingml.document:
 *             schema:
 *               type: string
 *               format: binary
 *         headers:
 *           Content-Disposition:
 *             description: 파일 다운로드를 위한 헤더
 *             schema:
 *               type: string
 *               example: "attachment; filename=manual.pdf"
 *       400:
 *         description: 잘못된 요청 (파일명 누락)
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: object
 *                   properties:
 *                     message:
 *                       type: string
 *                       example: "파일명이 필요합니다."
 *       404:
 *         description: 파일을 찾을 수 없음
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: object
 *                   properties:
 *                     message:
 *                       type: string
 *                       example: "파일을 찾을 수 없습니다."
 *       500:
 *         description: 서버 오류
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: object
 *                   properties:
 *                     message:
 *                       type: string
 *                       example: "파일 다운로드 중 오류가 발생했습니다."
 */

/**
 * @swagger
 * /api/v1/rag/thumbnail:
 *   get:
 *     operationId: getRAGThumbnail
 *     summary: 문서 페이지 썸네일 조회
 *     description: 지정한 문서의 특정 페이지 썸네일 이미지를 base64 형식으로 반환합니다.
 *     tags: [RAG]
 *     parameters:
 *       - in: query
 *         name: user_id
 *         schema:
 *           type: string
 *         required: true
 *         description: 사용자 ID
 *         example: "admin"
 *       - in: query
 *         name: filename
 *         schema:
 *           type: string
 *         required: true
 *         description: 문서 파일명
 *         example: "document.pdf"
 *       - in: query
 *         name: page
 *         schema:
 *           type: integer
 *           minimum: 1
 *         required: true
 *         description: 페이지 번호 (1부터 시작)
 *         example: 1
 *     responses:
 *       200:
 *         description: 썸네일 이미지 반환
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: true
 *                 data:
 *                   type: object
 *                   properties:
 *                     image:
 *                       type: string
 *                       description: base64 인코딩된 이미지 (data:image/webp;base64,...)
 *                     filename:
 *                       type: string
 *                       description: 파일명
 *                     page:
 *                       type: integer
 *                       description: 페이지 번호
 *                 timestamp:
 *                   type: string
 *                   format: date-time
 *       400:
 *         description: 잘못된 요청 (필수 파라미터 누락 또는 유효하지 않은 페이지 번호)
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       500:
 *         description: 서버 오류
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/rag/files:
 *   get:
 *     operationId: getRAGFiles
 *     summary: RAG 파일 목록 조회
 *     description: 사용자별로 RAG 시스템에 저장된 파일 목록을 조회합니다.
 *     tags: [RAG]
 *     security:
 *       - apiKeyAuth: []
 *     parameters:
 *       - in: query
 *         name: userId
 *         schema:
 *           type: string
 *         description: 사용자 ID (기본값 'all')
 *         example: "admin"
 *     responses:
 *       200:
 *         description: 파일 목록 반환
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     files:
 *                       type: array
 *                       items:
 *                         type: object
 *                         properties:
 *                           name:
 *                             type: string
 *                             description: 파일명
 *                           path:
 *                             type: string
 *                             description: 파일 경로
 *                           size:
 *                             type: integer
 *                             description: 파일 크기 (bytes)
 *                           modifiedAt:
 *                             type: string
 *                             format: date-time
 *                             description: 수정 시간
 *                     totalSize:
 *                       type: integer
 *                       description: 전체 파일 크기 합계
 *                     count:
 *                       type: integer
 *                       description: 파일 수
 *                     userId:
 *                       type: string
 *                       description: 사용자 ID
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/rag/files/all-users:
 *   get:
 *     operationId: getRAGFilesAllUsers
 *     summary: 전체 사용자 RAG 파일 목록 (관리자)
 *     description: 관리자 권한으로 모든 사용자의 RAG 파일 목록과 폴더 구조를 조회합니다.
 *     tags: [RAG]
 *     security:
 *       - apiKeyAuth: []
 *     responses:
 *       200:
 *         description: 전체 사용자 파일 목록 반환
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     users:
 *                       type: array
 *                       items:
 *                         type: object
 *                         properties:
 *                           userId:
 *                             type: string
 *                           fileCount:
 *                             type: integer
 *                           totalSize:
 *                             type: integer
 *                           files:
 *                             type: array
 *                             items:
 *                               type: object
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/rag/embedding-status:
 *   get:
 *     operationId: getRAGEmbeddingStatus
 *     summary: 파일 임베딩 상태 확인
 *     description: 특정 파일의 RAG 임베딩 처리 상태를 확인합니다.
 *     tags: [RAG]
 *     parameters:
 *       - in: query
 *         name: fileName
 *         schema:
 *           type: string
 *         required: true
 *         description: 확인할 파일명
 *         example: "document.pdf"
 *       - in: query
 *         name: userId
 *         schema:
 *           type: string
 *         description: 사용자 ID
 *         example: "admin"
 *     responses:
 *       200:
 *         description: 임베딩 상태 반환
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     status:
 *                       type: string
 *                       enum: [pending, processing, completed, error]
 *                       description: 임베딩 처리 상태
 *                     fileName:
 *                       type: string
 *                     chunkCount:
 *                       type: integer
 *                       description: 처리된 청크 수
 *                     message:
 *                       type: string
 *                 timestamp:
 *                   type: string
 *                   format: date-time
 *       400:
 *         description: 잘못된 요청 (파일명 누락)
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/rag/docs-path:
 *   get:
 *     operationId: getRAGDocsPath
 *     summary: RAG 문서 저장 경로 조회
 *     description: RAG 시스템의 문서 저장 기본 경로를 조회합니다.
 *     tags: [RAG]
 *     responses:
 *       200:
 *         description: 문서 경로 반환
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: true
 *                 data:
 *                   type: object
 *                   properties:
 *                     path:
 *                       type: string
 *                       description: RAG 문서 저장 경로
 *                       example: "/home/user/문서/rag-docs"
 *                 timestamp:
 *                   type: string
 *                   format: date-time
 */

/**
 * @swagger
 * /api/v1/rag/admin/document-stats:
 *   get:
 *     operationId: getRAGAdminDocumentStats
 *     summary: 관리자용 문서 통계
 *     description: 전체 RAG 시스템의 문서 및 임베딩 통계를 조회합니다.
 *     tags: [RAG]
 *     security:
 *       - apiKeyAuth: []
 *     responses:
 *       200:
 *         description: 문서 통계 반환
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     total_documents:
 *                       type: integer
 *                       description: 전체 문서 수
 *                     total_chunks:
 *                       type: integer
 *                       description: 전체 청크 수
 *                     by_user:
 *                       type: array
 *                       items:
 *                         type: object
 *                         properties:
 *                           user_id:
 *                             type: string
 *                           document_count:
 *                             type: integer
 *                           chunk_count:
 *                             type: integer
 *                 timestamp:
 *                   type: string
 *                   format: date-time
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/rag/total-time:
 *   get:
 *     operationId: getRAGTotalTime
 *     summary: RAG 초기화 소요 시간 조회
 *     description: RAG 시스템 초기화 작업의 전체 소요 시간을 조회합니다.
 *     tags: [RAG]
 *     security:
 *       - apiKeyAuth: []
 *     responses:
 *       200:
 *         description: 소요 시간 정보 반환
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/RAGTotalTimeResponse'
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

export const ragSchemas = {
    RAGSearchRequest: {
        type: 'object',
        required: ['query'],
        properties: {
            query: {
                type: 'string',
                description: '검색할 텍스트'
            },
            username: {
                type: 'string',
                description: '사용자명 (권장 방식, 특정 사용자의 문서만 검색할 경우)'
            },
            userId: {
                type: 'integer',
                description: '사용자 ID (정수형, 하위 호환성용)'
            }
        }
    },
    RAGPathRequest: {
        type: 'object',
        required: ['path'],
        properties: {
            path: {
                type: 'string',
                description: '문서 경로'
            }
        }
    },
    RAGDocument: {
        type: 'object',
        properties: {
            file_name: {
                type: 'string',
                description: '파일명'
            },
            source: {
                type: 'string',
                description: '파일 경로'
            },
            chunk_count: {
                type: 'integer',
                description: '청크 수'
            },
            last_modified: {
                type: 'string',
                format: 'date-time',
                description: '마지막 수정 시간'
            }
        }
    }
}; 