/**
 * 견적서 AI 에이전트 도구 모음
 *
 * 이 모듈은 AI가 견적서를 자동으로 생성하고 관리할 수 있도록 하는
 * 도구들을 제공합니다.
 *
 * 주요 기능:
 * - RAG 기반 과거 견적서/단가 검색
 * - 회사 정보 (로고, 인감) 조회
 * - AI 견적서 내용 생성 및 검산
 * - PDF 생성
 */

import { debugLog, logger } from '../../utils/logger.js';
import { URLs } from '../../config/urls.js';

// 데이터베이스 연결 가져오기
const getDBPool = async () => {
    const { default: dbManager } = await import('../../services/database/index.js');
    return dbManager;
};

/**
 * 단가 카탈로그 검색 도구
 * AI가 견적서 작성 시 기준 단가를 조회할 수 있습니다.
 */
const priceCatalogSearchTool = {
    name: 'price_catalog_search',
    category: 'quotation',
    description: '단가 카탈로그에서 품목과 가격 정보를 검색합니다. 견적서 작성 시 기준 단가를 확인할 때 사용합니다.',
    version: '1.0.0',
    parameters: {
        query: {
            type: 'string',
            required: false,
            description: '검색할 품목명 또는 카테고리 (빈 값이면 전체 조회)'
        },
        category: {
            type: 'string',
            required: false,
            description: '품목 카테고리 필터'
        }
    },
    handler: async (args, context) => {
        const { query, category } = args || {};
        const { username, updateProgress } = context;

        logger.debug(`[PriceCatalogSearch] 검색 시작: query=${query}, category=${category}`);

        try {
            if (updateProgress) {
                await updateProgress('📊 단가 카탈로그 검색 중...', 30);
            }

            const db = await getDBPool();

            // 사용자 ID 조회
            const userResult = await db.query(
                'SELECT id FROM users WHERE username = $1',
                [username || 'admin']
            );
            const userId = userResult.rows[0]?.id;

            // 단가 카탈로그 검색 쿼리 구성
            let sql = `
                SELECT id, category, item_code, item_name, description,
                       unit, unit_price, currency, tags, metadata
                FROM price_catalog
                WHERE is_active = true
            `;
            const params = [];
            let paramIndex = 1;

            // 사용자별 필터 (user_id가 NULL이면 공용 단가)
            if (userId) {
                sql += ` AND (user_id = $${paramIndex} OR user_id IS NULL)`;
                params.push(userId);
                paramIndex++;
            }

            // 검색어 필터
            if (query && query.trim()) {
                sql += ` AND (
                    item_name ILIKE $${paramIndex}
                    OR description ILIKE $${paramIndex}
                    OR item_code ILIKE $${paramIndex}
                    OR $${paramIndex + 1} = ANY(tags)
                )`;
                params.push(`%${query}%`);
                params.push(query);
                paramIndex += 2;
            }

            // 카테고리 필터
            if (category && category.trim()) {
                sql += ` AND category ILIKE $${paramIndex}`;
                params.push(`%${category}%`);
                paramIndex++;
            }

            sql += ' ORDER BY category, item_name LIMIT 50';

            const result = await db.query(sql, params);

            if (updateProgress) {
                await updateProgress(`✅ ${result.rows.length}개 단가 항목 검색 완료`, 70);
            }

            return {
                success: true,
                data: {
                    items: result.rows.map(row => ({
                        id: row.id,
                        category: row.category,
                        itemCode: row.item_code,
                        itemName: row.item_name,
                        description: row.description,
                        unit: row.unit,
                        unitPrice: parseFloat(row.unit_price),
                        currency: row.currency,
                        tags: row.tags || []
                    })),
                    count: result.rows.length
                },
                message: `단가 카탈로그에서 ${result.rows.length}개 항목을 찾았습니다.`
            };
        } catch (error) {
            logger.error('[PriceCatalogSearch] 오류:', error);
            return {
                success: false,
                error: error.message,
                message: `단가 검색 실패: ${error.message}`
            };
        }
    }
};

/**
 * 과거 견적서 검색 도구
 * RAG를 활용하여 과거 견적서와 유사 프로젝트의 견적 내역을 검색합니다.
 */
const pastQuotationSearchTool = {
    name: 'past_quotation_search',
    category: 'quotation',
    description: '과거 견적서와 유사 프로젝트의 견적 내역을 검색합니다. RAG 기반으로 관련 문서도 함께 검색합니다.',
    version: '1.0.0',
    parameters: {
        query: {
            type: 'string',
            required: true,
            description: '검색할 프로젝트명, 고객사, 또는 품목 키워드'
        },
        clientCompany: {
            type: 'string',
            required: false,
            description: '고객사명으로 필터링'
        }
    },
    handler: async (args, context) => {
        const { query, clientCompany } = args || {};
        const { username, updateProgress } = context;

        if (!query || typeof query !== 'string') {
            return {
                success: false,
                error: 'query 파라미터가 필요합니다.',
                data: { quotations: [], ragResults: [] }
            };
        }

        logger.debug(`[PastQuotationSearch] 검색 시작: query=${query}, clientCompany=${clientCompany}`);

        try {
            if (updateProgress) {
                await updateProgress('📋 과거 견적서 검색 중...', 20);
            }

            const db = await getDBPool();

            // 사용자 ID 조회
            const userResult = await db.query(
                'SELECT id FROM users WHERE username = $1',
                [username || 'admin']
            );
            const userId = userResult.rows[0]?.id;

            // 1. 데이터베이스에서 과거 견적서 검색
            let sql = `
                SELECT q.id, q.quotation_number, q.created_at as quotation_date, q.project_name as title,
                       q.client_company, q.client_name as client_contact, q.total_amount,
                       q.status,
                       (SELECT json_agg(json_build_object(
                           'itemName', qi.item_name,
                           'quantity', qi.quantity,
                           'unitPrice', qi.unit_price,
                           'amount', qi.amount
                       )) FROM quotation_items qi WHERE qi.quotation_id = q.id) as items
                FROM quotations q
                WHERE 1=1
            `;
            const params = [];
            let paramIndex = 1;

            if (userId) {
                sql += ` AND q.user_id = $${paramIndex}`;
                params.push(userId);
                paramIndex++;
            }

            // 검색어 필터
            sql += ` AND (
                q.project_name ILIKE $${paramIndex}
                OR q.client_company ILIKE $${paramIndex}
                OR q.notes ILIKE $${paramIndex}
                OR EXISTS (
                    SELECT 1 FROM quotation_items qi
                    WHERE qi.quotation_id = q.id
                    AND qi.item_name ILIKE $${paramIndex}
                )
            )`;
            params.push(`%${query}%`);
            paramIndex++;

            // 고객사 필터
            if (clientCompany && clientCompany.trim()) {
                sql += ` AND q.client_company ILIKE $${paramIndex}`;
                params.push(`%${clientCompany}%`);
                paramIndex++;
            }

            sql += ' ORDER BY q.created_at DESC LIMIT 10';

            const quotationResult = await db.query(sql, params);

            if (updateProgress) {
                await updateProgress('🔍 RAG 문서 검색 중...', 50);
            }

            // 2. RAG 검색으로 관련 문서 찾기
            let ragResults = [];
            try {
                const { RAGService } = await import('../../plugins/rag/ragService.js');
                const ragEnabled = await RAGService.checkIfEnabled(false);

                if (ragEnabled) {
                    const searchResults = await RAGService.searchDocuments(
                        `견적서 ${query} 단가 비용`,
                        username || 'admin',
                        null,
                        'personal'
                    );

                    if (searchResults && searchResults.length > 0) {
                        ragResults = searchResults.slice(0, 5).map(r => ({
                            content: r.content?.substring(0, 500),
                            source: r.metadata?.source || r.source,
                            score: r.score || r.metadata?.relevance_score
                        }));
                    }
                }
            } catch (ragError) {
                logger.warn('[PastQuotationSearch] RAG 검색 실패 (계속 진행):', ragError.message);
            }

            if (updateProgress) {
                await updateProgress(`✅ 과거 견적서 ${quotationResult.rows.length}건 검색 완료`, 80);
            }

            return {
                success: true,
                data: {
                    quotations: quotationResult.rows.map(q => ({
                        id: q.id,
                        quotationNumber: q.quotation_number,
                        date: q.quotation_date,
                        title: q.title,
                        clientCompany: q.client_company,
                        clientContact: q.client_contact,
                        totalAmount: parseFloat(q.total_amount),
                        status: q.status,
                        items: q.items || []
                    })),
                    ragResults,
                    quotationCount: quotationResult.rows.length,
                    ragCount: ragResults.length
                },
                message: `과거 견적서 ${quotationResult.rows.length}건, 관련 문서 ${ragResults.length}건을 찾았습니다.`
            };
        } catch (error) {
            logger.error('[PastQuotationSearch] 오류:', error);
            return {
                success: false,
                error: error.message,
                message: `과거 견적서 검색 실패: ${error.message}`
            };
        }
    }
};

/**
 * 회사 정보 조회 도구
 * 견적서에 들어갈 회사 정보(로고, 인감 등)를 조회합니다.
 */
const companyInfoTool = {
    name: 'company_info_lookup',
    category: 'quotation',
    description: '견적서에 사용할 회사 정보(로고, 인감, 은행계좌 등)를 조회합니다.',
    version: '1.0.0',
    parameters: {
        companyName: {
            type: 'string',
            required: false,
            description: '조회할 회사명 (미지정 시 기본 회사 정보 조회)'
        }
    },
    handler: async (args, context) => {
        const { companyName } = args || {};
        const { username, updateProgress } = context;

        logger.debug(`[CompanyInfoLookup] 조회 시작: companyName=${companyName}`);

        try {
            if (updateProgress) {
                await updateProgress('🏢 회사 정보 조회 중...', 30);
            }

            const db = await getDBPool();

            // 사용자 ID 조회
            const userResult = await db.query(
                'SELECT id FROM users WHERE username = $1',
                [username || 'admin']
            );
            const userId = userResult.rows[0]?.id;

            // 회사 정보 조회
            let sql = `
                SELECT id, company_name, business_number, ceo_name,
                       address, phone, fax, email, website,
                       logo_path, seal_path,
                       bank_name, bank_account, bank_holder,
                       default_validity_days, default_payment_terms, default_notes
                FROM company_profiles
                WHERE 1=1
            `;
            const params = [];
            let paramIndex = 1;

            if (userId) {
                sql += ` AND user_id = $${paramIndex}`;
                params.push(userId);
                paramIndex++;
            }

            if (companyName && companyName.trim()) {
                sql += ` AND company_name ILIKE $${paramIndex}`;
                params.push(`%${companyName}%`);
                paramIndex++;
            }

            sql += ' ORDER BY id LIMIT 1';

            const result = await db.query(sql, params);

            if (result.rows.length === 0) {
                // 기본 회사 정보가 없으면 혜안 기본값 반환
                return {
                    success: true,
                    data: {
                        companyName: '(주)혜안',
                        businessNumber: '000-00-00000',
                        ceoName: '대표이사',
                        address: '서울특별시',
                        phone: '02-0000-0000',
                        email: 'contact@hyean.com',
                        hasLogo: false,
                        hasSeal: false,
                        validityDays: 30,
                        isDefault: true
                    },
                    message: '기본 회사 정보를 반환합니다. 회사 정보를 등록하면 로고와 인감을 사용할 수 있습니다.'
                };
            }

            const company = result.rows[0];

            if (updateProgress) {
                await updateProgress(`✅ ${company.company_name} 정보 조회 완료`, 70);
            }

            return {
                success: true,
                data: {
                    id: company.id,
                    companyName: company.company_name,
                    businessNumber: company.business_number,
                    ceoName: company.ceo_name,
                    address: company.address,
                    phone: company.phone,
                    fax: company.fax,
                    email: company.email,
                    website: company.website,
                    logoPath: company.logo_path,
                    sealPath: company.seal_path,
                    hasLogo: !!company.logo_path,
                    hasSeal: !!company.seal_path,
                    bankName: company.bank_name,
                    bankAccount: company.bank_account,
                    bankHolder: company.bank_holder,
                    validityDays: company.default_validity_days || 30,
                    paymentTerms: company.default_payment_terms,
                    notes: company.default_notes
                },
                message: `${company.company_name} 회사 정보를 조회했습니다.`
            };
        } catch (error) {
            logger.error('[CompanyInfoLookup] 오류:', error);
            return {
                success: false,
                error: error.message,
                message: `회사 정보 조회 실패: ${error.message}`
            };
        }
    }
};

/**
 * 견적서 생성 도구
 * AI가 수집한 정보를 바탕으로 견적서를 생성합니다.
 */
const quotationGeneratorTool = {
    name: 'quotation_generator',
    category: 'quotation',
    description: '견적서를 생성합니다. 고객 정보, 품목, 총 금액을 기반으로 견적서를 작성하고 자동 검산을 수행합니다.',
    version: '1.0.0',
    parameters: {
        title: {
            type: 'string',
            required: true,
            description: '견적서 제목 (예: AI Agent 개발 견적서)'
        },
        clientCompany: {
            type: 'string',
            required: true,
            description: '고객사명'
        },
        clientContact: {
            type: 'string',
            required: false,
            description: '고객 담당자명'
        },
        items: {
            type: 'array',
            required: true,
            description: '견적 품목 배열 [{itemName, quantity, unitPrice, unit, description}]'
        },
        targetTotal: {
            type: 'number',
            required: false,
            description: '목표 총액 (부가세 포함). 지정 시 품목 금액을 조정하여 맞춤'
        },
        taxRate: {
            type: 'number',
            required: false,
            description: '부가세율 (기본값: 10)'
        },
        notes: {
            type: 'string',
            required: false,
            description: '비고/특이사항'
        },
        userPrompt: {
            type: 'string',
            required: false,
            description: '사용자가 요청한 원본 프롬프트'
        }
    },
    handler: async (args, context) => {
        const {
            title,
            clientCompany,
            clientContact,
            items,
            targetTotal,
            taxRate = 10,
            notes,
            userPrompt
        } = args || {};
        const { username, updateProgress } = context;

        // 필수 파라미터 검증
        if (!title || !clientCompany) {
            return {
                success: false,
                error: 'title과 clientCompany는 필수입니다.',
                data: null
            };
        }

        if (!items || !Array.isArray(items) || items.length === 0) {
            return {
                success: false,
                error: 'items 배열이 필요합니다.',
                data: null
            };
        }

        logger.debug(`[QuotationGenerator] 견적서 생성 시작: ${title}`);

        try {
            if (updateProgress) {
                await updateProgress('📝 견적서 생성 중...', 20);
            }

            const db = await getDBPool();

            // 사용자 ID 조회
            const userResult = await db.query(
                'SELECT id FROM users WHERE username = $1',
                [username || 'admin']
            );
            const userId = userResult.rows[0]?.id || 1;

            // 견적 품목 처리 및 금액 계산
            let processedItems = items.map((item, index) => {
                const quantity = parseFloat(item.quantity) || 1;
                const unitPrice = parseFloat(item.unitPrice) || 0;
                const amount = quantity * unitPrice;

                return {
                    itemOrder: index + 1,
                    itemName: item.itemName || item.name || `품목 ${index + 1}`,
                    description: item.description || '',
                    specification: item.specification || '',
                    unit: item.unit || '식',
                    quantity,
                    unitPrice,
                    amount,
                    discountRate: parseFloat(item.discountRate) || 0,
                    discountAmount: 0
                };
            });

            // 소계 계산
            let subtotal = processedItems.reduce((sum, item) => sum + item.amount, 0);

            // 목표 총액이 지정된 경우 금액 조정
            if (targetTotal && targetTotal > 0) {
                if (updateProgress) {
                    await updateProgress('🔢 금액 조정 및 검산 중...', 40);
                }

                // 목표 총액에서 부가세를 역산하여 순 금액 계산
                const targetSubtotal = Math.round(targetTotal / (1 + taxRate / 100));
                const ratio = targetSubtotal / subtotal;

                // 각 품목의 단가를 비율로 조정
                processedItems = processedItems.map(item => {
                    const newUnitPrice = Math.round(item.unitPrice * ratio);
                    const newAmount = item.quantity * newUnitPrice;
                    return {
                        ...item,
                        unitPrice: newUnitPrice,
                        amount: newAmount
                    };
                });

                // 조정 후 소계 재계산
                subtotal = processedItems.reduce((sum, item) => sum + item.amount, 0);

                // 반올림 오차 보정 (마지막 품목에서 조정)
                const actualTotal = Math.round(subtotal * (1 + taxRate / 100));
                if (actualTotal !== targetTotal && processedItems.length > 0) {
                    const diff = Math.round((targetTotal - actualTotal) / (1 + taxRate / 100));
                    const lastItem = processedItems[processedItems.length - 1];
                    lastItem.unitPrice += Math.round(diff / lastItem.quantity);
                    lastItem.amount = lastItem.quantity * lastItem.unitPrice;
                    subtotal = processedItems.reduce((sum, item) => sum + item.amount, 0);
                }
            }

            // 세금 및 총액 계산
            const taxAmount = Math.round(subtotal * taxRate / 100);
            const totalAmount = subtotal + taxAmount;

            if (updateProgress) {
                await updateProgress('💾 견적서 저장 중...', 60);
            }

            // 견적서 번호 생성
            const today = new Date();
            const datePrefix = today.toISOString().slice(0, 10).replace(/-/g, '');
            const countResult = await db.query(
                "SELECT COUNT(*) + 1 as seq FROM quotations WHERE quotation_number LIKE $1",
                [`Q${datePrefix}%`]
            );
            const seq = String(countResult.rows[0].seq).padStart(3, '0');
            const quotationNumber = `Q${datePrefix}-${seq}`;

            // 유효기간 계산 (30일)
            const validityDate = new Date(today);
            validityDate.setDate(validityDate.getDate() + 30);

            // 견적서 저장
            const insertResult = await db.query(
                `INSERT INTO quotations (
                    user_id, quotation_number, client_company, client_name,
                    project_name, supply_amount, vat_amount, total_amount,
                    validity_period, notes, status, generated_by_ai, ai_prompt
                ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, 'draft', true, $11)
                RETURNING id`,
                [
                    username || 'admin', quotationNumber, clientCompany, clientContact || '',
                    title, subtotal, taxAmount, totalAmount,
                    30, notes || '', userPrompt || null
                ]
            );

            const quotationId = insertResult.rows[0].id;

            // 견적 품목 저장
            for (const item of processedItems) {
                await db.query(
                    `INSERT INTO quotation_items (
                        quotation_id, item_order, item_name, specification,
                        unit, quantity, unit_price, amount, notes
                    ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)`,
                    [
                        quotationId, item.itemOrder, item.itemName,
                        item.specification || '', item.unit, item.quantity, item.unitPrice, item.amount,
                        item.description || ''
                    ]
                );
            }

            if (updateProgress) {
                await updateProgress('✅ 견적서 생성 완료', 90);
            }

            // 검산 결과
            const verification = {
                itemsTotal: processedItems.reduce((sum, item) => sum + item.amount, 0),
                subtotal,
                taxRate,
                taxAmount,
                totalAmount,
                targetTotal: targetTotal || null,
                isTargetMet: targetTotal ? totalAmount === targetTotal : true,
                variance: targetTotal ? totalAmount - targetTotal : 0
            };

            return {
                success: true,
                data: {
                    quotationId,
                    quotationNumber,
                    title,
                    clientCompany,
                    clientContact: clientContact || '',
                    quotationDate: today.toISOString().slice(0, 10),
                    validityDate: validityDate.toISOString().slice(0, 10),
                    items: processedItems,
                    subtotal,
                    taxRate,
                    taxAmount,
                    totalAmount,
                    currency: 'KRW',
                    notes: notes || '',
                    status: 'draft',
                    verification
                },
                message: `견적서 ${quotationNumber}이(가) 생성되었습니다. 총액: ${totalAmount.toLocaleString()}원 (VAT 포함)`
            };
        } catch (error) {
            logger.error('[QuotationGenerator] 오류:', error);
            return {
                success: false,
                error: error.message,
                message: `견적서 생성 실패: ${error.message}`
            };
        }
    }
};

/**
 * 견적서 목록 조회 도구
 */
const quotationListTool = {
    name: 'quotation_list',
    category: 'quotation',
    description: '견적서 목록을 조회합니다.',
    version: '1.0.0',
    parameters: {
        status: {
            type: 'string',
            required: false,
            description: '상태 필터 (draft, sent, accepted, rejected)'
        },
        limit: {
            type: 'number',
            required: false,
            description: '최대 조회 개수 (기본값: 20)'
        }
    },
    handler: async (args, context) => {
        const { status, limit = 20 } = args || {};
        const { username, updateProgress } = context;

        try {
            if (updateProgress) {
                await updateProgress('📋 견적서 목록 조회 중...', 30);
            }

            const db = await getDBPool();

            const userResult = await db.query(
                'SELECT id FROM users WHERE username = $1',
                [username || 'admin']
            );
            const userId = userResult.rows[0]?.id;

            let sql = `
                SELECT id, quotation_number, created_at as quotation_date, project_name as title,
                       client_company, total_amount, status
                FROM quotations
                WHERE 1=1
            `;
            const params = [];
            let paramIndex = 1;

            if (userId) {
                sql += ` AND user_id = $${paramIndex}`;
                params.push(userId);
                paramIndex++;
            }

            if (status) {
                sql += ` AND status = $${paramIndex}`;
                params.push(status);
                paramIndex++;
            }

            sql += ` ORDER BY created_at DESC LIMIT $${paramIndex}`;
            params.push(Math.min(limit, 100));

            const result = await db.query(sql, params);

            if (updateProgress) {
                await updateProgress(`✅ ${result.rows.length}건 조회 완료`, 70);
            }

            return {
                success: true,
                data: {
                    quotations: result.rows.map(q => ({
                        id: q.id,
                        quotationNumber: q.quotation_number,
                        date: q.quotation_date,
                        title: q.title,
                        clientCompany: q.client_company,
                        totalAmount: parseFloat(q.total_amount),
                        status: q.status,
                        createdAt: q.created_at
                    })),
                    count: result.rows.length
                },
                message: `${result.rows.length}건의 견적서를 조회했습니다.`
            };
        } catch (error) {
            logger.error('[QuotationList] 오류:', error);
            return {
                success: false,
                error: error.message,
                message: `견적서 목록 조회 실패: ${error.message}`
            };
        }
    }
};

// 도구 내보내기
export default [
    priceCatalogSearchTool,
    pastQuotationSearchTool,
    companyInfoTool,
    quotationGeneratorTool,
    quotationListTool
];
