// 도구 레지스트리 시스템
// 외부에서 도구를 정의하고 등록할 수 있도록 하는 시스템

import { updateToolProgress } from '../utils/progress-helper.js';
// 공통 로그 유틸리티 임포트
import { debugLog, logger } from '../utils/logger.js';
class ToolRegistry {
    constructor() {
        this.tools = new Map();
        this.categories = new Map();
    }

    /**
     * 도구 등록
     * @param {Object} toolDefinition - 도구 정의 객체
     */
    registerTool(toolDefinition) {
        const { name, category, description, parameters, handler, version = '1.0.0' } = toolDefinition;
        
        if (!name || !handler) {
            throw new Error('도구 이름과 핸들러는 필수입니다.');
        }

        if (typeof handler !== 'function') {
            throw new Error('핸들러는 함수여야 합니다.');
        }

        const tool = {
            name,
            category: category || 'general',
            description: description || '',
            parameters: parameters || {},
            handler,
            version,
            registeredAt: new Date().toISOString()
        };

        this.tools.set(name, tool);
        
        // 카테고리별 그룹화
        if (!this.categories.has(tool.category)) {
            this.categories.set(tool.category, []);
        }
        this.categories.get(tool.category).push(name);

        // logger.debug(`[ToolRegistry] 도구 등록됨: ${name} (${tool.category})`);
        return this;
    }

    /**
     * 도구 실행
     * @param {string} toolName - 도구 이름
     * @param {Object} args - 인수
     * @param {string} userId - 사용자 ID
     * @param {Object} job - 작업 객체
     * @param {Object} globalReferenceManager - 전역 참조 관리자
     */
    async executeTool(toolName, args, userId, job = null, globalReferenceManager = null) {
        const tool = this.tools.get(toolName);
        
        if (!tool) {
            throw new Error(`등록되지 않은 도구: ${toolName}`);
        }

        try {
            logger.debug(`[ToolRegistry] 도구 실행: ${toolName}`, {
                toolName,
                hasArgs: !!args,
                argsType: typeof args,
                argsKeys: args ? Object.keys(args) : [],
                userId,
                hasJob: !!job
            });
            
            // 세션에서 히스토리 가져오기
            let history = [];
            let session = null;
            if (job?.sessionId || job?.taskId) {
                try {
                    const { sharedContext } = await import('../aiFeatures.js');
                    session = await sharedContext.getSession(job.sessionId || job.taskId);
                    if (session) {
                        history = session.getHistory() || [];
                        logger.debug(`[ToolRegistry] 히스토리 로드됨: ${history.length}개 메시지`);
                    }
                } catch (error) {
                    logger.debug(`[ToolRegistry] 히스토리 로드 실패: ${error.message}`);
                }
            }
            
            // 도구 실행 컨텍스트 준비 - 히스토리와 진행 상황 업데이트 함수 포함
            const context = {
                userId,
                username: job?.username || session?.getUsername() || 'admin', // username 추가
                job,
                toolName,
                registeredAt: tool.registeredAt,
                version: tool.version,
                taskId: job?.taskId,
                sessionId: job?.sessionId,
                history: history, // 히스토리 직접 전달
                session: session, // 세션 객체도 전달 (필요시 사용)
                // 진행 상황 업데이트 헬퍼 함수 추가 (공통 헬퍼 사용)
                updateProgress: async (message, percent = null) => {
                    await updateToolProgress(job, message, percent);
                }
            };

            // 핸들러 실행
            const result = await tool.handler(args, context);

            // create_report 도구의 downloadInfo를 globalReferenceManager에 추가
            if (toolName === 'create_report' && result.success && result.data?.downloadInfo && globalReferenceManager) {
                logger.debug(`[ToolRegistry] 📄 create_report 도구에서 downloadInfo 처리:`, {
                    count: result.data.downloadInfo.length,
                    files: result.data.downloadInfo.map(info => info.filename || info.title).join(', ')
                });

                result.data.downloadInfo.forEach(info => {
                    if (info.source === 'generated' && info.filename) {
                        globalReferenceManager.addGeneratedDocument(info);
                        logger.debug(`[ToolRegistry] 📄 생성된 문서 추가: ${info.filename}`);
                    }
                });
            }

            logger.debug(`[ToolRegistry] 도구 실행 완료: ${toolName}`);
            return result;
        } catch (error) {
            console.error(`[ToolRegistry] 도구 실행 오류 ${toolName}:`, error);
            throw error;
        }
    }

    /**
     * 등록된 도구 목록 반환
     */
    getTools() {
        return Array.from(this.tools.keys());
    }

    /**
     * 도구 정보 반환
     */
    getToolInfo(toolName) {
        const tool = this.tools.get(toolName);
        if (!tool) {
            return null;
        }

        return {
            name: tool.name,
            category: tool.category,
            description: tool.description,
            parameters: tool.parameters,
            version: tool.version,
            registeredAt: tool.registeredAt
        };
    }


    /**
     * 도구 객체 반환 (핸들러 포함)
     * @param {string} toolName - 도구 이름
     * @returns {Object|null} 도구 객체 (handler 포함)
     */
    getTool(toolName) {
        return this.tools.get(toolName) || null;
    }

    /**
     * 카테고리별 도구 목록
     */
    getToolsByCategory() {
        const result = {};
        for (const [category, tools] of this.categories.entries()) {
            result[category] = tools.map(toolName => this.getToolInfo(toolName));
        }
        return result;
    }

    /**
     * 모든 도구 정보 반환 (CLI에서 사용)
     */
    getAllTools() {
        const result = [];
        for (const [name, tool] of this.tools.entries()) {
            result.push({
                name: tool.name,
                category: tool.category,
                description: tool.description,
                parameters: tool.parameters,
                version: tool.version,
                registeredAt: tool.registeredAt
            });
        }
        return result;
    }

    /**
     * 모든 도구 제거 (리로드용)
     */
    clearTools() {
        this.tools.clear();
        this.categories.clear();
        logger.debug('[ToolRegistry] 모든 도구가 제거되었습니다.');
    }

    /**
     * 도구 실행 통계 반환 (향후 구현용)
     */
    getStats() {
        // 향후 도구 실행 통계를 추적할 때 구현
        return {};
    }

    /**
     * 도구 제거
     */
    unregisterTool(toolName) {
        const tool = this.tools.get(toolName);
        if (!tool) {
            return false;
        }

        // 카테고리에서 제거
        const category = this.categories.get(tool.category);
        if (category) {
            const index = category.indexOf(toolName);
            if (index > -1) {
                category.splice(index, 1);
            }
            
            // 카테고리가 비었으면 제거
            if (category.length === 0) {
                this.categories.delete(tool.category);
            }
        }

        this.tools.delete(toolName);
        logger.debug(`[ToolRegistry] 도구 제거됨: ${toolName}`);
        return true;
    }

    /**
     * 외부 파일에서 도구 로드
     */
    async loadToolsFromFile(filePath) {
        try {
            const toolsModule = await import(filePath);
            const tools = toolsModule.default || toolsModule.tools;
            
            if (!Array.isArray(tools)) {
                throw new Error('도구 파일은 배열을 export해야 합니다.');
            }

            let loadedCount = 0;
            for (const tool of tools) {
                try {
                    this.registerTool(tool);
                    loadedCount++;
                } catch (error) {
                    // console.error(`도구 로드 실패 ${tool.name}:`, error.message);
                }
            }

            // logger.debug(`[ToolRegistry] ${filePath}에서 ${loadedCount}개 도구 로드됨`);
            return loadedCount;
        } catch (error) {
            // console.error(`[ToolRegistry] 도구 파일 로드 실패 ${filePath}:`, error);
            throw error;
        }
    }

    /**
     * 디렉토리에서 모든 도구 파일 로드
     */
    async loadToolsFromDirectory(dirPath) {
        const fs = await import('fs');
        const path = await import('path');
        
        try {
            const files = fs.readdirSync(dirPath);
            let totalLoaded = 0;
            
            for (const file of files) {
                if (file.endsWith('.js') && !file.startsWith('.')) {
                    const filePath = path.join(dirPath, file);
                    try {
                        const loaded = await this.loadToolsFromFile(filePath);
                        totalLoaded += loaded;
                    } catch (error) {
                        console.error(`도구 파일 로드 실패 ${file}:`, error.message);
                    }
                }
            }
            
            // logger.debug(`[ToolRegistry] ${dirPath}에서 총 ${totalLoaded}개 도구 로드됨`);
            return totalLoaded;
        } catch (error) {
            // console.error(`[ToolRegistry] 디렉토리 로드 실패 ${dirPath}:`, error);
            throw error;
        }
    }

    /**
     * 시스템 프롬프트용 동적 도구 정보 생성
     */
    generateSystemPromptToolInfo() {
        if (this.tools.size === 0) {
            return '현재 사용 가능한 도구가 없습니다.';
        }

        let toolInfo = '\n**🔧 현재 사용 가능한 도구 목록:**\n\n';
        
        // 단순하게 모든 도구를 순서대로 나열
        for (const [name, tool] of this.tools.entries()) {
            toolInfo += `- **${tool.name}**: ${tool.description}\n`;
            
            // 필수 매개변수만 간단하게 표시
            if (tool.parameters && Object.keys(tool.parameters).length > 0) {
                const requiredParams = Object.entries(tool.parameters)
                    .filter(([key, param]) => param.required)
                    .map(([key, param]) => key);
                
                if (requiredParams.length > 0) {
                    toolInfo += `  필수 매개변수: ${requiredParams.join(', ')}\n`;
                }
            }
            toolInfo += '\n';
        }
        
        // 간단한 사용 가이드
        toolInfo += '**사용법**: 위 도구들을 필요에 따라 호출하세요.';
        
        return toolInfo;
    }

    /**
     * Function Calling용 도구 스키마 생성 (OpenAI 형식)
     */
    generateFunctionCallSchemas() {
        const schemas = [];
        
        for (const [name, tool] of this.tools.entries()) {
            let description = tool.description;
            
            // 내장 도구의 파라미터를 JSON Schema 형식으로 변환
            const properties = {};
            const required = [];
            
            if (tool.parameters && typeof tool.parameters === 'object') {
                for (const [paramName, paramDef] of Object.entries(tool.parameters)) {
                    // 파라미터 정의를 JSON Schema 형식으로 변환
                    const paramSchema = {
                        type: paramDef.type || 'string',
                        description: paramDef.description || `${paramName} 파라미터`
                    };
                    
                    // 배열 타입인 경우 items 추가
                    if (paramDef.type === 'array') {
                        paramSchema.items = { type: 'string' }; // 기본값으로 string 배열
                        if (paramDef.items) {
                            paramSchema.items = paramDef.items;
                        }
                    }
                    
                    // enum이 있으면 추가
                    if (paramDef.enum) {
                        paramSchema.enum = paramDef.enum;
                    }
                    
                    // default 값이 있으면 추가 (OpenAI API에서는 default 지원하지 않으므로 제거)
                    // if (paramDef.default !== undefined) {
                    //     paramSchema.default = paramDef.default;
                    // }
                    
                    properties[paramName] = paramSchema;
                    
                    // 필수 파라미터인지 확인
                    if (paramDef.required === true) {
                        required.push(paramName);
                    }
                }
            }
            
            const schema = {
                type: "function",
                function: {
                    name: tool.name,
                    description: description,
                    parameters: {
                        type: "object",
                        properties: properties,
                        // required 배열이 비어있지 않을 때만 포함
                        ...(required.length > 0 ? { required: required } : {})
                    }
                }
            };
            schemas.push(schema);
        }
        
        return schemas;
    }

    /**
     * Anthropic용 도구 스키마 생성
     */
    generateAnthropicSchemas() {
        const schemas = [];
        
        for (const [name, tool] of this.tools.entries()) {
            let description = tool.description;
            
            // 내장 도구의 파라미터를 JSON Schema 형식으로 변환
            const properties = {};
            const required = [];
            
            if (tool.parameters && typeof tool.parameters === 'object') {
                for (const [paramName, paramDef] of Object.entries(tool.parameters)) {
                    // 파라미터 정의를 JSON Schema 형식으로 변환
                    const paramSchema = {
                        type: paramDef.type || 'string',
                        description: paramDef.description || `${paramName} 파라미터`
                    };
                    
                    // 배열 타입인 경우 items 추가
                    if (paramDef.type === 'array') {
                        paramSchema.items = { type: 'string' }; // 기본값으로 string 배열
                        if (paramDef.items) {
                            paramSchema.items = paramDef.items;
                        }
                    }
                    
                    // enum이 있으면 추가
                    if (paramDef.enum) {
                        paramSchema.enum = paramDef.enum;
                    }
                    
                    properties[paramName] = paramSchema;
                    
                    // 필수 파라미터인지 확인
                    if (paramDef.required === true) {
                        required.push(paramName);
                    }
                }
            }
            
            const schema = {
                name: tool.name,
                description: description,
                input_schema: {
                    type: "object",
                    properties: properties,
                    // required 배열이 비어있지 않을 때만 포함
                    ...(required.length > 0 ? { required: required } : {})
                }
            };
            schemas.push(schema);
        }
        
        return schemas;
    }
}

// 글로벌 도구 레지스트리 인스턴스
const toolRegistry = new ToolRegistry();

export { ToolRegistry, toolRegistry };
export default toolRegistry; 