# 청크 편집 기능 프론트엔드 통합 가이드

## 작업 완료 현황

### ✅ 완료된 작업
1. 청크 편집 모달 컴포넌트 생성 (`/src/components/rag/ChunkEditModal.tsx`)
2. 편집 버튼 UI 추가 (RAG admin page, line 1557-1572)
3. 모달 렌더링 추가 (RAG admin page, line 6310-6318)
4. 백엔드 API 엔드포인트 구현 (`chunk_edit_endpoints.py`)

### 🔄 남은 작업
1. `handleChunkSave` 함수 구현
2. `loadChunks` 함수 확인 또는 수정 (청크 목록 새로고침)
3. 백엔드 API를 `airun-rag.py`에 통합

## 1단계: handleChunkSave 함수 추가

### 위치
`/home/hamonikr/work/airun/workspaces/web/src/app/admin/rag/page.tsx`
**추가 위치**: 약 5180번째 줄 근처 (다른 `handle*` 함수들 다음)

### 추가할 코드

```typescript
// 청크 저장 핸들러
const handleChunkSave = async (
  chunkId: string,
  content: string,
  method: "direct" | "chandra"
) => {
  try {
    // 로딩 상태 설정 (선택사항)
    console.log(`[청크 저장] 시작: method=${method}, chunkId=${chunkId}`);

    // API 엔드포인트 결정
    const endpoint = method === "direct"
      ? "http://localhost:5600/chunks/update"
      : "http://localhost:5600/chunks/reanalyze";

    // 요청 바디 구성
    const requestBody = method === "direct"
      ? {
          chunk_id: chunkId,
          content: content,
          user_id: selectedUser || user?.username || "default"
        }
      : {
          chunk_id: chunkId,
          page_number: chunks[currentChunkIndex]?.metadata?.page_number || 1,
          user_id: selectedUser || user?.username || "default"
        };

    // API 호출
    const response = await fetch(endpoint, {
      method: method === "direct" ? "PUT" : "POST",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify(requestBody),
    });

    // 응답 처리
    if (!response.ok) {
      const errorData = await response.json();
      throw new Error(errorData.detail || "청크 저장에 실패했습니다.");
    }

    const result = await response.json();
    console.log("[청크 저장] 성공:", result);

    // 청크 목록 새로고침
    await loadChunks();

    // 성공 메시지 표시 (toast 사용 권장)
    alert(result.message || "청크가 성공적으로 저장되었습니다.");

  } catch (error: any) {
    console.error("[청크 저장] 오류:", error);
    // 사용자에게 오류 메시지 표시
    alert(`청크 저장 중 오류가 발생했습니다: ${error.message}`);
    throw error;
  }
};
```

## 2단계: loadChunks 함수 확인

### 확인 사항
`loadChunks` 함수가 이미 존재하는지 확인하세요. 이 함수는 청크 목록을 다시 불러오는 역할을 합니다.

### 예상 위치
RAG admin page에서 `const loadChunks` 또는 `function loadChunks`를 검색하세요.

### loadChunks가 없는 경우
청크 데이터를 다시 불러오는 기존 함수를 사용하거나, 다음 코드를 추가하세요:

```typescript
// 청크 목록 새로고침 함수
const loadChunks = async () => {
  try {
    if (!selectedDocument) {
      console.warn("[loadChunks] 선택된 문서가 없습니다.");
      return;
    }

    const response = await fetch(
      `http://localhost:5600/chunks?filename=${encodeURIComponent(selectedDocument.name)}&user_id=${selectedUser || user?.username || "default"}`
    );

    if (!response.ok) {
      throw new Error("청크 데이터를 불러오지 못했습니다.");
    }

    const data = await response.json();

    if (data.success && data.chunks) {
      setChunks(data.chunks);
      console.log(`[loadChunks] ${data.chunks.length}개의 청크를 불러왔습니다.`);
    } else {
      setChunks([]);
    }
  } catch (error) {
    console.error("[loadChunks] 오류:", error);
    setChunks([]);
  }
};
```

## 3단계: 백엔드 API 통합

### airun-rag.py에 엔드포인트 추가

1. **Request 모델 추가** (약 9260번째 줄 근처):

```python
class ChunkUpdateRequest(BaseModel):
    """청크 직접 수정 요청 모델"""
    chunk_id: str
    content: str
    user_id: Optional[str] = None

class ChunkReanalyzeRequest(BaseModel):
    """Chandra 페이지 재분석 요청 모델"""
    chunk_id: str
    page_number: int
    user_id: Optional[str] = None
```

2. **API 엔드포인트 추가** (약 14500번째 줄 근처, `/chunks` 엔드포인트 다음):

`/home/hamonikr/work/airun/plugins/rag/chunk_edit_endpoints.py` 파일의 두 엔드포인트를 복사하여 붙여넣으세요.

### RAG 서비스 재시작

```bash
echo "exitem08" | sudo -S systemctl restart airun-rag
```

## 4단계: 테스트

### 프론트엔드 테스트 (웹 브라우저)
1. `http://localhost:5700/admin/rag` 접속
2. 문서 선택 및 청크 미리보기
3. 편집 버튼 클릭
4. 직접 편집 모드에서 내용 수정 후 저장
5. Chandra 재분석 모드에서 저장 (페이지 정보 필요)
6. 청크 목록이 자동으로 새로고침되는지 확인

### 백엔드 API 테스트 (curl)

#### 직접 수정:
```bash
curl -X PUT http://localhost:5600/chunks/update \
  -H "Content-Type: application/json" \
  -d '{
    "chunk_id": "YOUR_CHUNK_ID",
    "content": "수정된 내용",
    "user_id": "admin"
  }'
```

#### Chandra 재분석:
```bash
curl -X POST http://localhost:5600/chunks/reanalyze \
  -H "Content-Type: application/json" \
  -d '{
    "chunk_id": "YOUR_CHUNK_ID",
    "page_number": 1,
    "user_id": "admin"
  }'
```

## 개선 사항 (선택사항)

### Toast Notification 사용
`alert()` 대신 더 나은 사용자 경험을 위해 toast 라이브러리 사용을 권장합니다:

```bash
npm install react-toastify
```

```typescript
import { toast } from 'react-toastify';

// alert 대신:
toast.success("청크가 성공적으로 저장되었습니다!");
toast.error(`청크 저장 실패: ${error.message}`);
```

### 로딩 스피너
청크 저장 중 로딩 상태를 표시하세요:

```typescript
const [isSavingChunk, setIsSavingChunk] = useState(false);

const handleChunkSave = async (...) => {
  setIsSavingChunk(true);
  try {
    // ... 저장 로직 ...
  } finally {
    setIsSavingChunk(false);
  }
};
```

## 트러블슈팅

### 1. "청크를 찾을 수 없습니다" 오류
- 청크 ID가 정확한지 확인
- `chunks[currentChunkIndex].metadata.db_id` 사용 확인

### 2. "원본 PDF 파일을 찾을 수 없습니다" 오류 (Chandra 모드)
- RAG_BASE_DIR 확인
- PDF 파일 위치 확인

### 3. Chandra 재분석이 작동하지 않음
- Chandra vLLM 서버 상태 확인
- 환경 변수 확인: `VLLM_API_BASE`, `VLLM_MODEL_NAME`
- 로그 확인: `/var/log/airun/airun-rag.log`

### 4. CORS 오류
프론트엔드와 백엔드가 다른 포트에서 실행 중인 경우, RAG 서버에 CORS 설정 확인:

```python
# airun-rag.py에 이미 CORS 설정이 있는지 확인
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)
```

## 참고 문서
- 백엔드 엔드포인트 구현: `/home/hamonikr/work/airun/plugins/rag/chunk_edit_endpoints.py`
- 통합 가이드: `/home/hamonikr/work/airun/plugins/rag/INTEGRATION_GUIDE.md`
- 모달 컴포넌트: `/home/hamonikr/work/airun/workspaces/web/src/components/rag/ChunkEditModal.tsx`
