-- 사용자가 이미 있는지 확인하고 없으면 생성
DO $$
BEGIN
    IF NOT EXISTS (SELECT FROM pg_catalog.pg_roles WHERE rolname = 'ivs') THEN
        CREATE USER ivs WITH PASSWORD "Exitem08**";
    END IF;
END
$$;

-- airun 데이터베이스는 이미 컨테이너에 의해 생성되므로 건너뜁니다.

-- 한국어 지원을 위한 설정
SET client_encoding = 'UTF8';

-- 스키마 생성
CREATE SCHEMA IF NOT EXISTS airun;

-- 예제 테이블: 사용자 정보
CREATE TABLE IF NOT EXISTS airun.users (
    id SERIAL PRIMARY KEY,
    username VARCHAR(100) NOT NULL UNIQUE,
    email VARCHAR(255) NOT NULL UNIQUE,
    full_name VARCHAR(255),
    created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
);

-- 예제 테이블: API 호출 기록
CREATE TABLE IF NOT EXISTS airun.api_calls (
    id SERIAL PRIMARY KEY,
    user_id INTEGER REFERENCES airun.users(id),
    api_endpoint VARCHAR(255) NOT NULL,
    request_data JSONB,
    response_data JSONB,
    status_code INTEGER,
    duration_ms INTEGER,
    timestamp TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
);

-- 예제 테이블: 모델 정보
CREATE TABLE IF NOT EXISTS airun.models (
    id SERIAL PRIMARY KEY,
    model_name VARCHAR(100) NOT NULL UNIQUE,
    provider VARCHAR(100) NOT NULL,
    description TEXT,
    parameters JSONB,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
);

-- 예제 테이블: 작업 큐
CREATE TABLE IF NOT EXISTS airun.job_queue (
    id SERIAL PRIMARY KEY,
    job_type VARCHAR(50) NOT NULL,
    status VARCHAR(20) NOT NULL DEFAULT 'pending',
    priority INTEGER DEFAULT 0,
    payload JSONB NOT NULL,
    result JSONB,
    error_message TEXT,
    created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
    started_at TIMESTAMP WITH TIME ZONE,
    completed_at TIMESTAMP WITH TIME ZONE
);

-- 예제 테이블: 한국어 콘텐츠
CREATE TABLE IF NOT EXISTS airun.korean_contents (
    id SERIAL PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    content TEXT NOT NULL,
    author VARCHAR(100),
    tags TEXT[],
    created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
);

-- 샘플 데이터 삽입
INSERT INTO airun.users (username, email, full_name)
VALUES 
    ('admin', 'admin@example.com', '관리자'),
    ('user1', 'user1@example.com', '사용자1'),
    ('user2', 'user2@example.com', '사용자2'),
    ('김철수', 'kim@example.com', '김철수'),
    ('이영희', 'lee@example.com', '이영희'),
    ('박지민', 'park@example.com', '박지민')
ON CONFLICT (username) DO NOTHING;

-- 샘플 모델 데이터 삽입
INSERT INTO airun.models (model_name, provider, description, parameters)
VALUES 
    ('gpt-4o', 'OpenAI', 'OpenAI GPT-4o 모델', '{"max_tokens": 8192, "temperature": 0.7}'),
    ('claude-3-5-sonnet', 'Anthropic', 'Anthropic Claude 3.5 Sonnet 모델', '{"max_tokens": 4096, "temperature": 0.7}'),
    ('gemini-pro', 'Google', 'Google Gemini Pro 모델', '{"max_tokens": 2048, "temperature": 0.7}'),
    ('clova-x', 'Naver', '네이버 클로바 X 모델', '{"max_tokens": 4096, "temperature": 0.8}'),
    ('kullm', 'KakaoBrain', '카카오브레인 한국어 LLM 모델', '{"max_tokens": 2048, "temperature": 0.7}')
ON CONFLICT (model_name) DO NOTHING;

-- 한국어 콘텐츠 샘플 데이터
INSERT INTO airun.korean_contents (title, content, author, tags)
VALUES
    ('인공지능의 미래', '인공지능 기술은 계속해서 발전하고 있으며, 우리의 일상생활과 업무 방식을 크게 변화시킬 것입니다.', '김철수', ARRAY['AI', '미래기술', '트렌드']),
    ('한국어 자연어처리의 발전', '한국어 자연어처리 기술은 최근 몇 년간 큰 발전을 이루었습니다. 특히 트랜스포머 기반 모델의 등장으로 번역, 요약, 질의응답 등의 작업에서 높은 성능을 보이고 있습니다.', '이영희', ARRAY['NLP', '한국어', '인공지능']),
    ('생성형 AI와 저작권 문제', '생성형 AI의 발전으로 저작권 문제가 새롭게 대두되고 있습니다. AI가 생성한 콘텐츠의 소유권과 책임 소재에 대한 논의가 필요한 시점입니다.', '박지민', ARRAY['생성형AI', '저작권', '법률']),
    ('한글과 AI', '한글은 과학적으로 설계된 문자 체계로, AI 학습에 있어 독특한 특성을 가지고 있습니다. 자모 분리와 조합의 특성은 자연어처리에 있어 도전과제가 되기도 합니다.', '이영희', ARRAY['한글', 'AI', '언어학']),
    ('국내 AI 기업 동향', '국내 AI 기업들은 글로벌 시장에서도 경쟁력을 갖추기 위해 다양한 시도를 하고 있습니다. 특히 한국어 특화 모델과 솔루션 개발에 집중하고 있습니다.', '김철수', ARRAY['스타트업', 'AI산업', '한국'])
ON CONFLICT DO NOTHING;

-- API 호출 샘플 데이터
INSERT INTO airun.api_calls (user_id, api_endpoint, request_data, response_data, status_code, duration_ms)
VALUES
    (1, '/api/chat', '{"model": "gpt-4o", "messages": [{"role": "user", "content": "안녕하세요"}]}', '{"id": "chat-123", "response": "안녕하세요! 무엇을 도와드릴까요?"}', 200, 450),
    (4, '/api/completions', '{"model": "clova-x", "prompt": "인공지능의 발전 방향은"}', '{"id": "comp-456", "completion": "인공지능의 발전 방향은 크게 세 가지로 나눌 수 있습니다..."}', 200, 320),
    (5, '/api/embeddings', '{"model": "kullm", "text": "한국어 임베딩 테스트"}', '{"embedding": [0.023, -0.045, 0.102, ...]}', 200, 150)
ON CONFLICT DO NOTHING;

-- 작업 큐 샘플 데이터
INSERT INTO airun.job_queue (job_type, status, priority, payload, result, started_at, completed_at)
VALUES
    ('문서요약', 'completed', 2, '{"document_id": 123, "max_length": 200}', '{"summary": "이 문서는 인공지능의 주요 발전 방향에 대해 설명하고 있습니다..."}', NOW() - INTERVAL '2 hours', NOW() - INTERVAL '1 hour 50 minutes'),
    ('번역작업', 'pending', 1, '{"text": "인공지능은 우리의 미래를 변화시킬 것입니다.", "source": "ko", "target": "en"}', NULL, NULL, NULL),
    ('이미지생성', 'processing', 3, '{"prompt": "한옥과 현대 건축이 조화된 미래 도시의 모습", "resolution": "1024x1024"}', NULL, NOW() - INTERVAL '5 minutes', NULL)
ON CONFLICT DO NOTHING;

-- 권한 설정
GRANT USAGE ON SCHEMA airun TO ivs;
GRANT ALL PRIVILEGES ON ALL TABLES IN SCHEMA airun TO ivs;
GRANT ALL PRIVILEGES ON ALL SEQUENCES IN SCHEMA airun TO ivs; 