
"use client";

import React, { useState, useEffect } from 'react';
import { useAuth } from '@/hooks/useAuth';
import {
  FolderOpen,
  Settings,
  BarChart3,
  TrendingUp,
  Database,
  Brain,
  Gamepad2
} from 'lucide-react';
import {
  Chart as ChartJS,
  CategoryScale,
  LinearScale,
  BarElement,
  RadialLinearScale,
  PointElement,
  LineElement,
  Filler,
  Title,
  Tooltip,
  Legend,
} from 'chart.js';
import { Bar, Radar } from 'react-chartjs-2';

ChartJS.register(
  CategoryScale,
  LinearScale,
  BarElement,
  RadialLinearScale,
  PointElement,
  LineElement,
  Filler,
  Title,
  Tooltip,
  Legend
);

// Dynamic imports for LLM components to avoid SSR issues
import dynamic from 'next/dynamic';

const LLMDatasetTab = dynamic(() => import('@/components/admin/LLMDatasetTab'), {
  ssr: false,
  loading: () => <div className="text-center py-8">로딩 중...</div>
});

const LLMTrainingTab = dynamic(() => import('@/components/admin/LLMTrainingTab'), {
  ssr: false,
  loading: () => <div className="text-center py-8">로딩 중...</div>
});

const LLMModelsTab = dynamic(() => import('@/components/admin/LLMModelsTab'), {
  ssr: false,
  loading: () => <div className="text-center py-8">로딩 중...</div>
});

const LLMPlaygroundTab = dynamic(() => import('@/components/admin/LLMPlaygroundTab'), {
  ssr: false,
  loading: () => <div className="text-center py-8">로딩 중...</div>
});

interface ModelInfo {
  id: string;
  name: string;
  version: string;
  type: 'base' | 'finetuned';
  size: string;
  description: string;
  createdAt: string;
  performance: {
    precision: number;
    recall: number;
    f1Score: number;
  };
  isActive: boolean;
  trainingData?: {
    documentCount: number;
    pairCount: number;
    domains: string[];
  };
  evaluated?: boolean;          // 평가 여부
  evaluationDate?: number;      // 평가 날짜 (timestamp)
  dataset_id?: string;          // 데이터셋 ID
  evaluationResults?: any;      // 평가 결과 전체 데이터
  evaluation_results?: any;     // 평가 결과 데이터 (API 응답용)
  test_samples?: number;       // 테스트 샘플 수
}

interface TrainingDataset {
  id: string;
  name: string;
  description: string;
  documentCount: number;
  pairCount: number;
  domains: string[];
  createdAt: string;
  status: 'ready' | 'processing' | 'error';
  type?: string;
  size?: number; // API에서 오는 실제 크기
  system_managed?: boolean; // 시스템 관리 여부
  deletable?: boolean; // 삭제 가능 여부
}

interface DatasetCreationMethod {
  id: string;
  title: string;
  description: string;
  icon: string;
  recommended?: boolean;
  difficulty: 'easy' | 'medium' | 'hard';
}

interface FinetuneTabs {
  overview: boolean;
  datasets: boolean;
  training: boolean;
  evaluation: boolean;
  llm: boolean;
  llmDatasets: boolean;
  llmTraining: boolean;
  llmModels: boolean;
  llmPlayground: boolean;
}

interface SystemResources {
  total_memory_gb: number;
  available_memory_gb: number;
  cpu_cores: number;
  gpu_available: boolean;
  gpu_memory_gb: number;
  gpu_available_memory_gb: number;
  recommendations: {
    gpu_mode: {
      batch_size: number;
      num_workers: number;
      use_amp: boolean;
    };
    cpu_mode: {
      batch_size: number;
      num_workers: number;
    };
  };
}

interface NotificationModal {
  isOpen: boolean;
  type: 'success' | 'error' | 'warning' | 'info';
  title: string;
  message: string;
  onConfirm?: () => void;
  confirmText?: string;
  showCancel?: boolean;
}

// 데이터셋 생성 프로그레스 바 컴포넌트
const DatasetCreationProgressBar: React.FC<{
  progress: {
    isCreating: boolean;
    progress: number;
    stage: string;
    message: string;
    method?: string;
  };
}> = ({ progress }) => {
  if (!progress.isCreating) return null;

  const getMethodIcon = () => {
    switch (progress.method) {
            case 'document-llm': return '🤖';
      case 'file-upload': return '📁';
      default: return '🔄';
    }
  };

  const getMethodTitle = () => {
    switch (progress.method) {
            case 'document-llm': return 'AI 질문-답변 생성';
      case 'file-upload': return '파일 업로드';
      default: return '데이터셋 생성';
    }
  };

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
      <div className="bg-white dark:bg-gray-800 rounded-lg p-6 max-w-md w-full mx-4 shadow-xl">
        <div className="text-center">
          <div className="text-4xl mb-4 animate-pulse">
            {getMethodIcon()}
          </div>
          <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
            {getMethodTitle()}
          </h3>
          <p className="text-sm text-gray-600 dark:text-gray-400 mb-4">
            {progress.stage}
          </p>
          
          {/* 프로그레스 바 */}
          <div className="w-full bg-gray-200 dark:bg-gray-700 rounded-full h-2.5 mb-4">
            <div
              className="bg-blue-600 h-2.5 rounded-full transition-all duration-300 ease-out"
              style={{ width: `${progress.progress}%` }}
            />
          </div>
          
          <p className="text-xs text-gray-500 dark:text-gray-400 mb-4">
            {progress.message}
          </p>
          
          <div className="flex justify-center">
            <div className="animate-spin rounded-full h-6 w-6 border-b-2 border-blue-600" />
          </div>
        </div>
      </div>
    </div>
  );
};

// 공통 모달 컴포넌트
const NotificationModalComponent: React.FC<{
  modal: NotificationModal;
  onClose: () => void;
}> = ({ modal, onClose }) => {
  if (!modal.isOpen) return null;

  const getModalIcon = () => {
    switch (modal.type) {
      case 'success': return '✅';
      case 'error': return '❌';
      case 'warning': return '⚠️';
      case 'info': return 'ℹ️';
      default: return '';
    }
  };

  const getModalColors = () => {
    switch (modal.type) {
      case 'success': return 'text-green-600 dark:text-green-400';
      case 'error': return 'text-red-600 dark:text-red-400';
      case 'warning': return 'text-yellow-600 dark:text-yellow-400';
      case 'info': return 'text-blue-600 dark:text-blue-400';
      default: return 'text-gray-600 dark:text-gray-400';
    }
  };

  const handleConfirm = () => {
    if (modal.onConfirm) {
      modal.onConfirm();
    }
    onClose();
  };

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
      <div className="bg-white dark:bg-gray-800 rounded-xl shadow-xl max-w-md w-full mx-4 transform transition-all">
        <div className="p-6">
          <div className="flex items-center mb-4">
            <div className="text-2xl mr-3">
              {getModalIcon()}
            </div>
            <h3 className={`text-lg font-semibold ${getModalColors()}`}>
              {modal.title}
            </h3>
          </div>
          
          <div className="mb-6">
            <p className="text-gray-700 dark:text-gray-300 whitespace-pre-line">
              {modal.message}
            </p>
          </div>
          
          <div className="flex justify-end gap-3">
            {modal.showCancel && (
              <button
                onClick={onClose}
                className="px-4 py-2 text-gray-600 hover:text-gray-800 dark:text-gray-400 dark:hover:text-gray-200 transition-colors"
              >
                취소
              </button>
            )}
            <button
              onClick={handleConfirm}
              className={`px-4 py-2 rounded-lg font-medium transition-colors ${
                modal.type === 'error' 
                  ? 'bg-red-600 text-white hover:bg-red-700 dark:bg-red-600 dark:hover:bg-red-700'
                  : modal.type === 'warning'
                  ? 'bg-yellow-600 text-white hover:bg-yellow-700 dark:bg-yellow-600 dark:hover:bg-yellow-700'
                  : modal.type === 'success'
                  ? 'bg-green-600 text-white hover:bg-green-700 dark:bg-green-600 dark:hover:bg-green-700'
                  : 'bg-blue-600 text-white hover:bg-blue-700 dark:bg-blue-600 dark:hover:bg-blue-700'
              }`}
            >
              {modal.confirmText || '확인'}
            </button>
          </div>
        </div>
      </div>
    </div>
  );
};

// 관리자 사이드바 컴포넌트
const AdminSidebar: React.FC<{
  activeTab: keyof FinetuneTabs;
  setActiveTab: (tab: keyof FinetuneTabs) => void;
  onRefreshModels?: () => void;
}> = ({ activeTab, setActiveTab, onRefreshModels }) => {
  // 임베딩 모델 메뉴 그룹
  const embeddingMenuItems = [
    { key: 'datasets', label: '데이터셋', icon: FolderOpen },
    { key: 'training', label: '모델 학습', icon: Settings },
    { key: 'overview', label: '모델 정보', icon: BarChart3 },
    { key: 'evaluation', label: '성능 평가', icon: TrendingUp }
  ];

  // LLM 파인튜닝 메뉴 그룹 (개발 중 - 주석처리)
  // const llmMenuItems = [
  //   { key: 'llmDatasets', label: '데이터셋', icon: Database },
  //   { key: 'llmTraining', label: '모델 학습', icon: Settings },
  //   { key: 'llmModels', label: '모델 관리', icon: Brain },
  //   { key: 'llmPlayground', label: '플레이그라운드', icon: Gamepad2 }
  // ];

  const MenuGroup: React.FC<{
    title: string;
    items: Array<{ key: string; label: string; icon: React.ComponentType<any> }>;
    titleIcon?: string;
  }> = ({ title, items, titleIcon }) => (
    <div className="mb-8">
      {/* 그룹 제목 */}
      <div className="px-3 mb-4 pb-2 border-b border-gray-200 dark:border-gray-700">
        <div className="flex items-center space-x-3">
          {titleIcon && <span className="text-2xl">{titleIcon}</span>}
          <h3 className="font-bold text-gray-800 dark:text-gray-200">
            {title}
          </h3>
        </div>
      </div>
      {/* 그룹 메뉴 항목들 */}
      <div className="space-y-1 px-1">
        {items.map(tab => (
          <button
            key={tab.key}
            onClick={() => {
              setActiveTab(tab.key as keyof FinetuneTabs);
              // 모델 정보 탭 클릭 시 자동 새로고침
              if (tab.key === 'overview' && onRefreshModels) {
                setTimeout(() => {
                  onRefreshModels();
                }, 100);
              }
            }}
            className={`w-full flex items-center px-3 py-3 text-base font-medium rounded-lg transition-colors text-left ${
              activeTab === tab.key
                ? 'bg-[var(--primary-100)] dark:bg-[var(--primary-900)] text-[var(--primary-800)] dark:text-[var(--primary-200)] border border-[var(--primary-200)] dark:border-[var(--primary-700)]'
                : 'text-[var(--text-secondary)] hover:text-[var(--text-primary)] hover:bg-[var(--card-hover)] dark:hover:bg-[var(--neutral-800)]'
            }`}
          >
            <tab.icon className="mr-3 w-5 h-5" />
            {tab.label}
          </button>
        ))}
      </div>
    </div>
  );

  return (
    <div className="w-64 min-w-64 max-w-64 flex-shrink-0 h-full bg-[var(--sidebar-bg)] border-r border-[var(--card-border)] flex flex-col">
      {/* 메뉴 그룹들 */}
      <div className="flex-1 overflow-y-auto">
        <div className="p-4">
          <MenuGroup 
            title="파인튜닝" 
            items={embeddingMenuItems}
            titleIcon="🔗"
          />

        </div>
      </div>
    </div>
  );
};






export default function AdminModelsPage() {
  const { user, isAuthenticated } = useAuth();
  const [models, setModels] = useState<ModelInfo[]>([]);
  const [datasets, setDatasets] = useState<TrainingDataset[]>([]);
  const [systemResources, setSystemResources] = useState<SystemResources | null>(null);
  const [ragServerConnected, setRagServerConnected] = useState<boolean>(true);
  const [activeTab, setActiveTab] = useState<keyof FinetuneTabs>('datasets');
  const [isLoading, setIsLoading] = useState(true);
  const [showDatasetModal, setShowDatasetModal] = useState(false);
  const [selectedDatasetMethod, setSelectedDatasetMethod] = useState<string>('');
  const [selectedDatasetForTraining, setSelectedDatasetForTraining] = useState<string>('');
  const [uploadedFile, setUploadedFile] = useState<File | null>(null);
  const [newDatasetName, setNewDatasetName] = useState('');
  const [newDatasetDescription, setNewDatasetDescription] = useState('');
  const [targetQACount, setTargetQACount] = useState<number>(25); // 기본값 25개 (더 안정적)
  const [finetuningStopped, setFinetuningStopped] = useState(false);
  const [trainingProgress, setTrainingProgress] = useState<{
    isTraining: boolean;
    progress: number;
    currentEpoch: number;
    totalEpochs: number;
    loss: number;
    eta: string;
    modelName?: string;
    datasetId?: string;
  }>({
    isTraining: false,
    progress: 0,
    currentEpoch: 0,
    totalEpochs: 0,
    loss: 0,
    eta: ''
  });

  const [evaluationProgress, setEvaluationProgress] = useState<{
    isEvaluating: boolean;
    progress: number;
    currentModel?: string;
    totalModels: number;
    completedModels: number;
  }>({
    isEvaluating: false,
    progress: 0,
    totalModels: 0,
    completedModels: 0
  });

  // 평가 모드 상태 (universal: 범용 평가, domain: 도메인 전문성 평가)
  const [evaluationMode, setEvaluationMode] = useState<'universal' | 'domain'>('universal');

  // 평가 데이터셋 선택 상태
  const [selectedEvaluationDataset, setSelectedEvaluationDataset] = useState<string>('');

  // 차트 표시 모드 상태
  const [chartMode, setChartMode] = useState<'bar' | 'radar'>('bar');

  // 데이터셋 생성 진행 상태
  const [datasetCreationProgress, setDatasetCreationProgress] = useState<{
    isCreating: boolean;
    progress: number;
    stage: string;
    message: string;
    method?: string;
  }>({
    isCreating: false,
    progress: 0,
    stage: '',
    message: ''
  });

  // 알림 모달 상태
  const [notificationModal, setNotificationModal] = useState<NotificationModal>({
    isOpen: false,
    type: 'info',
    title: '',
    message: '',
    confirmText: '확인',
    showCancel: false
  });

  // 모델 학습 설정 상태
  const [finetuningConfig, setFinetuningConfig] = useState<{
    epochs: number;
    batchSize: number;
    learningRate: number;
    forceCpu: boolean;
    baseModel: string;
  }>({
    epochs: 3,
    batchSize: 4,
    learningRate: 1e-5,
    forceCpu: false,
    baseModel: "nlpai-lab/KURE-v1"
  });

  // 사용자 설정 추적 상태 추가
  const [userConfigOverrides, setUserConfigOverrides] = useState<{
    epochs: boolean;
    batchSize: boolean;
    learningRate: boolean;
    forceCpu: boolean;
  }>({
    epochs: false,
    batchSize: false,
    learningRate: false,
    forceCpu: false
  });

  // 자동 최적화 사용 여부 상태
  const [useAutoOptimization, setUseAutoOptimization] = useState<boolean>(true);

  // 선택된 데이터셋 ID 상태
  const [selectedDatasetId, setSelectedDatasetId] = useState<string>('');
  
  // 데이터셋 상세보기 모달 상태
  const [showDatasetDetailModal, setShowDatasetDetailModal] = useState(false);
  const [selectedDatasetForDetail, setSelectedDatasetForDetail] = useState<TrainingDataset | null>(null);
  const [datasetDetails, setDatasetDetails] = useState<any>(null);
  const [loadingDatasetDetails, setLoadingDatasetDetails] = useState(false);
  
  // 드래그앤드롭 상태
  const [isDragging, setIsDragging] = useState(false);

  // 가이드라인 섹션 폴딩 상태
  const [isGuidelinesSectionFolded, setIsGuidelinesSectionFolded] = useState(true);

  // 프로바이더 관련 상태
  const [providers, setProviders] = useState<any>({});
  const [availableProviders, setAvailableProviders] = useState<any[]>([]);
  const [isLoadingProviders, setIsLoadingProviders] = useState(false);
  const [selectedProvider, setSelectedProvider] = useState<string>('ollama');
  const [selectedModel, setSelectedModel] = useState<string>('');

  // 프로바이더 로딩 함수
  const loadProviders = async () => {
    try {
      setIsLoadingProviders(true);
      console.log('🔄 [프로바이더 로딩] 시작...');
      
      // API 클라이언트를 사용하여 프로바이더 정보 로드
      const response = await fetch('/api/providers');
      // console.log('📡 [프로바이더 API] 응답 상태:', response.status, response.ok);
      
      if (response.ok) {
        const data = await response.json();
        // console.log('📊 [프로바이더 데이터]:', data);
        
        if (data.success) {
          // API 응답 구조 확인: data.data 객체에서 프로바이더 목록 추출
          const providersFromData = data.data ? Object.keys(data.data).map(key => ({
            key: key,
            name: data.data[key].name || key.charAt(0).toUpperCase() + key.slice(1),
            requiresApiKey: data.data[key].requiresApiKey || false,
            apiKeyConfigured: data.data[key].apiKeyConfigured !== false,
            isDynamic: data.data[key].isDynamic || false
          })) : [];
          
          setAvailableProviders(providersFromData);
          // console.log('✅ [사용 가능 프로바이더]:', providersFromData);
          
          // 각 프로바이더의 모델 목록 로드
          const providersData: any = {};
          for (const provider of providersFromData) {
            try {
              // console.log(`🔍 [${provider.key}] 모델 목록 로드 중...`);
              const modelsResponse = await fetch(`/api/providers/${provider.key}/models`);
              if (modelsResponse.ok) {
                const modelsData = await modelsResponse.json();
                // console.log(`📋 [${provider.key}] 모델 데이터:`, modelsData);
                if (modelsData.success) {
                  // API 응답 구조 확인: modelsData.data 배열에서 모델 목록 추출
                  const models = modelsData.data || modelsData.models || [];
                  providersData[provider.key] = {
                    name: provider.name,
                    requiresApiKey: provider.requiresApiKey,
                    apiKeyConfigured: provider.apiKeyConfigured !== false,
                    isDynamic: provider.isDynamic,
                    isAvailable: true,
                    models: models
                  };
                  // console.log(`✅ [${provider.key}] 모델 ${models.length}개 로드됨:`, models.map((m: any) => m.id || m.name));
                }
              } else {
                console.warn(`❌ [${provider.key}] 모델 API 응답 실패:`, modelsResponse.status);
              }
            } catch (modelError) {
              console.warn(`❌ [${provider.key}] 모델 목록 로드 실패:`, modelError);
              providersData[provider.key] = {
                name: provider.name,
                requiresApiKey: provider.requiresApiKey,
                apiKeyConfigured: provider.apiKeyConfigured !== false,
                isDynamic: provider.isDynamic,
                isAvailable: true,
                models: []
              };
            }
          }
          
          setProviders(providersData);
          // console.log('🗂️ [최종 프로바이더 데이터]:', providersData);
          
          // 기본 프로바이더와 모델 설정
          if (providersFromData && providersFromData.length > 0) {
            const firstProvider = providersFromData[0];
            setSelectedProvider(firstProvider.key);
            // console.log('🎯 [기본 프로바이더 설정]:', firstProvider.key);
            
            if (providersData[firstProvider.key]?.models?.length > 0) {
              const firstModel = providersData[firstProvider.key].models[0];
              setSelectedModel(firstModel.id || firstModel.name);
              // console.log('🎯 [기본 모델 설정]:', firstModel.id || firstModel.name);
            }
          }
        } else {
          console.warn('⚠️ [프로바이더 API] success: false');
        }
      } else {
        console.error('❌ [프로바이더 API] 요청 실패:', response.status);
      }
    } catch (error: any) {
      console.error('❌ [프로바이더 로딩] 오류:', error);
      
      // 오류 발생 시 기본 프로바이더 정보 설정
      const defaultProviders = {
        ollama: {
          name: 'Ollama',
          isAvailable: true,
          requiresApiKey: false,
          apiKeyConfigured: true,
          isDynamic: true,
          models: [{ id: 'hamonize:latest', name: 'hamonize:latest', size: 0 }]
        }
      };
      setProviders(defaultProviders);
      setAvailableProviders([
        { key: 'ollama', name: 'Ollama', requiresApiKey: false, apiKeyConfigured: true, isDynamic: true }
      ]);
      setSelectedProvider('ollama');
      setSelectedModel('hamonize:latest');
      // console.log('🔧 [기본값 설정] 완료');
    } finally {
      setIsLoadingProviders(false);
      console.log('✅ [프로바이더 로딩] 완료');
    }
  };

  // 프로바이더 변경 핸들러
  const handleProviderChange = (newProvider: string) => {
    setSelectedProvider(newProvider);
    setSelectedModel(''); // 모델 초기화
    
    // 첫 번째 사용 가능한 모델을 자동 선택
    if (providers[newProvider] && providers[newProvider].models.length > 0) {
      const firstModel = providers[newProvider].models[0];
      setSelectedModel(firstModel.id || firstModel.name);
    }
  };

  // 학습 시작 함수 (설정 카드에서 사용)
  const handleStartTrainingFromConfig = () => {
    if (!selectedDatasetId) {
      showNotification('warning', '데이터셋 선택 필요', '학습할 데이터셋을 선택해주세요.');
      return;
    }
    handleStartTraining(selectedDatasetId);
  };

  // 알림 모달 열기 함수
  const showNotification = (
    type: NotificationModal['type'],
    title: string,
    message: string,
    onConfirm?: () => void,
    confirmText?: string,
    showCancel?: boolean
  ) => {
    setNotificationModal({
      isOpen: true,
      type,
      title,
      message,
      onConfirm,
      confirmText: confirmText || '확인',
      showCancel: showCancel || false
    });
  };

  // 알림 모달 닫기 함수
  const closeNotification = () => {
    setNotificationModal(prev => ({ ...prev, isOpen: false }));
  };

  // 주기적으로 학습 진행률 체크
  useEffect(() => {
    let interval: NodeJS.Timeout;
    let consecutiveErrors = 0;
    const maxConsecutiveErrors = 5; // 연속 5번 실패하면 중지
    
    if (trainingProgress.isTraining) {
      interval = setInterval(async () => {
        try {
          console.log('🔄 [프론트엔드] API 호출 시작: /api/rag/models/finetune/progress');
          
          const response = await fetch('/api/rag/models/finetune/progress', {
            method: 'GET',
            headers: {
              'Content-Type': 'application/json',
            },
            // 타임아웃 설정 (빠른 응답을 위해 단축)
            signal: AbortSignal.timeout(5000) // 5초 타임아웃으로 단축
          });
          
          console.log('🔄 [프론트엔드] API 응답 상태:', response.status, response.statusText);
          
          if (!response.ok) {
            const errorText = await response.text();
            console.error('🔄 [프론트엔드] API 오류 응답:', errorText);
            
            // 타임아웃 오류인 경우 더 관대하게 처리
            if (response.status === 408 || errorText.includes('timeout')) {
              console.warn('🔄 [프론트엔드] 타임아웃 발생, 다음 주기에 재시도');
              return; // 에러를 던지지 않고 다음 주기를 기다림
            }
            
            throw new Error(`HTTP ${response.status}: ${response.statusText} - ${errorText}`);
          }
          
          const result = await response.json();
          console.log('🔄 [프론트엔드] API 응답 데이터:', result);
          
          if (result.success && result.progress) {
            // 성공하면 에러 카운터 리셋
            consecutiveErrors = 0;
            
            const progress = result.progress;
            console.log('🔄 [프론트엔드] 훈련 진행률 업데이트:', progress);
            
            setTrainingProgress(prev => ({
              ...prev,
              progress: progress.progress || 0,
              currentEpoch: progress.current_epoch || 0,
              totalEpochs: progress.total_epochs || 0,
              loss: progress.current_loss || 0,
              eta: progress.eta || '',
              modelName: prev.modelName || progress.model_name,
              datasetId: prev.datasetId || progress.dataset_id
            }));
            
            // 학습 상태에 따른 처리 (우선순위: completed > failed > 진행 중)
            if (progress.status === 'completed') {
              console.log('✅ [프론트엔드] 모델 학습 완료 감지 (상태: completed)');
              setTrainingProgress(prev => ({
                ...prev,
                isTraining: false,
                progress: 100,
                eta: '완료'
              }));
              
              // 모델 목록 새로고침
              await loadModels();
            } else if (progress.status === 'failed') {
              console.log('❌ [프론트엔드] 모델 학습 실패 감지');
              setTrainingProgress(prev => ({
                ...prev,
                isTraining: false,
                progress: 0,
                eta: '실패'
              }));
            } else if (progress.progress >= 100 && !progress.is_training) {
              console.log('✅ [프론트엔드] 모델 학습 완료 감지 (진행률 100%)');
              setTrainingProgress(prev => ({
                ...prev,
                isTraining: false,
                progress: 100,
                eta: '완료'
              }));
              
              // 모델 목록 새로고침
              await loadModels();
            } else if (progress.status === 'training' && progress.progress >= 100) {
              console.log('✅ [프론트엔드] 모델 학습 완료 감지 (training 상태이지만 진행률 100%)');
              setTrainingProgress(prev => ({
                ...prev,
                isTraining: false,
                progress: 100,
                eta: '완료'
              }));
              
              // 모델 목록 새로고침
              await loadModels();
            }
          } else if (!result.success) {
            console.warn('훈련 진행률 체크 실패:', result.error);
            consecutiveErrors++;
            
            // RAG 서버 연결 실패로 보이는 경우 훈련 상태 확인
            if (result.error?.includes('RAG 서버') || result.error?.includes('연결')) {
              // 훈련이 실제로 중지되었는지 확인하기 위해 상태 체크
              try {
                const statusResponse = await fetch('/api/rag/models/finetune/status');
                const statusResult = await statusResponse.json();
                
                if (statusResult.success && statusResult.status === 'completed') {
                  // 훈련이 완료된 경우
                  setTrainingProgress(prev => ({
                    ...prev,
                    isTraining: false,
                    progress: 100,
                    eta: '완료'
                  }));
                  await loadModels();
                } else if (statusResult.success && statusResult.status === 'failed') {
                  // 훈련이 실패한 경우
                  setTrainingProgress(prev => ({
                    ...prev,
                    isTraining: false,
                    progress: 0,
                    eta: '실패'
                  }));
                }
              } catch (statusError) {
                console.warn('훈련 상태 확인 실패:', statusError);
              }
            }
          }
          
          // 학습 진행 중에도 데이터셋 목록을 주기적으로 새로고침
          if (Math.random() < 0.3) { // 30% 확률로 데이터셋 목록 새로고침
            await loadDatasets();
          }
        } catch (error) {
          consecutiveErrors++;
          console.error(`훈련 진행률 체크 실패 (${consecutiveErrors}/${maxConsecutiveErrors}):`, error);
          
          // 타임아웃 에러인 경우 백엔드 상태 확인 시도
          if (error instanceof Error && (error.name === 'TimeoutError' || error.name === 'AbortError' || error.message.includes('timeout'))) {
            console.log('🔄 [프론트엔드] 타임아웃 에러 - 백엔드 상태 확인 시도');
            try {
              const statusResponse = await fetch('/api/rag/models/finetune/status', {
                method: 'GET',
                headers: { 'Content-Type': 'application/json' },
                signal: AbortSignal.timeout(5000) // 5초 타임아웃으로 단축
              });
              
              if (statusResponse.ok) {
                const statusResult = await statusResponse.json();
                console.log('🔄 [프론트엔드] 백엔드 상태 확인 결과:', statusResult);
                
                if (statusResult.success) {
                  const status = statusResult.status || statusResult.state?.status;
                  
                  if (status === 'training' || status === 'starting' || status === 'retrying_cpu') {
                    console.log('✅ [프론트엔드] 백엔드에서 훈련 진행 중 확인됨');
                    setTrainingProgress(prev => ({
                      ...prev,
                      eta: '훈련 진행 중... (서버 응답 지연)'
                    }));
                    // 에러 카운터 리셋 (실제로는 훈련이 진행 중)
                    consecutiveErrors = Math.max(0, consecutiveErrors - 1);
                  } else if (status === 'completed') {
                    console.log('✅ [프론트엔드] 백엔드에서 훈련 완료 확인됨');
                    setTrainingProgress(prev => ({
                      ...prev,
                      isTraining: false,
                      progress: 100,
                      eta: '완료'
                    }));
                    await loadModels();
                  } else if (status === 'failed') {
                    console.log('❌ [프론트엔드] 백엔드에서 훈련 실패 확인됨');
                    setTrainingProgress(prev => ({
                      ...prev,
                      isTraining: false,
                      progress: 0,
                      eta: '실패'
                    }));
                  }
                }
              }
            } catch (statusError) {
              console.warn('백엔드 상태 확인 실패:', statusError);
            }
          }
          
          // 연속으로 너무 많이 실패하면 훈련 상태를 중지로 변경
          if (consecutiveErrors >= maxConsecutiveErrors) {
            console.warn('연속 진행률 체크 실패로 인해 훈련 모니터링을 중지합니다.');
            setTrainingProgress(prev => ({
              ...prev,
              isTraining: false,
              eta: '모니터링 중지 (서버 연결 실패)'
            }));
          }
        }
      }, 1500); // 1.5초마다 체크 (더 빠른 업데이트)
    }
    
    return () => {
      if (interval) {
        clearInterval(interval);
      }
    };
  }, [trainingProgress.isTraining]); // 진행상태 변경 시 모니터링 재시작

  // 추가: 컴포넌트 마운트 시 진행 중인 훈련이 있는지 확인
  useEffect(() => {
    const checkOngoingTraining = async () => {
      try {
        const response = await fetch('/api/rag/models/finetune/progress');
        if (response.ok) {
          const result = await response.json();
          if (result.success && result.progress) {
            const progress = result.progress;
            console.log('🔄 [프론트엔드] 훈련 상태 확인:', progress);
            
            if (progress.is_training) {
              console.log('🔄 [프론트엔드] 진행 중인 훈련 발견, 모니터링 시작');
              setTrainingProgress({
                isTraining: true,
                progress: progress.progress || 0,
                currentEpoch: progress.current_epoch || 0,
                totalEpochs: progress.total_epochs || 0,
                loss: progress.current_loss || 0,
                eta: progress.eta || '진행 중...',
                modelName: progress.model_name || '',
                datasetId: progress.dataset_id || ''
              });
            } else if (progress.status === 'completed' && progress.progress >= 100) {
              console.log('✅ [프론트엔드] 완료된 훈련 발견, 상태 업데이트');
              setTrainingProgress({
                isTraining: false,
                progress: 100,
                currentEpoch: progress.current_epoch || 0,
                totalEpochs: progress.total_epochs || 0,
                loss: progress.current_loss || 0,
                eta: '완료',
                modelName: progress.model_name || '',
                datasetId: progress.dataset_id || ''
              });
              
              // 모델 목록 새로고침
              await loadModels();
            } else if (progress.status === 'failed') {
              console.log('❌ [프론트엔드] 실패한 훈련 발견, 상태 업데이트');
              setTrainingProgress({
                isTraining: false,
                progress: 0,
                currentEpoch: progress.current_epoch || 0,
                totalEpochs: progress.total_epochs || 0,
                loss: progress.current_loss || 0,
                eta: '실패',
                modelName: progress.model_name || '',
                datasetId: progress.dataset_id || ''
              });
            }
          }
        }
      } catch (error) {
        console.warn('진행 중인 훈련 확인 실패:', error);
      }
    };
    
    checkOngoingTraining();
  }, []);

  // 파인튜닝 상태 초기화 함수
  const resetTrainingState = async () => {
    try {
      console.log('🔄 파인튜닝 상태 초기화 시작');
      const response = await fetch('/api/rag/models/finetune/reset', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
      });
      
      if (response.ok) {
        const result = await response.json();
        if (result.success) {
          console.log('✅ 파인튜닝 상태 초기화 성공');
          // 훈련 상태 초기화
          setTrainingProgress({
            isTraining: false,
            progress: 0,
            currentEpoch: 0,
            totalEpochs: 0,
            loss: 0,
            eta: '',
            modelName: '',
            datasetId: ''
          });
          
          showNotification('success', '상태 초기화 완료', '파인튜닝 상태가 성공적으로 초기화되었습니다.');
        } else {
          console.error('❌ 파인튜닝 상태 초기화 실패:', result.error);
          showNotification('error', '상태 초기화 실패', result.error || '알 수 없는 오류가 발생했습니다.');
        }
      } else {
        console.error('❌ 파인튜닝 상태 초기화 요청 실패:', response.status);
        showNotification('error', '상태 초기화 실패', '서버 요청에 실패했습니다.');
      }
    } catch (error) {
      console.error('❌ 파인튜닝 상태 초기화 중 오류:', error);
      showNotification('error', '상태 초기화 실패', '네트워크 오류가 발생했습니다.');
    }
  };

  // 데이터셋 생성 방법들
  const datasetCreationMethods: DatasetCreationMethod[] = [
    {
      id: 'file-upload',
      title: '파일 업로드',
      description: 'JSON, CSV 형태의 데이터셋 파일을 직접 업로드',
      icon: '📁',
      recommended: true,
      difficulty: 'easy'
    },
    {
      id: 'document-llm',
      title: '문서 + LLM 생성',
      description: 'PDF, DOCX, HWP 등 다양한 문서에서 AI가 자동으로 질문-답변 쌍 생성',
      icon: '🤖',
      recommended: true,
      difficulty: 'medium'
    },
    {
      id: 'collection-based',
      title: '컬렉션 기반 생성',
      description: '기존 RAG 컬렉션의 문서들을 활용하여 임베딩 훈련용 데이터셋 자동 생성',
      icon: '📚',
      recommended: false,
      difficulty: 'easy'
    },
    {
      id: 'evaluation-universal',
      title: '범용 평가 데이터셋',
      description: '일반적인 검색 성능을 평가하기 위한 평가 전용 데이터셋 업로드 [데이터 누수 방지]',
      icon: '🎯',
      recommended: true,
      difficulty: 'easy'
    },
    {
      id: 'evaluation-domain',
      title: '도메인 평가 데이터셋',
      description: '특정 도메인의 전문성을 평가하기 위한 평가 전용 데이터셋 업로드 [데이터 누수 방지]',
      icon: '🔬',
      recommended: true,
      difficulty: 'easy'
    }
  ];

  // 모델 목록 로드
  useEffect(() => {
    loadModels();
    loadDatasets();
    loadSystemResources();
    loadProviders(); // 프로바이더 정보 로드 추가
  }, []);

  // 시스템 자원에 따른 모델 학습 설정 초기값 업데이트 (자동 최적화 사용 시에만)
  useEffect(() => {
    if (systemResources && useAutoOptimization) {
      setFinetuningConfig(prev => ({
        ...prev,
        batchSize: userConfigOverrides.batchSize ? prev.batchSize : (systemResources.gpu_available 
          ? systemResources.recommendations.gpu_mode.batch_size 
          : systemResources.recommendations.cpu_mode.batch_size),
        forceCpu: userConfigOverrides.forceCpu ? prev.forceCpu : !systemResources.gpu_available,
        epochs: userConfigOverrides.epochs ? prev.epochs : (systemResources.gpu_available ? 3 : 1) // GPU 사용 가능할 때는 3, CPU 모드는 1로 설정
      }));
    }
  }, [systemResources, useAutoOptimization, userConfigOverrides]);

  // 모델 목록이 업데이트될 때 기본 모델 설정 확인
  useEffect(() => {
    if (models.length > 0) {
      const baseModels = models.filter(m => m.type === 'base');
      if (baseModels.length > 0 && !baseModels.find(m => m.id === finetuningConfig.baseModel)) {
        // 현재 설정된 모델이 기본 모델 목록에 없으면 첫 번째 기본 모델로 설정
        setFinetuningConfig(prev => ({
          ...prev,
          baseModel: baseModels[0].id
        }));
      }
    }
  }, [models, finetuningConfig.baseModel]);

  // 데이터셋 목록이 업데이트될 때 선택된 데이터셋 유효성 확인 및 자동 선택
  useEffect(() => {
    const availableDatasets = datasets.filter(d => d.status === 'ready' && (d.size ?? 0) > 0);
    
    // 현재 선택된 데이터셋이 사용 가능한 목록에 없는 경우
    if (selectedDatasetId && !availableDatasets.find(d => d.id === selectedDatasetId)) {
      setSelectedDatasetId('');
    }
    
    // 데이터셋이 선택되지 않았고 사용 가능한 데이터셋이 있는 경우 첫 번째를 자동 선택
    if (!selectedDatasetId && availableDatasets.length > 0) {
      setSelectedDatasetId(availableDatasets[0].id);
    }
  }, [datasets, selectedDatasetId]);

  const loadModels = async () => {
    try {
      // 먼저 현재 활성 모델 설정 조회
      const configResponse = await fetch('/api/rag/models', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ action: 'get-config' })
      });
      
      let currentActiveModel = 'nlpai-lab/KURE-v1'; // 기본값
      
      if (configResponse.ok) {
        const configResult = await configResponse.json();
        if (configResult.success && configResult.config?.current_embedding_model) {
          currentActiveModel = configResult.config.current_embedding_model;
        }
      }
      
      // 모델 목록 조회 (기본 모델 + 모델 학습 모델 모두 포함)
      const response = await fetch('/api/rag/models');
      const result = await response.json();
      
      if (result.success) {
        // 평가 결과 별도 조회
        let evaluationResults: Record<string, any> = {};
        try {
          const evalResponse = await fetch('/api/rag/models', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ action: 'get-evaluation-results' })
          });
          
          if (evalResponse.ok) {
            const evalResult = await evalResponse.json();
            if (evalResult.success && evalResult.evaluation_results) {
              // 평가 결과를 모델 ID별로 매핑
              evalResult.evaluation_results.forEach((evalItem: any) => {
                evaluationResults[evalItem.model_id] = evalItem;
              });
            }
          }
        } catch (evalError) {
          console.warn('평가 결과 로드 실패 (무시):', evalError);
        }
        
        // RAG 서버에서 받은 모델들을 웹 UI 형식으로 변환
        const allModels: ModelInfo[] = result.models.map((model: any) => {
          const evalData = evaluationResults[model.id];
          const hasEvaluation = evalData && (evalData.precision !== undefined || evalData.recall !== undefined || evalData.f1_score !== undefined);
          
          return {
            id: model.id,
            name: model.name || (model.id.includes('/') ? model.id.split('/')[1] : model.id),
            version: '1.0.0',
            type: (model.type || 'base') as 'base' | 'finetuned',
            size: model.size_gb ? `${model.size_gb}GB` : '2.1GB',
            description: model.type === 'base' 
              ? '기본 한국어 임베딩 모델' 
              : `${model.base_model || model.id} 기반 모델 학습 모델`,
            createdAt: model.created_at || '2024-01-15',
            performance: { 
              precision: hasEvaluation ? (evalData.precision || model.precision || 0) : (model.precision || 0.72), 
              recall: hasEvaluation ? (evalData.recall || model.recall || 0) : (model.recall || 0.68), 
              f1Score: hasEvaluation ? (evalData.f1_score || evalData.best_score || model.f1_score || model.best_score || 0) : (model.f1_score || model.best_score || 0.70)
            },
            isActive: model.id === currentActiveModel ||
                      model.path === currentActiveModel ||
                      currentActiveModel.includes(model.id) ||
                      (model.path && currentActiveModel.includes(model.path)),
            trainingData: model.dataset_info ? {
              documentCount: model.dataset_info.document_count || 0,
              pairCount: model.dataset_info.pair_count || model.dataset_info.size || 0,
              domains: model.dataset_info.domains || [model.dataset_info.description || '도메인 특화']
            } : (model.training_samples ? {
              documentCount: Math.floor(model.training_samples / 10) || 1,
              pairCount: model.training_samples,
              domains: ['도메인 특화']
            } : undefined),
            evaluated: hasEvaluation || model.evaluated || false,
            evaluationDate: evalData?.evaluation_date || model.evaluation_date,
            evaluationResults: evalData // 평가 결과 전체 데이터 추가
          };
        });
        
        setModels(allModels);
        setRagServerConnected(true);
        
        // 디버깅: 평가 데이터 확인

        
      } else {
        console.error('모델 목록 로드 실패:', result.error);
        setRagServerConnected(false);
        // 실패 시 기본 모델만 표시
        setModels([{
          id: currentActiveModel,
          name: currentActiveModel.includes('/') ? currentActiveModel.split('/')[1] : currentActiveModel,
          version: '1.0.0',
          type: 'base',
          size: '2.1GB',
          description: '현재 활성 임베딩 모델',
          createdAt: '2024-01-15',
          performance: { precision: 0.72, recall: 0.68, f1Score: 0.70 },
          isActive: true,
          evaluated: false
        }]);
      }
    } catch (error) {
      console.error('모델 로드 실패:', error);
      setRagServerConnected(false);
      // 에러 시 기본 모델만 표시
      setModels([{
        id: 'nlpai-lab/KURE-v1',
        name: 'KURE-v1',
        version: '1.0.0',
        type: 'base',
        size: '2.1GB',
        description: '기본 한국어 임베딩 모델',
        createdAt: '2024-01-15',
        performance: { precision: 0.72, recall: 0.68, f1Score: 0.70 },
        isActive: true,
        evaluated: false
      }]);
    } finally {
      setIsLoading(false);
    }
  };

  const loadDatasets = async () => {
    try {
      console.log('🔄 [데이터셋 로드] 시작...');
      
      // RAG 데이터셋, 피드백 데이터셋, LLM 서버 데이터셋을 모두 로드
      const [ragResponse, feedbackResponse] = await Promise.all([
        fetch('/api/rag/datasets'),
        fetch('/api/feedback/dataset'),
      ]);
      
      console.log('📡 [RAG 데이터셋] 응답 상태:', ragResponse.status, ragResponse.ok);
      console.log('📡 [피드백 데이터셋] 응답 상태:', feedbackResponse.status, feedbackResponse.ok);
      // console.log('📡 [LLM 데이터셋] 응답 상태:', llmResponse.status, llmResponse.ok);
      
      const ragResult = await ragResponse.json();
      const feedbackResult = await feedbackResponse.json();
      // const llmResult = llmResponse.ok ? await llmResponse.json() : { success: false, datasets: [] };
      
      console.log('📊 [RAG 데이터셋] 응답 데이터:', ragResult);
      console.log('🔍 [디버깅] 각 데이터셋의 deletable 속성:', ragResult.datasets?.map((d: any) => ({id: d.id, deletable: d.deletable, system_managed: d.system_managed})));
      console.log('📊 [피드백 데이터셋] 응답 데이터:', feedbackResult);
      // console.log('📊 [LLM 데이터셋] 응답 데이터:', llmResult);
      
      const allDatasets: TrainingDataset[] = [];
      
      // RAG 데이터셋 추가 (중복 제거)
      if (ragResult.success) {
        const ragDatasets = ragResult.datasets.map((dataset: any) => ({
          id: dataset.id,
          name: dataset.name,
          description: dataset.description,
          documentCount: Math.floor(dataset.size / 10) || 0, // 추정값
          pairCount: dataset.size || 0,
          domains: [dataset.type || '기본'],
          createdAt: dataset.created_at,
          status: 'ready' as const,
          type: dataset.type,
          size: dataset.size,
          system_managed: dataset.system_managed,  // 시스템 관리 여부 추가
          deletable: dataset.deletable  // 삭제 가능 여부 추가
        }));

        // RAG 데이터셋 내부의 중복 ID 제거
        const uniqueRagDatasets = ragDatasets.filter((dataset: TrainingDataset, index: number, self: TrainingDataset[]) =>
          index === self.findIndex((d: TrainingDataset) => d.id === dataset.id)
        );

        allDatasets.push(...uniqueRagDatasets);
        console.log('✅ [RAG 데이터셋] 추가됨:', uniqueRagDatasets.length, '개');

        if (ragDatasets.length > uniqueRagDatasets.length) {
          console.log('ℹ️ [RAG 데이터셋] 중복 제거됨:', ragDatasets.length - uniqueRagDatasets.length, '개');
        }
      } else {
        console.warn('⚠️ [RAG 데이터셋] 로드 실패:', ragResult.error);
      }
      
      // 피드백 데이터셋 추가 (API 서버 응답 구조에 맞게 수정)
      if (feedbackResult.success && feedbackResult.data?.datasets && feedbackResult.data.datasets.length > 0) {
        const feedbackDatasets = feedbackResult.data.datasets.map((dataset: any) => ({
          id: dataset.id,
          name: dataset.name,
          description: dataset.description,
          documentCount: dataset.documentCount || 1,
          pairCount: dataset.pairCount || 1,
          domains: dataset.domains || ['피드백 기반'],
          createdAt: dataset.createdAt,
          status: dataset.status || 'ready',
          type: 'feedback',
          size: dataset.size || 1,
          system_managed: true,  // 피드백 데이터셋은 시스템 관리
          deletable: false  // 삭제 불가능
        }));

        // 중복 ID 제거: 이미 존재하는 ID는 추가하지 않음 (RAG 데이터셋 우선)
        const existingIds = new Set(allDatasets.map((d: TrainingDataset) => d.id));
        const uniqueFeedbackDatasets = feedbackDatasets.filter((dataset: TrainingDataset) => !existingIds.has(dataset.id));

        allDatasets.push(...uniqueFeedbackDatasets);
        console.log('✅ [피드백 데이터셋] 추가됨:', uniqueFeedbackDatasets.length, '개 (중복 제거됨)');

        if (feedbackDatasets.length > uniqueFeedbackDatasets.length) {
          console.log('ℹ️ [피드백 데이터셋] 중복 제거됨:', feedbackDatasets.length - uniqueFeedbackDatasets.length, '개');
        }
      } else {
        console.warn('⚠️ [피드백 데이터셋] 로드 실패 또는 데이터 없음:', feedbackResult);
      }

      // LLM 데이터셋 추가 (평가용 데이터셋 포함)
      // if (llmResult.success && llmResult.datasets && llmResult.datasets.length > 0) {
      //   const llmDatasets = llmResult.datasets.map((dataset: any) => ({
      //     id: dataset.id,
      //     name: dataset.name,
      //     description: dataset.description || `${dataset.type || 'LLM'} 데이터셋`,
      //     documentCount: Math.floor(dataset.size / 10) || 0,
      //     pairCount: dataset.size || 0,
      //     domains: [dataset.type === 'universal_eval' ? '범용 평가' :
      //              dataset.type === 'domain_eval' ? '도메인 평가' :
      //              dataset.type || 'LLM 학습'],
      //     createdAt: dataset.createdAt,
      //     status: dataset.status || 'ready',
      //     type: dataset.type,
      //     size: dataset.size
      //   }));

      //   // 중복 ID 제거: 이미 존재하는 ID는 추가하지 않음
      //   const existingIds = new Set(allDatasets.map((d: TrainingDataset) => d.id));
      //   const uniqueLlmDatasets = llmDatasets.filter((dataset: TrainingDataset) => !existingIds.has(dataset.id));

      //   allDatasets.push(...uniqueLlmDatasets);
      //   console.log('✅ [LLM 데이터셋] 추가됨:', uniqueLlmDatasets.length, '개 (평가용 포함)');

      //   if (llmDatasets.length > uniqueLlmDatasets.length) {
      //     console.log('ℹ️ [LLM 데이터셋] 중복 제거됨:', llmDatasets.length - uniqueLlmDatasets.length, '개');
      //   }
      // } else {
      //   console.warn('⚠️ [LLM 데이터셋] 로드 실패 또는 데이터 없음:', llmResult);
      // }

      console.log('📋 [최종 데이터셋 목록]:', allDatasets.length, '개');
      setDatasets(allDatasets);
      setRagServerConnected(true);
    } catch (error) {
      console.error('❌ [데이터셋 로드] 오류:', error);
      setDatasets([]);
      setRagServerConnected(false);
    }
  };

  const loadSystemResources = async () => {
    try {
      const response = await fetch('/api/rag/models', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ action: 'analyze-resources' })
      });
      
      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: RAG 서버에 연결할 수 없습니다`);
      }
      
      const result = await response.json();
      
      if (result.success) {
        setSystemResources(result.resources);
        setRagServerConnected(true);
      } else {
        // error 필드가 없거나 undefined인 경우 기본 메시지 사용
        const errorMessage = result.error || '알 수 없는 오류가 발생했습니다';
        console.warn('시스템 자원 분석 실패:', errorMessage, '전체 응답:', result);
        setRagServerConnected(false);
        
        // 기본 시스템 자원 정보 설정 (추정값)
        setSystemResources({
          total_memory_gb: 16,
          available_memory_gb: 8,
          cpu_cores: 8,
          gpu_available: false,
          gpu_memory_gb: 0,
          gpu_available_memory_gb: 0,
          recommendations: {
            gpu_mode: {
              batch_size: 4,
              num_workers: 2,
              use_amp: true
            },
            cpu_mode: {
              batch_size: 2,
              num_workers: 4
            }
          }
        });
      }
    } catch (error) {
      console.error('시스템 자원 로드 실패:', error);
      setRagServerConnected(false);
      // RAG 서버 연결 실패 시 기본값 설정
      setSystemResources({
        total_memory_gb: 16,
        available_memory_gb: 8,
        cpu_cores: 8,
        gpu_available: false,
        gpu_memory_gb: 0,
        gpu_available_memory_gb: 0,
        recommendations: {
          gpu_mode: {
            batch_size: 4,
            num_workers: 2,
            use_amp: true
          },
          cpu_mode: {
            batch_size: 2,
            num_workers: 4
          }
        }
      });
    }
  };

  const handleModelActivation = async (modelId: string) => {
    try {
      const response = await fetch('/api/rag/models', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ 
          action: 'set-active-model',
          modelId: modelId
        })
      });
      
      const result = await response.json();
      
      if (result.success) {
        // UI 업데이트
        setModels(models.map(m => ({ ...m, isActive: m.id === modelId })));
        
        const modelName = models.find(m => m.id === modelId)?.name;
        showNotification(
          'success',
          '모델 활성화 완료',
          `${modelName} 모델이 활성화되었습니다.\n\n⚠️ 변경사항을 적용하려면 RAG 서비스를 재시작하세요`
        );
      } else {
        showNotification(
          'error',
          '모델 활성화 실패',
          `모델 활성화 실패: ${result.error || '알 수 없는 오류'}`
        );
      }
    } catch (error) {
      console.error('모델 활성화 실패:', error);
      showNotification(
        'error',
        '모델 활성화 실패',
        '모델 활성화 중 오류가 발생했습니다.'
      );
    }
  };

  // 모델 삭제 함수
  const handleDeleteModel = async (modelId: string, modelName: string) => {
    // 활성화된 모델은 삭제 불가
    const model = models.find(m => m.id === modelId);
    if (model?.isActive) {
      showNotification(
        'warning',
        '삭제 불가',
        '현재 활성화된 모델은 삭제할 수 없습니다. 다른 모델을 활성화한 후 삭제해주세요.'
      );
      return;
    }

    // 삭제 확인 모달
    showNotification(
      'warning',
      '모델 삭제 확인',
      `정말로 "${modelName}" 모델을 삭제하시겠습니까?\n\n이 작업은 되돌릴 수 없으며, 모델 파일이 완전히 제거됩니다.`,
      async () => {
        try {
          const response = await fetch('/api/rag/models', {
            method: 'DELETE',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ 
              modelId: modelId 
            })
          });
          
          const result = await response.json();
          
          if (result.success) {
            // UI에서 모델 제거
            setModels(models.filter(m => m.id !== modelId));
            
            showNotification(
              'success',
              '모델 삭제 완료',
              `${modelName} 모델이 성공적으로 삭제되었습니다.`
            );
          } else {
            showNotification(
              'error',
              '모델 삭제 실패',
              `모델 삭제 실패: ${result.error || '알 수 없는 오류'}`
            );
          }
        } catch (error) {
          console.error('모델 삭제 실패:', error);
          showNotification(
            'error',
            '모델 삭제 실패',
            '모델 삭제 중 오류가 발생했습니다.'
          );
        }
      },
      '삭제',
      true
    );
  };

  const handleStartTraining = async (datasetId: string) => {
    // RAG 서버 연결 상태 확인
    if (!ragServerConnected) {
      showNotification(
        'warning',
        'RAG 서버 연결 필요',
        'RAG 서버가 실행되지 않았습니다.\n먼저 RAG 서버를 시작해주세요:\n\nsystemctl --user start airun-rag'
      );
      return;
    }

    try {
      // 모델 학습 탭으로 자동 전환
      setActiveTab('training');
      
      setSelectedDatasetForTraining(datasetId);
      setFinetuningStopped(false);
      
      const dataset = datasets.find(d => d.id === datasetId);
      const trainingParams = {
        action: 'finetune',
        model_name: finetuningConfig.baseModel,
        dataset_id: datasetId,
        epochs: finetuningConfig.epochs,
        batch_size: finetuningConfig.batchSize,
        learning_rate: finetuningConfig.learningRate,
        force_cpu: finetuningConfig.forceCpu,
        // 사용자 설정 추적 정보 전달
        user_set_epochs: userConfigOverrides.epochs || !useAutoOptimization,
        user_set_batch_size: userConfigOverrides.batchSize || !useAutoOptimization,
        user_set_learning_rate: userConfigOverrides.learningRate || !useAutoOptimization
      };

      console.log('🚀 [프론트엔드] 모델 학습 시작, polling 활성화');
      
      // 이전 훈련 상태 초기화
      try {
        await fetch('/api/rag/models/finetune/reset', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' }
        });
        console.log('🔄 [프론트엔드] 이전 훈련 상태 초기화 완료');
      } catch (error) {
        console.warn('이전 훈련 상태 초기화 실패:', error);
      }
      
      setTrainingProgress({
        isTraining: true,
        progress: 0,
        currentEpoch: 0,
        totalEpochs: trainingParams.epochs,
        loss: 0,
        eta: '훈련 준비 중...',
        modelName: models.find(m => m.id === trainingParams.model_name)?.name || trainingParams.model_name,
        datasetId: datasetId
      });

      const response = await fetch('/api/rag/models', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(trainingParams)
      });

      const result = await response.json();
      
      if (result.success) {
        console.log('✅ [프론트엔드] 모델 학습 API 호출 성공 - 백그라운드 훈련 시작:', result);
        
        // 백그라운드 훈련이 시작되었으므로 훈련 상태 유지
        setTrainingProgress(prev => ({
          ...prev,
          isTraining: true,
          progress: 5, // 시작 단계 진행률
          eta: '훈련 시작됨...',
          modelName: trainingParams.model_name,
          datasetId: datasetId
        }));

        // 훈련 시작 후 즉시 진행상황 확인 (모니터링 강제 시작)
        setTimeout(async () => {
          try {
            console.log('🔄 [프론트엔드] 훈련 시작 후 진행상황 강제 확인');
            const progressResponse = await fetch('/api/rag/models/finetune/progress');
            if (progressResponse.ok) {
              const progressResult = await progressResponse.json();
              if (progressResult.success && progressResult.progress) {
                const progress = progressResult.progress;
                console.log('🔄 [프론트엔드] 훈련 시작 후 진행상황:', progress);

                setTrainingProgress(prev => ({
                  ...prev,
                  isTraining: progress.is_training || prev.isTraining,
                  progress: progress.progress || prev.progress,
                  currentEpoch: progress.current_epoch || prev.currentEpoch,
                  totalEpochs: progress.total_epochs || prev.totalEpochs,
                  loss: progress.current_loss || prev.loss,
                  eta: progress.eta || prev.eta,
                  modelName: progress.model_name || prev.modelName,
                  datasetId: progress.dataset_id || prev.datasetId
                }));
              }
            }
          } catch (error) {
            console.warn('🔄 [프론트엔드] 훈련 시작 후 진행상황 확인 실패:', error);
          }
        }, 2000); // 2초 후 확인
        
        // 안전성 기능 정보 포함하여 알림 표시
        const safetyInfo = result.safety_features ? 
          '\n\n🛡️ 안전성 기능:\n• 실시간 메모리 모니터링\n• 자동 CPU 모드 전환\n• 비상 메모리 정리\n• 리소스 임계값 검사' : '';
        
        showNotification(
          'success',
          '모델 학습 시작',
          `모델 학습이 안전성 모니터링과 함께 백그라운드에서 시작되었습니다.\n\n모델: ${trainingParams.model_name}\n데이터셋: ${dataset?.name || datasetId}\n\n진행 상황은 실시간으로 업데이트됩니다.${safetyInfo}`
        );
      } else {
        setTrainingProgress(prev => ({
          ...prev,
          isTraining: false,
          progress: 0,
          eta: '실패'
        }));
        
        // 리소스 부족으로 인한 실패 처리
        if (result.warnings && result.warnings.length > 0) {
          const warningDetails = result.warnings.join('\n• ');
          const resourceInfo = result.resource_info ? 
            `\n\n현재 시스템 상태:\n• 사용 가능한 메모리: ${result.resource_info.available_memory_gb?.toFixed(1)}GB\n• GPU 사용 가능: ${result.resource_info.gpu_available ? '예' : '아니오'}\n• GPU 메모리: ${result.resource_info.gpu_memory_gb?.toFixed(1)}GB` : '';
          
          const recommendations = result.recommendations ? 
            `\n\n권장 해결 방법:\n• ${result.recommendations.join('\n• ')}` : '';
          
          showNotification(
            'warning',
            '시스템 리소스 부족',
            `모델 학습을 시작할 수 없습니다.\n\n문제점:\n• ${warningDetails}${resourceInfo}${recommendations}`,
            undefined,
            undefined,
            true
          );
        } else {
          showNotification(
            'error',
            '모델 학습 실패',
            `모델 학습 실패: ${result.error}`
          );
        }
      }
    } catch (error) {
      console.error('모델 학습 시작 실패:', error);
      setTrainingProgress(prev => ({
        ...prev,
        isTraining: false,
        progress: 0,
        eta: '오류'
      }));
      
      // 모델 학습 상태 확인하여 구체적인 오류 메시지 제공
      try {
        const statusResponse = await fetch('/api/rag/finetune/status');
        const statusResult = await statusResponse.json();
        
        if (statusResult.success && statusResult.is_resource_failure) {
          // 시스템 자원 부족으로 인한 실패
          showNotification(
            'error',
            '시스템 자원 부족',
            `모델 학습에 필요한 시스템 자원이 부족합니다.\n\n원인: ${statusResult.failure_reason}\n\n권장 해결 방법:\n• 다른 프로그램을 종료하여 메모리 확보\n• CPU 모드로 모델 학습 재시도\n• 배치 크기를 더 작게 설정\n• 시스템 재시작 후 재시도`
          );
                 } else {
           // 일반적인 연결 오류 또는 기타 오류
           const errorMessage = error instanceof Error ? error.message : String(error);
           const isConnectionError = errorMessage.includes('fetch') || 
                                    errorMessage.includes('network') ||
                                    errorMessage.includes('connection');
           
           if (isConnectionError) {
             showNotification(
               'error',
               'RAG 서버 연결 오류',
               `RAG 서버에 연결할 수 없습니다.\n\n가능한 원인:\n• RAG 서버가 중지됨 (systemd 서비스 확인)\n• 시스템 자원 부족으로 서버 재시작됨\n• 네트워크 연결 문제\n\n해결 방법:\n• systemctl --user status airun-rag\n• systemctl --user restart airun-rag\n• 시스템 리소스 상태 확인`
             );
           } else {
             showNotification(
               'error',
               '모델 학습 오류',
               `모델 학습 중 오류가 발생했습니다:\n\n${errorMessage}\n\n자세한 내용은 로그를 확인해주세요.`
             );
           }
        }
      } catch (statusError) {
        // 상태 확인도 실패한 경우 (서버가 완전히 중지된 상태)
        showNotification(
          'error',
          'RAG 서버 중지',
          `RAG 서버가 중지되었습니다.\n\n모델 학습 중 시스템 자원 부족으로 인해 서버가 중지되었을 가능성이 높습니다.\n\n해결 방법:\n• systemctl --user restart airun-rag\n• 메모리 사용량 확인 후 재시도\n• CPU 모드로 모델 학습 시도\n• 시스템 재시작 권장`
        );
      }
    }
  };

  const handleCreateDatasetFromRAG = async () => {
    // RAG 서버 연결 상태 확인
    if (!ragServerConnected) {
      showNotification(
        'warning',
        'RAG 서버 연결 필요',
        '컬렉션 데이터셋 생성을 위해 RAG 서버가 필요합니다.\n먼저 RAG 서버를 시작해주세요:\n\nsystemctl --user start airun-rag'
      );
      return;
    }

    try {
      const response = await fetch('/api/rag/datasets', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          method: 'rag-default',
          provider: selectedProvider,
          model: selectedModel
        })
      });
      
      const result = await response.json();
      
      if (result.success) {
        showNotification(
          'success',
          '컬렉션 데이터셋 생성 완료',
          `컬렉션에서 데이터셋이 생성되었습니다!\n\n데이터셋 ID: ${result.dataset_id}\n문서 수: ${result.document_count}개`,
          async () => {
            await loadDatasets(); // 데이터셋 목록 새로고침
          }
        );
      } else {
        // Next.js API route는 flat structure를 반환
        const errorMessage = result.error || '알 수 없는 오류가 발생했습니다.';
        showNotification(
          'error',
          '컬렉션 데이터셋 생성 실패',
          `컬렉션 데이터셋 생성 실패: ${errorMessage}`
        );
      }
    } catch (error) {
      console.error('컬렉션 데이터셋 생성 실패:', error);
      const errorMessage = error instanceof Error ? error.message : String(error);
      showNotification(
        'error',
        '컬렉션 데이터셋 생성 오류',
        `컬렉션 데이터셋 생성 오류: ${errorMessage}`
      );
    }
  };

  // 파일 업로드 데이터셋 생성
  const handleCreateDatasetFromFile = async () => {
    if (!uploadedFile || !newDatasetName) {
      showNotification('warning', '입력 필요', '파일과 데이터셋 이름을 입력해주세요.');
      return;
    }

    // 진행 상태 시작
    setDatasetCreationProgress({
      isCreating: true,
      progress: 20,
      stage: '파일 업로드 중...',
      message: `${uploadedFile.name} 파일을 업로드하고 있습니다.`,
      method: 'file-upload'
    });

    try {
      const formData = new FormData();
      formData.append('file', uploadedFile);
      formData.append('name', newDatasetName);
      formData.append('description', newDatasetDescription);
      formData.append('method', 'file-upload');

      // 파일 처리 단계
      setDatasetCreationProgress(prev => ({
        ...prev,
        progress: 60,
        stage: '파일 분석 중...',
        message: '업로드된 파일을 분석하고 형식을 검증하고 있습니다.'
      }));

      const response = await fetch('/api/rag/datasets', {
        method: 'POST',
        body: formData
      });

      // 데이터셋 생성 단계
      setDatasetCreationProgress(prev => ({
        ...prev,
        progress: 80,
        stage: '데이터셋 생성 중...',
        message: '검증된 데이터를 데이터셋으로 변환하고 있습니다.'
      }));

      const result = await response.json();

      if (result.success) {
        // 완료 진행률 100%
        setDatasetCreationProgress(prev => ({
          ...prev,
          progress: 100,
          stage: '완료!',
          message: `${result.examples_count}개의 예제가 성공적으로 생성되었습니다.`
        }));

        // 잠시 완료 상태 표시 후 초기화
        setTimeout(() => {
          setDatasetCreationProgress({
            isCreating: false,
            progress: 0,
            stage: '',
            message: ''
          });
        }, 1500);

        showNotification(
          'success',
          '파일 데이터셋 생성 완료',
          `파일에서 데이터셋이 생성되었습니다!\n\n데이터셋 ID: ${result.dataset_id}\n예제 수: ${result.examples_count}`,
          async () => {
            await loadDatasets();
            setShowDatasetModal(false);
            resetDatasetForm();
          }
        );
      } else {
        // 실패 시 진행 상태 초기화
        setDatasetCreationProgress({
          isCreating: false,
          progress: 0,
          stage: '',
          message: ''
        });
        showNotification('error', '데이터셋 생성 실패', result.error);
      }
    } catch (error) {
      // 오류 시 진행 상태 초기화
      setDatasetCreationProgress({
        isCreating: false,
        progress: 0,
        stage: '',
        message: ''
      });
      console.error('파일 데이터셋 생성 실패:', error);
      showNotification('error', '데이터셋 생성 오류', `오류: ${error}`);
    }
  };

  // 문서 + LLM 기반 데이터셋 생성
  const handleCreateDatasetFromDocument = async () => {
    if (!uploadedFile || !newDatasetName) {
      showNotification('warning', '입력 필요', '문서 파일과 데이터셋 이름을 입력해주세요.');
      return;
    }

    // 🔍 디버깅: 요청 시작 로그
    console.log('🚀 [프론트엔드] 데이터셋 생성 시작:', {
      fileName: uploadedFile.name,
      fileSize: uploadedFile.size,
      datasetName: newDatasetName,
      description: newDatasetDescription
    });

    // 진행 상태 시작
    setDatasetCreationProgress({
      isCreating: true,
      progress: 10,
      stage: '파일 업로드 중...',
      message: `${uploadedFile.name} 파일을 업로드하고 있습니다.`,
      method: 'document-llm'
    });

    try {
      const formData = new FormData();
      formData.append('document', uploadedFile);
      formData.append('name', newDatasetName);
      formData.append('description', newDatasetDescription);
      formData.append('method', 'document-llm');
      // 프로바이더 정보 추가
      formData.append('provider', selectedProvider);
      formData.append('model', selectedModel);
      // 사용자 지정 QA 개수 추가
      formData.append('target_qa_count', targetQACount.toString());

      console.log('📤 [프론트엔드] API 요청 전송 중...');

      // 파일 업로드 단계
      setDatasetCreationProgress(prev => ({
        ...prev,
        progress: 30,
        stage: '문서 분석 중...',
        message: '업로드된 문서를 분석하고 있습니다.'
      }));

      const response = await fetch('/api/rag/datasets', {
        method: 'POST',
        body: formData
      });

      console.log('📥 [프론트엔드] API 응답 수신:', {
        status: response.status,
        statusText: response.statusText,
        ok: response.ok
      });

      // AI 처리 단계
      setDatasetCreationProgress(prev => ({
        ...prev,
        progress: 60,
        stage: 'AI 질문-답변 생성 중...',
        message: 'AI가 문서를 바탕으로 질문-답변 쌍을 생성하고 있습니다.'
      }));

      const result = await response.json();
      console.log('📊 [프론트엔드] 파싱된 응답:', result);

      // 완료 단계
      setDatasetCreationProgress(prev => ({
        ...prev,
        progress: 90,
        stage: '데이터셋 저장 중...',
        message: '생성된 데이터셋을 저장하고 있습니다.'
      }));

      if (result.success) {
        // 완료 진행률 100%
        setDatasetCreationProgress(prev => ({
          ...prev,
          progress: 100,
          stage: '완료!',
          message: `${result.questions_generated}개의 질문-답변 쌍이 생성되었습니다.`
        }));

        // 잠시 완료 상태 표시 후 초기화
        setTimeout(() => {
          setDatasetCreationProgress({
            isCreating: false,
            progress: 0,
            stage: '',
            message: ''
          });
        }, 1500);

        showNotification(
          'success',
          'LLM 데이터셋 생성 완료',
          `문서에서 AI가 질문-답변 쌍을 생성했습니다!\n\n데이터셋 ID: ${result.dataset_id}\n생성된 질문 수: ${result.questions_generated}`,
          async () => {
            await loadDatasets();
            setShowDatasetModal(false);
            resetDatasetForm();
          }
        );
      } else {
        // 실패 시 진행 상태 초기화
        setDatasetCreationProgress({
          isCreating: false,
          progress: 0,
          stage: '',
          message: ''
        });
        showNotification('error', '데이터셋 생성 실패', result.error);
      }
    } catch (error) {
      // 오류 시 진행 상태 초기화
      setDatasetCreationProgress({
        isCreating: false,
        progress: 0,
        stage: '',
        message: ''
      });
      console.error('문서 기반 데이터셋 생성 실패:', error);
      showNotification('error', '데이터셋 생성 오류', `오류: ${error}`);
    }
      };
  
  // 데이터셋 삭제 처리
  const handleDeleteDataset = async (datasetId: string, datasetName: string) => {
    // 삭제 확인 모달 표시
    showNotification(
      'warning',
      '데이터셋 삭제 확인',
      `정말로 "${datasetName}" 데이터셋을 삭제하시겠습니까?\n\n삭제된 데이터셋은 복구할 수 없습니다.\n이 데이터셋으로 훈련된 모델은 영향을 받지 않습니다.`,
      async () => {
        try {
          const response = await fetch(`/api/rag/datasets`, {
            method: 'DELETE',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ datasetId: datasetId })
          });

          const result = await response.json();

          if (result.success) {
            showNotification(
              'success',
              '데이터셋 삭제 완료',
              `"${datasetName}" 데이터셋이 성공적으로 삭제되었습니다.`,
              async () => {
                await loadDatasets(); // 데이터셋 목록 새로고침
              }
            );
          } else {
            showNotification(
              'error',
              '데이터셋 삭제 실패',
              `데이터셋 삭제 실패: ${result.error}`
            );
          }
        } catch (error) {
          console.error('데이터셋 삭제 실패:', error);
          showNotification(
            'error',
            '데이터셋 삭제 오류',
            `데이터셋 삭제 중 오류가 발생했습니다: ${error}`
          );
        }
      },
      '삭제',
      true // showCancel = true
    );
  };

  // 데이터셋 상세 정보 조회 함수
  const handleViewDatasetDetails = async (dataset: TrainingDataset) => {
    setSelectedDatasetForDetail(dataset);
    setShowDatasetDetailModal(true);
    setLoadingDatasetDetails(true);
    
    try {
      let response;
      let result;
      
      // 데이터셋 타입에 따른 처리 - 모든 데이터셋은 RAG 서버에서 관리
      if (dataset.type === 'feedback') {
        // 피드백 데이터셋은 특별한 엔드포인트 사용
        response = await fetch(`/api/rag/datasets/feedback_live`);
        result = await response.json();

        if (result.success && result.dataset) {
          setDatasetDetails(result.dataset);
        } else {
          throw new Error(result.error || '피드백 데이터셋을 불러올 수 없습니다.');
        }
      } else {
        // 모든 다른 데이터셋 (RAG, 평가 데이터셋 포함)은 동일한 RAG 서버 API 사용
        response = await fetch(`/api/rag/datasets/${dataset.id}`);
        result = await response.json();

        if (result.success && result.dataset) {
          setDatasetDetails(result.dataset);
        } else {
          throw new Error(result.error || '데이터셋을 찾을 수 없습니다');
        }
      }
    } catch (error) {
      console.error('데이터셋 상세 정보 조회 오류:', error);
      showNotification(
        'error',
        '조회 실패',
        error instanceof Error ? error.message : '데이터셋 상세 정보를 불러오는 중 오류가 발생했습니다.'
      );
    } finally {
      setLoadingDatasetDetails(false);
    }
  };

  // 데이터셋 폼 리셋
  const resetDatasetForm = () => {
    setSelectedDatasetMethod('');
    setUploadedFile(null);
    setNewDatasetName('');
    setNewDatasetDescription('');
    setTargetQACount(25); // QA 개수도 기본값으로 리셋
    
    // 프로바이더와 모델을 기본값으로 재설정
    if (availableProviders.length > 0) {
      const firstProvider = availableProviders[0];
      setSelectedProvider(firstProvider.key);
      
      if (providers[firstProvider.key]?.models?.length > 0) {
        const firstModel = providers[firstProvider.key].models[0];
        setSelectedModel(firstModel.id || firstModel.name);
      }
    }
  };

  // 드래그앤드롭 핸들러 함수들
  const handleDragEnter = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragging(true);
  };

  const handleDragLeave = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragging(false);
  };

  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
  };

  const handleDrop = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragging(false);
    
    const files = e.dataTransfer.files;
    if (files && files.length > 0) {
      const file = files[0];
      
      // 파일 형식 검증
      const isEvaluationDataset = selectedDatasetMethod === 'evaluation-universal' || selectedDatasetMethod === 'evaluation-domain';
      const isValidFile = (selectedDatasetMethod === 'file-upload' || isEvaluationDataset)
        ? file.name.match(/\.(json|csv|txt)$/i) || file.type.match(/^(application\/json|text\/csv|text\/plain)$/)
        : file.name.match(/\.(txt|md|pdf|docx|doc|hwp|hwpx|pptx|ppt)$/i) || file.type.match(/^(text\/plain|text\/markdown|application\/pdf|application\/vnd\.openxmlformats-officedocument\.wordprocessingml\.document|application\/msword|application\/vnd\.ms-powerpoint|application\/vnd\.openxmlformats-officedocument\.presentationml\.presentation)$/);

      if (isValidFile) {
        setUploadedFile(file);
      } else {
        const fileTypes = (selectedDatasetMethod === 'file-upload' || isEvaluationDataset)
          ? 'JSON, CSV, TXT'
          : 'TXT, MD, PDF, DOCX, DOC, HWP, HWPX, PPTX, PPT';
        showNotification('warning', '지원하지 않는 파일 형식', `${fileTypes} 파일만 업로드 가능합니다.`);
      }
    }
  };

  // 데이터셋 생성 실행
  // 평가용 데이터셋 생성 함수
  const handleCreateEvaluationDataset = async (evaluationType: 'universal' | 'domain') => {
    if (!uploadedFile || !newDatasetName) {
      showNotification('warning', '입력 필요', '파일과 데이터셋 이름을 입력해주세요.');
      return;
    }

    // 진행 상태 시작
    setDatasetCreationProgress({
      isCreating: true,
      progress: 20,
      stage: '평가 데이터셋 업로드 중...',
      message: `${uploadedFile.name} 파일을 업로드하고 있습니다.`,
      method: `evaluation-${evaluationType}`
    });

    try {
      const formData = new FormData();
      formData.append('file', uploadedFile);
      formData.append('name', newDatasetName);
      formData.append('description', newDatasetDescription || `${evaluationType === 'universal' ? '범용' : '도메인'} 평가 데이터셋`);
      formData.append('type', `${evaluationType}_eval`); // universal_eval 또는 domain_eval

      setDatasetCreationProgress(prev => ({
        ...prev,
        progress: 50,
        stage: '평가 데이터셋 처리 중...',
        message: '업로드된 파일을 평가용 데이터셋으로 변환하고 있습니다.'
      }));

      // RAG 서버의 평가데이터셋 업로드 엔드포인트 사용 (기존 datasets API를 통해)
      const evaluationFormData = new FormData();
      evaluationFormData.append('file', uploadedFile);
      evaluationFormData.append('name', newDatasetName);
      evaluationFormData.append('description', newDatasetDescription || `${evaluationType === 'universal' ? '범용' : '도메인'} 평가 데이터셋`);
      evaluationFormData.append('method', `evaluation-${evaluationType}`);

      const response = await fetch('/api/rag/datasets', {
        method: 'POST',
        body: evaluationFormData
      });

      const data = await response.json();

      if (response.ok) {
        setDatasetCreationProgress(prev => ({
          ...prev,
          progress: 100,
          stage: '완료',
          message: '평가 데이터셋이 성공적으로 생성되었습니다.'
        }));

        showNotification('success', '데이터셋 생성 완료', `${evaluationType === 'universal' ? '범용' : '도메인'} 평가 데이터셋이 성공적으로 생성되었습니다.`);

        // 상태 초기화
        setTimeout(() => {
          setSelectedDatasetMethod('');
          setNewDatasetName('');
          setNewDatasetDescription('');
          setUploadedFile(null);
          setDatasetCreationProgress({
            isCreating: false,
            progress: 0,
            stage: '',
            message: ''
          });
          loadDatasets(); // 데이터셋 목록 새로고침
        }, 2000);

      } else {
        throw new Error(data.error || '평가 데이터셋 생성에 실패했습니다.');
      }
    } catch (error: any) {
      console.error('평가 데이터셋 생성 오류:', error);
      setDatasetCreationProgress({
        isCreating: false,
        progress: 0,
        stage: '',
        message: ''
      });
      showNotification('error', '데이터셋 생성 실패', error.message);
    }
  };

  const handleCreateDataset = async () => {
    // 필수값 검증
    if (!newDatasetName.trim()) {
      showNotification('warning', '필수 정보 누락', '데이터셋 이름을 입력해주세요.');
      return;
    }

    // 메서드별 추가 검증
    if ((selectedDatasetMethod === 'file-upload' || selectedDatasetMethod === 'document-llm' ||
         selectedDatasetMethod === 'evaluation-universal' || selectedDatasetMethod === 'evaluation-domain') && !uploadedFile) {
      const fileType = selectedDatasetMethod === 'file-upload' ? '데이터셋 파일' :
                      selectedDatasetMethod === 'document-llm' ? '문서 파일' : '평가 데이터셋 파일';
      showNotification('warning', '필수 정보 누락', `${fileType}을 업로드해주세요.`);
      return;
    }

    switch (selectedDatasetMethod) {
      case 'file-upload':
        await handleCreateDatasetFromFile();
        break;
      case 'document-llm':
        await handleCreateDatasetFromDocument();
        break;
      case 'evaluation-universal':
        await handleCreateEvaluationDataset('universal');
        break;
      case 'evaluation-domain':
        await handleCreateEvaluationDataset('domain');
        break;
      default:
        showNotification('warning', '방법 선택 필요', '데이터셋 생성 방법을 선택해주세요.');
    }
  };

  const handleStartEvaluation = async () => {
    // RAG 서버 연결 상태 확인
    if (!ragServerConnected) {
      showNotification(
        'warning',
        'RAG 서버 연결 필요',
        'RAG 서버가 실행되지 않았습니다.\n먼저 RAG 서버를 시작해주세요.'
      );
      return;
    }

    // 디버깅용 로그 추가
    console.log('전체 모델 평가 시작');
    console.log('models 배열:', models);
    console.log('models.length:', models.length);

    if (models.length === 0) {
      showNotification(
        'warning',
        '평가할 모델 없음',
        '평가할 모델이 없습니다. 먼저 모델을 로드해주세요.'
      );
      return;
    }

    try {
      setEvaluationProgress({
        isEvaluating: true,
        progress: 0,
        totalModels: models.length,
        completedModels: 0,
        currentModel: '평가 준비 중...'
      });

      // 모든 모델 평가 실행
      for (let i = 0; i < models.length; i++) {
        const model = models[i];
        console.log(`모델 ${i+1}/${models.length} 평가 시작:`, model.name, '(ID:', model.id, ')');
        
        setEvaluationProgress(prev => ({
          ...prev,
          currentModel: model.name,
          progress: Math.floor((i / models.length) * 100)
        }));

        const startTime = Date.now();
        const response = await fetch('/api/rag/models', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            action: 'evaluate-model',
            modelId: model.id,
            datasetId: model.dataset_id,  // 모델의 dataset_id 추가 (학습용)
            evaluationDatasetId: selectedEvaluationDataset || undefined,  // 평가용 데이터셋 ID 추가
            evaluation_mode: evaluationMode  // 평가 모드 추가
          })
        });

        const result = await response.json();
        const duration = Date.now() - startTime;
        console.log(`모델 ${model.name} 평가 완료 (${duration}ms):`, result);
        
        if (result.success) {
                  // 모델 성능 업데이트
        setModels(prevModels => prevModels.map(m => 
          m.id === model.id 
            ? {
                ...m,
                performance: {
                  precision: result.metrics.precision || m.performance.precision,
                  recall: result.metrics.recall || m.performance.recall,
                  f1Score: result.metrics.f1_score || m.performance.f1Score
                },
                evaluated: true,
                evaluationDate: Date.now()
              }
            : m
        ));
        }

        setEvaluationProgress(prev => ({
          ...prev,
          completedModels: i + 1,
          progress: Math.floor(((i + 1) / models.length) * 100)
        }));
      }

      setEvaluationProgress(prev => ({
        ...prev,
        isEvaluating: false,
        progress: 100,
        currentModel: '평가 완료'
      }));

      showNotification(
        'success',
        '모델 평가 완료',
        `${models.length}개 모델의 성능 평가가 완료되었습니다.`,
        async () => {
          // 모든 모델 평가 완료 후 모델 목록 새로고침하여 최신 성능 지표 반영
          await loadModels();
        }
      );

      // 전체 평가 완료 후 자동으로 모델 목록 새로고침
      setTimeout(() => {
        loadModels();
      }, 1000);

    } catch (error) {
      console.error('모델 평가 실패:', error);
      setEvaluationProgress(prev => ({
        ...prev,
        isEvaluating: false,
        currentModel: '평가 실패'
      }));
      showNotification(
        'error',
        '모델 평가 실패',
        `모델 평가 중 오류가 발생했습니다: ${error}`
      );
    }
  };

  // 차트 데이터 생성 함수들
  const prepareBarChartData = () => {
    const evaluatedModels = models.filter(model => model.evaluated);
    if (evaluatedModels.length === 0) return null;

    const modelNames = evaluatedModels.map(model => model.name);

    const precisionData = evaluatedModels.map(model => {
      try {
        const evalData = model.evaluation_results || model.evaluationResults;
        // 실제 저장된 데이터 구조에 맞춰 수정
        return model.performance?.precision ||                 // 파인튜닝 모델 performance
               evalData?.precision ||                          // 평가 결과 최상위
               evalData?.evaluation_metrics?.precision ||      // 파인튜닝 모델 evaluation_metrics
               evalData?.metrics?.precision ||                 // 기본 모델 metrics
               0;
      } catch { return 0; }
    });

    const recallData = evaluatedModels.map(model => {
      try {
        const evalData = model.evaluation_results || model.evaluationResults;
        return model.performance?.recall ||                    // 파인튜닝 모델 performance
               evalData?.recall ||                             // 평가 결과 최상위
               evalData?.evaluation_metrics?.recall ||         // 파인튜닝 모델 evaluation_metrics
               evalData?.metrics?.recall ||                    // 기본 모델 metrics
               0;
      } catch { return 0; }
    });

    const f1Data = evaluatedModels.map(model => {
      try {
        const evalData = model.evaluation_results || model.evaluationResults;
        return model.performance?.f1Score ||                  // 파인튜닝 모델 performance
               evalData?.f1_score ||                          // 평가 결과 최상위
               evalData?.evaluation_metrics?.f1_score ||      // 파인튜닝 모델 evaluation_metrics
               evalData?.metrics?.f1_score ||                 // 기본 모델 metrics
               0;
      } catch { return 0; }
    });

    const ndcgData = evaluatedModels.map(model => {
      try {
        const evalData = model.evaluation_results || model.evaluationResults;
        // 정보 검색 성능 - 핵심 지표
        return evalData?.evaluation_metrics?.retrieval_metrics?.ndcg_at_5 ||
               evalData?.metrics?.retrieval_metrics?.ndcg_at_5 ||
               evalData?.retrieval_metrics?.ndcg_at_5 || 0;
      } catch { return 0; }
    });

    const mrrData = evaluatedModels.map(model => {
      try {
        const evalData = model.evaluation_results || model.evaluationResults;
        // 정보 검색 성능 - 핵심 지표
        return evalData?.evaluation_metrics?.retrieval_metrics?.mrr ||
               evalData?.metrics?.retrieval_metrics?.mrr ||
               evalData?.retrieval_metrics?.mrr || 0;
      } catch { return 0; }
    });

    const recallAt1Data = evaluatedModels.map(model => {
      try {
        const evalData = model.evaluation_results || model.evaluationResults;
        // 정보 검색 성능 - 핵심 지표
        return evalData?.evaluation_metrics?.retrieval_metrics?.recall_at_1 ||
               evalData?.metrics?.retrieval_metrics?.recall_at_1 ||
               evalData?.retrieval_metrics?.recall_at_1 || 0;
      } catch { return 0; }
    });

    return {
      labels: modelNames,
      datasets: [
        // 정보 검색 성능 지표를 앞쪽에 배치하여 강조
        {
          label: '🎯 nDCG@5 (핵심지표)',
          data: ndcgData,
          backgroundColor: 'rgba(168, 85, 247, 0.8)',
          borderColor: 'rgba(168, 85, 247, 1)',
          borderWidth: 2,
        },
        {
          label: '🎯 MRR (핵심지표)',
          data: mrrData,
          backgroundColor: 'rgba(236, 72, 153, 0.8)',
          borderColor: 'rgba(236, 72, 153, 1)',
          borderWidth: 2,
        },
        {
          label: '🎯 Recall@1 (핵심지표)',
          data: recallAt1Data,
          backgroundColor: 'rgba(34, 197, 94, 0.8)',
          borderColor: 'rgba(34, 197, 94, 1)',
          borderWidth: 2,
        },
        // 분류 성능 지표
        {
          label: '정밀도',
          data: precisionData,
          backgroundColor: 'rgba(59, 130, 246, 0.5)',
          borderColor: 'rgba(59, 130, 246, 1)',
          borderWidth: 1,
        },
        {
          label: '재현율',
          data: recallData,
          backgroundColor: 'rgba(16, 185, 129, 0.5)',
          borderColor: 'rgba(16, 185, 129, 1)',
          borderWidth: 1,
        },
        {
          label: 'F1 Score',
          data: f1Data,
          backgroundColor: 'rgba(245, 158, 11, 0.5)',
          borderColor: 'rgba(245, 158, 11, 1)',
          borderWidth: 1,
        },
      ],
    };
  };

  const prepareRadarChartData = () => {
    const evaluatedModels = models.filter(model => model.evaluated);
    if (evaluatedModels.length === 0) return null;

    const metricsLabels = ['정밀도', '재현율', 'F1 Score', '🎯 nDCG@5 (핵심)', '🎯 MRR (핵심)', '🎯 Recall@1 (핵심)'];

    const datasets = evaluatedModels.map((model, index) => {
      const colors = [
        'rgba(59, 130, 246, 0.3)', // 파란색
        'rgba(16, 185, 129, 0.3)', // 초록색
        'rgba(245, 158, 11, 0.3)', // 주황색
        'rgba(168, 85, 247, 0.3)', // 보라색
        'rgba(236, 72, 153, 0.3)', // 핑크색
        'rgba(34, 197, 94, 0.3)', // 연두색
      ];

      const borderColors = [
        'rgba(59, 130, 246, 1)',
        'rgba(16, 185, 129, 1)',
        'rgba(245, 158, 11, 1)',
        'rgba(168, 85, 247, 1)',
        'rgba(236, 72, 153, 1)',
        'rgba(34, 197, 94, 1)',
      ];

      try {
        const evalData = model.evaluation_results || model.evaluationResults;
        const data = [
          // 정밀도
          model.performance?.precision || evalData?.precision || evalData?.evaluation_metrics?.precision || evalData?.metrics?.precision || 0,
          // 재현율
          model.performance?.recall || evalData?.recall || evalData?.evaluation_metrics?.recall || evalData?.metrics?.recall || 0,
          // F1 스코어
          model.performance?.f1Score || evalData?.f1_score || evalData?.evaluation_metrics?.f1_score || evalData?.metrics?.f1_score || 0,
          // nDCG@5
          evalData?.evaluation_metrics?.retrieval_metrics?.ndcg_at_5 || evalData?.metrics?.retrieval_metrics?.ndcg_at_5 || evalData?.retrieval_metrics?.ndcg_at_5 || 0,
          // MRR
          evalData?.evaluation_metrics?.retrieval_metrics?.mrr || evalData?.metrics?.retrieval_metrics?.mrr || evalData?.retrieval_metrics?.mrr || 0,
          // Recall@1
          evalData?.evaluation_metrics?.retrieval_metrics?.recall_at_1 || evalData?.metrics?.retrieval_metrics?.recall_at_1 || evalData?.retrieval_metrics?.recall_at_1 || 0,
        ];

        return {
          label: model.name,
          data: data,
          backgroundColor: colors[index % colors.length],
          borderColor: borderColors[index % borderColors.length],
          borderWidth: 2,
          pointBackgroundColor: borderColors[index % borderColors.length],
          pointBorderColor: '#fff',
          pointHoverBackgroundColor: '#fff',
          pointHoverBorderColor: borderColors[index % borderColors.length],
        };
      } catch {
        return {
          label: model.name,
          data: [0, 0, 0, 0, 0, 0],
          backgroundColor: colors[index % colors.length],
          borderColor: borderColors[index % borderColors.length],
          borderWidth: 2,
        };
      }
    });

    return {
      labels: metricsLabels,
      datasets: datasets,
    };
  };

  const handleEvaluateModel = async (modelId: string) => {
    if (!ragServerConnected) {
      showNotification(
        'warning',
        'RAG 서버 연결 필요',
        'RAG 서버가 실행되지 않았습니다.'
      );
      return;
    }

    try {
      const model = models.find(m => m.id === modelId);
      if (!model) return;

      setEvaluationProgress({
        isEvaluating: true,
        progress: 0,
        totalModels: 1,
        completedModels: 0,
        currentModel: model.name
      });

      const response = await fetch('/api/rag/models', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          action: 'evaluate-model',
          modelId: modelId,
          datasetId: models.find(m => m.id === modelId)?.dataset_id,  // 모델의 dataset_id 추가
          evaluation_mode: evaluationMode  // 평가 모드 추가
        })
      });

      const result = await response.json();
      
      if (result.success) {
        // 해당 모델 성능 업데이트
        setModels(prevModels => prevModels.map(m => 
          m.id === modelId 
            ? {
                ...m,
                performance: {
                  precision: result.metrics.precision || m.performance.precision,
                  recall: result.metrics.recall || m.performance.recall,
                  f1Score: result.metrics.f1_score || m.performance.f1Score
                },
                evaluated: true,
                evaluationDate: Date.now()
              }
            : m
        ));

        setEvaluationProgress({
          isEvaluating: false,
          progress: 100,
          totalModels: 1,
          completedModels: 1,
          currentModel: '평가 완료'
        });

        showNotification(
          'success',
          '모델 평가 완료',
          `${model.name} 모델의 성능 평가가 완료되었습니다.\n\nF1 Score: ${(result.metrics.f1_score * 100).toFixed(1)}%\n정밀도: ${(result.metrics.precision * 100).toFixed(1)}%\n재현율: ${(result.metrics.recall * 100).toFixed(1)}%`,
          async () => {
            // 모델 목록 새로고침하여 최신 성능 지표 반영
            await loadModels();
          }
        );
        
        // 평가 완료 후 자동으로 모델 목록 새로고침
        setTimeout(() => {
          loadModels();
        }, 1000);
      } else {
        setEvaluationProgress({
          isEvaluating: false,
          progress: 0,
          totalModels: 1,
          completedModels: 0,
          currentModel: '평가 실패'
        });
        
        showNotification(
          'error',
          '모델 평가 실패',
          `모델 평가 실패: ${result.error}`
        );
      }
    } catch (error) {
      console.error('개별 모델 평가 실패:', error);
      setEvaluationProgress({
        isEvaluating: false,
        progress: 0,
        totalModels: 1,
        completedModels: 0,
        currentModel: '평가 실패'
      });
      
      showNotification(
        'error',
        '모델 평가 오류',
        `모델 평가 중 오류가 발생했습니다: ${error}`
      );
    }
  };

  if (!isAuthenticated || user?.role !== 'admin') {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-gray-900 dark:text-white mb-4">
            접근 권한이 없습니다
          </h1>
          <p className="text-gray-600 dark:text-gray-400">
            이 페이지는 관리자만 접근할 수 있습니다.
          </p>
        </div>
      </div>
    );
  }

  return (
    <div className="h-full bg-[var(--body-bg)] flex">
      {/* 사이드바 */}
      <AdminSidebar
        activeTab={activeTab}
        setActiveTab={setActiveTab}
        onRefreshModels={loadModels}
      />

      {/* 메인 컨텐츠 영역 */}
      <div className="flex-1 flex flex-col min-w-0 bg-[var(--body-bg)]">
        {/* 헤더 - 임베딩 모델 탭에서만 표시 */}
        {(['overview', 'datasets', 'training', 'evaluation'] as (keyof FinetuneTabs)[]).includes(activeTab) && (
          <div className="flex-none px-6 py-4">
            <div className="flex items-center justify-between">
              <div className="flex items-center space-x-4">
                <div>
                  <h3 className="font-bold text-gray-900 dark:text-white">
                    {activeTab === 'overview' && '모델 관리'}
                    {activeTab === 'datasets' && '데이터셋 관리'}
                    {activeTab === 'training' && '모델 학습 관리'}
                    {activeTab === 'evaluation' && '성능 평가'}
                  </h3>
                  <p className="text-gray-600 dark:text-gray-400 mt-1">
                    {activeTab === 'overview' && '현재 사용 가능한 임베딩 모델 현황 및 관리'}

  
                    {activeTab === 'datasets' && '모델 학습을 위한 훈련 데이터셋 생성 및 관리'}
                    {activeTab === 'training' && '도메인 특화 임베딩 모델 모델 학습 실행'}
                    {activeTab === 'evaluation' && '모델 성능 평가 및 비교 분석'}
                  </p>
                </div>
              </div>
              
            </div>
          </div>
        )}

        {/* 컨텐츠 스크롤 영역 */}
        <div className="flex-1 overflow-y-auto bg-[var(--body-bg)]">

      {/* 모델 정보 탭 */}
      {activeTab === 'overview' && (
          <div className="p-6 space-y-6">

            {/* 모델 정보 설명 카드 */}
            <div className="bg-gradient-to-r from-blue-50 to-indigo-50 dark:from-blue-900/20 dark:to-indigo-900/20 rounded-xl p-6 border border-blue-200 dark:border-blue-700">
              <div className="flex items-start space-x-4">
                <div className="flex-shrink-0">
                  <div className="w-12 h-12 bg-blue-100 dark:bg-blue-800 rounded-lg flex items-center justify-center">
                    <span className="text-2xl">📊</span>
                  </div>
                </div>
                <div className="flex-1">
                  <h3 className="text-lg font-semibold text-blue-900 dark:text-blue-200 mb-2">
                    모델 현황 및 관리
                  </h3>
                  <p className="text-blue-700 dark:text-blue-300 text-sm leading-relaxed">
                    현재 시스템에서 사용 가능한 모든 임베딩 모델의 정보를 확인하고 관리할 수 있습니다. 
                    기본 모델과 학습된 모델의 성능 지표를 비교하고, 원하는 모델을 활성화하여 RAG 시스템의 품질을 향상시키세요.
                  </p>
                  <div className="flex flex-wrap gap-2 mt-3">
                    <span className="px-2 py-1 bg-blue-100 dark:bg-blue-800 text-blue-700 dark:text-blue-300 text-xs rounded-full">모델 상태 확인</span>
                    <span className="px-2 py-1 bg-blue-100 dark:bg-blue-800 text-blue-700 dark:text-blue-300 text-xs rounded-full">성능 지표 비교</span>
                    <span className="px-2 py-1 bg-blue-100 dark:bg-blue-800 text-blue-700 dark:text-blue-300 text-xs rounded-full">모델 활성화</span>
                  </div>
                </div>
              </div>
            </div>

            {/* RAG 서비스 재시작 안내 */}
            <div className="bg-gradient-to-r from-amber-50 to-orange-50 dark:from-amber-900/20 dark:to-orange-900/20 rounded-xl p-4 border border-amber-200 dark:border-amber-700">
              <div className="flex items-start space-x-3">
                <div className="flex-shrink-0">
                  <div className="w-8 h-8 bg-amber-100 dark:bg-amber-800 rounded-lg flex items-center justify-center">
                    <span className="text-lg">ℹ️</span>
                  </div>
                </div>
                <div className="flex-1">
                  <h4 className="text-sm font-semibold text-amber-900 dark:text-amber-200 mb-1">
                    새 모델 적용을 위해서는 서비스 재시작이 필요합니다.
                  </h4>
                  <p className="text-amber-700 dark:text-amber-300 text-xs leading-relaxed">
                    파인튜닝된 모델을 활성화한 후에는 변경사항을 적용하기 위해 RAG 서비스를 재시작해야 합니다. 
                  </p>
                </div>
              </div>
            </div>

            {/* 모델 평가 액션 바 */}
            <div className="bg-[var(--card-bg)] rounded-xl shadow-sm border border-[var(--card-border)] p-4">
              <div className="flex items-center justify-between">
                <div className="flex items-center space-x-4">
                  <div className="text-sm text-gray-600 dark:text-gray-400">
                    {models.filter(m => m.evaluated).length} / {models.length} 모델 평가 완료
                  </div>
                  {evaluationProgress.isEvaluating && (
                    <div className="flex items-center space-x-2">
                      <div className="w-2 h-2 bg-orange-500 rounded-full animate-pulse" />
                      <span className="text-sm text-orange-600 dark:text-orange-400">
                        평가 진행 중: {evaluationProgress.currentModel}
                      </span>
                    </div>
                  )}
                </div>
                
                <div className="flex items-center space-x-2">
                  {!ragServerConnected && (
                    <span className="text-xs text-yellow-600 dark:text-yellow-400 mr-2">
                      ⚠️ RAG 서버 연결 필요
                    </span>
                  )}
                  <button
                    onClick={handleStartEvaluation}
                    disabled={!ragServerConnected || evaluationProgress.isEvaluating}
                    className={`px-4 py-2 text-sm font-medium rounded-lg transition-colors ${
                      !ragServerConnected || evaluationProgress.isEvaluating
                        ? 'bg-gray-300 text-gray-500 dark:bg-gray-600 dark:text-gray-400 cursor-not-allowed'
                        : 'bg-blue-600 text-white hover:bg-blue-700 dark:bg-blue-600 dark:hover:bg-blue-700'
                    }`}
                  >
                    {evaluationProgress.isEvaluating ? '⏳ 평가 진행 중...' : '🧪 모든 모델 평가'}
                  </button>
                </div>
              </div>
              
              {/* 평가 진행률 표시 */}
              {evaluationProgress.isEvaluating && (
                <div className="mt-3">
                  <div className="flex justify-between text-xs text-gray-500 mb-1">
                    <span>평가 진행률</span>
                    <span>{evaluationProgress.completedModels} / {evaluationProgress.totalModels} 완료</span>
                  </div>
                  <div className="w-full bg-gray-200 dark:bg-gray-700 rounded-full h-2">
                    <div 
                      className="bg-blue-600 h-2 rounded-full transition-all duration-300"
                      style={{ width: `${evaluationProgress.progress}%` }}
                    />
                  </div>
                </div>
              )}
            </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              {models.map(model => (
                <div key={model.id} className={`bg-[var(--card-bg)] rounded-xl shadow-sm border p-6 ${
                  model.isActive ? 'border-green-500 bg-green-50 dark:bg-green-900/20' : 'border-[var(--card-border)]'
                }`}>
                  <div className="flex items-center justify-between mb-4">
                    <div className="flex items-center space-x-3">
                      <div className={`w-3 h-3 rounded-full ${
                        model.isActive ? 'bg-green-500' : 'bg-gray-300'
                      }`} />
                      <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                        {model.name}
                      </h3>
                    </div>
                    <span className={`px-2 py-1 text-xs font-medium rounded-full whitespace-nowrap ${
                      model.type === 'base' 
                        ? 'bg-gray-100 text-gray-700 dark:bg-gray-700 dark:text-gray-300'
                        : 'bg-blue-100 text-blue-700 dark:bg-blue-900 dark:text-blue-300'
                    }`}>
                      {model.type === 'base' ? '기본' : '파인튜닝'}
                    </span>
                  </div>
                  
                  <p className="text-sm text-gray-600 dark:text-gray-400 mb-4">
                    {model.description}
                  </p>
                  
                  <div className="space-y-2 mb-4">
                    <div className="flex justify-between text-sm">
                      <span className="text-gray-500">버전</span>
                      <span className="font-medium">{model.version}</span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span className="text-gray-500">크기</span>
                      <span className="font-medium">{model.size}</span>
                    </div>
                  </div>

                  {/* 성능 평가 결과 섹션 */}
                  <div className="mb-4">
                    <div className="flex items-center justify-between mb-3">
                      <h4 className="text-sm font-medium text-gray-700 dark:text-gray-300">
                        📊 성능 지표
                      </h4>
                      {model.evaluated && model.evaluationDate && (
                        <span className="text-xs text-gray-500">
                          {new Date(model.evaluationDate).toLocaleDateString('ko-KR')} 평가
                        </span>
                      )}
                    </div>
                    
                    {/* 성능 우수 모델 표시 */}
                    {model.evaluated && model.performance.f1Score >= 0.8 && (
                      <div className="mb-3 p-2 bg-gradient-to-r from-green-50 to-emerald-50 dark:from-green-900/20 dark:to-emerald-900/20 border border-green-200 dark:border-green-700 rounded-lg">
                        <div className="flex items-center space-x-2">
                          <span className="text-green-600 dark:text-green-400">🏆</span>
                          <span className="text-sm font-medium text-green-800 dark:text-green-200">
                            우수 성능 모델
                          </span>
                          <span className="px-2 py-0.5 text-xs bg-green-100 text-green-700 dark:bg-green-900 dark:text-green-300 rounded-full">
                            권장
                          </span>
                        </div>
                        <div className="text-xs text-green-700 dark:text-green-300 mt-1">
                          F1 Score {(model.performance.f1Score * 100).toFixed(1)}%로 높은 성능을 보입니다
                        </div>
                      </div>
                    )}
                    
                    <div className="grid grid-cols-3 gap-3">
                      <div className="bg-gray-50 dark:bg-gray-700/50 rounded-lg p-2">
                        <div className="text-xs text-gray-500 dark:text-gray-400 mb-1">정밀도</div>
                        <div className={`text-sm font-semibold ${
                          !model.evaluated 
                            ? 'text-gray-500' 
                            : model.performance.precision >= 0.8 
                              ? 'text-green-600' 
                              : model.performance.precision >= 0.6 
                                ? 'text-yellow-600' 
                                : 'text-red-600'
                        }`}>
                          {!model.evaluated ? '미평가' : (model.performance.precision * 100).toFixed(1) + '%'}
                        </div>
                      </div>
                      
                      <div className="bg-gray-50 dark:bg-gray-700/50 rounded-lg p-2">
                        <div className="text-xs text-gray-500 dark:text-gray-400 mb-1">재현율</div>
                        <div className={`text-sm font-semibold ${
                          !model.evaluated 
                            ? 'text-gray-500' 
                            : model.performance.recall >= 0.8 
                              ? 'text-green-600' 
                              : model.performance.recall >= 0.6 
                                ? 'text-yellow-600' 
                                : 'text-red-600'
                        }`}>
                          {!model.evaluated ? '미평가' : (model.performance.recall * 100).toFixed(1) + '%'}
                        </div>
                      </div>
                      
                      <div className="bg-gray-50 dark:bg-gray-700/50 rounded-lg p-2">
                        <div className="text-xs text-gray-500 dark:text-gray-400 mb-1">F1 Score</div>
                        <div className={`text-sm font-semibold ${
                          !model.evaluated 
                            ? 'text-gray-500' 
                            : model.performance.f1Score >= 0.8 
                              ? 'text-green-600' 
                              : model.performance.f1Score >= 0.6 
                                ? 'text-yellow-600' 
                                : 'text-red-600'
                        }`}>
                          {!model.evaluated ? '미평가' : (model.performance.f1Score * 100).toFixed(1) + '%'}
                        </div>
                      </div>
                    </div>

                    {/* IR 메트릭 추가 */}
                    {model.evaluated && (
                      <div className="mt-3">
                        <div className="text-xs text-gray-500 dark:text-gray-400 mb-2">🎯 검색 성능 지표</div>
                        <div className="grid grid-cols-3 gap-2">
                          <div className="bg-blue-50 dark:bg-blue-900/20 rounded-lg p-2">
                            <div className="text-xs text-blue-600 dark:text-blue-400 mb-1">nDCG@5</div>
                            <div className={`text-sm font-semibold ${
                              (() => {
                                try {
                                  const evalData = model.evaluation_results || model.evaluationResults;
                                  const ndcg = evalData?.retrieval_metrics?.ndcg_at_5 || evalData?.metrics?.retrieval_metrics?.ndcg_at_5;
                                  if (typeof ndcg !== 'number') return 'text-gray-500';
                                  return ndcg >= 0.8 ? 'text-green-600'
                                       : ndcg >= 0.6 ? 'text-yellow-600'
                                       : 'text-red-600';
                                } catch { return 'text-gray-500'; }
                              })()
                            }`}>
                              {(() => {
                                try {
                                  const evalData = model.evaluation_results || model.evaluationResults;
                                  const ndcg = evalData?.retrieval_metrics?.ndcg_at_5 || evalData?.metrics?.retrieval_metrics?.ndcg_at_5;
                                  return typeof ndcg === 'number' ? ndcg.toFixed(3) : '미평가';
                                } catch { return '미평가'; }
                              })()}
                            </div>
                          </div>

                          <div className="bg-purple-50 dark:bg-purple-900/20 rounded-lg p-2">
                            <div className="text-xs text-purple-600 dark:text-purple-400 mb-1">MRR</div>
                            <div className={`text-sm font-semibold ${
                              (() => {
                                try {
                                  const evalData = model.evaluation_results || model.evaluationResults;
                                  const mrr = evalData?.retrieval_metrics?.mrr || evalData?.metrics?.retrieval_metrics?.mrr;
                                  if (typeof mrr !== 'number') return 'text-gray-500';
                                  return mrr >= 0.8 ? 'text-green-600'
                                       : mrr >= 0.6 ? 'text-yellow-600'
                                       : 'text-red-600';
                                } catch { return 'text-gray-500'; }
                              })()
                            }`}>
                              {(() => {
                                try {
                                  const evalData = model.evaluation_results || model.evaluationResults;
                                  const mrr = evalData?.retrieval_metrics?.mrr || evalData?.metrics?.retrieval_metrics?.mrr;
                                  return typeof mrr === 'number' ? mrr.toFixed(3) : '미평가';
                                } catch { return '미평가'; }
                              })()}
                            </div>
                          </div>

                          <div className="bg-green-50 dark:bg-green-900/20 rounded-lg p-2">
                            <div className="text-xs text-green-600 dark:text-green-400 mb-1">Recall@1</div>
                            <div className={`text-sm font-semibold ${
                              (() => {
                                try {
                                  const evalData = model.evaluation_results || model.evaluationResults;
                                  const recall_at_1 = evalData?.retrieval_metrics?.recall_at_1 || evalData?.metrics?.retrieval_metrics?.recall_at_1;
                                  if (typeof recall_at_1 !== 'number') return 'text-gray-500';
                                  return recall_at_1 >= 0.8 ? 'text-green-600'
                                       : recall_at_1 >= 0.6 ? 'text-yellow-600'
                                       : 'text-red-600';
                                } catch { return 'text-gray-500'; }
                              })()
                            }`}>
                              {(() => {
                                try {
                                  const evalData = model.evaluation_results || model.evaluationResults;
                                  const recall_at_1 = evalData?.retrieval_metrics?.recall_at_1 || evalData?.metrics?.retrieval_metrics?.recall_at_1;
                                  return typeof recall_at_1 === 'number' ? recall_at_1.toFixed(3) : '미평가';
                                } catch { return '미평가'; }
                              })()}
                            </div>
                          </div>
                        </div>
                      </div>
                    )}

                    {/* 평가 상태 배지 */}
                    <div className="mt-3 flex items-center justify-between">
                      <span className={`px-2 py-1 text-xs font-medium rounded-full ${
                        evaluationProgress.isEvaluating && evaluationProgress.currentModel === model.name
                          ? 'bg-orange-100 text-orange-700 dark:bg-orange-900 dark:text-orange-300'
                          : !model.evaluated 
                            ? 'bg-gray-100 text-gray-600 dark:bg-gray-700 dark:text-gray-400'
                            : 'bg-green-100 text-green-700 dark:bg-green-900 dark:text-green-300'
                      }`}>
                        {evaluationProgress.isEvaluating && evaluationProgress.currentModel === model.name
                          ? '⏳ 평가 중...'
                          : !model.evaluated 
                            ? '🔄 평가 대기' 
                            : '✅ 평가 완료'}
                      </span>
                      
                      <button
                        onClick={() => handleEvaluateModel(model.id)}
                        disabled={!ragServerConnected || evaluationProgress.isEvaluating}
                        className={`px-2 py-1 text-xs font-medium rounded transition-colors ${
                          !ragServerConnected || evaluationProgress.isEvaluating
                            ? 'bg-gray-200 text-gray-500 cursor-not-allowed'
                            : 'bg-blue-100 text-blue-700 hover:bg-blue-200 dark:bg-blue-900 dark:text-blue-300'
                        }`}
                      >
                        {model.evaluated 
                          ? (evaluationProgress.isEvaluating && evaluationProgress.currentModel === model.name 
                             ? '⏳ 평가 중...' 
                             : '🔄 재평가')
                          : '🧪 평가하기'}
                      </button>
                    </div>
                  </div>
                  
                  {model.trainingData && (
                    <div className="mb-4 p-3 bg-blue-50 dark:bg-blue-900/20 rounded-lg">
                      <div className="text-xs text-blue-700 dark:text-blue-300">
                        훈련 데이터: {model.trainingData.pairCount}개 
                      </div>
                    </div>
                  )}
                  
                  <div className="space-y-2">
                    <div className="flex gap-2">
                      {!model.isActive && (
                        <button
                          onClick={() => handleModelActivation(model.id)}
                          className="flex-1 px-4 py-2 bg-blue-600 text-white text-sm font-bold rounded-lg hover:bg-blue-700 dark:bg-blue-600 dark:hover:bg-blue-700 transition-colors"
                        >
                          활성화
                        </button>
                      )}
                      {model.type === 'finetuned' && (
                        <button
                          onClick={() => handleDeleteModel(model.id, model.name)}
                          disabled={model.isActive}
                          className={`px-4 py-2 text-sm font-medium rounded-lg transition-colors ${
                            model.isActive
                              ? 'bg-gray-300 text-gray-500 dark:bg-gray-600 dark:text-gray-400 cursor-not-allowed'
                              : 'bg-red-600 text-white hover:bg-red-700 dark:bg-red-600 dark:hover:bg-red-700'
                          }`}
                          title={model.isActive ? '활성화된 모델은 삭제할 수 없습니다' : '모델 삭제'}
                        >
                          🗑️ 삭제
                        </button>
                      )}
                    </div>
                    {!model.isActive && model.type === 'finetuned' && (
                      <p className="text-xs text-amber-600 dark:text-amber-400 bg-amber-50 dark:bg-amber-900/20 px-2 py-1 rounded">
                        ⚠️ 활성화 후 RAG 서비스 재시작 필요
                      </p>
                    )}
                  </div>
                </div>
              ))}
            </div>
          </div>
        )}

              {/* 데이터셋 탭 */}
      {activeTab === 'datasets' && (
          <div className="p-6 space-y-6">
            {/* 데이터셋 관리 설명 카드 */}
            <div className="bg-gradient-to-r from-green-50 to-emerald-50 dark:from-green-900/20 dark:to-emerald-900/20 rounded-xl p-6 border border-green-200 dark:border-green-700">
              <div className="flex items-start space-x-4">
                <div className="flex-shrink-0">
                  <div className="w-12 h-12 bg-green-100 dark:bg-green-800 rounded-lg flex items-center justify-center">
                    <span className="text-2xl">📁</span>
                  </div>
                </div>
                <div className="flex-1">
                  <h3 className="text-lg font-semibold text-green-900 dark:text-green-200 mb-2">
                    학습 데이터셋 생성 및 관리
                  </h3>
                  <p className="text-green-700 dark:text-green-300 text-sm leading-relaxed">
                    모델 학습에 필요한 고품질 훈련 데이터셋을 생성하고 관리합니다. 
                    문서에서 AI가 자동으로 질문-답변 쌍을 생성하거나, 파일을 직접 업로드하여 데이터셋을 만들 수 있습니다.
                  </p>
                  <div className="flex flex-wrap gap-2 mt-3">
                    <span className="px-2 py-1 bg-green-100 dark:bg-green-800 text-green-700 dark:text-green-300 text-xs rounded-full">AI 자동 생성</span>
                    <span className="px-2 py-1 bg-green-100 dark:bg-green-800 text-green-700 dark:text-green-300 text-xs rounded-full">파일 업로드</span>
                    <span className="px-2 py-1 bg-green-100 dark:bg-green-800 text-green-700 dark:text-green-300 text-xs rounded-full">컬렉션 활용</span>
                  </div>
                </div>
              </div>
            </div>

            {/* 데이터셋 품질 가이드라인 & 대조학습 파인튜닝 과정 */}
            <div className="bg-[var(--card-bg)] rounded-xl shadow-sm border border-[var(--card-border)]">
              <div 
                className="flex items-center justify-between p-4 cursor-pointer hover:bg-gray-50 dark:hover:bg-gray-700/50 transition-colors"
                onClick={() => setIsGuidelinesSectionFolded(!isGuidelinesSectionFolded)}
              >
                <p className="text-base font-semibold text-gray-900 dark:text-white flex items-center">
                  <span className="mr-3">📚</span>
                  학습 가이드라인
                </p>
                <button className="text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-200 transition-colors">
                  <svg 
                    className={`w-5 h-5 transform transition-transform duration-200 ${isGuidelinesSectionFolded ? 'rotate-180' : ''}`}
                    fill="none" 
                    stroke="currentColor" 
                    viewBox="0 0 24 24"
                  >
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                  </svg>
                </button>
              </div>
              
              {!isGuidelinesSectionFolded && (
                <div className="p-6 pt-0">
                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {/* 데이터셋 품질 가이드라인 */}
              <div className="bg-gradient-to-r from-cyan-50 to-blue-50 dark:from-cyan-900/20 dark:to-blue-900/20 rounded-xl p-6 border border-cyan-200 dark:border-cyan-700">
                <p className="text-base font-semibold text-cyan-900 dark:text-cyan-200 mb-4 flex items-center">
                  <span className="mr-3">✨</span>
                  좋은 데이터셋을 만들기 위한 가이드라인
                </p>
                <div className="space-y-4">
                  <div className="flex items-start space-x-3">
                    <div className="flex-shrink-0 w-8 h-8 bg-cyan-100 dark:bg-cyan-800 rounded-lg flex items-center justify-center">
                      <span className="text-cyan-600 dark:text-cyan-400">📈</span>
                    </div>
                    <div>
                      <p className="text-sm font-semibold text-cyan-800 dark:text-cyan-200 mb-1">데이터 품질</p>
                      <p className="text-sm text-cyan-700 dark:text-cyan-300">
                        명확하고 정확한 질문-답변 쌍을 구성하세요. 모호하거나 부정확한 데이터는 모델 성능을 저하시킵니다.
                      </p>
                    </div>
                  </div>
                  <div className="flex items-start space-x-3">
                    <div className="flex-shrink-0 w-8 h-8 bg-cyan-100 dark:bg-cyan-800 rounded-lg flex items-center justify-center">
                      <span className="text-cyan-600 dark:text-cyan-400">🎯</span>
                    </div>
                    <div>
                      <p className="text-sm font-semibold text-cyan-800 dark:text-cyan-200 mb-1">도메인 특화</p>
                      <p className="text-sm text-cyan-700 dark:text-cyan-300">
                        목적에 맞는 전문 용어와 맥락을 포함하여 특정 도메인에 최적화된 데이터를 구성하세요.
                      </p>
                    </div>
                  </div>
                  <div className="flex items-start space-x-3">
                    <div className="flex-shrink-0 w-8 h-8 bg-cyan-100 dark:bg-cyan-800 rounded-lg flex items-center justify-center">
                      <span className="text-cyan-600 dark:text-cyan-400">⚖️</span>
                    </div>
                    <div>
                      <p className="text-sm font-semibold text-cyan-800 dark:text-cyan-200 mb-1">균형잡힌 구성</p>
                      <p className="text-sm text-cyan-700 dark:text-cyan-300">
                        다양한 주제와 난이도로 구성하여 편향을 방지하고 일반화 성능을 높이세요.
                      </p>
                    </div>
                  </div>
                  <div className="flex items-start space-x-3">
                    <div className="flex-shrink-0 w-8 h-8 bg-cyan-100 dark:bg-cyan-800 rounded-lg flex items-center justify-center">
                      <span className="text-cyan-600 dark:text-cyan-400">🔄</span>
                    </div>
                    <div>
                      <p className="text-sm font-semibold text-cyan-800 dark:text-cyan-200 mb-1">대조 학습 쌍</p>
                      <p className="text-sm text-cyan-700 dark:text-cyan-300">
                        Positive(관련 문서)와 Negative(무관한 문서) 샘플을 포함하여 대조학습 효과를 극대화하세요.
                      </p>
                    </div>
                  </div>
                  <div className="flex items-start space-x-3">
                    <div className="flex-shrink-0 w-8 h-8 bg-cyan-100 dark:bg-cyan-800 rounded-lg flex items-center justify-center">
                      <span className="text-cyan-600 dark:text-cyan-400">📊</span>
                    </div>
                    <div>
                      <p className="text-sm font-semibold text-cyan-800 dark:text-cyan-200 mb-1">충분한 데이터량</p>
                      <p className="text-sm text-cyan-700 dark:text-cyan-300">
                        최소 100개 이상의 예제를 권장합니다. 더 많은 데이터일수록 더 안정적인 학습이 가능합니다.
                      </p>
                    </div>
                  </div>
                  <div className="flex items-start space-x-3">
                    <div className="flex-shrink-0 w-8 h-8 bg-cyan-100 dark:bg-cyan-800 rounded-lg flex items-center justify-center">
                      <span className="text-cyan-600 dark:text-cyan-400">🧹</span>
                    </div>
                    <div>
                      <p className="text-sm font-semibold text-cyan-800 dark:text-cyan-200 mb-1">데이터 정제</p>
                      <p className="text-sm text-cyan-700 dark:text-cyan-300">
                        중복 제거, 노이즈 필터링을 통해 깨끗하고 일관성 있는 데이터를 유지하세요.
                      </p>
                    </div>
                  </div>
                </div>
              </div>

              {/* 대조학습 파인튜닝 과정 설명 */}
              <div className="bg-gradient-to-r from-violet-50 to-purple-50 dark:from-violet-900/20 dark:to-purple-900/20 rounded-xl p-6 border border-violet-200 dark:border-violet-700">
                <p className="text-base font-semibold text-violet-900 dark:text-violet-200 mb-4 flex items-center">
                  <span className="mr-3">🧠</span>
                  대조학습을 통한 파인튜닝 과정
                </p>
                
                <div className="mb-6">
                  <p className="text-sm text-violet-700 dark:text-violet-300 leading-relaxed">
                    대조학습(Contrastive Learning)은 의미적으로 유사한 문서들을 임베딩 공간에서 가깝게 배치하고, 
                    관련 없는 문서들을 멀리 배치하여 검색 정확도를 향상시키는 학습 방법입니다.
                  </p>
                </div>

                <div className="space-y-4">
                  <div className="flex items-start space-x-3">
                    <div className="flex-shrink-0 w-8 h-8 bg-violet-600 !text-white rounded-full flex items-center justify-center text-sm font-bold">1</div>
                    <div>
                      <p className="text-sm font-semibold text-violet-800 dark:text-violet-200 mb-1">질문 임베딩 생성</p>
                      <p className="text-sm text-violet-700 dark:text-violet-300">
                        사용자의 질문을 고차원 벡터 공간으로 변환하여 의미적 표현을 생성합니다.
                      </p>
                    </div>
                  </div>
                  <div className="flex items-start space-x-3">
                    <div className="flex-shrink-0 w-8 h-8 bg-violet-600 !text-white rounded-full flex items-center justify-center text-sm font-bold">2</div>
                    <div>
                      <p className="text-sm font-semibold text-violet-800 dark:text-violet-200 mb-1">Positive/Negative 비교</p>
                      <p className="text-sm text-violet-700 dark:text-violet-300">
                        관련 문서(Positive)는 질문과 가깝게, 무관한 문서(Negative)는 멀게 배치되도록 학습합니다.
                      </p>
                    </div>
                  </div>
                  <div className="flex items-start space-x-3">
                    <div className="flex-shrink-0 w-8 h-8 bg-violet-600 !text-white rounded-full flex items-center justify-center text-sm font-bold">3</div>
                    <div>
                      <p className="text-sm font-semibold text-violet-800 dark:text-violet-200 mb-1">모델 가중치 조정</p>
                      <p className="text-sm text-violet-700 dark:text-violet-300">
                        손실 함수(Contrastive Loss)를 통해 임베딩 품질을 개선하는 방향으로 가중치를 업데이트합니다.
                      </p>
                    </div>
                  </div>
                  <div className="flex items-start space-x-3">
                    <div className="flex-shrink-0 w-8 h-8 bg-violet-600 !text-white rounded-full flex items-center justify-center text-sm font-bold">4</div>
                    <div>
                      <p className="text-sm font-semibold text-violet-800 dark:text-violet-200 mb-1">성능 평가</p>
                      <p className="text-sm text-violet-700 dark:text-violet-300">
                        정밀도, 재현율, F1 Score를 통해 검색 성능을 측정하고 모델의 개선 정도를 확인합니다.
                      </p>
                    </div>
                  </div>
                </div>

                <div className="mt-6 p-4 bg-violet-100 dark:bg-violet-900/40 rounded-lg border border-violet-200 dark:border-violet-700">
                  <div className="flex items-start space-x-3">
                    <span className="text-violet-600 dark:text-violet-400 text-lg">💡</span>
                    <div>
                      <h6 className="font-semibold text-violet-800 dark:text-violet-200 mb-1">핵심 포인트</h6>
                      <p className="text-sm text-violet-700 dark:text-violet-300">
                        <strong>질 좋은 Negative 샘플</strong>이 학습 효과를 크게 향상시킵니다. 
                        단순히 무작위로 선택한 문서보다는 유사하지만 관련 없는 문서를 Negative로 사용할 때 
                        모델이 더 정교한 의미 구분 능력을 학습할 수 있습니다.
                      </p>
                    </div>
                  </div>
                </div>
              </div>
            </div>
                  </div>
                )}
              </div>

            {/* RAG 서버 연결 상태 알림 */}
            {!ragServerConnected && (
              <div className="bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg p-4 mb-4">
                <div className="flex items-start">
                  <div className="flex-shrink-0">
                    <svg className="h-5 w-5 text-yellow-400" viewBox="0 0 20 20" fill="currentColor">
                      <path fillRule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <div className="ml-3">
                    <h3 className="text-sm font-medium text-yellow-800 dark:text-yellow-200">
                      RAG 서버에 연결할 수 없습니다
                    </h3>
                    <div className="mt-2 text-sm text-yellow-700 dark:text-yellow-300">
                      <p>임베딩 모델 학습 기능을 사용하려면 RAG 서버를 시작해주세요:</p>
                      <div className="mt-2 bg-yellow-100 dark:bg-yellow-900/30 rounded px-2 py-1 font-mono text-xs">
                        systemctl --user start airun-rag
                      </div>
                      <button
                        onClick={() => {
                          loadSystemResources();
                          loadDatasets();
                        }}
                        className="mt-3 px-3 py-1 bg-yellow-600 text-white text-xs font-medium rounded hover:bg-yellow-700 dark:bg-yellow-600 dark:hover:bg-yellow-700 transition-colors"
                      >
                        🔄 연결 상태 다시 확인
                      </button>
                    </div>
                  </div>
                </div>
              </div>
            )}

            {/* 메인 컨텐츠 - 좌우 분할 레이아웃 */}
            <div className="grid grid-cols-1 lg:grid-cols-4 gap-6">
              {/* 왼쪽 패널 - 데이터셋 생성 옵션들 */}
              <div className="lg:col-span-1 space-y-4">
                <div className="bg-[var(--card-bg)] rounded-xl shadow-sm border border-[var(--card-border)] p-6">
                  <h4 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">
                    📊 새 데이터셋 생성
                  </h4>
                  
                  <div className="space-y-4">
                    {/* 컬렉션에서 생성 */}
                    <div className="border border-green-200 dark:border-green-700 rounded-lg p-4 bg-green-50 dark:bg-green-900/20">
                      <div className="flex items-center space-x-3 mb-3">
                        <span className="text-2xl">📚</span>
                        <div>
                          <h5 className="font-semibold text-green-800 dark:text-green-200">
                            컬렉션에서 생성
                          </h5>
                          <p className="text-xs text-green-600 dark:text-green-400">
                            기존 RAG 컬렉션 활용
                          </p>
                        </div>
                      </div>
                      <p className="text-sm text-green-700 dark:text-green-300 mb-4">
                        현재 RAG 시스템에 저장된 문서들을 활용하여 자동으로 데이터셋을 생성합니다.
                      </p>
                      <button 
                        onClick={handleCreateDatasetFromRAG}
                        disabled={!ragServerConnected}
                        className={`w-full px-4 py-2 text-sm font-medium rounded-lg transition-colors ${
                          !ragServerConnected
                            ? 'bg-gray-300 text-gray-500 dark:bg-gray-600 dark:text-gray-400 cursor-not-allowed'
                            : 'bg-green-600 text-white hover:bg-green-700 dark:bg-green-600 dark:hover:bg-green-700'
                        }`}
                      >
                        컬렉션에서 생성
                      </button>
                    </div>

                    {/* 파일 업로드 */}
                    <div className="border border-amber-200 dark:border-amber-700 rounded-lg p-4 bg-amber-50 dark:bg-amber-900/20">
                      <div className="flex items-center space-x-3 mb-3">
                        <span className="text-2xl">📁</span>
                        <div>
                          <h5 className="font-semibold text-amber-800 dark:text-amber-200">
                            파일 업로드
                          </h5>
                          <p className="text-xs text-amber-600 dark:text-amber-400">
                            JSON, CSV 직접 업로드
                          </p>
                        </div>
                      </div>
                      <p className="text-sm text-amber-700 dark:text-amber-300 mb-4">
                        미리 준비된 JSON, CSV 형태의 질문-답변 데이터셋 파일을 직접 업로드합니다.
                      </p>
                      <button 
                        onClick={() => {
                          setSelectedDatasetMethod('file-upload');
                          setShowDatasetModal(true);
                        }}
                        className="w-full px-4 py-2 bg-amber-600 text-white text-sm font-medium rounded-lg hover:bg-amber-700 dark:bg-amber-600 dark:hover:bg-amber-700 transition-colors"
                      >
                        파일 업로드
                      </button>
                    </div>

                    {/* 문서 + LLM 생성 */}
                    <div className="border border-purple-200 dark:border-purple-700 rounded-lg p-4 bg-purple-50 dark:bg-purple-900/20">
                      <div className="flex items-center space-x-3 mb-3">
                        <span className="text-2xl">🤖</span>
                        <div>
                          <h5 className="font-semibold text-purple-800 dark:text-purple-200">
                            문서 + LLM 생성
                          </h5>
                          <p className="text-xs text-purple-600 dark:text-purple-400">
                            AI 자동 질문-답변 생성
                          </p>
                        </div>
                      </div>
                      <p className="text-sm text-purple-700 dark:text-purple-300 mb-4">
                        PDF, DOCX, HWP 등 문서를 업로드하면 AI가 자동으로 질문-답변 쌍을 생성합니다.
                      </p>
                      <button
                        onClick={() => {
                          setSelectedDatasetMethod('document-llm');
                          setShowDatasetModal(true);
                        }}
                        className="w-full px-4 py-2 bg-purple-600 text-white text-sm font-medium rounded-lg hover:bg-purple-700 transition-colors"
                      >
                        문서 업로드
                      </button>
                    </div>

                    {/* 범용 평가 데이터셋 */}
                    <div className="border border-blue-200 dark:border-blue-700 rounded-lg p-4 bg-blue-50 dark:bg-blue-900/20">
                      <div className="flex items-center space-x-3 mb-3">
                        <span className="text-2xl">🎯</span>
                        <div>
                          <h5 className="font-semibold text-blue-800 dark:text-blue-200">
                            범용 평가 데이터셋
                          </h5>
                          <p className="text-xs text-blue-600 dark:text-blue-400">
                            데이터 누수 방지
                          </p>
                        </div>
                      </div>
                      <p className="text-sm text-blue-700 dark:text-blue-300 mb-4">
                        일반적인 검색 성능을 평가하기 위한 평가 전용 데이터셋을 사용합니다.
                      </p>
                      <button
                        onClick={() => {
                          setSelectedDatasetMethod('evaluation-universal');
                          setShowDatasetModal(true);
                        }}
                        className="w-full px-4 py-2 bg-blue-600 text-white text-sm font-medium rounded-lg hover:bg-blue-700 transition-colors"
                      >
                        📁 평가 데이터셋 업로드
                      </button>
                    </div>

                    {/* 도메인 평가 데이터셋 */}
                    <div className="border border-cyan-200 dark:border-cyan-700 rounded-lg p-4 bg-cyan-50 dark:bg-cyan-900/20">
                      <div className="flex items-center space-x-3 mb-3">
                        <span className="text-2xl">🔬</span>
                        <div>
                          <h5 className="font-semibold text-cyan-800 dark:text-cyan-200">
                            도메인 평가 데이터셋
                          </h5>
                          <p className="text-xs text-cyan-600 dark:text-cyan-400">
                            데이터 누수 방지
                          </p>
                        </div>
                      </div>
                      <p className="text-sm text-cyan-700 dark:text-cyan-300 mb-4">
                        특정 도메인의 전문성을 평가하기 위한 평가 전용 데이터셋을 사용합니다.
                      </p>
                      <button
                        onClick={() => {
                          setSelectedDatasetMethod('evaluation-domain');
                          setShowDatasetModal(true);
                        }}
                        className="w-full px-4 py-2 bg-cyan-600 text-white text-sm font-medium rounded-lg hover:bg-cyan-700 transition-colors"
                      >
                        📁 평가 데이터셋 업로드
                      </button>
                    </div>
                  </div>
                </div>
              </div>

              {/* 오른쪽 패널 - 데이터셋 목록 */}
              <div className="lg:col-span-3">
                {datasets.length === 0 ? (
                  <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700 p-12">
                    <div className="text-center">
                      <div className="text-gray-400 mb-4">
                        <svg className="w-16 h-16 mx-auto" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                        </svg>
                      </div>
                      <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                        데이터셋이 없습니다
                      </h3>
                      <p className="text-gray-600 dark:text-gray-400 mb-6">
                        왼쪽 패널에서 데이터셋 생성 방법을 선택하여 첫 번째 데이터셋을 만들어보세요.
                      </p>
                      <div className="text-sm text-gray-500">
                        💡 추천: 컬렉션에서 생성하여 빠르게 시작하거나, 문서를 업로드하여 AI가 자동으로 데이터셋을 생성하도록 할 수 있습니다.
                      </div>
                    </div>
                  </div>
                ) : (
                  <div className="grid grid-cols-1 xl:grid-cols-2 gap-6">
                    {datasets.map(dataset => (
                      <div key={dataset.id} className="bg-[var(--card-bg)] rounded-xl shadow-sm border border-[var(--card-border)] p-6">
                        <div className="mb-4">
                          <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                            {dataset.type === 'feedback' ? '💬 ' : ''}{dataset.name}
                          </h3>
                          <div className="flex items-center gap-2">
                            <span className={`px-2 py-1 text-xs font-medium whitespace-nowrap rounded-full ${
                              dataset.status === 'ready' 
                                ? 'bg-green-100 text-green-700 dark:bg-green-900 dark:text-green-300'
                                : 'bg-yellow-100 text-yellow-700 dark:bg-yellow-900 dark:text-yellow-300'
                            }`}>
                              {dataset.status === 'ready' ? '준비됨' : '처리중'}
                            </span>
                            <span className={`px-2 py-1 text-xs font-medium rounded-full ${
                              dataset.type === 'feedback'
                                ? 'bg-purple-100 text-purple-700 dark:bg-purple-900 dark:text-purple-300'
                                : 'bg-gray-100 text-gray-700 dark:bg-gray-700 dark:text-gray-300'
                            }`}>
                              {dataset.type === 'feedback' ? '💬 피드백 기반' : dataset.type || '문서 기반'}
                            </span>
                          </div>
                        </div>
                        
                        <p className="text-sm text-gray-600 dark:text-gray-400 mb-4">
                          {dataset.description}
                          {dataset.id === 'feedback_live' && (
                            <span className="block mt-1 text-xs text-blue-600 dark:text-blue-400">
                              ℹ️ 이 데이터셋은 페이지 로드 시 DB에서 자동으로 생성됩니다
                            </span>
                          )}
                        </p>
                        
                        <div className="space-y-2 mb-4">
                          <div className="flex justify-between text-sm">
                            <span className="text-gray-500">총 예제 수</span>
                            <span className="font-medium">{((dataset.size || dataset.pairCount) || 0).toLocaleString()}</span>
                          </div>
                          <div className="flex justify-between text-sm">
                            <span className="text-gray-500">데이터셋 타입</span>
                            <span className="font-medium">{dataset.type || '문서 기반'}</span>
                          </div>
                          <div className="flex justify-between text-sm">
                            <span className="text-gray-500">생성일</span>
                            <span className="font-medium">{new Date(dataset.createdAt).toLocaleDateString('ko-KR')}</span>
                          </div>
                        </div>

                        {/* 모델 학습 설정 */}
                        <div className="bg-gray-50 dark:bg-gray-700/50 rounded-lg p-3 mb-4">
                          <div className="flex items-center justify-between mb-2">
                            <p className="text-sm font-medium text-gray-600 dark:text-gray-400">🔧 학습 설정</p>
                            <div className="flex items-center space-x-2">
                              <label className="flex items-center text-xs text-gray-600 dark:text-gray-400">
                                <input
                                  type="checkbox"
                                  checked={useAutoOptimization}
                                  onChange={(e) => {
                                    setUseAutoOptimization(e.target.checked);
                                    if (e.target.checked) {
                                      // 자동 최적화 활성화 시 사용자 설정 플래그 초기화
                                      setUserConfigOverrides({
                                        epochs: false,
                                        batchSize: false,
                                        learningRate: false,
                                        forceCpu: false
                                      });
                                    }
                                  }}
                                  className="mr-1"
                                  disabled={trainingProgress.isTraining}
                                />
                                자동 최적화
                              </label>
                            </div>
                          </div>
                          <div className="grid grid-cols-4 gap-2">
                            {/* 에포크 수 */}
                            <div>
                              <label className="block text-xs text-gray-500 mb-1">에포크</label>
                              <select
                                value={finetuningConfig.epochs}
                                onChange={(e) => {
                                  setFinetuningConfig(prev => ({ ...prev, epochs: parseInt(e.target.value) }));
                                  setUserConfigOverrides(prev => ({ ...prev, epochs: true }));
                                }}
                                className="w-full text-xs px-2 py-1 border border-gray-300 dark:border-gray-600 rounded focus:outline-none focus:ring-1 focus:ring-blue-500 dark:bg-gray-700 dark:text-white"
                                disabled={trainingProgress.isTraining}
                              >
                                <option value={1}>1</option>
                                <option value={2}>2</option>
                                <option value={3}>3</option>
                                <option value={4}>4</option>
                                <option value={5}>5</option>
                              </select>
                            </div>

                            {/* 배치 크기 */}
                            <div>
                              <label className="block text-xs text-gray-500 mb-1">배치</label>
                              <select
                                value={finetuningConfig.batchSize}
                                onChange={(e) => {
                                  setFinetuningConfig(prev => ({ ...prev, batchSize: parseInt(e.target.value) }));
                                  setUserConfigOverrides(prev => ({ ...prev, batchSize: true }));
                                }}
                                className="w-full text-xs px-2 py-1 border border-gray-300 dark:border-gray-600 rounded focus:outline-none focus:ring-1 focus:ring-blue-500 dark:bg-gray-700 dark:text-white"
                                disabled={trainingProgress.isTraining}
                              >
                                <option value={1}>1</option>
                                <option value={2}>2</option>
                                <option value={4}>4</option>
                                <option value={8}>8</option>
                                <option value={16}>16</option>
                              </select>
                            </div>

                            {/* 학습률 */}
                            <div>
                              <label className="block text-xs text-gray-500 mb-1">학습률</label>
                              <select
                                value={finetuningConfig.learningRate}
                                onChange={(e) => {
                                  setFinetuningConfig(prev => ({ ...prev, learningRate: parseFloat(e.target.value) }));
                                  setUserConfigOverrides(prev => ({ ...prev, learningRate: true }));
                                }}
                                className="w-full text-xs px-2 py-1 border border-gray-300 dark:border-gray-600 rounded focus:outline-none focus:ring-1 focus:ring-blue-500 dark:bg-gray-700 dark:text-white"
                                disabled={trainingProgress.isTraining}
                              >
                                <option value={5e-6}>5e-6</option>
                                <option value={1e-5}>1e-5</option>
                                <option value={2e-5}>2e-5</option>
                                <option value={5e-5}>5e-5</option>
                              </select>
                            </div>

                            {/* 디바이스 모드 */}
                            <div>
                              <label className="block text-xs text-gray-500 mb-1">디바이스</label>
                              <select
                                value={finetuningConfig.forceCpu ? 'cpu' : 'gpu'}
                                onChange={(e) => {
                                  setFinetuningConfig(prev => ({ ...prev, forceCpu: e.target.value === 'cpu' }));
                                  setUserConfigOverrides(prev => ({ ...prev, forceCpu: true }));
                                }}
                                className="w-full text-xs px-2 py-1 border border-gray-300 dark:border-gray-600 rounded focus:outline-none focus:ring-1 focus:ring-blue-500 dark:bg-gray-700 dark:text-white"
                                disabled={trainingProgress.isTraining}
                              >
                                <option value="gpu" disabled={!systemResources?.gpu_available}>
                                  {systemResources?.gpu_available ? 'GPU' : 'GPU(없음)'}
                                </option>
                                <option value="cpu">CPU</option>
                              </select>
                            </div>
                          </div>
                          <div className="text-xs text-gray-500 mt-2 flex items-center justify-between">
                            <span>
                              {!useAutoOptimization 
                                ? '👤 수동 설정 모드'
                                : (userConfigOverrides.epochs || userConfigOverrides.batchSize || userConfigOverrides.learningRate || userConfigOverrides.forceCpu)
                                ? '🔧 혼합 모드 (일부 사용자 설정)'
                                : '🤖 자동 최적화 활성화'} - 
                              권장: {systemResources?.gpu_available ? 'GPU' : 'CPU'} 모드, 
                              배치 {systemResources?.gpu_available 
                                ? systemResources.recommendations.gpu_mode.batch_size 
                                : systemResources?.recommendations.cpu_mode.batch_size || 2}
                            </span>
                            <span className={`px-2 py-0.5 text-xs rounded-full ${
                              finetuningConfig.forceCpu 
                                ? 'bg-orange-100 text-orange-600 dark:bg-orange-900 dark:text-orange-300'
                                : 'bg-green-100 text-green-600 dark:bg-green-900 dark:text-green-300'
                            }`}>
                              {finetuningConfig.forceCpu ? '🖥️ CPU' : '🚀 GPU'}
                            </span>
                          </div>
                        </div>
                        
                        <div className="flex gap-2">
                          <button 
                            onClick={() => handleViewDatasetDetails(dataset)}
                            className="px-4 py-2 text-purple-700 dark:text-purple-300 text-sm font-medium rounded-lg hover:bg-purple-100 dark:hover:bg-purple-800 transition-colors"
                            title="데이터셋 상세보기"
                          >
                            👁️ 상세보기
                          </button>
                          <button 
                            onClick={() => handleStartTraining(dataset.id)}
                            disabled={!dataset.size || dataset.size === 0 || trainingProgress.isTraining}
                            className={`flex-1 px-4 py-2 text-sm font-medium rounded-lg transition-colors ${
                              !dataset.size || dataset.size === 0 || trainingProgress.isTraining
                                ? 'bg-gray-300 text-gray-500 dark:bg-gray-600 dark:text-gray-400 cursor-not-allowed'
                                : 'bg-blue-600 text-white hover:bg-blue-700 dark:bg-blue-600 dark:hover:bg-blue-700'
                            }`}
                          >
                            {trainingProgress.isTraining && trainingProgress.datasetId === dataset.id 
                              ? '모델 학습 중...' 
                              : '모델 학습 시작'}
                          </button>
                          {/* 시스템 관리 데이터셋은 deletable 속성에 따라 삭제 버튼 숨김 */}
                          {dataset.deletable === false ? (
                            <span className="px-4 py-2 text-gray-500 dark:text-gray-400 text-sm italic">
                              {dataset.system_managed ? '시스템 관리됨' : '자동 생성됨'}
                            </span>
                          ) : (
                            <button
                              onClick={() => handleDeleteDataset(dataset.id, dataset.name)}
                              className="px-4 py-2 text-red-700 dark:text-white text-sm font-bold rounded-lg hover:bg-red-200 dark:hover:bg-red-800 transition-colors"
                              title="데이터셋 삭제"
                            >
                              🗑️ 삭제
                            </button>
                          )}
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </div>
            </div>
          </div>
        )}

      {/* 모델 학습 탭 */}
      {activeTab === 'training' && (
          <div className="p-6 space-y-6">
            {/* 모델 학습 관리 설명 카드 */}
            <div className="bg-gradient-to-r from-purple-50 to-violet-50 dark:from-purple-900/20 dark:to-violet-900/20 rounded-xl p-6 border border-purple-200 dark:border-purple-700">
              <div className="flex items-start space-x-4">
                <div className="flex-shrink-0">
                  <div className="w-12 h-12 bg-purple-100 dark:bg-purple-800 rounded-lg flex items-center justify-center">
                    <span className="text-2xl">🔧</span>
                  </div>
                </div>
                <div className="flex-1">
                  <h3 className="text-lg font-semibold text-purple-900 dark:text-purple-200 mb-2">
                    도메인 특화 모델 학습
                  </h3>
                  <p className="text-purple-700 dark:text-purple-300 text-sm leading-relaxed">
                    생성된 데이터셋을 사용하여 기본 임베딩 모델을 도메인에 특화된 모델로 학습합니다. 
                    시스템 자원 상황에 맞는 최적의 설정으로 안정적인 훈련을 진행하고, 실시간으로 진행 상황을 모니터링할 수 있습니다.
                  </p>
                  <div className="flex flex-wrap gap-2 mt-3">
                    <span className="px-2 py-1 bg-purple-100 dark:bg-purple-800 text-purple-700 dark:text-purple-300 text-xs rounded-full">자원 최적화</span>
                    <span className="px-2 py-1 bg-purple-100 dark:bg-purple-800 text-purple-700 dark:text-purple-300 text-xs rounded-full">실시간 모니터링</span>
                    <span className="px-2 py-1 bg-purple-100 dark:bg-purple-800 text-purple-700 dark:text-purple-300 text-xs rounded-full">GPU/CPU 지원</span>
                  </div>
                </div>
              </div>
            </div>

            {/* 모델 학습 설정 섹션 */}
            <div className="bg-[var(--card-bg)] rounded-xl shadow-sm border border-[var(--card-border)] p-6">
              <h4 className="text-base font-semibold text-gray-900 dark:text-white mb-4">
                ⚙️ 모델 학습 설정
                {trainingProgress.isTraining && (
                  <span className="ml-2 text-sm text-orange-600 dark:text-orange-400 font-normal">
                    (훈련 중에는 설정 변경 불가)
                  </span>
                )}
              </h4>
              
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
                {/* 왼쪽 영역: 설정 가이드 */}
                <div className="space-y-6">
                  <div className="bg-gradient-to-r from-blue-50 to-indigo-50 dark:from-blue-900/20 dark:to-indigo-900/20 rounded-lg p-4 border border-blue-200 dark:border-blue-700">
                    <h5 className="text-sm font-semibold text-blue-800 dark:text-blue-200 mb-3">
                      📚 모델 학습 설정 가이드
                    </h5>
                    <div className="space-y-3 text-sm text-blue-700 dark:text-blue-300">
                      <div className="flex items-start space-x-2">
                        <span className="text-blue-500 font-bold">•</span>
                        <div>
                          <strong>기본 모델:</strong> 학습의 기반이 되는 사전 훈련된 모델을 선택합니다.
                        </div>
                      </div>
                      <div className="flex items-start space-x-2">
                        <span className="text-blue-500 font-bold">•</span>
                        <div>
                          <strong>에포크:</strong> 전체 데이터셋을 몇 번 반복 학습할지 결정합니다. 과적합 방지를 위해 적절한 값을 설정하세요.
                        </div>
                      </div>
                      <div className="flex items-start space-x-2">
                        <span className="text-blue-500 font-bold">•</span>
                        <div>
                          <strong>배치 크기:</strong> 한 번에 처리할 데이터 샘플 수입니다. 시스템 메모리에 맞게 조정하세요.
                        </div>
                      </div>
                      <div className="flex items-start space-x-2">
                        <span className="text-blue-500 font-bold">•</span>
                        <div>
                          <strong>학습률:</strong> 모델 가중치 업데이트 속도를 조절합니다. 너무 높으면 불안정, 너무 낮으면 학습 속도가 느려집니다.
                        </div>
                      </div>
                    </div>
                  </div>

                  <div className="bg-gradient-to-r from-green-50 to-emerald-50 dark:from-green-900/20 dark:to-emerald-900/20 rounded-lg p-4 border border-green-200 dark:border-green-700">
                    <h5 className="text-sm font-semibold text-green-800 dark:text-green-200 mb-3">
                      🎯 권장 설정값
                    </h5>
                    <div className="space-y-2 text-sm text-green-700 dark:text-green-300">
                      <div className="grid grid-cols-2 gap-2">
                        <div className="font-medium">GPU 환경:</div>
                        <div>에포크 3-5, 배치 4-8</div>
                      </div>
                      <div className="grid grid-cols-2 gap-2">
                        <div className="font-medium">CPU 환경:</div>
                        <div>에포크 1-2, 배치 2-4</div>
                      </div>
                      <div className="grid grid-cols-2 gap-2">
                        <div className="font-medium">학습률:</div>
                        <div>1e-5 ~ 2e-5 (권장)</div>
                      </div>
                      <div className="grid grid-cols-2 gap-2">
                        <div className="font-medium">데이터셋 크기:</div>
                        <div>1000+ 샘플 권장</div>
                      </div>
                    </div>
                  </div>

                  {/* 시스템 자원 현황 */}
                   {systemResources && (
                     <div className={`bg-gradient-to-r ${ragServerConnected ? 'from-purple-50 to-indigo-50 dark:from-purple-900/20 dark:to-indigo-900/20' : 'from-gray-50 to-slate-50 dark:from-gray-900/20 dark:to-slate-900/20'} rounded-lg p-4 border ${ragServerConnected ? 'border-purple-200 dark:border-purple-700' : 'border-gray-200 dark:border-gray-700'}`}>
                       <h5 className={`text-sm font-semibold ${ragServerConnected ? 'text-purple-800 dark:text-purple-200' : 'text-gray-700 dark:text-gray-300'} mb-3`}>
                         🖥️ 시스템 자원 현황 {!ragServerConnected && '(추정값)'}
                       </h5>
                       <div className="grid grid-cols-4 gap-3">
                         <div className="bg-[var(--card-bg)] rounded-lg p-3 border border-[var(--card-border)]">
                           <div className={`${ragServerConnected ? 'text-purple-700 dark:text-purple-300' : 'text-gray-600 dark:text-gray-400'} text-xs font-medium mb-1`}>메모리</div>
                           <div className={`${ragServerConnected ? 'text-purple-800 dark:text-purple-200' : 'text-gray-700 dark:text-gray-300'} text-sm font-semibold`}>
                             {systemResources.available_memory_gb.toFixed(1)}GB
                           </div>
                           <div className={`${ragServerConnected ? 'text-purple-600 dark:text-purple-400' : 'text-gray-500 dark:text-gray-500'} text-xs`}>
                             / {systemResources.total_memory_gb.toFixed(1)}GB
                           </div>
                         </div>
                         
                         <div className="bg-[var(--card-bg)] rounded-lg p-3 border border-[var(--card-border)]">
                           <div className={`${ragServerConnected ? 'text-purple-700 dark:text-purple-300' : 'text-gray-600 dark:text-gray-400'} text-xs font-medium mb-1`}>CPU</div>
                           <div className={`${ragServerConnected ? 'text-purple-800 dark:text-purple-200' : 'text-gray-700 dark:text-gray-300'} text-sm font-semibold`}>
                             {systemResources.cpu_cores}코어
                           </div>
                           <div className={`${ragServerConnected ? 'text-purple-600 dark:text-purple-400' : 'text-gray-500 dark:text-gray-500'} text-xs`}>
                             물리적
                           </div>
                         </div>
                         
                         <div className="bg-[var(--card-bg)] rounded-lg p-3 border border-[var(--card-border)]">
                           <div className={`${ragServerConnected ? 'text-purple-700 dark:text-purple-300' : 'text-gray-600 dark:text-gray-400'} text-xs font-medium mb-1`}>GPU</div>
                           <div className={`${ragServerConnected ? 'text-purple-800 dark:text-purple-200' : 'text-gray-700 dark:text-gray-300'} text-sm font-semibold`}>
                             {systemResources.gpu_available ? `${systemResources.gpu_available_memory_gb.toFixed(1)}GB` : '❌'}
                           </div>
                           <div className={`${ragServerConnected ? 'text-purple-600 dark:text-purple-400' : 'text-gray-500 dark:text-gray-500'} text-xs`}>
                             {systemResources.gpu_available ? '사용 가능' : '사용 불가'}
                           </div>
                         </div>
                         
                         <div className="bg-[var(--card-bg)] rounded-lg p-3 border border-[var(--card-border)]">
                           <div className={`${ragServerConnected ? 'text-purple-700 dark:text-purple-300' : 'text-gray-600 dark:text-gray-400'} text-xs font-medium mb-1`}>권장 배치</div>
                           <div className={`${ragServerConnected ? 'text-purple-800 dark:text-purple-200' : 'text-gray-700 dark:text-gray-300'} text-sm font-semibold`}>
                             {systemResources.gpu_available 
                               ? systemResources.recommendations.gpu_mode.batch_size 
                               : systemResources.recommendations.cpu_mode.batch_size}
                           </div>
                           <div className={`${ragServerConnected ? 'text-purple-600 dark:text-purple-400' : 'text-gray-500 dark:text-gray-500'} text-xs`}>
                             {systemResources.gpu_available ? 'GPU 모드' : 'CPU 모드'}
                           </div>
                         </div>
                       </div>
                     </div>
                   )}

                   <div className="bg-gradient-to-r from-amber-50 to-orange-50 dark:from-amber-900/20 dark:to-orange-900/20 rounded-lg p-4 border border-amber-200 dark:border-amber-700">
                     <h5 className="text-sm font-semibold text-amber-800 dark:text-amber-200 mb-3">
                       ⚠️ 주의사항
                     </h5>
                     <div className="space-y-2 text-sm text-amber-700 dark:text-amber-300">
                       <div className="flex items-start space-x-2">
                         <span className="text-amber-500 font-bold">•</span>
                         <div>모델 학습 중에는 시스템 리소스를 많이 사용하므로 다른 작업을 자제하세요.</div>
                       </div>
                       <div className="flex items-start space-x-2">
                         <span className="text-amber-500 font-bold">•</span>
                         <div>메모리 부족 시 배치 크기를 줄이거나 CPU 모드를 사용하세요.</div>
                       </div>
                       <div className="flex items-start space-x-2">
                         <span className="text-amber-500 font-bold">•</span>
                         <div>과적합 방지를 위해 검증 데이터셋을 별도로 준비하는 것을 권장합니다.</div>
                       </div>
                     </div>
                   </div>
                </div>

                {/* 오른쪽 영역: 설정 폼 */}
                <div className={`${trainingProgress.isTraining ? 'opacity-50 pointer-events-none' : ''}`}>
                  <div className="space-y-6">
                    {/* 기본 모델 선택 */}
                    <div>
                      <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        기본 모델
                      </label>
                      <select
                        value={finetuningConfig.baseModel}
                        onChange={(e) => setFinetuningConfig(prev => ({ ...prev, baseModel: e.target.value }))}
                        className="text-sm w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:text-white"
                      >
                         {models.filter(m => m.type === 'base').length > 0 ? (
                           models.filter(m => m.type === 'base').map(model => (
                             <option key={model.id} value={model.id}>
                               {model.name} ({model.size})
                             </option>
                           ))
                         ) : (
                           <option value="">기본 모델 없음</option>
                         )}
                      </select>
                      <div className="text-xs text-gray-500 mt-1">
                        모델 학습할 기본 모델을 선택하세요
                      </div>
                    </div>

                    {/* 자동 최적화 설정 */}
                    <div className="bg-blue-50 dark:bg-blue-900/20 rounded-lg p-4 mb-4">
                      <div className="flex items-center justify-between mb-3">
                        <h5 className="text-sm font-medium text-blue-800 dark:text-blue-200">
                          ⚙️ 설정 모드
                        </h5>
                        <label className="flex items-center">
                          <input
                            type="checkbox"
                            checked={useAutoOptimization}
                            onChange={(e) => {
                              setUseAutoOptimization(e.target.checked);
                              if (e.target.checked) {
                                // 자동 최적화 활성화 시 사용자 설정 플래그 초기화
                                setUserConfigOverrides({
                                  epochs: false,
                                  batchSize: false,
                                  learningRate: false,
                                  forceCpu: false
                                });
                              }
                            }}
                            className="mr-2"
                          />
                          <span className="text-sm text-blue-700 dark:text-blue-300">자동 최적화 사용</span>
                        </label>
                      </div>
                      <div className="text-sm text-blue-700 dark:text-blue-300">
                        {!useAutoOptimization ? (
                          <div className="flex items-center space-x-2">
                            <span>👤</span>
                            <span>사용자가 직접 설정한 값들을 그대로 사용합니다. 자동 최적화가 적용되지 않습니다.</span>
                          </div>
                        ) : (userConfigOverrides.epochs || userConfigOverrides.batchSize || userConfigOverrides.learningRate || userConfigOverrides.forceCpu) ? (
                          <div className="flex items-center space-x-2">
                            <span>🔧</span>
                            <span>일부 설정은 사용자가 변경한 값으로 고정되고, 나머지는 시스템이 자동으로 최적화합니다.</span>
                          </div>
                        ) : (
                          <div className="flex items-center space-x-2">
                            <span>🤖</span>
                            <span>시스템 자원에 맞춰 자동으로 최적화된 설정값을 사용합니다. 사용자가 직접 변경하면 해당 값은 고정됩니다.</span>
                          </div>
                        )}
                      </div>
                    </div>

                    {/* 에포크 수 & 배치 크기 */}
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      <div>
                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                          에포크 수
                          {userConfigOverrides.epochs && (
                            <span className="ml-2 text-xs text-blue-600 dark:text-blue-400">👤 사용자 설정</span>
                          )}
                        </label>
                        <input
                          type="number"
                          min="1"
                          max="10"
                          value={finetuningConfig.epochs}
                          onChange={(e) => {
                            setFinetuningConfig(prev => ({ ...prev, epochs: parseInt(e.target.value) || 1 }));
                            setUserConfigOverrides(prev => ({ ...prev, epochs: true }));
                          }}
                          className="text-sm w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:text-white"
                        />
                        <div className="text-xs text-gray-500 mt-1">
                          GPU: 3-5 권장, CPU: 1-2 권장
                        </div>
                      </div>

                      <div>
                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                          배치 크기
                          {userConfigOverrides.batchSize && (
                            <span className="ml-2 text-xs text-blue-600 dark:text-blue-400">👤 사용자 설정</span>
                          )}
                        </label>
                        <input
                          type="number"
                          min="1"
                          max="32"
                          value={finetuningConfig.batchSize}
                          onChange={(e) => {
                            setFinetuningConfig(prev => ({ ...prev, batchSize: parseInt(e.target.value) || 1 }));
                            setUserConfigOverrides(prev => ({ ...prev, batchSize: true }));
                          }}
                          className="text-sm w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:text-white"
                        />
                        <div className="text-xs text-gray-500 mt-1">
                          {systemResources?.gpu_available 
                            ? `GPU 권장: ${systemResources.recommendations.gpu_mode.batch_size}` 
                            : `CPU 권장: ${systemResources?.recommendations.cpu_mode.batch_size || 2}`}
                        </div>
                      </div>
                    </div>

                    {/* 학습률 & 디바이스 모드 */}
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      <div>
                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                          학습률
                          {userConfigOverrides.learningRate && (
                            <span className="ml-2 text-xs text-blue-600 dark:text-blue-400">👤 사용자 설정</span>
                          )}
                        </label>
                        <select
                          value={finetuningConfig.learningRate}
                          onChange={(e) => {
                            setFinetuningConfig(prev => ({ ...prev, learningRate: parseFloat(e.target.value) }));
                            setUserConfigOverrides(prev => ({ ...prev, learningRate: true }));
                          }}
                          className="text-sm w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:text-white"
                        >
                          <option value={5e-6}>5e-6 (매우 낮음)</option>
                          <option value={1e-5}>1e-5 (낮음, 권장)</option>
                          <option value={2e-5}>2e-5 (보통)</option>
                          <option value={5e-5}>5e-5 (높음)</option>
                          <option value={1e-4}>1e-4 (매우 높음)</option>
                        </select>
                        <div className="text-xs text-gray-500 mt-1">
                          낮은 학습률 권장 (과적합 방지)
                        </div>
                      </div>

                      <div>
                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                          디바이스 모드
                          {userConfigOverrides.forceCpu && (
                            <span className="ml-2 text-xs text-blue-600 dark:text-blue-400">👤 사용자 설정</span>
                          )}
                        </label>
                        <div className="flex items-center space-x-4 mb-2">
                          <label className="flex items-center">
                            <input
                              type="radio"
                              name="deviceMode"
                              checked={!finetuningConfig.forceCpu}
                              onChange={() => {
                                setFinetuningConfig(prev => ({ ...prev, forceCpu: false }));
                                setUserConfigOverrides(prev => ({ ...prev, forceCpu: true }));
                              }}
                              disabled={!systemResources?.gpu_available}
                              className="mr-2"
                            />
                            <span className={`text-sm ${!systemResources?.gpu_available ? 'text-gray-400' : 'text-gray-700 dark:text-gray-300'}`}>
                              GPU 모드
                            </span>
                          </label>
                          <label className="flex items-center">
                            <input
                              type="radio"
                              name="deviceMode"
                              checked={finetuningConfig.forceCpu}
                              onChange={() => {
                                setFinetuningConfig(prev => ({ ...prev, forceCpu: true }));
                                setUserConfigOverrides(prev => ({ ...prev, forceCpu: true }));
                              }}
                              className="mr-2"
                            />
                            <span className="text-sm text-gray-700 dark:text-gray-300">
                              CPU 모드
                            </span>
                          </label>
                        </div>
                        <div className="text-xs text-gray-500">
                          {systemResources?.gpu_available 
                            ? 'GPU: 빠른 훈련, CPU: 안정적'
                            : 'GPU 미감지, CPU 모드 사용'}
                        </div>
                      </div>
                    </div>

                    {/* 데이터셋 선택 */}
                    <div>
                      <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        학습 데이터셋
                      </label>
                      <select
                        value={selectedDatasetId}
                        onChange={(e) => setSelectedDatasetId(e.target.value)}
                        className="text-sm w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:text-white"
                      >
                        <option value="">데이터셋을 선택하세요</option>
                        {datasets.filter(d => d.status === 'ready' && (d.size ?? 0) > 0).map(dataset => (
                          <option key={dataset.id} value={dataset.id}>
                            {dataset.name} ({(dataset.size ?? 0).toLocaleString()}개 샘플)
                          </option>
                        ))}
                      </select>
                      <div className="text-xs text-gray-500 mt-1">
                        {datasets.filter(d => d.status === 'ready' && (d.size ?? 0) > 0).length === 0 
                          ? '사용 가능한 데이터셋이 없습니다. 먼저 데이터셋을 생성해주세요.'
                          : '학습에 사용할 데이터셋을 선택하세요'
                        }
                      </div>
                    </div>

                    {/* 설정 요약 */}
                    <div className="p-4 bg-gray-50 dark:bg-gray-700/50 rounded-lg">
                      <div className="flex items-center justify-between mb-3">
                        <h5 className="text-sm font-medium text-gray-700 dark:text-gray-300">
                          📋 현재 설정 요약
                        </h5>
                        <span className={`px-2 py-1 text-xs font-medium rounded-full ${
                          !useAutoOptimization 
                            ? 'bg-purple-100 text-purple-700 dark:bg-purple-900 dark:text-purple-300'
                            : (userConfigOverrides.epochs || userConfigOverrides.batchSize || userConfigOverrides.learningRate || userConfigOverrides.forceCpu)
                            ? 'bg-orange-100 text-orange-700 dark:bg-orange-900 dark:text-orange-300'
                            : 'bg-blue-100 text-blue-700 dark:bg-blue-900 dark:text-blue-300'
                        }`}>
                          {!useAutoOptimization 
                            ? '👤 수동 설정'
                            : (userConfigOverrides.epochs || userConfigOverrides.batchSize || userConfigOverrides.learningRate || userConfigOverrides.forceCpu)
                            ? '🔧 혼합 모드'
                            : '🤖 자동 최적화'}
                        </span>
                      </div>
                      
                      {/* 설정 모드 설명 */}
                      <div className="mb-3 p-2 bg-gray-100 dark:bg-gray-600/50 rounded text-xs">
                        {!useAutoOptimization 
                          ? '👤 모든 설정이 사용자 지정값으로 고정됩니다.'
                          : (userConfigOverrides.epochs || userConfigOverrides.batchSize || userConfigOverrides.learningRate || userConfigOverrides.forceCpu)
                          ? '🔧 일부 설정은 사용자 지정, 나머지는 자동 최적화됩니다.'
                          : '🤖 모든 설정이 시스템 자원에 맞게 자동 최적화됩니다.'}
                      </div>
                      
                      <div className="grid grid-cols-1 gap-3 text-sm">
                        <div className="flex justify-between">
                          <span className="text-gray-500">기본 모델:</span>
                          <span className="font-medium text-gray-800 dark:text-gray-200">
                            {models.find(m => m.id === finetuningConfig.baseModel)?.name || 'Unknown'}
                          </span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-gray-500">학습 데이터셋:</span>
                          <span className="font-medium text-gray-800 dark:text-gray-200">
                            {selectedDatasetId 
                              ? datasets.find(d => d.id === selectedDatasetId)?.name || 'Unknown'
                              : '선택되지 않음'
                            }
                          </span>
                        </div>
                        <div className="flex justify-between items-center">
                          <span className="text-gray-500">에포크:</span>
                          <div className="flex items-center space-x-2">
                            <span className="font-medium text-gray-800 dark:text-gray-200">
                              {finetuningConfig.epochs}회
                            </span>
                            {userConfigOverrides.epochs && (
                              <span className="text-xs text-blue-600 dark:text-blue-400">👤</span>
                            )}
                          </div>
                        </div>
                        <div className="flex justify-between items-center">
                          <span className="text-gray-500">배치 크기:</span>
                          <div className="flex items-center space-x-2">
                            <span className="font-medium text-gray-800 dark:text-gray-200">
                              {finetuningConfig.batchSize}
                            </span>
                            {userConfigOverrides.batchSize && (
                              <span className="text-xs text-blue-600 dark:text-blue-400">👤</span>
                            )}
                          </div>
                        </div>
                        <div className="flex justify-between items-center">
                          <span className="text-gray-500">학습률:</span>
                          <div className="flex items-center space-x-2">
                            <span className="font-medium text-gray-800 dark:text-gray-200">
                              {finetuningConfig.learningRate.toExponential(0)}
                            </span>
                            {userConfigOverrides.learningRate && (
                              <span className="text-xs text-blue-600 dark:text-blue-400">👤</span>
                            )}
                          </div>
                        </div>
                        <div className="flex justify-between items-center">
                          <span className="text-gray-500">디바이스:</span>
                          <div className="flex items-center space-x-2">
                            <span className={`px-2 py-1 text-xs font-medium rounded-full ${
                              finetuningConfig.forceCpu 
                                ? 'bg-orange-100 text-orange-700 dark:bg-orange-900 dark:text-orange-300'
                                : 'bg-green-100 text-green-700 dark:bg-green-900 dark:text-green-300'
                            }`}>
                              {finetuningConfig.forceCpu ? '🖥️ CPU 모드' : '🚀 GPU 모드'}
                            </span>
                            {userConfigOverrides.forceCpu && (
                              <span className="text-xs text-blue-600 dark:text-blue-400">👤</span>
                            )}
                          </div>
                        </div>
                        {systemResources && (
                          <div className="flex justify-between">
                            <span className="text-gray-500">시스템 자원:</span>
                            <span className="text-xs text-gray-500">
                              {finetuningConfig.forceCpu 
                                ? `${systemResources.cpu_cores}코어` 
                                : `${systemResources.gpu_memory_gb.toFixed(1)}GB GPU`}
                            </span>
                          </div>
                        )}
                      </div>
                    </div>

                    {/* 학습 시작 버튼 */}
                    <div className="flex justify-center">
                      <button
                        onClick={handleStartTrainingFromConfig}
                        disabled={!selectedDatasetId || !ragServerConnected || trainingProgress.isTraining}
                        className={`px-8 py-3 rounded-lg font-medium transition-all duration-200 ${
                          !selectedDatasetId || !ragServerConnected || trainingProgress.isTraining
                            ? 'bg-gray-300 text-gray-500 dark:bg-gray-600 dark:text-gray-400 cursor-not-allowed'
                            : 'bg-blue-600 text-white hover:bg-blue-700 dark:bg-blue-600 dark:hover:bg-blue-700 transform hover:scale-105'
                        }`}
                      >
                        {trainingProgress.isTraining ? (
                          <div className="flex items-center space-x-2">
                            <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white" />
                            <span>🔧 모델 학습 진행 중...</span>
                          </div>
                        ) : (
                          <div className="flex items-center space-x-2">
                            <span>🚀</span>
                            <span>모델 학습 시작</span>
                          </div>
                        )}
                      </button>
                    </div>
                    
                    {/* 학습 시작 조건 안내 */}
                    {(!selectedDatasetId || !ragServerConnected) && (
                      <div className="bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg p-3">
                        <div className="flex items-start space-x-2">
                          <div className="text-yellow-500 text-sm">⚠️</div>
                          <div className="text-sm text-yellow-700 dark:text-yellow-300">
                            <strong>학습 시작 조건:</strong>
                            <ul className="mt-1 space-y-1">
                              {!selectedDatasetId && (
                                <li>• 학습 데이터셋을 선택해주세요</li>
                              )}
                              {!ragServerConnected && (
                                <li>• RAG 서버가 실행되어야 합니다</li>
                              )}
                            </ul>
                          </div>
                        </div>
                      </div>
                    )}
                  </div>
                </div>
              </div>
            </div>

            <div className="bg-[var(--card-bg)] rounded-xl shadow-sm border border-[var(--card-border)] p-6">
              <h4 className="text-base font-semibold text-gray-900 dark:text-white mb-4">
                모델 학습 진행 상황
              </h4>
              
              {trainingProgress.isTraining || (trainingProgress.progress > 0 && trainingProgress.eta) ? (
                <div className="space-y-4">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center space-x-3">
                      <div className={`w-2 h-2 rounded-full ${
                        trainingProgress.isTraining 
                          ? 'bg-blue-500 animate-pulse' 
                          : trainingProgress.eta === '완료' 
                            ? 'bg-green-500' 
                            : trainingProgress.eta === '실패' 
                              ? 'bg-red-500' 
                              : 'bg-gray-500'
                      }`} />
                      <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
                        {trainingProgress.isTraining 
                          ? '모델 학습 진행 중...' 
                          : trainingProgress.eta === '완료' 
                            ? '모델 학습 완료!' 
                            : trainingProgress.eta === '실패' 
                              ? '모델 학습 실패' 
                              : '모델 학습 상태'
                        }
                        {trainingProgress.modelName && (
                          <span className={`${
                            trainingProgress.eta === '완료' 
                              ? 'text-green-600' 
                              : trainingProgress.eta === '실패' 
                                ? 'text-red-600' 
                                : 'text-blue-600'
                          }`}> ({trainingProgress.modelName})</span>
                        )}
                      </span>
                    </div>
                    <span className="text-sm text-gray-500">
                      {trainingProgress.progress.toFixed(1)}% 완료
                    </span>
                  </div>
                  
                  <div className="w-full bg-gray-200 dark:bg-gray-700 rounded-full h-3">
                    <div 
                      className={`h-3 rounded-full transition-all duration-300 ${
                        trainingProgress.isTraining 
                          ? 'bg-blue-600' 
                          : trainingProgress.eta === '완료' 
                            ? 'bg-green-600' 
                            : trainingProgress.eta === '실패' 
                              ? 'bg-red-600' 
                              : 'bg-gray-600'
                      }`}
                      style={{ width: `${trainingProgress.progress}%` }}
                    />
                  </div>
                  
                  <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                    <div>
                      <span className="text-gray-500">데이터셋</span>
                      <div className="font-medium">
                        {datasets.find(d => d.id === trainingProgress.datasetId)?.name || 'Unknown'}
                      </div>
                    </div>
                    <div>
                      <span className="text-gray-500">에포크</span>
                      <div className="font-medium">{trainingProgress.currentEpoch}/{trainingProgress.totalEpochs}</div>
                    </div>
                    <div>
                      <span className="text-gray-500">배치 크기</span>
                      <div className="font-medium">{finetuningConfig.batchSize}</div>
                    </div>
                    <div>
                      <span className="text-gray-500">학습률</span>
                      <div className="font-medium">{finetuningConfig.learningRate.toExponential(0)}</div>
                    </div>
                  </div>
                  
                  <div className="grid grid-cols-2 md:grid-cols-3 gap-4 text-sm mt-3">
                    <div>
                      <span className="text-gray-500">손실값</span>
                      <div className="font-medium">{trainingProgress.loss.toFixed(4)}</div>
                    </div>
                    <div>
                      <span className="text-gray-500">상태</span>
                      <div className="font-medium">{trainingProgress.eta}</div>
                    </div>
                    <div>
                      <span className="text-gray-500">디바이스</span>
                      <div className="font-medium">
                        <span className={`px-2 py-1 text-xs rounded-full ${
                          finetuningConfig.forceCpu 
                            ? 'bg-orange-100 text-orange-700 dark:bg-orange-900 dark:text-orange-300'
                            : 'bg-green-100 text-green-700 dark:bg-green-900 dark:text-green-300'
                        }`}>
                          {finetuningConfig.forceCpu ? '🖥️ CPU' : '🚀 GPU'}
                        </span>
                      </div>
                    </div>
                  </div>
                  
                  {systemResources && (
                    <div className={`rounded-lg p-3 ${
                      trainingProgress.eta === '완료' 
                        ? 'bg-green-50 dark:bg-green-900/20' 
                        : trainingProgress.eta === '실패' 
                          ? 'bg-red-50 dark:bg-red-900/20' 
                          : 'bg-blue-50 dark:bg-blue-900/20'
                    }`}>
                      <div className={`text-sm ${
                        trainingProgress.eta === '완료' 
                          ? 'text-green-700 dark:text-green-300' 
                          : trainingProgress.eta === '실패' 
                            ? 'text-red-700 dark:text-red-300' 
                            : 'text-blue-700 dark:text-blue-300'
                      }`}>
                        🖥️ 훈련 환경: {systemResources.gpu_available ? 'GPU' : 'CPU'} 모드
                        {systemResources.gpu_available && (
                          <span className="ml-2">
                            ({systemResources.gpu_memory_gb.toFixed(1)}GB GPU 메모리)
                          </span>
                        )}
                        {trainingProgress.eta === '완료' && (
                          <span className="ml-2">✅ 성공적으로 완료됨</span>
                        )}
                        {trainingProgress.eta === '실패' && (
                          <span className="ml-2">❌ 훈련 실패</span>
                        )}
                      </div>
                    </div>
                  )}
                  
                  {/* 액션 버튼들 */}
                  {!trainingProgress.isTraining && trainingProgress.progress > 0 && (
                    <div className="mt-4 flex justify-center space-x-3">
                      {trainingProgress.progress === 100 && trainingProgress.eta === '완료' && (
                        <>
                          <button
                            onClick={() => setActiveTab('overview')}
                            className="px-4 py-2 text-sm font-medium text-green-700 bg-green-100 hover:bg-green-200 dark:bg-green-900 dark:text-green-300 dark:hover:bg-green-800 rounded-lg transition-colors"
                            title="완료된 모델 확인하기"
                          >
                            ✅ 완료된 모델 확인
                          </button>
                          <button
                            onClick={resetTrainingState}
                            className="px-4 py-2 text-sm font-medium text-gray-700 bg-gray-100 hover:bg-gray-200 dark:bg-gray-700 dark:text-gray-300 dark:hover:bg-gray-600 rounded-lg transition-colors"
                            title="훈련 상태 초기화"
                          >
                            🔄 상태 초기화
                          </button>
                        </>
                      )}
                      
                      {trainingProgress.eta === '모니터링 중지 (서버 연결 실패)' && (
                        <>
                          <button
                            onClick={async () => {
                              console.log('🔄 [프론트엔드] 모니터링 재시작 시도');
                              try {
                                const response = await fetch('/api/rag/models/finetune/progress');
                                const result = await response.json();
                                if (result.success && result.progress) {
                                  const progress = result.progress;
                                  if (progress.is_training) {
                                    setTrainingProgress(prev => ({
                                      ...prev,
                                      isTraining: true,
                                      progress: progress.progress || prev.progress,
                                      eta: progress.eta || '훈련 진행 중...'
                                    }));
                                  } else if (progress.status === 'completed') {
                                    setTrainingProgress(prev => ({
                                      ...prev,
                                      isTraining: false,
                                      progress: 100,
                                      eta: '완료'
                                    }));
                                    await loadModels();
                                  }
                                }
                              } catch (error) {
                                console.error('모니터링 재시작 실패:', error);
                              }
                            }}
                            className="px-4 py-2 text-sm font-medium text-blue-700 bg-blue-100 hover:bg-blue-200 dark:bg-blue-900 dark:text-blue-300 dark:hover:bg-blue-800 rounded-lg transition-colors"
                            title="훈련 모니터링을 재시작합니다"
                          >
                            🔄 모니터링 재시작
                          </button>
                          <button
                            onClick={resetTrainingState}
                            className="px-4 py-2 text-sm font-medium text-gray-700 bg-gray-100 hover:bg-gray-200 dark:bg-gray-700 dark:text-gray-300 dark:hover:bg-gray-600 rounded-lg transition-colors"
                            title="훈련 상태 초기화"
                          >
                            🔄 상태 초기화
                          </button>
                        </>
                      )}
                      
                      {trainingProgress.progress < 100 && trainingProgress.eta !== '완료' && trainingProgress.eta !== '모니터링 중지 (서버 연결 실패)' && (
                        <button
                          onClick={resetTrainingState}
                          className="px-4 py-2 text-sm font-medium text-orange-700 bg-orange-100 hover:bg-orange-200 dark:bg-orange-900 dark:text-orange-300 dark:hover:bg-orange-800 rounded-lg transition-colors"
                          title="중단된 훈련 상태를 초기화합니다"
                        >
                          🔄 훈련 상태 초기화
                        </button>
                      )}
                      
                      {trainingProgress.eta === '실패' && (
                        <button
                          onClick={resetTrainingState}
                          className="px-4 py-2 text-sm font-medium text-red-700 bg-red-100 hover:bg-red-200 dark:bg-red-900 dark:text-red-300 dark:hover:bg-red-800 rounded-lg transition-colors"
                          title="실패한 훈련 상태를 초기화합니다"
                        >
                          🔄 실패 상태 초기화
                        </button>
                      )}
                    </div>
                  )}
                </div>
              ) : (
                <div className="text-center py-8">
                  <div className="text-gray-400 mb-4">
                    <svg className="w-16 h-16 mx-auto" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1} d="M19 11H5m14 0a2 2 0 012 2v6a2 2 0 01-2 2H5a2 2 0 01-2-2v-6a2 2 0 012-2m14 0V9a2 2 0 00-2-2M5 11V9a2 2 0 012-2m0 0V5a2 2 0 012-2h6a2 2 0 012 2v2M7 7h10" />
                    </svg>
                  </div>
                  <p className="text-gray-600 dark:text-gray-400 mb-4">
                    현재 실행 중인 모델 학습 작업이 없습니다.
                  </p>
                  <p className="text-sm text-gray-500 mb-4">
                    데이터셋 메뉴에서 훈련할 데이터셋을 선택하여 모델 학습을 시작하세요.
                  </p>
                  
                  {/* 훈련 상태 초기화 버튼 (진행률이 0이 아니고 학습이 진행 중이지 않을 때 표시) */}
                  {trainingProgress.progress > 0 && trainingProgress.progress < 100 && (
                    <div className="mt-4">
                      <p className="text-sm text-gray-500 mb-2">
                        이전 훈련 상태가 남아있습니다. ({trainingProgress.progress.toFixed(1)}%)
                      </p>
                      <button
                        onClick={resetTrainingState}
                        className="px-4 py-2 text-sm font-medium text-orange-700 bg-orange-100 hover:bg-orange-200 dark:bg-orange-900 dark:text-orange-300 dark:hover:bg-orange-800 rounded-lg transition-colors"
                        title="중단된 훈련 상태를 초기화합니다"
                      >
                        🔄 훈련 상태 초기화
                      </button>
                    </div>
                  )}
                </div>
              )}
            </div>
            
            {/* 모델 학습 성능 팁 */}
            {systemResources && (
              <div className="bg-[var(--card-bg)] rounded-xl shadow-sm border border-[var(--card-border)] p-6">
                <h4 className={`text-base font-semibold ${ragServerConnected ? 'text-blue-800 dark:text-blue-300' : 'text-gray-700 dark:text-gray-400'} mb-4`}>
                  💡 모델 학습 성능 팁
                </h4>
                <div className={`text-sm ${ragServerConnected ? 'text-blue-700 dark:text-blue-400' : 'text-gray-600 dark:text-gray-500'} space-y-3`}>
                  {systemResources.gpu_available ? (
                    <>
                      <div className="flex items-start space-x-2">
                        <div className="text-green-500 font-semibold">•</div>
                        <div>GPU 가속으로 빠른 훈련 가능 (권장 배치 크기: {systemResources.recommendations.gpu_mode.batch_size})</div>
                      </div>
                      <div className="flex items-start space-x-2">
                        <div className="text-yellow-500 font-semibold">•</div>
                        <div>메모리 부족 시 배치 크기를 줄여보세요</div>
                      </div>
                      <div className="flex items-start space-x-2">
                        <div className="text-blue-500 font-semibold">•</div>
                        <div>Mixed Precision (AMP) 활성화로 메모리 절약 가능</div>
                      </div>
                    </>
                  ) : (
                    <>
                      <div className="flex items-start space-x-2">
                        <div className="text-orange-500 font-semibold">•</div>
                        <div>CPU 모드: 안정적이지만 느린 훈련 (권장 배치 크기: {systemResources.recommendations.cpu_mode.batch_size})</div>
                      </div>
                      <div className="flex items-start space-x-2">
                        <div className="text-purple-500 font-semibold">•</div>
                        <div>작은 데이터셋과 적은 에포크 권장</div>
                      </div>
                      <div className="flex items-start space-x-2">
                        <div className="text-indigo-500 font-semibold">•</div>
                        <div><code className="bg-gray-100 dark:bg-gray-700 px-2 py-1 rounded text-xs">force_cpu: true</code> 옵션으로 메모리 효율성 향상</div>
                      </div>
                    </>
                  )}
                </div>
              </div>
            )}
          </div>
        )}

      {/* 성능 평가 탭 */}
      {activeTab === 'evaluation' && (
          <div className="p-6 space-y-6">
            {/* 성능 평가 설명 카드 */}
            <div className="bg-gradient-to-r from-orange-50 to-amber-50 dark:from-orange-900/20 dark:to-amber-900/20 rounded-xl p-6 border border-orange-200 dark:border-orange-700">
              <div className="flex items-start space-x-4">
                <div className="flex-shrink-0">
                  <div className="w-12 h-12 bg-orange-100 dark:bg-orange-800 rounded-lg flex items-center justify-center">
                    <span className="text-2xl">📈</span>
                  </div>
                </div>
                <div className="flex-1">
                  <h3 className="text-lg font-semibold text-orange-900 dark:text-orange-200 mb-2">
                    모델 성능 평가 및 비교 분석
                  </h3>
                  <p className="text-orange-700 dark:text-orange-300 text-sm leading-relaxed">
                    기본 모델과 모델 학습된 모델들의 성능을 객관적으로 평가하고 비교합니다. 
                    실제 검색 시나리오를 시뮬레이션하여 정밀도, 재현율, F1 Score 등의 표준 메트릭으로 모델 품질을 측정합니다.
                  </p>
                  <div className="flex flex-wrap gap-2 mt-3">
                    <span className="px-2 py-1 bg-orange-100 dark:bg-orange-800 text-orange-700 dark:text-orange-300 text-xs rounded-full">실제 시나리오 테스트</span>
                    <span className="px-2 py-1 bg-orange-100 dark:bg-orange-800 text-orange-700 dark:text-orange-300 text-xs rounded-full">표준 메트릭 측정</span>
                    <span className="px-2 py-1 bg-orange-100 dark:bg-orange-800 text-orange-700 dark:text-orange-300 text-xs rounded-full">성능 비교 분석</span>
                  </div>
                </div>
              </div>
            </div>

            {/* 평가 실행 섹션 */}
            <div className="bg-[var(--card-bg)] rounded-xl shadow-sm border border-[var(--card-border)] p-6">
              <div className="flex items-center justify-between mb-4">
                <h4 className="text-base font-semibold text-gray-900 dark:text-white">
                  모델 성능 평가
                </h4>
                <div className="flex items-center space-x-3">
                  {/* 평가 모드 선택 드롭다운 */}
                  <div className="flex items-center space-x-2">
                    <label className="text-sm text-gray-600 dark:text-gray-400">
                      평가 모드:
                    </label>
                    <select
                      value={evaluationMode}
                      onChange={(e) => setEvaluationMode(e.target.value as 'universal' | 'domain')}
                      disabled={evaluationProgress.isEvaluating}
                      className="px-3 py-1 text-sm border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 focus:border-blue-500 disabled:opacity-50 disabled:cursor-not-allowed"
                    >
                      <option value="universal">🌐 범용 평가</option>
                      <option value="domain">🎯 도메인 전문성 평가</option>
                    </select>
                  </div>

                  {/* 평가용 데이터셋 선택 드롭다운 */}
                  <div className="flex items-center space-x-2">
                    <label className="text-sm text-gray-600 dark:text-gray-400">
                      평가 데이터셋:
                    </label>
                    <select
                      value={selectedEvaluationDataset}
                      onChange={(e) => setSelectedEvaluationDataset(e.target.value)}
                      disabled={evaluationProgress.isEvaluating}
                      className="px-3 py-1 text-sm border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 focus:border-blue-500 disabled:opacity-50 disabled:cursor-not-allowed"
                    >
                      <option value="">기본 평가 데이터 사용</option>
                      {datasets
                        .filter((dataset: any) => {
                          const type = dataset.type || 'training';
                          if (evaluationMode === 'universal') {
                            return type === 'universal_eval';
                          } else if (evaluationMode === 'domain') {
                            return type === 'domain_eval';
                          }
                          return false;
                        })
                        .map((dataset: any) => (
                          <option key={dataset.id} value={dataset.id}>
                            {dataset.name} ({dataset.size || 0}개 샘플)
                          </option>
                        ))
                      }
                    </select>
                  </div>
                  <button
                    onClick={handleStartEvaluation}
                    disabled={!ragServerConnected || evaluationProgress.isEvaluating}
                    className={`px-4 py-2 text-sm font-medium rounded-lg transition-colors ${
                      !ragServerConnected || evaluationProgress.isEvaluating
                        ? 'bg-gray-300 text-gray-500 dark:bg-gray-600 dark:text-gray-400 cursor-not-allowed'
                        : 'bg-blue-600 text-white hover:bg-blue-700 dark:bg-blue-600 dark:hover:bg-blue-700'
                    }`}
                  >
                    {evaluationProgress.isEvaluating ? '평가 진행 중...' : '🧪 모든 모델 평가 실행'}
                  </button>
                </div>
              </div>

              {/* RAG 서버 연결 상태 알림 */}
              {!ragServerConnected && (
                <div className="bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg p-4 mb-4">
                  <div className="flex items-start">
                    <div className="flex-shrink-0">
                      <svg className="h-5 w-5 text-yellow-400" viewBox="0 0 20 20" fill="currentColor">
                        <path fillRule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                      </svg>
                    </div>
                    <div className="ml-3">
                      <h3 className="text-sm font-medium text-yellow-800 dark:text-yellow-200">
                        RAG 서버에 연결할 수 없습니다
                      </h3>
                      <div className="mt-2 text-sm text-yellow-700 dark:text-yellow-300">
                        <p>모델 성능 평가를 수행하려면 RAG 서버를 시작해주세요.</p>
                      </div>
                    </div>
                  </div>
                </div>
              )}

              {/* 평가 진행 상황 */}
              {evaluationProgress.isEvaluating && (
                <div className="bg-blue-50 dark:bg-blue-900/20 rounded-lg p-4 mb-4">
                  <div className="flex items-center justify-between mb-2">
                    <div className="flex items-center space-x-3">
                      <div className="w-2 h-2 bg-blue-500 rounded-full animate-pulse" />
                      <span className="text-sm font-medium text-blue-800 dark:text-blue-200">
                        모델 성능 평가 중...
                      </span>
                    </div>
                    <span className="text-sm text-blue-600 dark:text-blue-400">
                      {evaluationProgress.progress.toFixed(1)}% 완료
                    </span>
                  </div>
                  
                  <div className="w-full bg-blue-200 dark:bg-blue-800 rounded-full h-2 mb-3">
                    <div 
                      className="bg-blue-600 h-2 rounded-full transition-all duration-300"
                      style={{ width: `${evaluationProgress.progress}%` }}
                    />
                  </div>
                  
                  <div className="text-sm text-blue-700 dark:text-blue-300">
                    현재 평가 중: {evaluationProgress.currentModel || '준비 중...'}
                  </div>
                </div>
              )}

              {/* 평가 방법 설명 */}
              <div className="bg-[var(--card-bg)] rounded-lg p-4 mb-6">
                <h3 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  📊 평가 방법
                </h3>
                <div className="text-sm text-gray-600 dark:text-gray-400 space-y-1">
                  {evaluationMode === 'universal' ? (
                    <>
                      <div>• <strong>테스트 데이터셋:</strong> 다중 도메인 범용 벤치마크 (환경, 경제, 정책, 언론 등)</div>
                      <div>• <strong>목적:</strong> 모든 모델의 일관성 있는 범용 성능 비교</div>
                      <div>• <strong>메트릭:</strong> 코사인 유사도 기반 정밀도, 재현율, F1 Score 계산</div>
                      <div>• <strong>특징:</strong> MTEB/BEIR 스타일의 표준화된 평가</div>
                    </>
                  ) : (
                    <>
                      <div>• <strong>테스트 데이터셋:</strong> 각 훈련 데이터셋의 20% 홀드아웃 데이터 사용</div>
                      <div>• <strong>목적:</strong> 특정 도메인에 대한 모델 적합성 평가</div>
                      <div>• <strong>메트릭:</strong> 코사인 유사도 기반 정밀도, 재현율, F1 Score 계산</div>
                      <div>• <strong>특징:</strong> 도메인별 맞춤형 성능 측정</div>
                    </>
                  )}
                </div>
              </div>
            </div>

            {/* 성능 비교 테이블 */}
            <div className="bg-[var(--card-bg)] rounded-xl shadow-sm border border-[var(--card-border)] p-6">
              <div className="flex items-center justify-between mb-6">
                <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                  모델 성능 비교 결과
                </h3>
                <div className="flex items-center space-x-3">
                  <span className="text-sm text-gray-600 dark:text-gray-400">차트 모드:</span>
                  <button
                    onClick={() => setChartMode('bar')}
                    className={`px-3 py-1 text-sm font-medium rounded-md transition-colors ${
                      chartMode === 'bar'
                        ? 'bg-blue-100 text-blue-700 dark:bg-blue-900 dark:text-blue-300'
                        : 'text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-200'
                    }`}
                  >
                    📊 막대 그래프
                  </button>
                  <button
                    onClick={() => setChartMode('radar')}
                    className={`px-3 py-1 text-sm font-medium rounded-md transition-colors ${
                      chartMode === 'radar'
                        ? 'bg-blue-100 text-blue-700 dark:bg-blue-900 dark:text-blue-300'
                        : 'text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-200'
                    }`}
                  >
                    🕸️ 방사형 차트
                  </button>
                </div>
              </div>

              {/* 성능 비교 차트 */}
              {models.filter(model => model.evaluated).length > 0 && (
                <div className="mb-8">
                  <div className="bg-gray-50 dark:bg-gray-800 rounded-lg p-6">
                    <div className="h-80">
                      {chartMode === 'bar' && prepareBarChartData() && (
                        <Bar
                          data={prepareBarChartData()!}
                          options={{
                            responsive: true,
                            maintainAspectRatio: false,
                            plugins: {
                              title: {
                                display: true,
                                text: '모델 성능 메트릭 비교 (막대 그래프)',
                                font: { size: 16, weight: 'bold' },
                                color: 'rgb(75, 85, 99)',
                              },
                              legend: {
                                position: 'top' as const,
                                labels: {
                                  usePointStyle: true,
                                  padding: 15,
                                },
                              },
                              tooltip: {
                                callbacks: {
                                  label: function(context) {
                                    return `${context.dataset.label}: ${(context.parsed.y * 100).toFixed(1)}%`;
                                  }
                                }
                              }
                            },
                            scales: {
                              x: {
                                grid: {
                                  display: false,
                                },
                                ticks: {
                                  maxRotation: 45,
                                  minRotation: 0,
                                },
                              },
                              y: {
                                beginAtZero: true,
                                max: 1.0,
                                grid: {
                                  color: 'rgba(156, 163, 175, 0.2)',
                                },
                                ticks: {
                                  callback: function(value) {
                                    return (Number(value) * 100).toFixed(0) + '%';
                                  }
                                }
                              },
                            },
                            elements: {
                              bar: {
                                borderRadius: 4,
                              }
                            }
                          }}
                        />
                      )}

                      {chartMode === 'radar' && prepareRadarChartData() && (
                        <Radar
                          data={prepareRadarChartData()!}
                          options={{
                            responsive: true,
                            maintainAspectRatio: false,
                            plugins: {
                              title: {
                                display: true,
                                text: '모델 성능 패턴 비교 (방사형 차트)',
                                font: { size: 16, weight: 'bold' },
                                color: 'rgb(75, 85, 99)',
                              },
                              legend: {
                                position: 'top' as const,
                                labels: {
                                  usePointStyle: true,
                                  padding: 15,
                                },
                              },
                              tooltip: {
                                callbacks: {
                                  label: function(context) {
                                    return `${context.dataset.label}: ${(context.parsed.r * 100).toFixed(1)}%`;
                                  }
                                }
                              }
                            },
                            scales: {
                              r: {
                                beginAtZero: true,
                                max: 1.0,
                                grid: {
                                  color: 'rgba(156, 163, 175, 0.2)',
                                },
                                pointLabels: {
                                  font: {
                                    size: 12,
                                    weight: 'bold',
                                  },
                                },
                                ticks: {
                                  callback: function(value) {
                                    return (Number(value) * 100).toFixed(0) + '%';
                                  },
                                  stepSize: 0.2,
                                }
                              },
                            },
                          }}
                        />
                      )}
                    </div>

                    {/* 차트 설명 */}
                    <div className="mt-4 text-sm text-gray-600 dark:text-gray-400">
                      {chartMode === 'bar' ? (
                        <div className="flex items-start space-x-2">
                          <span className="text-blue-500">💡</span>
                          <div>
                            <div className="font-medium mb-1">막대 그래프 해석 가이드:</div>
                            <div>• <strong>🎯 핵심지표(nDCG@5, MRR, Recall@1)</strong>: 실제 검색 성능을 나타내는 가장 중요한 지표</div>
                            <div>• <strong>파인튜닝 효과</strong>는 특히 핵심지표에서 크게 나타납니다</div>
                            <div>• 정밀도/재현율/F1은 분류 성능으로 보조 참고 지표입니다</div>
                            <div>• 핵심지표에서 높은 점수를 보이는 모델이 실제 검색에서 우수합니다</div>
                          </div>
                        </div>
                      ) : (
                        <div className="flex items-start space-x-2">
                          <span className="text-purple-500">🕸️</span>
                          <div>
                            <div className="font-medium mb-1">방사형 차트 해석 가이드:</div>
                            <div>• <strong>🎯 핵심지표 영역</strong>(nDCG@5, MRR, Recall@1)이 바깥쪽으로 확장된 모델이 우수</div>
                            <div>• 파인튜닝된 모델은 오른쪽 하단(핵심지표)에서 크게 개선됩니다</div>
                            <div>• 전체적으로 균형잡힌 육각형보다 <strong>핵심지표가 강한 형태</strong>가 더 실용적</div>
                            <div>• 중심에서 멀수록 해당 메트릭에서 높은 성능을 의미합니다</div>
                          </div>
                        </div>
                      )}
                    </div>
                  </div>
                </div>
              )}

              {/* 평가된 모델이 없는 경우 안내 메시지 */}
              {models.filter(model => model.evaluated).length === 0 && (
                <div className="mb-6 text-center py-8 text-gray-500 dark:text-gray-400">
                  <div className="text-4xl mb-2">📊</div>
                  <div className="text-lg font-medium mb-1">성능 차트를 표시할 데이터가 없습니다</div>
                  <div className="text-sm">모델을 평가한 후 성능 비교 차트가 표시됩니다</div>
                </div>
              )}
              
              <div className="overflow-x-auto">
                <table className="w-full text-sm">
                  <thead>
                    <tr className="border-b border-gray-200 dark:border-gray-700">
                      <th className="text-left py-3 px-4">모델</th>
                      <th className="text-left py-3 px-4">정밀도</th>
                      <th className="text-left py-3 px-4">재현율</th>
                      <th className="text-left py-3 px-4">F1 Score</th>
                      <th className="text-left py-3 px-4">nDCG@5</th>
                      <th className="text-left py-3 px-4">MRR</th>
                      <th className="text-left py-3 px-4">Recall@1</th>
                      <th className="text-left py-3 px-4">테스트 샘플 수</th>
                      <th className="text-left py-3 px-4">평가 상태</th>
                      <th className="text-left py-3 px-4">액션</th>
                    </tr>
                  </thead>
                  <tbody>
                    {models.map(model => (
                      <tr key={model.id} className="border-b border-gray-100 dark:border-gray-800">
                        <td className="py-3 px-4">
                          <div className="flex items-center space-x-2">
                            <div className={`w-2 h-2 rounded-full ${
                              model.isActive ? 'bg-green-500' : 'bg-gray-300'
                            }`} />
                            <span className="font-medium">{model.name}</span>
                            {model.type === 'finetuned' && (
                              <span className="px-1.5 py-0.5 text-xs bg-blue-100 text-blue-700 dark:bg-blue-900 dark:text-blue-300 rounded">
                                파인튜닝
                              </span>
                            )}
                          </div>
                        </td>
                        <td className="py-3 px-4">
                          <span className={`${
                            !model.evaluated 
                              ? 'text-gray-500' 
                              : model.performance.precision >= 0.8 
                                ? 'text-green-600 font-semibold' 
                                : model.performance.precision >= 0.6 
                                  ? 'text-yellow-600 font-semibold' 
                                  : 'text-red-600 font-semibold'
                          }`}>
                            {!model.evaluated 
                              ? '미평가' 
                              : model.performance.precision.toFixed(3)}
                          </span>
                        </td>
                        <td className="py-3 px-4">
                          <span className={`${
                            !model.evaluated 
                              ? 'text-gray-500' 
                              : model.performance.recall >= 0.8 
                                ? 'text-green-600 font-semibold' 
                                : model.performance.recall >= 0.6 
                                  ? 'text-yellow-600 font-semibold' 
                                  : 'text-red-600 font-semibold'
                          }`}>
                            {!model.evaluated 
                              ? '미평가' 
                              : model.performance.recall.toFixed(3)}
                          </span>
                        </td>
                        <td className="py-3 px-4">
                          <span className={`${
                            !model.evaluated 
                              ? 'text-gray-500' 
                              : model.performance.f1Score >= 0.8 
                                ? 'text-green-600 font-semibold' 
                                : model.performance.f1Score >= 0.6 
                                  ? 'text-yellow-600 font-semibold' 
                                  : 'text-red-600 font-semibold'
                          }`}>
                            {!model.evaluated 
                              ? '미평가' 
                              : model.performance.f1Score.toFixed(3)}
                          </span>
                        </td>

                        {/* nDCG@5 컬럼 */}
                        <td className="py-3 px-4">
                          <span className={`${
                            !model.evaluated
                              ? 'text-gray-500'
                              : (() => {
                                  try {
                                    const evalData = model.evaluation_results || model.evaluationResults;
                                    const ndcg = evalData?.retrieval_metrics?.ndcg_at_5 || evalData?.metrics?.retrieval_metrics?.ndcg_at_5;
                                    if (typeof ndcg !== 'number') return 'text-gray-500';
                                    return ndcg >= 0.8 ? 'text-green-600 font-semibold'
                                         : ndcg >= 0.6 ? 'text-yellow-600 font-semibold'
                                         : 'text-red-600 font-semibold';
                                  } catch { return 'text-gray-500'; }
                                })()
                          }`}>
                            {(() => {
                              try {
                                if (!model.evaluated) return '미평가';
                                const evalData = model.evaluation_results || model.evaluationResults;
                                const ndcg = evalData?.retrieval_metrics?.ndcg_at_5 || evalData?.metrics?.retrieval_metrics?.ndcg_at_5;
                                return typeof ndcg === 'number' ? ndcg.toFixed(3) : '미평가';
                              } catch { return '미평가'; }
                            })()}
                          </span>
                        </td>

                        {/* MRR 컬럼 */}
                        <td className="py-3 px-4">
                          <span className={`${
                            !model.evaluated
                              ? 'text-gray-500'
                              : (() => {
                                  try {
                                    const evalData = model.evaluation_results || model.evaluationResults;
                                    const mrr = evalData?.retrieval_metrics?.mrr || evalData?.metrics?.retrieval_metrics?.mrr;
                                    if (typeof mrr !== 'number') return 'text-gray-500';
                                    return mrr >= 0.8 ? 'text-green-600 font-semibold'
                                         : mrr >= 0.6 ? 'text-yellow-600 font-semibold'
                                         : 'text-red-600 font-semibold';
                                  } catch { return 'text-gray-500'; }
                                })()
                          }`}>
                            {(() => {
                              try {
                                if (!model.evaluated) return '미평가';
                                const evalData = model.evaluation_results || model.evaluationResults;
                                const mrr = evalData?.retrieval_metrics?.mrr || evalData?.metrics?.retrieval_metrics?.mrr;
                                return typeof mrr === 'number' ? mrr.toFixed(3) : '미평가';
                              } catch { return '미평가'; }
                            })()}
                          </span>
                        </td>

                        {/* Recall@1 컬럼 */}
                        <td className="py-3 px-4">
                          <span className={`${
                            !model.evaluated
                              ? 'text-gray-500'
                              : (() => {
                                  try {
                                    const evalData = model.evaluation_results || model.evaluationResults;
                                    const recall1 = evalData?.retrieval_metrics?.recall_at_1 || evalData?.metrics?.retrieval_metrics?.recall_at_1;
                                    if (typeof recall1 !== 'number') return 'text-gray-500';
                                    return recall1 >= 0.8 ? 'text-green-600 font-semibold'
                                         : recall1 >= 0.6 ? 'text-yellow-600 font-semibold'
                                         : 'text-red-600 font-semibold';
                                  } catch { return 'text-gray-500'; }
                                })()
                          }`}>
                            {(() => {
                              try {
                                if (!model.evaluated) return '미평가';
                                const evalData = model.evaluation_results || model.evaluationResults;
                                const recall1 = evalData?.retrieval_metrics?.recall_at_1 || evalData?.metrics?.retrieval_metrics?.recall_at_1;
                                return typeof recall1 === 'number' ? recall1.toFixed(3) : '미평가';
                              } catch { return '미평가'; }
                            })()}
                          </span>
                        </td>

                        <td className="py-3 px-4 text-gray-600 dark:text-gray-400">
                          {(() => {
                            try {
                              // 1. 모델 객체에서 직접 test_samples 확인 (API 응답의 최상위 레벨)
                              if (typeof model.test_samples === 'number' && model.test_samples > 0) {
                                return model.test_samples.toLocaleString();
                              }

                              // 2. evaluation_results 객체에서 확인
                              const evalData = model.evaluation_results || model.evaluationResults;
                              if (evalData && typeof evalData.test_samples === 'number' && evalData.test_samples > 0) {
                                return evalData.test_samples.toLocaleString();
                              }

                              // 3. 평가 데이터에 총 샘플 수 정보가 있는 경우
                              if (evalData && typeof evalData.total_samples === 'number' && evalData.total_samples > 0) {
                                return evalData.total_samples.toLocaleString();
                              }

                              // 4. 데이터셋 정보 기반 추정치 (20% 테스트셋 사용)
                              if (model.trainingData?.pairCount && typeof model.trainingData.pairCount === 'number') {
                                const estimatedTestSamples = Math.floor(model.trainingData.pairCount * 0.2);
                                return `${estimatedTestSamples}`;
                              }

                              // 5. 기본값
                              return '일반 테스트셋';
                            } catch (error) {
                              console.error('Error processing test samples:', error);
                              return '일반 테스트셋';
                            }
                          })()}
                        </td>
                        <td className="py-3 px-4">
                          <span className={`px-2 py-1 text-xs font-medium rounded-full ${
                            !model.evaluated 
                              ? 'bg-gray-100 text-gray-600 dark:bg-gray-700 dark:text-gray-400'
                              : 'bg-green-100 text-green-700 dark:bg-green-900 dark:text-green-300'
                          }`}>
                            {!model.evaluated ? '미평가' : '평가완료'}
                            {model.evaluated && model.evaluationDate && (
                              <span className="ml-1 text-xs opacity-75">
                                ({new Date(model.evaluationDate).toLocaleDateString('ko-KR')})
                              </span>
                            )}
                          </span>
                        </td>
                        <td className="py-3 px-4">
                          <button
                            onClick={() => handleEvaluateModel(model.id)}
                            disabled={!ragServerConnected || evaluationProgress.isEvaluating}
                            className={`px-3 py-1 text-xs font-medium rounded transition-colors ${
                              !ragServerConnected || evaluationProgress.isEvaluating
                                ? 'bg-blue-100 text-blue-700 dark:bg-blue-900 dark:text-blue-300 cursor-not-allowed'
                                : 'bg-blue-100 hover:bg-blue-200 dark:bg-blue-900'
                            }`}
                          >
                            개별 평가
                          </button>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>

              {models.every(m => !m.evaluated) && (
                <div className="text-center py-8">
                  <div className="text-gray-400 mb-4">
                    <svg className="w-16 h-16 mx-auto" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1} d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
                    </svg>
                  </div>
                  <p className="text-gray-600 dark:text-gray-400 mb-4">
                    아직 실제 성능 평가가 수행되지 않았습니다.
                  </p>
                  <p className="text-sm text-gray-500 mb-4">
                    위의 "모든 모델 평가 실행" 버튼을 클릭하여 실제 성능을 측정하세요.
                  </p>
                </div>
              )}
            </div>

            {/* 성능 개선 권장사항 */}
            <div className="bg-[var(--card-bg)] rounded-xl shadow-sm border border-[var(--card-border)] p-6">
              <h4 className="text-base font-semibold text-gray-900 dark:text-white mb-4">
                💡 성능 개선 권장사항
              </h4>
              
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="bg-blue-50 dark:bg-blue-900/20 rounded-lg p-4">
                  <h4 className="font-semibold text-blue-800 dark:text-blue-200 mb-2">
                    📈 성능 향상 방법
                  </h4>
                  <ul className="text-sm text-blue-700 dark:text-blue-300 space-y-1">
                    <li>• 더 많은 도메인 특화 데이터 수집</li>
                    <li>• 네거티브 샘플 품질 개선</li>
                    <li>• 하이퍼파라미터 튜닝 (learning rate, batch size)</li>
                    <li>• 데이터 증강 기법 적용</li>
                  </ul>
                </div>
                
                <div className="bg-green-50 dark:bg-green-900/20 rounded-lg p-4">
                  <h4 className="font-semibold text-green-800 dark:text-green-200 mb-2">
                    🎯 평가 지표 이해
                  </h4>
                  <ul className="text-sm text-green-700 dark:text-green-300 space-y-1">
                    <li>• <strong>정밀도:</strong> 검색된 결과 중 관련 문서 비율</li>
                    <li>• <strong>재현율:</strong> 전체 관련 문서 중 검색된 비율</li>
                    <li>• <strong>F1 Score:</strong> 정밀도와 재현율의 조화평균</li>
                    <li>• <strong>nDCG@5:</strong> 상위 5개 결과의 순위 품질 (높을수록 좋음)</li>
                    <li>• <strong>MRR:</strong> 첫 번째 관련 문서 찾기 성능 (높을수록 좋음)</li>
                    <li>• <strong>Recall@1:</strong> 상위 1개에서 관련 문서 찾기 비율</li>
                    <li>• <strong>실제 검색 성능:</strong> nDCG@5, MRR이 더 중요한 지표</li>
                  </ul>
                </div>
              </div>
            </div>
          </div>
        )}
        </div>
      </div>

      {/* 데이터셋 생성 모달 */}
      {showDatasetModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white dark:bg-gray-800 rounded-xl shadow-xl max-w-3xl w-full mx-4 max-h-[90vh] overflow-y-auto">
            <div className="p-6">
              <div className="flex justify-between items-center mb-6">
                <h2 className="text-xl font-bold text-gray-900 dark:text-white">
                  새 데이터셋 생성
                </h2>
                <button
                  onClick={() => {
                    setShowDatasetModal(false);
                    resetDatasetForm();
                  }}
                  className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-200"
                >
                  ✕
                </button>
              </div>

              {/* 단계 1: 생성 방법 선택 */}
              {!selectedDatasetMethod && (
                <div className="space-y-6">
                  <div className="text-center mb-6">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                      데이터셋 생성 방법을 선택하세요
                    </h3>
                    <p className="text-sm text-gray-600 dark:text-gray-400">
                      용도에 맞는 방법으로 고품질 훈련 데이터셋을 생성할 수 있습니다
                    </p>
                  </div>

                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    {datasetCreationMethods.map((method) => (
                      <div
                        key={method.id}
                        onClick={() => setSelectedDatasetMethod(method.id)}
                        className={`border-2 rounded-lg p-4 cursor-pointer transition-all hover:border-blue-300 hover:bg-blue-50 dark:hover:bg-blue-900/20 ${
                          method.recommended 
                            ? 'border-blue-200 bg-blue-50/50 dark:bg-blue-900/10 dark:border-blue-700' 
                            : 'border-gray-200 dark:border-gray-700'
                        }`}
                      >
                        <div className="flex items-start justify-between mb-3">
                          <div className="flex items-center space-x-2">
                            <span className="text-2xl">{method.icon}</span>
                            <div>
                              <h3 className="font-semibold text-gray-900 dark:text-white">
                                {method.title}
                              </h3>
                            </div>
                          </div>
                          <div className="flex items-center space-x-2">
                          <span className={`px-2 py-0.5 text-xs font-medium rounded-full ${
                            method.difficulty === 'easy' 
                              ? 'bg-green-100 text-green-700 dark:bg-green-900 dark:text-green-300'
                              : method.difficulty === 'medium'
                              ? 'bg-yellow-100 text-yellow-700 dark:bg-yellow-900 dark:text-yellow-300'
                              : 'bg-red-100 text-red-700 dark:bg-red-900 dark:text-red-300'
                          }`}>
                            {method.difficulty === 'easy' ? '쉬움' : method.difficulty === 'medium' ? '보통' : '어려움'}
                          </span>
                          {method.recommended && (
                            <span className="inline-block px-2 py-0.5 text-xs font-medium bg-green-100 text-green-700 dark:bg-green-900 dark:text-green-300 rounded-full">
                              추천
                            </span>
                          )}
                          </div>
                        </div>
                        <p className="text-sm text-gray-600 dark:text-gray-400">
                          {method.description}
                        </p>
                      </div>
                    ))}
                  </div>
                </div>
              )}

              {/* 단계 2: 선택된 방법에 따른 폼 */}
              {selectedDatasetMethod && (
                <div className="space-y-6">
                  <div className="bg-blue-50 dark:bg-blue-900/20 rounded-lg p-4">
                    <div className="flex items-center space-x-2 mb-2">
                      <span className="text-lg">
                        {datasetCreationMethods.find(m => m.id === selectedDatasetMethod)?.icon}
                      </span>
                      <h3 className="font-semibold text-blue-800 dark:text-blue-200">
                        {datasetCreationMethods.find(m => m.id === selectedDatasetMethod)?.title}
                      </h3>
                      <button
                        onClick={() => setSelectedDatasetMethod('')}
                        className="text-blue-600 hover:text-blue-800 dark:text-blue-300 dark:hover:text-blue-100 text-sm"
                      >
                        방법 변경
                      </button>
                    </div>
                    <p className="text-sm text-blue-700 dark:text-blue-300">
                      {datasetCreationMethods.find(m => m.id === selectedDatasetMethod)?.description}
                    </p>
                  </div>

                  {/* 공통 입력 필드 */}
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        데이터셋 이름 *
                      </label>
                      <input
                        type="text"
                        value={newDatasetName}
                        onChange={(e) => setNewDatasetName(e.target.value)}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                        placeholder="예: 의료 AI 도메인 데이터셋"
                      />
                    </div>
                    
                    <div>
                      <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        설명
                      </label>
                      <input
                        type="text"
                        value={newDatasetDescription}
                        onChange={(e) => setNewDatasetDescription(e.target.value)}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                        placeholder="데이터셋에 대한 설명"
                      />
                    </div>

                    {/* QA 개수 입력 필드 (문서 + LLM 생성 방식에서만 표시) */}
                    {selectedDatasetMethod === 'document-llm' && (
                      <div>
                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                          목표 QA 개수
                        </label>
                        <input
                          type="number"
                          min="10"
                          max="500"
                          value={targetQACount}
                          onChange={(e) => setTargetQACount(parseInt(e.target.value) || 25)}
                          className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                          placeholder="25"
                        />
                        <div className="text-xs text-gray-500 mt-1">
                                                      10-500개 (기본: 25개)
                        </div>
                      </div>
                    )}
                  </div>

                  {/* 프로바이더 선택 (문서 + LLM 생성 방식에서만 표시) */}
                  {selectedDatasetMethod === 'document-llm' && (
                    <div className="space-y-4 mb-6 p-4 bg-purple-50 dark:bg-purple-900/20 rounded-lg border border-purple-200 dark:border-purple-700">
                      <h4 className="text-sm font-semibold text-purple-800 dark:text-purple-200 flex items-center">
                        <span className="text-lg mr-2">🤖</span>
                        AI 모델 선택
                      </h4>
                      
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        {/* 프로바이더 선택 */}
                        <div>
                          <label className="block text-sm font-medium text-purple-700 dark:text-purple-300 mb-2">
                            프로바이더
                          </label>
                          <select
                            value={selectedProvider}
                            onChange={(e) => handleProviderChange(e.target.value)}
                            disabled={isLoadingProviders}
                            className="w-full px-3 py-2 border border-purple-300 dark:border-purple-600 rounded-md focus:outline-none focus:ring-2 focus:ring-purple-500 dark:bg-gray-700 dark:text-white"
                          >
                            {isLoadingProviders ? (
                              <option>로딩 중...</option>
                            ) : availableProviders.length === 0 ? (
                              <option>프로바이더 없음 (디버그: {JSON.stringify({isLoadingProviders, availableProvidersCount: availableProviders.length})})</option>
                            ) : (
                              availableProviders.map((provider) => (
                                <option 
                                  key={provider.key} 
                                  value={provider.key}
                                  disabled={provider.requiresApiKey && !provider.apiKeyConfigured}
                                >
                                  {provider.name} {provider.requiresApiKey && !provider.apiKeyConfigured ? '(API 키 필요)' : ''}
                                </option>
                              ))
                            )}
                          </select>
                        </div>

                        {/* 모델 선택 */}
                        <div>
                          <label className="block text-sm font-medium text-purple-700 dark:text-purple-300 mb-2">
                            모델
                          </label>
                          <select
                            value={selectedModel}
                            onChange={(e) => setSelectedModel(e.target.value)}
                            disabled={isLoadingProviders || !providers[selectedProvider]?.models?.length}
                            className="w-full px-3 py-2 border border-purple-300 dark:border-purple-600 rounded-md focus:outline-none focus:ring-2 focus:ring-purple-500 dark:bg-gray-700 dark:text-white"
                          >
                            {!providers[selectedProvider]?.models?.length ? (
                              <option>모델 없음 (디버그: provider={selectedProvider}, modelsLength={providers[selectedProvider]?.models?.length || 0})</option>
                            ) : (
                              providers[selectedProvider].models.map((model: any) => (
                                <option key={model.id || model.name} value={model.id || model.name}>
                                  {model.name || model.id}
                                </option>
                              ))
                            )}
                          </select>
                        </div>
                      </div>
                    </div>
                  )}

                  {/* 파일 업로드 */}
                  {(selectedDatasetMethod === 'file-upload' || selectedDatasetMethod === 'document-llm' ||
                    selectedDatasetMethod === 'evaluation-universal' || selectedDatasetMethod === 'evaluation-domain') && (
                    <div>
                      <label className="block text-sm font-medium text-amber-800 dark:text-amber-300 mb-2">
                        {selectedDatasetMethod === 'file-upload' ? '데이터셋 파일 *' :
                         selectedDatasetMethod === 'document-llm' ? '문서 파일 *' :
                         selectedDatasetMethod === 'evaluation-universal' ? '범용 평가 데이터셋 파일 *' :
                         '도메인 평가 데이터셋 파일 *'}
                      </label>
                      <div 
                        className={`border-2 border-dashed rounded-lg p-6 transition-all duration-300 ${
                          isDragging 
                            ? 'border-blue-400 bg-blue-50 dark:border-blue-500 dark:bg-blue-900/20' 
                            : 'border-amber-300 dark:border-amber-600 bg-amber-50/30 dark:bg-amber-900/10'
                        }`}
                        onDragEnter={handleDragEnter}
                        onDragLeave={handleDragLeave}
                        onDragOver={handleDragOver}
                        onDrop={handleDrop}
                      >
                        <div className="text-center">
                          <input
                            type="file"
                            onChange={(e) => setUploadedFile(e.target.files?.[0] || null)}
                            accept={(selectedDatasetMethod === 'file-upload' ||
                                    selectedDatasetMethod === 'evaluation-universal' ||
                                    selectedDatasetMethod === 'evaluation-domain')
                              ? '.json,.csv,.txt,application/json,text/csv,text/plain'
                              : '.txt,.md,.pdf,.docx,.doc,.hwp,.hwpx,.pptx,.ppt,text/plain,text/markdown,application/pdf,application/vnd.openxmlformats-officedocument.wordprocessingml.document,application/msword,application/vnd.ms-powerpoint,application/vnd.openxmlformats-officedocument.presentationml.presentation'
                            }
                            className="hidden"
                            id="file-upload"
                          />
                          <label htmlFor="file-upload" className="cursor-pointer">
                            <div className={`mb-2 ${isDragging ? 'text-blue-500' : 'text-amber-500'}`}>
                              {isDragging ? (
                                <svg className="w-12 h-12 mx-auto animate-bounce" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M9 12l3 3m0 0l3-3m-3 3V9" />
                                </svg>
                              ) : (
                                <svg className="w-12 h-12 mx-auto" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1} d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12" />
                                </svg>
                              )}
                            </div>
                            <div className={`text-sm ${isDragging ? 'text-blue-700 dark:text-blue-300' : 'text-amber-700 dark:text-amber-400'}`}>
                              {isDragging ? (
                                '파일을 여기에 놓으세요'
                              ) : (
                                selectedDatasetMethod === 'file-upload'
                                  ? '파일을 드래그하여 놓거나 클릭하여 선택하세요'
                                  : selectedDatasetMethod === 'document-llm'
                                  ? '문서를 드래그하여 놓거나 클릭하여 선택하세요'
                                  : selectedDatasetMethod === 'evaluation-universal'
                                  ? '범용 평가 데이터셋을 드래그하여 놓거나 클릭하여 선택하세요'
                                  : '도메인 평가 데이터셋을 드래그하여 놓거나 클릭하여 선택하세요'
                              )}
                            </div>
                            <div className={`text-xs mt-1 ${isDragging ? 'text-blue-600 dark:text-blue-400' : 'text-amber-600 dark:text-amber-500'}`}>
                              {(selectedDatasetMethod === 'file-upload' ||
                                selectedDatasetMethod === 'evaluation-universal' ||
                                selectedDatasetMethod === 'evaluation-domain')
                                ? 'JSON, CSV, TXT 파일 지원'
                                : 'TXT, MD, PDF, DOCX, DOC, HWP, HWPX, PPTX, PPT 파일 지원'
                              }
                            </div>
                          </label>
                          {uploadedFile && (
                            <div className="mt-3 text-sm text-green-600 dark:text-green-400">
                              ✓ {uploadedFile.name} ({(uploadedFile.size / 1024).toFixed(1)}KB)
                            </div>
                          )}
                        </div>
                      </div>
                      {selectedDatasetMethod === 'file-upload' && (
                        <div className="mt-2 text-xs text-gray-500">
                          <strong>JSON 형식 예시:</strong> {`{"examples": [{"query": "질문", "positive": "관련 문서", "negative": "무관한 문서"}]}`}
                        </div>
                      )}
                      {(selectedDatasetMethod === 'evaluation-universal' || selectedDatasetMethod === 'evaluation-domain') && (
                        <div className="mt-2 p-3 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-200 dark:border-blue-700">
                          <div className="text-xs text-blue-800 dark:text-blue-200 space-y-2">
                            <div className="font-semibold flex items-center">
                              <span className="text-base mr-2">📋</span>
                              평가 데이터셋 안내
                            </div>
                            <div>
                              <strong>목적:</strong> {selectedDatasetMethod === 'evaluation-universal'
                                ? '범용 임베딩 모델의 일반적인 검색 성능을 평가합니다.'
                                : '특정 도메인에 특화된 임베딩 모델의 전문 성능을 평가합니다.'
                              }
                            </div>
                            <div>
                              <strong>데이터 분리:</strong> 학습 데이터와 완전히 분리된 데이터로 구성하여 객관적인 평가가 가능합니다.
                            </div>
                            <div>
                              <strong>JSON 형식:</strong> {`{"examples": [{"query": "평가 질문", "positive": "정답 문서", "negative": "오답 문서"}]}`}
                            </div>
                          </div>
                        </div>
                      )}
                    </div>
                  )}




                </div>
              )}
              
              <div className="flex justify-end gap-4 mt-6">
                <button
                  onClick={() => {
                    setShowDatasetModal(false);
                    resetDatasetForm();
                  }}
                  className="px-4 py-2 text-gray-600 hover:text-gray-800 dark:text-gray-400 dark:hover:text-gray-200 transition-colors"
                >
                  취소
                </button>
                {selectedDatasetMethod && (
                  <div className="space-y-2">
                    <button
                      onClick={handleCreateDataset}
                      disabled={
                        datasetCreationProgress.isCreating || 
                        (selectedDatasetMethod === 'document-llm' && (!selectedProvider || !selectedModel))
                      }
                      className={`px-4 py-2 rounded-lg font-medium transition-colors ${
                        datasetCreationProgress.isCreating || 
                        (selectedDatasetMethod === 'document-llm' && (!selectedProvider || !selectedModel))
                          ? 'bg-gray-400 text-gray-700 dark:bg-gray-600 dark:text-gray-400 cursor-not-allowed'
                          : 'bg-blue-600 text-white hover:bg-blue-700 dark:bg-blue-600 dark:hover:bg-blue-700'
                      }`}
                    >
                      {datasetCreationProgress.isCreating ? (
                        <div className="flex items-center space-x-2">
                          <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white" />
                          <span>생성 중...</span>
                        </div>
                      ) : (
                        '데이터셋 생성'
                      )}
                    </button>

                  </div>
                )}
              </div>
            </div>
          </div>
        </div>
      )}

      {/* 알림 모달 */}
      <NotificationModalComponent modal={notificationModal} onClose={closeNotification} />
      
      {/* 데이터셋 생성 프로그레스 바 */}
      <DatasetCreationProgressBar progress={datasetCreationProgress} />
      
      {/* 데이터셋 상세보기 모달 */}
      {showDatasetDetailModal && (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
          <div className="bg-white dark:bg-gray-800 rounded-xl max-w-4xl w-full max-h-[90vh] overflow-hidden">
            <div className="flex items-center justify-between p-6 border-b border-gray-200 dark:border-gray-700">
              <h2 className="text-xl font-semibold text-gray-900 dark:text-white">
                📊 데이터셋 상세보기
              </h2>
              <button
                onClick={() => {
                  setShowDatasetDetailModal(false);
                  setSelectedDatasetForDetail(null);
                  setDatasetDetails(null);
                }}
                className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-200 transition-colors"
              >
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                </svg>
              </button>
            </div>

            {loadingDatasetDetails ? (
              <div className="p-8 text-center">
                <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500 mx-auto mb-4"></div>
                <p className="text-gray-600 dark:text-gray-400">데이터셋 내용을 불러오는 중...</p>
              </div>
            ) : (
              <div className="overflow-y-auto max-h-[calc(90vh-120px)]">
                {/* 데이터셋 정보 헤더 */}
                {selectedDatasetForDetail && (
                  <div className="p-6 bg-gray-50 dark:bg-gray-700/50 border-b border-gray-200 dark:border-gray-600">
                    <div className="flex items-center justify-between mb-4">
                      <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                        {selectedDatasetForDetail.name}
                      </h3>
                      <div className="flex items-center gap-2">
                        <span className={`px-2 py-1 text-xs font-medium whitespace-nowrap rounded-full ${
                          selectedDatasetForDetail.status === 'ready' 
                            ? 'bg-green-100 text-green-700 dark:bg-green-900 dark:text-green-300'
                            : 'bg-yellow-100 text-yellow-700 dark:bg-yellow-900 dark:text-yellow-300'
                        }`}>
                          {selectedDatasetForDetail.status === 'ready' ? '준비됨' : '처리중'}
                        </span>
                        <span className="px-2 py-1 text-xs font-medium rounded-full bg-blue-100 text-blue-700 dark:bg-blue-900 dark:text-blue-300">
                          {selectedDatasetForDetail.type || '문서 기반'}
                        </span>
                      </div>
                    </div>
                    
                    <p className="text-sm text-gray-600 dark:text-gray-400 mb-4">
                      {selectedDatasetForDetail.description}
                    </p>
                    
                    <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                      <div>
                        <span className="text-gray-500">총 예제 수</span>
                        <div className="font-medium">{((selectedDatasetForDetail.size || selectedDatasetForDetail.pairCount) || 0).toLocaleString()}</div>
                      </div>
                      <div>
                        <span className="text-gray-500">데이터셋 타입</span>
                        <div className="font-medium">{selectedDatasetForDetail.type || '문서 기반'}</div>
                      </div>
                      <div>
                        <span className="text-gray-500">생성일</span>
                        <div className="font-medium">{new Date(selectedDatasetForDetail.createdAt).toLocaleDateString('ko-KR')}</div>
                      </div>
                      <div>
                        <span className="text-gray-500">ID</span>
                        <div className="font-medium font-mono text-xs">{selectedDatasetForDetail.id}</div>
                      </div>
                    </div>
                  </div>
                )}

                {/* 데이터셋 예제 목록 */}
                {datasetDetails && datasetDetails.examples ? (
                  <div className="p-6">
                    <div className="mb-4">
                      <h4 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                        🔍 데이터셋 예제 ({datasetDetails.examples.length.toLocaleString()}개)
                      </h4>
                      <p className="text-sm text-gray-600 dark:text-gray-400">
                        아래는 데이터셋에 포함된 질문-답변 쌍들입니다. 스크롤하여 모든 예제를 확인할 수 있습니다.
                      </p>
                    </div>
                    
                    <div className="space-y-4 max-h-96 overflow-y-auto">
                      {datasetDetails.examples.map((example: any, index: number) => (
                        <div key={index} className="bg-white dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600 p-4">
                          <div className="flex items-start justify-between mb-3">
                            <span className="inline-block px-2 py-1 bg-blue-100 text-blue-700 dark:bg-blue-900 dark:text-blue-300 text-xs font-medium rounded-full">
                              #{index + 1}
                            </span>
                            {example.type && (
                              <span className={`px-2 py-1 text-xs font-medium rounded-full ${
                                example.type.includes('positive') 
                                  ? 'bg-green-100 text-green-700 dark:bg-green-900 dark:text-green-300'
                                  : 'bg-red-100 text-red-700 dark:bg-red-900 dark:text-red-300'
                              }`}>
                                {example.type}
                              </span>
                            )}
                          </div>
                          
                          <div className="space-y-3">
                            <div>
                              <div className="flex items-center gap-2 mb-1">
                                <span className="text-sm font-medium text-gray-700 dark:text-gray-300">❓ 질문</span>
                              </div>
                              <div className="bg-gray-50 dark:bg-gray-600 rounded-md p-3 text-sm">
                                {example.query || example.question || 'N/A'}
                              </div>
                            </div>
                            
                            <div>
                              <div className="flex items-center gap-2 mb-1">
                                <span className="text-sm font-medium text-green-700 dark:text-green-300">✅ 긍정적 답변</span>
                              </div>
                              <div className="bg-green-50 dark:bg-green-900/20 rounded-md p-3 text-sm">
                                {example.positive || example.document || example.answer || 'N/A'}
                              </div>
                            </div>
                            
                            {example.negative && (
                              <div>
                                <div className="flex items-center gap-2 mb-1">
                                  <span className="text-sm font-medium text-red-700 dark:text-red-300">❌ 부정적 답변</span>
                                </div>
                                <div className="bg-red-50 dark:bg-red-900/20 rounded-md p-3 text-sm">
                                  {example.negative}
                                </div>
                              </div>
                            )}
                            
                            {example.score !== undefined && (
                              <div>
                                <div className="flex items-center gap-2 mb-1">
                                  <span className="text-sm font-medium text-gray-700 dark:text-gray-300">📊 점수</span>
                                </div>
                                <div className="bg-gray-50 dark:bg-gray-600 rounded-md p-3 text-sm">
                                  {example.score}
                                </div>
                              </div>
                            )}
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                ) : (
                  <div className="p-8 text-center text-gray-500 dark:text-gray-400">
                    데이터셋 예제를 불러올 수 없습니다.
                  </div>
                )}
              </div>
            )}
          </div>
        </div>
      )}


      {/* LLM 데이터셋 관리 탭 */}
      {activeTab === 'llmDatasets' && (
        <div className="w-full max-w-none p-6 space-y-6">
          {/* 데이터셋 관리 설명 카드 */}
          <div className="bg-gradient-to-r from-green-50 to-emerald-50 dark:from-green-900/20 dark:to-emerald-900/20 rounded-xl p-6 border border-green-200 dark:border-green-700">
            <div className="flex items-start space-x-4">
              <div className="flex-shrink-0">
                <div className="w-12 h-12 bg-green-100 dark:bg-green-800 rounded-lg flex items-center justify-center">
                  <span className="text-2xl">📁</span>
                </div>
              </div>
              <div className="flex-1">
                <h3 className="text-lg font-semibold text-green-900 dark:text-green-200 mb-2">
                  LLM 학습 데이터셋 생성 및 관리
                </h3>
                <p className="text-green-700 dark:text-green-300 text-sm leading-relaxed">
                  LLM 파인튜닝에 필요한 고품질 훈련 데이터셋을 생성하고 관리합니다. 
                  웹 크롤링, RAG 변환, 문서 기반 생성 등 다양한 방법으로 데이터셋을 만들 수 있습니다.
                </p>
                <div className="flex flex-wrap gap-2 mt-3">
                  <span className="px-2 py-1 bg-green-100 dark:bg-green-800 text-green-700 dark:text-green-300 text-xs rounded-full">웹 크롤링</span>
                  <span className="px-2 py-1 bg-green-100 dark:bg-green-800 text-green-700 dark:text-green-300 text-xs rounded-full">RAG 변환</span>
                  <span className="px-2 py-1 bg-green-100 dark:bg-green-800 text-green-700 dark:text-green-300 text-xs rounded-full">문서 기반</span>
                  <span className="px-2 py-1 bg-green-100 dark:bg-green-800 text-green-700 dark:text-green-300 text-xs rounded-full">파일 업로드</span>
                </div>
              </div>
            </div>
          </div>
          <LLMDatasetTab />
        </div>
      )}

      {/* LLM 모델 학습 탭 */}
      {activeTab === 'llmTraining' && (
        <div className="w-full max-w-none p-6 space-y-6">
          {/* 모델 학습 관리 설명 카드 */}
          <div className="bg-gradient-to-r from-purple-50 to-violet-50 dark:from-purple-900/20 dark:to-violet-900/20 rounded-xl p-6 border border-purple-200 dark:border-purple-700">
            <div className="flex items-start space-x-4">
              <div className="flex-shrink-0">
                <div className="w-12 h-12 bg-purple-100 dark:bg-purple-800 rounded-lg flex items-center justify-center">
                  <span className="text-2xl">🔧</span>
                </div>
              </div>
              <div className="flex-1">
                <h3 className="text-lg font-semibold text-purple-900 dark:text-purple-200 mb-2">
                  LLM 파인튜닝 학습
                </h3>
                <p className="text-purple-700 dark:text-purple-300 text-sm leading-relaxed">
                  생성된 데이터셋을 사용하여 LLM 모델을 파인튜닝합니다. 
                  LoRA(저순위 적응) 또는 전체 파인튜닝을 선택할 수 있으며, 실시간으로 진행 상황을 모니터링할 수 있습니다.
                </p>
                <div className="flex flex-wrap gap-2 mt-3">
                  <span className="px-2 py-1 bg-purple-100 dark:bg-purple-800 text-purple-700 dark:text-purple-300 text-xs rounded-full">LoRA 지원</span>
                  <span className="px-2 py-1 bg-purple-100 dark:bg-purple-800 text-purple-700 dark:text-purple-300 text-xs rounded-full">풀 파인튜닝</span>
                  <span className="px-2 py-1 bg-purple-100 dark:bg-purple-800 text-purple-700 dark:text-purple-300 text-xs rounded-full">실시간 모니터링</span>
                  <span className="px-2 py-1 bg-purple-100 dark:bg-purple-800 text-purple-700 dark:text-purple-300 text-xs rounded-full">GPU/CPU 지원</span>
                </div>
              </div>
            </div>
          </div>
          <LLMTrainingTab />
        </div>
      )}

      {/* LLM 모델 관리 탭 */}
      {activeTab === 'llmModels' && (
        <div className="w-full max-w-none p-6 space-y-6">
          {/* 모델 관리 설명 카드 */}
          <div className="bg-gradient-to-r from-blue-50 to-indigo-50 dark:from-blue-900/20 dark:to-indigo-900/20 rounded-xl p-6 border border-blue-200 dark:border-blue-700">
            <div className="flex items-start space-x-4">
              <div className="flex-shrink-0">
                <div className="w-12 h-12 bg-blue-100 dark:bg-blue-800 rounded-lg flex items-center justify-center">
                  <span className="text-2xl">🧠</span>
                </div>
              </div>
              <div className="flex-1">
                <h3 className="text-lg font-semibold text-blue-900 dark:text-blue-200 mb-2">
                  LLM 모델 현황 및 관리
                </h3>
                <p className="text-blue-700 dark:text-blue-300 text-sm leading-relaxed">
                  파인튜닝된 LLM 모델들을 관리하고 배포 상태를 확인할 수 있습니다. 
                  모델의 성능 지표를 비교하고, 원하는 모델을 활성화하여 실제 서비스에 적용할 수 있습니다.
                </p>
                <div className="flex flex-wrap gap-2 mt-3">
                  <span className="px-2 py-1 bg-blue-100 dark:bg-blue-800 text-blue-700 dark:text-blue-300 text-xs rounded-full">모델 상태 확인</span>
                  <span className="px-2 py-1 bg-blue-100 dark:bg-blue-800 text-blue-700 dark:text-blue-300 text-xs rounded-full">성능 지표 비교</span>
                  <span className="px-2 py-1 bg-blue-100 dark:bg-blue-800 text-blue-700 dark:text-blue-300 text-xs rounded-full">모델 다운로드</span>
                  <span className="px-2 py-1 bg-blue-100 dark:bg-blue-800 text-blue-700 dark:text-blue-300 text-xs rounded-full">모델 업로드</span>
                </div>
              </div>
            </div>
          </div>
          <LLMModelsTab />
        </div>
      )}

      {/* LLM 플레이그라운드 탭 */}
      {activeTab === 'llmPlayground' && (
        <div className="w-full max-w-none p-6 space-y-6 h-[calc(100vh-8rem)]">
          {/* 플레이그라운드 설명 카드 */}
          <div className="bg-gradient-to-r from-orange-50 to-red-50 dark:from-orange-900/20 dark:to-red-900/20 rounded-xl p-6 border border-orange-200 dark:border-orange-700">
            <div className="flex items-start space-x-4">
              <div className="flex-shrink-0">
                <div className="w-12 h-12 bg-orange-100 dark:bg-orange-800 rounded-lg flex items-center justify-center">
                  <span className="text-2xl">🎮</span>
                </div>
              </div>
              <div className="flex-1">
                <h3 className="text-lg font-semibold text-orange-900 dark:text-orange-200 mb-2">
                  LLM 플레이그라운드
                </h3>
                <p className="text-orange-700 dark:text-orange-300 text-sm leading-relaxed">
                  파인튜닝된 LLM 모델과 직접 대화하며 성능을 테스트할 수 있습니다. 
                  다양한 파라미터를 조정하며 최적의 설정을 찾아보세요.
                </p>
                <div className="flex flex-wrap gap-2 mt-3">
                  <span className="px-2 py-1 bg-orange-100 dark:bg-orange-800 text-orange-700 dark:text-orange-300 text-xs rounded-full">실시간 대화</span>
                  <span className="px-2 py-1 bg-orange-100 dark:bg-orange-800 text-orange-700 dark:text-orange-300 text-xs rounded-full">파라미터 조정</span>
                  <span className="px-2 py-1 bg-orange-100 dark:bg-orange-800 text-orange-700 dark:text-orange-300 text-xs rounded-full">대화 이력 저장</span>
                  <span className="px-2 py-1 bg-orange-100 dark:bg-orange-800 text-orange-700 dark:text-orange-300 text-xs rounded-full">성능 테스트</span>
                </div>
              </div>
            </div>
          </div>
          <div className="h-[calc(100%-10rem)]">
            <LLMPlaygroundTab />
          </div>
        </div>
      )}
    </div>
  );
} 