"use client";
import type { User } from "../../../types/user";
import { useState } from "react";
import { externalApiClient } from "@/lib/apiClient";

interface UserTableProps {
  users: User[];
  onEdit: (user: User) => void;
  onDelete: (user: User) => void;
  onRegenApiKey?: () => void;
  onShowStats?: (user: User) => void;
  onEditSystemPrompt?: (user: User) => void;
}

// 안전한 클립보드 복사 함수
const copyToClipboard = async (text: string): Promise<boolean> => {
  try {
    // 최신 Clipboard API 시도
    if (navigator.clipboard && navigator.clipboard.writeText) {
      await navigator.clipboard.writeText(text);
      return true;
    }
    
    // 폴백: 임시 textarea 사용
    const textarea = document.createElement('textarea');
    textarea.value = text;
    textarea.style.position = 'fixed';
    textarea.style.left = '-999999px';
    textarea.style.top = '-999999px';
    document.body.appendChild(textarea);
    textarea.focus();
    textarea.select();
    
    try {
      // @ts-ignore - deprecated API이지만 폴백으로 필요
      const result = document.execCommand('copy');
      document.body.removeChild(textarea);
      return result;
    } catch {
      document.body.removeChild(textarea);
      return false;
    }
  } catch (err) {
    console.error('클립보드 복사 실패:', err);
    return false;
  }
};

export default function UserTable({ users, onEdit, onDelete, onRegenApiKey, onShowStats, onEditSystemPrompt }: UserTableProps) {
  const [showCopyModal, setShowCopyModal] = useState(false);
  const [showRegenModal, setShowRegenModal] = useState(false);
  const [copyError, setCopyError] = useState(false);

  if (!users || users.length === 0) {
    return (
      <div className="text-center text-gray-500 py-8 dark:text-gray-400">
        사용자 데이터가 없습니다.
      </div>
    );
  }
  return (
    <div className="overflow-x-auto rounded-lg shadow border border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-900 w-full">
      <table className="min-w-full w-full font-sans text-sm table-auto">
        <thead>
          <tr className="bg-gradient-to-r from-gray-100 via-gray-50 to-gray-100 dark:from-gray-800 dark:via-gray-900 dark:to-gray-800 text-gray-700 dark:text-gray-200 font-semibold">
            <th className="px-3 py-3 text-left w-20">아이디</th>
            <th className="px-3 py-3 text-left w-44">이메일</th>
            <th className="px-3 py-3 text-left w-24">이름</th>
            <th className="px-3 py-3 text-center w-16">역할</th>
            <th className="px-3 py-3 text-center w-48 min-w-[200px]">API키</th>
            <th className="px-3 py-3 text-center w-12">상태</th>
            <th className="px-3 py-3 text-center w-20">생성일</th>
            <th className="px-3 py-3 text-center w-32 min-w-[130px] whitespace-nowrap">관리</th>
          </tr>
        </thead>
        <tbody>
          {users.map((user, idx) => (
            <tr
              key={user.id}
              className={`border-t border-gray-200 dark:border-gray-700 ${
                idx % 2 === 0
                  ? 'bg-white dark:bg-gray-900'
                  : 'bg-gray-50 dark:bg-gray-800'
              } hover:bg-blue-50 dark:hover:bg-gray-800 transition`}
            >
              <td className="px-3 py-2 text-left align-middle">
                <button
                  onClick={() => onShowStats?.(user)}
                  className="text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-300 underline hover:no-underline transition font-medium"
                  title="사용자 통계 보기"
                >
                  {user.username}
                </button>
              </td>
              <td className="px-3 py-2 text-left align-middle">{user.email}</td>
              <td className="px-3 py-2 text-left align-middle">{user.name}</td>
              <td className="px-3 py-2 text-center align-middle capitalize">{user.role}</td>
              <td className="px-3 py-2 text-center align-middle min-w-[200px] whitespace-nowrap">
                {(user.apiKeys?.length ?? 0) > 0 && user.apiKeys?.[0]?.key ? (
                  <div className="flex items-center gap-1 justify-center">
                    <span className="font-mono text-[11px] select-all text-gray-700 dark:text-gray-200">
                      {user.apiKeys[0].key.slice(0, 6)}...{user.apiKeys[0].key.slice(-4)}
                    </span>
                    <button
                      className="px-1.5 py-0.5 text-[10px] border border-gray-400 dark:border-gray-600 rounded bg-white dark:bg-gray-900 hover:bg-gray-100 dark:hover:bg-gray-800 transition ml-1"
                      onClick={async () => {
                        if (user.apiKeys?.[0]?.key) {
                          const success = await copyToClipboard(user.apiKeys[0].key);
                          if (success) {
                            setShowCopyModal(true);
                            setCopyError(false);
                            setTimeout(() => setShowCopyModal(false), 1000);
                          } else {
                            setCopyError(true);
                            setTimeout(() => setCopyError(false), 3000);
                          }
                        }
                      }}
                    >복사</button>
                    <button
                      className="px-1.5 py-0.5 text-[10px] border border-gray-400 dark:border-gray-600 rounded bg-white dark:bg-gray-900 hover:bg-gray-100 dark:hover:bg-gray-800 transition ml-1"
                      disabled={user.apiKeys?.length === 0}
                      onClick={async () => {
                        if (!window.confirm('정말로 이 사용자의 API 키를 재생성하시겠습니까? 기존 키는 비활성화됩니다.')) return;
                        setShowCopyModal(false);
                        setShowRegenModal(true);
                        try {
                          const res = await externalApiClient.regenerateUserApiKey(String(user.id));
                          console.log('API 재발급 응답:', res);
                          if (res.success && res.data?.key) {
                            const success = await copyToClipboard(res.data.key);
                            if (success) {
                              setShowRegenModal(true);
                              setCopyError(false);
                              setTimeout(() => setShowRegenModal(false), 1500);
                            } else {
                              setCopyError(true);
                              setTimeout(() => setCopyError(false), 3000);
                            }
                            if (onRegenApiKey) onRegenApiKey();
                          } else {
                            alert(res.error?.message || 'API 키 재생성 실패');
                          }
                        } catch (e) {
                          alert('API 키 재생성 중 오류 발생: ' + (e instanceof Error ? e.message : e));
                        }
                      }}
                    >재발급</button>
                  </div>
                ) : (
                  <span className="text-gray-400 text-xs">none</span>
                )}
              </td>
              <td className="px-3 py-2 text-center align-middle">
                {user.status === 'active' ? (
                  <span className="inline-block w-3 h-3 rounded-full bg-green-500" title="활성"></span>
                ) : (
                  <span className="inline-block w-3 h-3 rounded-full bg-red-400" title="비활성"></span>
                )}
              </td>
              <td className="px-3 py-2 text-center align-middle text-xs">{user.created_at?.slice(0, 10)}</td>
              <td className="px-3 py-2 text-right align-middle flex gap-1 justify-end min-w-[130px] whitespace-nowrap">
                <button
                  onClick={() => onEditSystemPrompt?.(user)}
                  className="px-2 py-1 rounded bg-purple-600 hover:bg-purple-700 text-white font-medium text-xs shadow transition"
                  title="시스템 프롬프트 편집"
                >
                  프롬프트
                </button>
                <button
                  onClick={() => onEdit(user)}
                  className="px-2 py-1 rounded bg-blue-600 hover:bg-blue-700 text-white font-medium text-xs shadow transition"
                >
                  수정
                </button>
                <button
                  onClick={() => onDelete(user)}
                  className="px-2 py-1 rounded bg-red-500 hover:bg-red-600 text-white font-medium text-xs shadow transition"
                >
                  삭제
                </button>
              </td>
            </tr>
          ))}
        </tbody>
      </table>
      {/* 복사 성공 모달 */}
      {showCopyModal && (
        <div className="fixed inset-0 flex items-center justify-center z-50">
          <div className="bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-700 rounded-lg shadow-lg px-8 py-6 text-center">
            <span className="text-sm font-semibold text-gray-800 dark:text-gray-100">API 키가 복사되었습니다!</span>
          </div>
        </div>
      )}
      {/* 재생성 성공 모달 */}
      {showRegenModal && (
        <div className="fixed inset-0 flex items-center justify-center z-50">
          <div className="bg-white dark:bg-gray-800 border border-blue-300 dark:border-blue-700 rounded-lg shadow-lg px-8 py-6 text-center">
            <span className="text-sm font-semibold text-blue-700 dark:text-blue-200">새 API 키가 복사되었습니다!</span>
          </div>
        </div>
      )}
      {/* 복사 실패 모달 */}
      {copyError && (
        <div className="fixed inset-0 flex items-center justify-center z-50">
          <div className="bg-white dark:bg-gray-800 border border-red-300 dark:border-red-700 rounded-lg shadow-lg px-8 py-6 text-center">
            <div className="text-sm font-semibold text-red-700 dark:text-red-200 mb-2">복사에 실패했습니다</div>
            <div className="text-xs text-gray-600 dark:text-gray-400">API 키를 수동으로 선택하여 복사해주세요</div>
          </div>
        </div>
      )}
    </div>
  );
}