import { NextRequest, NextResponse } from 'next/server';
import pg from 'pg';
import { getDatabaseConfig } from '@/config/serverConfig';

const { Pool } = pg;

// 데이터베이스 연결
function getDbConnection() {
  const config = getDatabaseConfig();
  return new Pool({
    host: config.host,
    port: config.port,
    user: config.user,
    password: config.password,
    database: config.name,
    max: 5
  });
}

// 임의의 비교 분석 데이터 생성 (실제로는 산업연관표 기반 계산 필요)
function generateComparisonData(documentName: string, industryName: string, revenue: number) {
  // 산업별 기본 탄소 집약도 (tCO2e/백만원) - 임의 데이터
  const carbonIntensityMap: { [key: string]: number } = {
    '농업': 3.50,
    '제조업': 2.20,
    '전기전자': 1.85,
    '화학': 3.80,
    '정보통신': 0.95,
    '운수': 2.60,
    '도소매': 0.75,
    '금융보험': 0.45,
    '부동산': 0.62,
    '전문과학': 0.88,
    '건설': 2.40,
    '교육': 0.55,
    '보건': 0.70
  };

  const baseCarbonIntensity = carbonIntensityMap[industryName] || 2.0;
  const revenueInMillionKRW = revenue; // 이미 백만원 단위

  // Baseline (기존 산업구조)
  const baselineEmission = baseCarbonIntensity * revenueInMillionKRW;

  // Target (신기술 적용 시) - 15-30% 감소 가정
  const reductionRate = 0.15 + Math.random() * 0.15; // 15-30%
  const targetEmission = baselineEmission * (1 - reductionRate);
  const emissionChange = targetEmission - baselineEmission;
  const changeRate = (emissionChange / baselineEmission) * 100;

  // 산업별 매출 분배 (임의 데이터)
  const salesDistribution = [
    { component: '핵심 제품/서비스', function: '주요 사업', industry: industryName, revenue: revenue * 0.50, ratio: 50.0 },
    { component: '부가 서비스', function: '지원 사업', industry: '전문과학', revenue: revenue * 0.20, ratio: 20.0 },
    { component: 'IT 인프라', function: '시스템 운영', industry: '정보통신', revenue: revenue * 0.15, ratio: 15.0 },
    { component: '물류/유통', function: '배송 및 유통', industry: '운수', revenue: revenue * 0.10, ratio: 10.0 },
    { component: '기타', function: '기타 비용', industry: '도소매', revenue: revenue * 0.05, ratio: 5.0 }
  ];

  // 산업별 배출량 (임의 계산)
  const industryEmissions = [
    { name: industryName, baseline: baselineEmission * 0.45, target: targetEmission * 0.40 },
    { name: '정보통신', baseline: baselineEmission * 0.10, target: targetEmission * 0.15 },
    { name: '전문과학', baseline: baselineEmission * 0.08, target: targetEmission * 0.12 },
    { name: '운수', baseline: baselineEmission * 0.15, target: targetEmission * 0.12 },
    { name: '제조업', baseline: baselineEmission * 0.12, target: targetEmission * 0.11 },
    { name: '도소매', baseline: baselineEmission * 0.10, target: targetEmission * 0.10 }
  ];

  // 증감 요인 분석
  const factors = [
    { label: 'Baseline', value: baselineEmission, color: '#ef4444' },
    { label: '효율 개선', value: baselineEmission * -0.15, color: '#10b981' },
    { label: '기술 혁신', value: baselineEmission * -0.10, color: '#10b981' },
    { label: '프로세스 최적화', value: baselineEmission * -0.08, color: '#10b981' },
    { label: '신규 투입', value: baselineEmission * 0.03, color: '#ef4444' },
    { label: 'Target', value: targetEmission, color: '#3b82f6' }
  ];

  // 생산 유발 효과 (임의 데이터)
  const productionInduction = [
    { rank: 1, industry: industryName, coefficient: 1.85, amount: revenue * 1.85 },
    { rank: 2, industry: '정보통신', coefficient: 1.75, amount: revenue * 0.15 * 1.75 },
    { rank: 3, industry: '전문과학', coefficient: 1.68, amount: revenue * 0.20 * 1.68 },
    { rank: 4, industry: '운수', coefficient: 1.52, amount: revenue * 0.10 * 1.52 },
    { rank: 5, industry: '제조업', coefficient: 1.90, amount: revenue * 0.05 * 1.90 }
  ];

  // 산업 연관 효과 (Radar 차트용)
  const linkageEffects = [
    {
      label: industryName,
      data: [0.88, 0.85, 1.85, 0.42, 7.5, 0.22],
      backgroundColor: 'rgba(102, 126, 234, 0.2)',
      borderColor: '#667eea'
    },
    {
      label: '정보통신',
      data: [0.85, 0.75, 1.75, 0.62, 8.5, 0.15],
      backgroundColor: 'rgba(118, 75, 162, 0.2)',
      borderColor: '#764ba2'
    }
  ];

  return {
    summary: {
      baseline: Math.round(baselineEmission * 10) / 10,
      target: Math.round(targetEmission * 10) / 10,
      change: Math.round(emissionChange * 10) / 10,
      changeRate: Math.round(changeRate * 10) / 10
    },
    salesDistribution,
    industryEmissions,
    factors,
    productionInduction,
    linkageEffects
  };
}

// HTML 보고서 생성
function generateHTMLReport(
  documentName: string,
  industryName: string,
  revenue: number,
  comparisonData: any
) {
  const { summary, salesDistribution, industryEmissions, factors, productionInduction, linkageEffects } = comparisonData;

  return `<!DOCTYPE html>
<html lang="ko">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>${documentName} - ESG 환경영향 심층분석 보고서</title>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/Chart.js/4.4.0/chart.umd.min.js"></script>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', sans-serif;
            line-height: 1.8;
            color: #333;
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            padding: 20px;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            box-shadow: 0 10px 40px rgba(0,0,0,0.1);
            border-radius: 12px;
            overflow: hidden;
        }

        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 50px 40px;
            text-align: center;
        }

        .header h1 {
            font-size: 2.2em;
            margin-bottom: 10px;
        }

        .content {
            padding: 40px;
        }

        .section {
            margin-bottom: 50px;
        }

        .section-title {
            font-size: 1.8em;
            color: #667eea;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 3px solid #667eea;
        }

        .subsection-title {
            font-size: 1.3em;
            color: #764ba2;
            margin: 25px 0 15px 0;
        }

        .summary-box {
            background: linear-gradient(135deg, #667eea15 0%, #764ba215 100%);
            border-left: 5px solid #667eea;
            padding: 25px;
            border-radius: 8px;
            margin: 20px 0;
        }

        .summary-box h3 {
            color: #667eea;
            margin-bottom: 15px;
            font-size: 1.2em;
        }

        .summary-box ul {
            list-style: none;
            padding-left: 0;
        }

        .summary-box li {
            padding: 8px 0;
            padding-left: 25px;
            position: relative;
        }

        .summary-box li:before {
            content: "▪";
            position: absolute;
            left: 0;
            color: #667eea;
            font-weight: bold;
        }

        table {
            width: 100%;
            border-collapse: collapse;
            margin: 25px 0;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            border-radius: 8px;
            overflow: hidden;
        }

        th {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 15px;
            text-align: center;
            font-weight: 600;
        }

        td {
            padding: 12px 15px;
            text-align: center;
            border-bottom: 1px solid #eee;
        }

        tr:hover {
            background-color: #f8f9ff;
        }

        .chart-container {
            position: relative;
            margin: 30px auto;
            padding: 20px;
            background: #f8f9ff;
            border-radius: 12px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.08);
        }

        .chart-wrapper {
            position: relative;
            height: 400px;
        }

        .highlight {
            background: linear-gradient(120deg, #f6d365 0%, #fda085 100%);
            padding: 3px 8px;
            border-radius: 4px;
            font-weight: 600;
        }

        .positive {
            color: #10b981;
            font-weight: 600;
        }

        .negative {
            color: #ef4444;
            font-weight: 600;
        }

        .info-card {
            background: white;
            border: 2px solid #667eea;
            border-radius: 10px;
            padding: 20px;
            margin: 20px 0;
        }

        .metric {
            display: inline-block;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 8px 16px;
            border-radius: 20px;
            font-weight: 600;
            margin: 5px;
        }

        .conclusion-box {
            background: linear-gradient(135deg, #10b98115 0%, #34d39915 100%);
            border: 2px solid #10b981;
            border-radius: 12px;
            padding: 30px;
            margin: 30px 0;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>${documentName} - ESG 환경영향 심층분석 보고서</h1>
            <p>Environmental Impact Assessment Report</p>
        </div>

        <div class="content">
            <div class="section">
                <h2 class="section-title">1. 분석 개요</h2>
                <div class="summary-box">
                    <h3>분석 대상</h3>
                    <p>${documentName} - ${industryName} 산업</p>
                    <p>연간 매출액: <strong>${(revenue / 100000000).toFixed(1)}억 원</strong></p>
                </div>
                <div class="summary-box">
                    <h3>핵심 결론</h3>
                    <ul>
                        <li><span class="highlight">총 탄소배출량 변화:</span> 기존 산업구조 대비 <span class="${summary.changeRate < 0 ? 'positive' : 'negative'}">${summary.changeRate.toFixed(1)}%</span>의 탄소배출량 ${summary.changeRate < 0 ? '감소' : '증가'} (연간 <span class="${summary.changeRate < 0 ? 'positive' : 'negative'}">${summary.change.toFixed(0)} tCO2eq</span>)</li>
                        <li><span class="highlight">주요 영향 산업:</span> ${industryName}, 정보통신, 전문과학서비스</li>
                    </ul>
                </div>
            </div>

            <div class="section">
                <h2 class="section-title">2. 분석 대상 및 방법론</h2>
                <h3 class="subsection-title">2.1. 분석 대상 상세</h3>
                <div class="info-card">
                    <p><strong>산업 분류:</strong> ${industryName}</p>
                    <p style="margin-top:10px;"><strong>기술 개요:</strong> 본 사업은 ${industryName} 분야의 혁신적 솔루션으로, 기존 대비 효율성을 크게 개선하고 환경 부담을 줄이는 것을 목표로 합니다.</p>
                </div>
                <h3 class="subsection-title">2.2. 분석 프레임워크</h3>
                <div class="info-card">
                    <p><strong>분석 모델:</strong> 환경확장 산업연관분석 (EEIOA)</p>
                    <p style="margin-top:10px;"><strong>시나리오:</strong></p>
                    <ul style="margin-left:20px;">
                        <li>Baseline: 기존 산업 구조</li>
                        <li>Target: 신기술/서비스 도입 시나리오</li>
                    </ul>
                </div>
            </div>

            <div class="section">
                <h2 class="section-title">3. 산업 분류 및 경제적 가정</h2>
                <h3 class="subsection-title">3.1. AI 기반 산업 분류 결과</h3>
                <table>
                    <thead>
                        <tr>
                            <th>구성요소</th>
                            <th>주요기능</th>
                            <th>연관 산업분류</th>
                            <th>매출액(억원)</th>
                            <th>비중(%)</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${salesDistribution.map(item => `
                        <tr>
                            <td>${item.component}</td>
                            <td>${item.function}</td>
                            <td>${item.industry}</td>
                            <td>${(item.revenue / 100000000).toFixed(1)}</td>
                            <td>${item.ratio.toFixed(1)}</td>
                        </tr>
                        `).join('')}
                    </tbody>
                </table>

                <h3 class="subsection-title">3.2. 매출 구조 시각화</h3>
                <div class="chart-container">
                    <div class="chart-wrapper">
                        <canvas id="salesChart"></canvas>
                    </div>
                </div>
            </div>

            <div class="section">
                <h2 class="section-title">4. 환경영향 정량 분석 결과</h2>
                <h3 class="subsection-title">4.1. 총 탄소배출량 변화</h3>
                <table>
                    <thead>
                        <tr>
                            <th>구분</th>
                            <th>Baseline<br>(tCO2eq)</th>
                            <th>Target<br>(tCO2eq)</th>
                            <th>변화량<br>(tCO2eq)</th>
                            <th>변화율(%)</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td style="font-weight:600;">총 탄소배출량</td>
                            <td>${summary.baseline.toLocaleString()}</td>
                            <td>${summary.target.toLocaleString()}</td>
                            <td class="${summary.changeRate < 0 ? 'positive' : 'negative'}">${summary.change.toLocaleString()}</td>
                            <td class="${summary.changeRate < 0 ? 'positive' : 'negative'}">${summary.changeRate.toFixed(1)}</td>
                        </tr>
                    </tbody>
                </table>

                <h3 class="subsection-title">4.2. 주요 산업별 배출량 변화 분석</h3>
                <div class="chart-container">
                    <div class="chart-wrapper">
                        <canvas id="industryChart"></canvas>
                    </div>
                </div>

                <h3 class="subsection-title">4.3. 탄소배출량 증감 요인 분석</h3>
                <div class="chart-container">
                    <div class="chart-wrapper">
                        <canvas id="waterfallChart"></canvas>
                    </div>
                </div>
            </div>

            <div class="section">
                <h2 class="section-title">5. 산업 연관효과 분석</h2>
                <h3 class="subsection-title">5.1. 생산 유발효과</h3>
                <table>
                    <thead>
                        <tr>
                            <th>순위</th>
                            <th>산업 부문</th>
                            <th>생산유발계수</th>
                            <th>생산유발액(억원)</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${productionInduction.map(item => `
                        <tr>
                            <td>${item.rank}</td>
                            <td>${item.industry}</td>
                            <td>${item.coefficient.toFixed(2)}</td>
                            <td>${(item.amount / 100000000).toFixed(1)}</td>
                        </tr>
                        `).join('')}
                    </tbody>
                </table>

                <h3 class="subsection-title">5.2. 산업 간 연관관계</h3>
                <div class="chart-container">
                    <div class="chart-wrapper">
                        <canvas id="linkageChart"></canvas>
                    </div>
                </div>
            </div>

            <div class="section">
                <h2 class="section-title">6. 결론 및 제언</h2>
                <h3 class="subsection-title">6.1. 종합 평가</h3>
                <div class="conclusion-box">
                    <h3>✓ 환경적 적합성: ${summary.changeRate < 0 ? '긍정적' : '주의 필요'}</h3>
                    <p>본 사업은 연간 약 ${Math.abs(summary.change).toLocaleString()} tCO2eq의 탄소배출량 ${summary.changeRate < 0 ? '감소' : '증가'} 효과를 보입니다.</p>
                    <p style="margin-top:15px;"><strong>핵심 영향 경로:</strong></p>
                    <ul style="margin-left:20px; list-style:disc;">
                        <li>프로세스 효율 개선으로 ${Math.abs(summary.baseline * 0.15).toFixed(0)} tCO2eq ${summary.changeRate < 0 ? '감소' : '증가'}</li>
                        <li>기술 혁신을 통한 ${Math.abs(summary.baseline * 0.10).toFixed(0)} tCO2eq ${summary.changeRate < 0 ? '감소' : '증가'}</li>
                        <li>최적화를 통한 ${Math.abs(summary.baseline * 0.08).toFixed(0)} tCO2eq ${summary.changeRate < 0 ? '감소' : '증가'}</li>
                    </ul>
                </div>

                <h3 class="subsection-title">6.2. 한계 및 참고사항</h3>
                <div class="info-card" style="border-color:#fda085;">
                    <p><strong>분석의 한계:</strong></p>
                    <ul style="margin-left:20px; list-style:disc;">
                        <li>산업연관표 기준 데이터로 최근 변화 반영 필요</li>
                        <li>제품 전 생애주기(LCA) 분석 미포함</li>
                        <li>세부 산업별 분석 필요</li>
                        <li>간접 효과 및 시너지 효과 추가 분석 필요</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>

    <script>
        // 매출 구조 차트
        const salesCtx = document.getElementById('salesChart').getContext('2d');
        new Chart(salesCtx, {
            type: 'doughnut',
            data: {
                labels: ${JSON.stringify(salesDistribution.map(item => `${item.component} (${item.ratio.toFixed(0)}%)`))},
                datasets: [{
                    data: ${JSON.stringify(salesDistribution.map(item => item.ratio))},
                    backgroundColor: ['#667eea', '#764ba2', '#f093fb', '#4facfe', '#43e97b']
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { position: 'right' },
                    title: { display: true, text: '산업 분류별 매출 비중 (총 ${(revenue / 100000000).toFixed(0)}억 원)', font: { size: 16 } }
                }
            }
        });

        // 산업별 배출량 비교 차트
        const industryCtx = document.getElementById('industryChart').getContext('2d');
        new Chart(industryCtx, {
            type: 'bar',
            data: {
                labels: ${JSON.stringify(industryEmissions.map(item => item.name))},
                datasets: [{
                    label: 'Baseline',
                    data: ${JSON.stringify(industryEmissions.map(item => Math.round(item.baseline)))},
                    backgroundColor: 'rgba(239, 68, 68, 0.7)'
                }, {
                    label: 'Target',
                    data: ${JSON.stringify(industryEmissions.map(item => Math.round(item.target)))},
                    backgroundColor: 'rgba(16, 185, 129, 0.7)'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: { display: true, text: '주요 산업별 탄소배출량 비교 (tCO2eq)', font: { size: 16 } }
                },
                scales: { y: { beginAtZero: true } }
            }
        });

        // 증감 요인 차트
        const waterfallCtx = document.getElementById('waterfallChart').getContext('2d');
        new Chart(waterfallCtx, {
            type: 'bar',
            data: {
                labels: ${JSON.stringify(factors.map(f => f.label))},
                datasets: [{
                    data: ${JSON.stringify(factors.map(f => Math.round(f.value)))},
                    backgroundColor: ${JSON.stringify(factors.map(f => f.color))}
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { display: false },
                    title: { display: true, text: '탄소배출량 증감 요인 분석', font: { size: 16 } }
                },
                scales: { y: { beginAtZero: true } }
            }
        });

        // 산업 연관효과 차트
        const linkageCtx = document.getElementById('linkageChart').getContext('2d');
        new Chart(linkageCtx, {
            type: 'radar',
            data: {
                labels: ['전방연쇄', '후방연쇄', '생산유발', '부가가치', '고용유발', '수입유발'],
                datasets: ${JSON.stringify(linkageEffects)}
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: { display: true, text: '주요 산업 연관효과 비교', font: { size: 16 } }
                }
            }
        });
    </script>
</body>
</html>`;
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { documentId, jobId } = body;

    if (!documentId || !jobId) {
      return NextResponse.json({
        success: false,
        error: 'documentId 및 jobId가 필요합니다.'
      }, { status: 400 });
    }

    const dbManager = getDbConnection();

    // 문서 정보 조회
    const docResult = await dbManager.query(`
      SELECT
        document_name,
        industry_major_name,
        estimated_revenue,
        total_carbon_emission
      FROM esg_batch_documents
      WHERE id = $1 AND job_id = $2
    `, [documentId, jobId]);

    if (docResult.rows.length === 0) {
      await dbManager.end();
      return NextResponse.json({
        success: false,
        error: '문서를 찾을 수 없습니다.'
      }, { status: 404 });
    }

    const doc = docResult.rows[0];
    const documentName = doc.document_name || '분석 대상 문서';
    const industryName = doc.industry_major_name || '정보통신';
    const revenue = doc.estimated_revenue || 10000000000; // 기본 100억

    // 비교 분석 데이터 생성
    const comparisonData = generateComparisonData(documentName, industryName, revenue);

    // HTML 보고서 생성
    const htmlReport = generateHTMLReport(documentName, industryName, revenue, comparisonData);

    // 보고서를 데이터베이스에 저장
    await dbManager.query(`
      UPDATE esg_batch_documents
      SET
        report_content = $1,
        report_format = 'html',
        updated_at = CURRENT_TIMESTAMP
      WHERE id = $2 AND job_id = $3
    `, [htmlReport, documentId, jobId]);

    await dbManager.end();

    return NextResponse.json({
      success: true,
      data: {
        documentId,
        jobId,
        reportGenerated: true,
        summary: comparisonData.summary,
        message: '보고서가 성공적으로 생성되었습니다.'
      }
    });

  } catch (error) {
    console.error('보고서 생성 오류:', error);

    return NextResponse.json({
      success: false,
      error: error instanceof Error ? error.message : '보고서 생성에 실패했습니다.'
    }, { status: 500 });
  }
}
