import { NextRequest, NextResponse } from 'next/server';
import pg from 'pg';
import { getDatabaseConfig } from '@/config/serverConfig';
import fs from 'fs';
import path from 'path';

const { Pool } = pg;

// 데이터베이스 연결
function getDbConnection() {
  const config = getDatabaseConfig();
  return new Pool({
    host: config.host,
    port: config.port,
    user: config.user,
    password: config.password,
    database: config.name,
    max: 5
  });
}

// 표준산업분류 데이터 로드
function loadIndustryClassifications() {
  const dataPath = path.join(process.cwd(), '../../data/industry-classification.json');
  const jsonData = fs.readFileSync(dataPath, 'utf-8');
  return JSON.parse(jsonData);
}

// AI를 사용한 표준산업분류 및 매출액 식별
async function identifyIndustryWithAI(documentContent: string): Promise<{
  majorCode: string;
  majorName: string;
  subCode: string;
  subName: string;
  confidence: number;
  carbonIntensity: number;
  estimatedRevenue: number | null;
}> {
  // 표준산업분류 데이터 로드
  const industryData = loadIndustryClassifications();

  // LLM API 호출하여 산업분류 및 매출액 추출 (/api/v1/callai 사용)
  try {
    const apiServerUrl = process.env.API_SERVER_URL || 'http://localhost:5500';

    // 산업분류 코드 설명
    const industryCodesExplanation = `한국표준산업분류(KSIC) 코드:
A: 농업, 임업 및 어업
B: 광업
C: 제조업
D: 전기, 가스, 증기 및 공기조절 공급업
E: 수도, 하수 및 폐기물 처리, 원료 재생업
F: 건설업
G: 도매 및 소매업
H: 운수 및 창고업
I: 숙박 및 음식점업
J: 정보통신업
K: 금융 및 보험업
L: 부동산업
M: 전문, 과학 및 기술 서비스업
N: 사업시설관리, 사업지원 및 임대 서비스업
O: 공공행정, 국방 및 사회보장 행정
P: 교육 서비스업
Q: 보건업 및 사회복지 서비스업
R: 예술, 스포츠 및 여가관련 서비스업
S: 협회 및 단체, 수리 및 기타 개인 서비스업`;

    // AI에게 전달할 메시지 구성
    const messages = [
      {
        role: 'system',
        content: `당신은 문서 분석 전문가입니다. 주어진 문서에서 산업분류와 매출액 정보를 추출합니다.
${industryCodesExplanation}

작업:
1. 문서의 사업 내용을 분석하여 가장 적합한 산업분류 코드(A-S)를 선택
2. 문서에서 매출액을 찾아 원(KRW) 단위 숫자로 변환 (예: "50억원" → 5000000000, "200억원" → 20000000000)
3. 매출액이 없으면 null 반환

응답 형식: 반드시 JSON만 출력 (설명 없이)
{"industry_code": "A-S 중 하나", "revenue_krw": 숫자 또는 null}`
      },
      {
        role: 'user',
        content: `다음 문서를 분석해주세요:\n\n${documentContent.substring(0, 3000)}`
      }
    ];

    console.log('AI API 호출 시작 (/api/v1/callai):', {
      url: `${apiServerUrl}/api/v1/callai`,
      messageCount: messages.length
    });

    const response = await fetch(`${apiServerUrl}/api/v1/callai`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-API-Key': process.env.API_KEY || 'airun_1_b76483e3b0f6809be057259ec1568337'
      },
      body: JSON.stringify({
        messages,
        options: {
          provider: 'ollama',
          model: 'hamonize:latest',
          temperature: 0.1,
          max_tokens: 500
        },
        stream: false
      })
    });

    console.log('AI API 응답 상태:', response.status);

    if (response.ok) {
      const data = await response.json();
      // callai 응답 구조: data.data.response
      const responseData = data.data?.response || data.response || {};

      // content가 문자열이면 그대로, 객체면 content 속성 추출
      let aiResponseText = '';
      if (typeof responseData === 'string') {
        aiResponseText = responseData;
      } else if (typeof responseData === 'object' && responseData.content) {
        aiResponseText = responseData.content;
      }

      console.log('AI 응답 내용:', aiResponseText);

      // JSON 추출 시도
      let jsonMatch = aiResponseText.match(/\{[\s\S]*?"industry_code"[\s\S]*?"revenue_krw"[\s\S]*?\}/);

      if (!jsonMatch) {
        // 대체 패턴 시도
        jsonMatch = aiResponseText.match(/\{[^}]*"industry_code"[^}]*"revenue_krw"[^}]*\}/);
      }

      if (jsonMatch) {
        try {
          const parsed = JSON.parse(jsonMatch[0]);
          const industryCode = parsed.industry_code;
          let revenue = parsed.revenue_krw;

          // revenue가 문자열인 경우 숫자로 변환
          if (typeof revenue === 'string') {
            revenue = parseFloat(revenue.replace(/,/g, ''));
          }

          console.log('파싱된 결과:', { industryCode, revenue });

          // 산업 정보 찾기
          const industry = industryData.classifications.find((c: any) => c.code === industryCode);
          if (industry) {
            const subCategory = industry.sub_categories?.[0] || { code: '00', name: '기타' };
            const result = {
              majorCode: industry.code,
              majorName: industry.major_category,
              subCode: subCategory.code,
              subName: subCategory.name,
              confidence: 0.9,
              carbonIntensity: industry.carbon_intensity,
              estimatedRevenue: (revenue !== null && !isNaN(revenue)) ? revenue : null
            };
            console.log('AI 분석 성공:', result);
            return result;
          } else {
            console.log('산업 코드를 찾을 수 없음:', industryCode);
          }
        } catch (parseError) {
          console.error('JSON 파싱 오류:', parseError);
        }
      } else {
        console.log('JSON 매칭 실패, AI 응답:', aiResponse);
      }
    } else {
      const errorText = await response.text();
      console.error('AI API 호출 실패:', response.status, errorText);
    }
  } catch (error) {
    console.error('AI API 호출 오류:', error);
    // 에러 발생시 fallback으로 키워드 매칭 사용
  }

  // 간단한 키워드 매칭 (실제로는 AI 모델 사용 권장)
  const keywords: { [key: string]: string[] } = {
    'A': ['농업', '임업', '어업', '농작물', '축산', '수산', '양식'],
    'B': ['광업', '석탄', '석유', '가스', '금속', '광물', '채굴'],
    'C': ['제조', '생산', '공장', '화학', '전자', '자동차', '기계', '의약품', '플라스틱'],
    'D': ['전기', '가스', '발전', '에너지', '전력'],
    'E': ['수도', '하수', '폐기물', '환경', '재활용'],
    'F': ['건설', '건축', '토목', '시공'],
    'G': ['도매', '소매', '판매', '유통', '상점'],
    'H': ['운수', '운송', '물류', '창고', '배송', '택배'],
    'I': ['숙박', '호텔', '음식', '식당', '카페', '레스토랑'],
    'J': ['정보통신', 'IT', '소프트웨어', '개발', '프로그래밍', '앱', '웹', '서버', '클라우드'],
    'K': ['금융', '은행', '보험', '증권', '투자'],
    'L': ['부동산', '건물', '임대', '매매'],
    'M': ['연구', '과학', '기술', '컨설팅', '전문'],
    'N': ['사업지원', '관리', '임대', '청소'],
    'O': ['공공', '행정', '정부', '국방'],
    'P': ['교육', '학교', '학원', '강의'],
    'Q': ['보건', '의료', '병원', '클리닉', '복지'],
    'R': ['예술', '스포츠', '문화', '여가', '오락'],
    'S': ['수리', '개인서비스', '협회']
  };

  let bestMatch = {
    code: 'J',
    name: '정보통신업',
    confidence: 0.3
  };

  // 키워드 매칭
  for (const [code, keywordList] of Object.entries(keywords)) {
    let matchCount = 0;
    for (const keyword of keywordList) {
      if (documentContent.includes(keyword)) {
        matchCount++;
      }
    }

    const confidence = Math.min(0.95, 0.3 + (matchCount * 0.15));
    if (matchCount > 0 && confidence > bestMatch.confidence) {
      const industry = industryData.classifications.find((c: any) => c.code === code);
      if (industry) {
        bestMatch = {
          code,
          name: industry.major_category,
          confidence
        };
      }
    }
  }

  // 해당 산업의 세부 분류 찾기
  const industry = industryData.classifications.find((c: any) => c.code === bestMatch.code);
  const subCategory = industry?.sub_categories?.[0] || { code: '62', name: '컴퓨터 프로그래밍, 시스템 통합 및 관리업' };

  return {
    majorCode: bestMatch.code,
    majorName: bestMatch.name,
    subCode: subCategory.code,
    subName: subCategory.name,
    confidence: bestMatch.confidence,
    carbonIntensity: industry?.carbon_intensity || 0.28,
    estimatedRevenue: null  // fallback에서는 매출액 추출 안 함
  };
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { documentId, jobId } = body;

    if (!documentId || !jobId) {
      return NextResponse.json({
        success: false,
        error: '문서 ID 또는 작업 ID가 필요합니다.'
      }, { status: 400 });
    }

    // 데이터베이스 연결
    const dbManager = getDbConnection();

    // 문서 내용 조회
    const documentResult = await dbManager.query(`
      SELECT raw_content, estimated_revenue
      FROM esg_batch_documents
      WHERE id = $1 AND job_id = $2
    `, [documentId, jobId]);

    if (documentResult.rows.length === 0) {
      await dbManager.end();
      return NextResponse.json({
        success: false,
        error: '문서를 찾을 수 없습니다.'
      }, { status: 404 });
    }

    const document = documentResult.rows[0];
    const content = document.raw_content;

    // AI를 사용하여 표준산업분류 및 매출액 식별
    const industryResult = await identifyIndustryWithAI(content);

    // 데이터베이스 업데이트 (AI가 추출한 매출액도 함께 저장)
    const updateQuery = industryResult.estimatedRevenue !== null
      ? `UPDATE esg_batch_documents
         SET
           industry_major_code = $1,
           industry_major_name = $2,
           industry_sub_code = $3,
           industry_sub_name = $4,
           industry_confidence = $5,
           carbon_intensity = $6,
           estimated_revenue = $7,
           updated_at = CURRENT_TIMESTAMP
         WHERE id = $8`
      : `UPDATE esg_batch_documents
         SET
           industry_major_code = $1,
           industry_major_name = $2,
           industry_sub_code = $3,
           industry_sub_name = $4,
           industry_confidence = $5,
           carbon_intensity = $6,
           updated_at = CURRENT_TIMESTAMP
         WHERE id = $7`;

    const updateParams = industryResult.estimatedRevenue !== null
      ? [
          industryResult.majorCode,
          industryResult.majorName,
          industryResult.subCode,
          industryResult.subName,
          industryResult.confidence,
          industryResult.carbonIntensity,
          industryResult.estimatedRevenue,
          documentId
        ]
      : [
          industryResult.majorCode,
          industryResult.majorName,
          industryResult.subCode,
          industryResult.subName,
          industryResult.confidence,
          industryResult.carbonIntensity,
          documentId
        ];

    await dbManager.query(updateQuery, updateParams);

    // 업데이트된 매출액 조회
    const updatedDoc = await dbManager.query(
      'SELECT estimated_revenue FROM esg_batch_documents WHERE id = $1',
      [documentId]
    );

    await dbManager.end();

    return NextResponse.json({
      success: true,
      data: {
        documentId,
        industry: {
          majorCode: industryResult.majorCode,
          majorName: industryResult.majorName,
          subCode: industryResult.subCode,
          subName: industryResult.subName,
          confidence: industryResult.confidence
        },
        carbonIntensity: industryResult.carbonIntensity,
        estimatedRevenue: updatedDoc.rows[0]?.estimated_revenue || null
      }
    });

  } catch (error) {
    console.error('표준산업분류 식별 오류:', error);

    return NextResponse.json({
      success: false,
      error: error instanceof Error ? error.message : '표준산업분류 식별에 실패했습니다.'
    }, { status: 500 });
  }
}
