import { NextRequest, NextResponse } from 'next/server';
import ExcelJS from 'exceljs';
import { v4 as uuidv4 } from 'uuid';
import path from 'path';
import fs from 'fs/promises';

// Redis for job status tracking
let redisClient: any = null;

// Initialize Redis connection
async function getRedisClient() {
  if (!redisClient) {
    try {
      const { createClient } = await import('redis');
      redisClient = createClient({
        url: process.env.REDIS_URL || 'redis://localhost:6379'
      });
      await redisClient.connect();
    } catch (error) {
      console.error('Redis connection failed:', error);
      redisClient = null;
    }
  }
  return redisClient;
}

// Mock Redis for fallback
const fallbackJobStore = new Map();

async function setJobData(jobId: string, data: any) {
  const redis = await getRedisClient();
  if (redis) {
    await redis.setEx(`health-batch:${jobId}`, 3600, JSON.stringify(data)); // 1시간 TTL
  } else {
    fallbackJobStore.set(jobId, data);
  }
}

async function getJobData(jobId: string) {
  const redis = await getRedisClient();
  if (redis) {
    const data = await redis.get(`health-batch:${jobId}`);
    return data ? JSON.parse(data) : null;
  } else {
    return fallbackJobStore.get(jobId) || null;
  }
}

// 건강검진 분석 기준 (builtin-tools.js에서 가져온 기준)
const HEALTH_EVALUATION_CRITERIA = {
  bloodPressure: {
    systolic: 140,    // mmHg 이상 시 해당(1)
    diastolic: 90     // mmHg 이상 시 해당(1)
  },
  lipid: {
    triglycerides: 150,  // mg/dl 이상 시 해당(1)
    hdl: 40,            // mg/dl 미만 시 해당(1)
    ldl: 140            // mg/dl 이상 시 해당(1)
  },
  bloodSugar: {
    general: 100,       // mg/dl 이상 시 해당(1)
    fasting: 100,       // mg/dl 이상 시 해당(1)
    hba1c: 5.6          // % 이상 시 해당(1)
  },
  obesity: {
    bmi: 25,           // kg/m² 이상 시 해당(1)
    waist: { male: 85, female: 90 }  // cm 이상 시 해당(1)
  },
  urineProtein: {
    // ー(음성)→1, ±→3, ＋→5, ＋＋→7, ＋＋＋→9
    codes: { '－': 1, '±': 3, '＋': 5, '＋＋': 7, '＋＋＋': 9 }
  }
};

// 환자 데이터 분석 함수 (builtin-tools.js 로직 재사용)
function analyzePatientData(patientData: any): any {
  const analysis = {
    bloodPressureScore: 3, // 기본값: 비해당
    cholesterolScore: 3,
    hba1cReceived: 3,
    bloodSugarScore: 3,
    obesityScore: 3,
    urineProteinScore: 1,
    pastHistoryTreatment: '',
    currentHistoryTreatment: '',
    finalJudgment: 3,
    hasValidData: false
  };

  // 실제 분석이 수행되었는지 확인 (완화된 검증)
  // 환자 이름이 있고, 원시 데이터가 존재하면 기본적으로 유효한 것으로 판정
  const hasPatientName = patientData?.name && patientData.name.trim().length > 0;
  const hasRawData = patientData?.rawData && Object.keys(patientData.rawData).length > 0;
  
  // 혈압, 콜레스테롤, 혈당 중 최소 하나라도 있으면 분석 가능
  const hasValidBloodPressure = patientData?.bloodPressure?.systolic && patientData?.bloodPressure?.diastolic;
  const hasValidCholesterol = patientData?.cholesterol && (patientData.cholesterol.hdl || patientData.cholesterol.ldl);
  const hasValidBloodSugar = patientData?.bloodSugar?.glucose;
  const hasAnyHealthData = hasValidBloodPressure || hasValidCholesterol || hasValidBloodSugar;
  
  // 기본적으로 환자명과 원시데이터가 있으면 성공으로 판정
  const hasMinimumRequiredData = hasPatientName && hasRawData;

  try {
    // 1. 혈압 판정
    if (patientData.bloodPressure) {
      const { systolic, diastolic } = patientData.bloodPressure;
      if (systolic >= HEALTH_EVALUATION_CRITERIA.bloodPressure.systolic || 
          diastolic >= HEALTH_EVALUATION_CRITERIA.bloodPressure.diastolic) {
        analysis.bloodPressureScore = 1; // 해당
      }

    }

    // 2. 혈중지질 판정
    if (patientData.cholesterol) {
      const { hdl, ldl, triglycerides } = patientData.cholesterol;
      if (hdl < HEALTH_EVALUATION_CRITERIA.lipid.hdl || 
          ldl >= HEALTH_EVALUATION_CRITERIA.lipid.ldl || 
          triglycerides >= HEALTH_EVALUATION_CRITERIA.lipid.triglycerides) {
        analysis.cholesterolScore = 1; // 해당
      }

    }

    // 3. HbA1c 수진 여부 (있으면 수진(3), 없으면 미수진(1))
    if (patientData.bloodSugar?.hba1c > 0) {
      analysis.hba1cReceived = 3; // 수진
    } else {
      analysis.hba1cReceived = 1; // 미수진
    }

    // 4. 혈당 판정
    if (patientData.bloodSugar) {
      const { glucose, hba1c } = patientData.bloodSugar;
      // 수시혈당만 있고 HbA1c가 없는 경우는 비해당(3)
      if (hba1c === 0 && glucose >= 100) {
        analysis.bloodSugarScore = 3; // 비해당 (특별 규칙)
      } else if (glucose >= HEALTH_EVALUATION_CRITERIA.bloodSugar.general || 
                 hba1c >= HEALTH_EVALUATION_CRITERIA.bloodSugar.hba1c) {
        analysis.bloodSugarScore = 1; // 해당
      }

    }

    // 5. 비만 판정
    if (patientData.bmi >= HEALTH_EVALUATION_CRITERIA.obesity.bmi) {
      analysis.obesityScore = 1; // 해당

    }
    if (patientData.waist) {
      // 성별이 없으면 남성 기준 적용
      const waistThreshold = HEALTH_EVALUATION_CRITERIA.obesity.waist.male;
      if (patientData.waist >= waistThreshold) {
        analysis.obesityScore = 1; // 해당
      }

    }

    // 6. 요단백 점수 변환
    if (patientData.urineProtein) {
      const protein = patientData.urineProtein.toString().trim();
      const proteinCode = HEALTH_EVALUATION_CRITERIA.urineProtein.codes[protein as keyof typeof HEALTH_EVALUATION_CRITERIA.urineProtein.codes];
      if (proteinCode) {
        analysis.urineProteinScore = proteinCode;
      }
    }

    // 7. 기왕력/현병력 처리
    analysis.pastHistoryTreatment = patientData.pastHistory || '';
    analysis.currentHistoryTreatment = patientData.currentHistory || '';

    // 8. 최종 판정 계산
    const scores = [
      analysis.bloodPressureScore,
      analysis.cholesterolScore,
      analysis.bloodSugarScore,
      analysis.obesityScore
    ];
    
    const relevantCount = scores.filter(score => score === 1).length;
    
    // 중증질환 대상외 확인 (특정 질환의 치료중/복용중/경과관찰중)
    const criticalConditions = ['간질', '심질환', '뇌질환', '협심증', '심근경색', '심부전', '뇌경색', '뇌졸중'];
    const hasCriticalTreatment = criticalConditions.some(condition => 
      (analysis.pastHistoryTreatment.includes(condition) && 
       (analysis.pastHistoryTreatment.includes('치료중') || 
        analysis.pastHistoryTreatment.includes('복용중') || 
        analysis.pastHistoryTreatment.includes('경과관찰중'))) ||
      (analysis.currentHistoryTreatment.includes(condition) && 
       (analysis.currentHistoryTreatment.includes('치료중') || 
        analysis.currentHistoryTreatment.includes('복용중') || 
        analysis.currentHistoryTreatment.includes('경과관찰중')))
    );

    if (hasCriticalTreatment) {
      analysis.finalJudgment = 4; // 대상외
    } else if (scores.includes(1) && scores.filter(score => score === 1).length >= 1) {
      // 미수진 항목이 있으면 3
      if (scores.includes(1)) { // 하나라도 미수진
        analysis.finalJudgment = 3;
      } else if (relevantCount === 4) {
        analysis.finalJudgment = 1; // 고위험군
      } else if (relevantCount >= 1) {
        analysis.finalJudgment = 2; // 중위험군
      } else {
        analysis.finalJudgment = 3; // 저위험군
      }
    } else {
      analysis.finalJudgment = 3; // 저위험군
    }

    // 최소 필수 데이터가 모두 있어야 성공으로 처리
    analysis.hasValidData = hasMinimumRequiredData;
    
    return analysis;

  } catch (error) {
    console.error('환자 데이터 분석 오류:', error);
    return {
      ...analysis,
      error: error instanceof Error ? error.message : '분석 중 오류 발생',
      hasValidData: false
    };
  }
}

// 배치 분석 백그라운드 프로세스
async function processBatchAnalysis(jobId: string, patients: any[]) {
  let completed = 0;
  const results: any[] = [];
  
  try {
    // 작업 초기화
    await setJobData(jobId, {
      status: 'running',
      progress: 0,
      completed: 0,
      total: patients.length,
      startTime: new Date().toISOString(),
      currentPatient: '',
      results: []
    });

    // 각 환자별 분석 수행
    for (const patient of patients) {
      try {
        // 현재 환자 정보 업데이트
        await setJobData(jobId, {
          status: 'running',
          progress: (completed / patients.length) * 100,
          completed,
          total: patients.length,
          currentPatient: patient.name,
          results
        });

        // 환자 데이터 분석
        const analysisResult = analyzePatientData(patient);
        
        const result = {
          patientName: patient.name,
          bloodPressureScore: analysisResult.bloodPressureScore,
          cholesterolScore: analysisResult.cholesterolScore,
          hba1cReceived: analysisResult.hba1cReceived,
          bloodSugarScore: analysisResult.bloodSugarScore,
          obesityScore: analysisResult.obesityScore,
          urineProteinScore: analysisResult.urineProteinScore,
          pastHistoryTreatment: analysisResult.pastHistoryTreatment,
          currentHistoryTreatment: analysisResult.currentHistoryTreatment,
          finalJudgment: analysisResult.finalJudgment,
          success: analysisResult.hasValidData,
          error: analysisResult.error
        };
        
        results.push(result);
        completed++;
        
        // 진행률 업데이트
        await setJobData(jobId, {
          status: 'running',
          progress: (completed / patients.length) * 100,
          completed,
          total: patients.length,
          currentPatient: patient.name,
          results
        });
        
        // 1초 대기 (시뮬레이션)
        await new Promise(resolve => setTimeout(resolve, 1000));
        
      } catch (error) {
        console.error(`환자 ${patient.name} 분석 오류:`, error);
        results.push({
          patientName: patient.name,
          success: false,
          error: error instanceof Error ? error.message : '분석 실패'
        });
        completed++;
      }
    }

    // 작업 완료
    await setJobData(jobId, {
      status: 'completed',
      progress: 100,
      completed,
      total: patients.length,
      endTime: new Date().toISOString(),
      results,
      originalData: patients // 원본 환자 데이터 저장
    });

  } catch (error) {
    console.error('배치 분석 오류:', error);
    await setJobData(jobId, {
      status: 'failed',
      error: error instanceof Error ? error.message : '배치 분석 실패',
      completed,
      total: patients.length,
      results
    });
  }
}

export async function POST(request: NextRequest) {
  try {
    const formData = await request.formData();
    const file = formData.get('file') as File;
    const patientCount = parseInt(formData.get('patientCount') as string);
    
    if (!file) {
      return NextResponse.json({
        success: false,
        error: '파일이 업로드되지 않았습니다.'
      }, { status: 400 });
    }

    // Excel 파일 다시 파싱 (환자 데이터 추출)
    const buffer = await file.arrayBuffer();
    const workbook = new ExcelJS.Workbook();
    await workbook.xlsx.load(buffer);
    const worksheet = workbook.worksheets[0];

    const jsonData: any[] = [];
    const headers: string[] = [];

    // 헤더 추출
    worksheet.getRow(1).eachCell((cell, colNumber) => {
      const value = cell.value;
      const textValue = typeof value === 'object' && value !== null && 'richText' in value
        ? value.richText.map((t: any) => t.text).join('')
        : (value?.toString() || `Column${colNumber}`);
      headers.push(textValue);
    });

    // 데이터 추출 (2행부터)
    worksheet.eachRow((row, rowNumber) => {
      if (rowNumber > 1) {
        const rowData: any = {};
        row.eachCell((cell, colNumber) => {
          const value = cell.value;
          // richText 객체를 텍스트로 변환
          const textValue = typeof value === 'object' && value !== null && 'richText' in value
            ? value.richText.map((t: any) => t.text).join('')
            : value;
          rowData[headers[colNumber - 1]] = textValue;
        });
        jsonData.push(rowData);
      }
    });

    // 환자 데이터 변환
    const patients: any[] = [];
    
    for (const row of jsonData) {
      const rowData = row as any;
      
      // 환자명 추출
      const name = rowData['환자명'] || rowData['이름'] || rowData['성명'] || 
                   rowData['patient_name'] || rowData['name'] || 
                   rowData['氏名'] || rowData['患者名'];
      
      if (!name || typeof name !== 'string' || name.trim() === '') {
        continue;
      }

      // 혈압 데이터
      const systolic = parseFloat(rowData['수축기혈압'] || rowData['수축기'] || rowData['SBP'] || '0');
      const diastolic = parseFloat(rowData['이완기혈압'] || rowData['이완기'] || rowData['DBP'] || '0');
      
      // 혈중지질 데이터
      const hdl = parseFloat(rowData['HDL'] || rowData['hdl'] || '0');
      const ldl = parseFloat(rowData['LDL'] || rowData['ldl'] || '0');
      const triglycerides = parseFloat(rowData['중성지방'] || rowData['TG'] || '0');
      
      // 혈당 데이터
      const glucose = parseFloat(rowData['혈당'] || rowData['공복혈당'] || rowData['glucose'] || '0');
      const hba1c = parseFloat(rowData['HbA1c'] || rowData['당화혈색소'] || '0');
      
      // 기타 데이터
      const bmi = parseFloat(rowData['BMI'] || rowData['bmi'] || '0');
      const waist = parseFloat(rowData['복부둘레'] || rowData['허리둘레'] || '0');
      const urineProtein = rowData['요단백'] || rowData['urine_protein'] || '';
      const pastHistory = rowData['기왕력'] || rowData['과거병력'] || '';
      const currentHistory = rowData['현병력'] || rowData['현재병력'] || '';

      patients.push({
        name: name.trim(),
        bloodPressure: systolic > 0 && diastolic > 0 ? { systolic, diastolic } : null,
        cholesterol: (hdl > 0 || ldl > 0 || triglycerides > 0) ? { hdl, ldl, triglycerides } : null,
        bloodSugar: (glucose > 0 || hba1c > 0) ? { glucose, hba1c } : null,
        bmi: bmi > 0 ? bmi : null,
        waist: waist > 0 ? waist : null,
        urineProtein,
        pastHistory,
        currentHistory,
        rawData: rowData
      });
    }

    if (patients.length === 0) {
      return NextResponse.json({
        success: false,
        error: '유효한 환자 데이터를 찾을 수 없습니다.'
      }, { status: 400 });
    }

    // 배치 작업 ID 생성
    const jobId = uuidv4();
    
    // 백그라운드에서 분석 시작
    processBatchAnalysis(jobId, patients).catch(error => {
      console.error('배치 분석 프로세스 오류:', error);
    });

    return NextResponse.json({
      success: true,
      data: {
        jobId,
        patientCount: patients.length,
        estimatedTime: Math.ceil(patients.length * 2 / 60) // 분 단위 예상 시간
      }
    });

  } catch (error) {
    console.error('배치 분석 시작 오류:', error);
    return NextResponse.json({
      success: false,
      error: error instanceof Error ? error.message : '배치 분석 시작에 실패했습니다.'
    }, { status: 500 });
  }
}