import { NextRequest, NextResponse } from 'next/server';

/**
 * 보안 로그 인터페이스
 */
interface SecurityLog {
  timestamp: string;
  type: string;
  path: string;
  ip: string;
  userAgent: string;
  reason?: string;
}

/**
 * 보안 로그 저장소 (메모리 기반, 프로덕션에서는 DB 사용 권장)
 * 최대 1000개의 로그를 저장하고, 오래된 로그는 자동 삭제
 */
class SecurityLogStore {
  private logs: SecurityLog[] = [];
  private readonly MAX_LOGS = 1000;

  addLog(log: SecurityLog) {
    this.logs.push(log);
    if (this.logs.length > this.MAX_LOGS) {
      this.logs.shift(); // 가장 오래된 로그 제거
    }
  }

  getLogs(limit: number = 100, offset: number = 0): SecurityLog[] {
    return this.logs.slice(-limit - offset, -offset || undefined).reverse();
  }

  getStats() {
    const now = Date.now();
    const oneHourAgo = now - 60 * 60 * 1000;
    const oneDayAgo = now - 24 * 60 * 60 * 1000;

    const recentLogs = this.logs.filter(
      log => new Date(log.timestamp).getTime() > oneHourAgo
    );
    const dailyLogs = this.logs.filter(
      log => new Date(log.timestamp).getTime() > oneDayAgo
    );

    // 타입별 통계
    const typeStats = this.logs.reduce((acc, log) => {
      acc[log.type] = (acc[log.type] || 0) + 1;
      return acc;
    }, {} as Record<string, number>);

    // IP별 통계 (상위 10개)
    const ipStats = this.logs.reduce((acc, log) => {
      acc[log.ip] = (acc[log.ip] || 0) + 1;
      return acc;
    }, {} as Record<string, number>);

    const topIps = Object.entries(ipStats)
      .sort(([, a], [, b]) => b - a)
      .slice(0, 10)
      .map(([ip, count]) => ({ ip, count }));

    // 경로별 통계 (상위 10개)
    const pathStats = this.logs.reduce((acc, log) => {
      acc[log.path] = (acc[log.path] || 0) + 1;
      return acc;
    }, {} as Record<string, number>);

    const topPaths = Object.entries(pathStats)
      .sort(([, a], [, b]) => b - a)
      .slice(0, 10)
      .map(([path, count]) => ({ path, count }));

    return {
      total: this.logs.length,
      lastHour: recentLogs.length,
      last24Hours: dailyLogs.length,
      byType: typeStats,
      topIps,
      topPaths,
    };
  }

  clear() {
    this.logs = [];
  }
}

// 전역 보안 로그 저장소
export const securityLogStore = new SecurityLogStore();

/**
 * GET /api/admin/security
 * 보안 로그 및 통계 조회
 */
export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const action = searchParams.get('action') || 'logs';
    const limit = parseInt(searchParams.get('limit') || '100');
    const offset = parseInt(searchParams.get('offset') || '0');

    // 관리자 권한 체크 (실제 구현에서는 JWT 토큰 검증 필요)
    const authToken = request.cookies.get('auth_token');
    if (!authToken) {
      return NextResponse.json(
        { error: 'Unauthorized' },
        { status: 401 }
      );
    }

    if (action === 'stats') {
      // 통계 조회
      const stats = securityLogStore.getStats();
      return NextResponse.json({ success: true, data: stats });
    } else if (action === 'logs') {
      // 로그 조회
      const logs = securityLogStore.getLogs(limit, offset);
      return NextResponse.json({ success: true, data: logs });
    } else {
      return NextResponse.json(
        { error: 'Invalid action' },
        { status: 400 }
      );
    }
  } catch (error) {
    console.error('Security API error:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}

/**
 * POST /api/admin/security
 * 보안 로그 추가 (middleware에서 호출)
 */
export async function POST(request: NextRequest) {
  try {
    const log: SecurityLog = await request.json();
    securityLogStore.addLog(log);
    return NextResponse.json({ success: true });
  } catch (error) {
    console.error('Security log add error:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}

/**
 * DELETE /api/admin/security
 * 보안 로그 삭제 (관리자만)
 */
export async function DELETE(request: NextRequest) {
  try {
    // 관리자 권한 체크
    const authToken = request.cookies.get('auth_token');
    if (!authToken) {
      return NextResponse.json(
        { error: 'Unauthorized' },
        { status: 401 }
      );
    }

    securityLogStore.clear();
    return NextResponse.json({ success: true, message: 'Security logs cleared' });
  } catch (error) {
    console.error('Security log clear error:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}
