import { NextRequest, NextResponse } from 'next/server';

const OLLAMA_URL = process.env.NEXT_PUBLIC_OLLAMA_PROXY_SERVER || process.env.OLLAMA_URL || "https://api.hamonize.com/ollama";

export async function POST(request: NextRequest) {
  try {
    const { modelName } = await request.json();
    
    if (!modelName) {
      return NextResponse.json({ error: '모델명이 필요합니다.' }, { status: 400 });
    }

    // Ollama 모델 상세 정보 조회
    const response = await fetch(`${OLLAMA_URL}/api/show`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ name: modelName }),
    });

    if (!response.ok) {
      throw new Error(`Ollama API 오류: ${response.status}`);
    }

    const data = await response.json();
    
    // Modelfile에서 메타데이터 추출
    const modelfile = data.modelfile || '';
    const parameters = data.parameters || {};
    const template = data.template || '';
    
    // 파라미터 수 추출 (모델파일에서)
    let parameterCount = 'Unknown';
    const sizeMatch = modelfile.match(/(\d+\.?\d*)[BM]/i);
    if (sizeMatch) {
      parameterCount = sizeMatch[0].toUpperCase();
    }

    // 컨텍스트 길이 추출
    let contextLength = parameters.num_ctx || 2048;
    
    // 도구 지원 여부 (템플릿에서 추론)
    const supportsTools = template.includes('tool') || 
                         template.includes('function') || 
                         modelfile.toLowerCase().includes('tool');

    // 비전 지원 여부
    const supportsVision = modelfile.toLowerCase().includes('vision') ||
                          modelfile.toLowerCase().includes('multimodal') ||
                          modelName.toLowerCase().includes('vision') ||
                          modelName.toLowerCase().includes('vl');

    const modelInfo = {
      name: modelName,
      parameters: parameterCount,
      context_length: contextLength,
      supports_tools: supportsTools,
      supports_vision: supportsVision,
      modelfile: modelfile,
      template: template,
      system: data.system || '',
      details: data.details || {}
    };

    return NextResponse.json(modelInfo);
  } catch (error) {
    console.error('모델 정보 조회 오류:', error);
    return NextResponse.json({ 
      error: '모델 정보를 조회할 수 없습니다.',
      details: String(error)
    }, { status: 500 });
  }
} 