import { NextRequest, NextResponse } from 'next/server';

/**
 * 분석 요청을 기존 Chandra 처리 라우트로 프록시합니다.
 * 프론트에서는 /api/formbuilder/analyze 를 호출하지만 실제 처리는
 * /api/formbuilder/chandra/process 로 위임합니다.
 */
export async function POST(request: NextRequest) {
  try {
    const formData = await request.formData();
    const file = formData.get('file') as File | null;
    const pagesJson = formData.get('pages') as string | null;

    if (!file) {
      return NextResponse.json({ success: false, error: '파일이 없습니다.' }, { status: 400 });
    }

    // Chandra 라우트가 기대하는 form-data 구성으로 변환
    const proxyFormData = new FormData();
    proxyFormData.append('file', file);

    // 선택된 페이지가 있다면 pageRange 로 전달 (1-based)
    if (pagesJson) {
      try {
        const pages = JSON.parse(pagesJson) as number[];
        if (Array.isArray(pages) && pages.length > 0) {
          const pageRange = pages.map((p) => String(p)).join(',');
          proxyFormData.append('pageRange', pageRange);
        }
      } catch (e) {
        console.warn('[analyze] pages 파싱 실패, 전체 페이지 처리로 진행', e);
      }
    }

    // 같은 호스트의 Chandra 라우트로 프록시 호출
    const targetUrl = new URL('/api/formbuilder/chandra/process', request.nextUrl.origin);
    const proxyResponse = await fetch(targetUrl, {
      method: 'POST',
      body: proxyFormData,
    });

    const contentType = proxyResponse.headers.get('content-type') || '';
    let data: any = null;
    if (contentType.includes('application/json')) {
      data = await proxyResponse.json();
    } else {
      // HTML 등 비-JSON 응답을 안전하게 처리
      const text = await proxyResponse.text();
      return NextResponse.json(
        { success: false, error: '문서 분석 응답이 올바른 JSON이 아닙니다.', details: text.slice(0, 500) },
        { status: 502 }
      );
    }

    return NextResponse.json(data, { status: proxyResponse.status });
  } catch (error: any) {
    console.error('formbuilder/analyze proxy error:', error);
    return NextResponse.json(
      { success: false, error: '문서 분석 중 오류가 발생했습니다.', details: error?.message || String(error) },
      { status: 500 }
    );
  }
}
