import { NextRequest, NextResponse } from 'next/server';
import { writeFile, mkdir } from 'fs/promises';
import { existsSync } from 'fs';
import path from 'path';
import { exec } from 'child_process';
import { promisify } from 'util';

const execAsync = promisify(exec);

const CHANDRA_PATH = process.env.CHANDRA_PATH || path.join(process.env.HOME!, 'airun', 'chandra');
const UPLOAD_DIR = process.env.FORMBUILDER_UPLOAD_DIR || '/tmp/formbuilder-uploads';
const OUTPUT_DIR = process.env.FORMBUILDER_OUTPUT_DIR || '/tmp/formbuilder-outputs';

/**
 * HTML 내 이미지 경로를 API 엔드포인트로 변환
 */
function convertImagePathsToAPI(html: string, timestamp: number, baseFilename: string): string {
  // img 태그의 src 속성을 찾아서 API 경로로 변경
  return html.replace(/src="([^"]+\.webp)"/g, (match, imagePath) => {
    // 이미 절대 경로이거나 http로 시작하는 경우는 변경하지 않음
    if (imagePath.startsWith('http') || imagePath.startsWith('/')) {
      return match;
    }
    // 상대 경로를 API 엔드포인트 경로로 변환
    const apiPath = `/api/formbuilder/images/${timestamp}/${baseFilename}/${imagePath}`;
    return `src="${apiPath}"`;
  });
}

/**
 * Markdown 내 이미지 경로를 API 엔드포인트로 변환
 */
function convertMarkdownImagePathsToAPI(markdown: string, timestamp: number, baseFilename: string): string {
  // Markdown 이미지 형식 ![alt](src) 찾아서 API 경로로 변경
  return markdown.replace(/!\[([^\]]*)\]\(([^)]+\.webp)\)/g, (match, alt, imagePath) => {
    // 이미 절대 경로이거나 http로 시작하는 경우는 변경하지 않음
    if (imagePath.startsWith('http') || imagePath.startsWith('/')) {
      return match;
    }
    // 상대 경로를 API 엔드포인트 경로로 변환
    const apiPath = `/api/formbuilder/images/${timestamp}/${baseFilename}/${imagePath}`;
    return `![${alt}](${apiPath})`;
  });
}

// 번역 함수 제거: Chandra CLI 자체에서 번역을 수행하므로 불필요

// 지원하는 문서 포맷
const SUPPORTED_FORMATS = {
  pdf: ['.pdf'],
  image: ['.png', '.jpg', '.jpeg', '.webp', '.gif', '.tiff', '.bmp'],
  document: ['.doc', '.docx', '.odt', '.hwp', '.hwpx', '.ppt', '.pptx', '.xls', '.xlsx'],
};

/**
 * LibreOffice를 사용하여 문서를 PDF로 변환
 */
async function convertToPDF(filePath: string): Promise<string> {
  try {
    const outputDir = path.dirname(filePath);
    const fileName = path.basename(filePath);

    console.log(`📄 Converting document to PDF: ${fileName}`);

    // LibreOffice 변환 명령 실행
    const { stdout, stderr } = await execAsync(
      `cd "${outputDir}" && libreoffice --headless --convert-to pdf "${fileName}"`,
      { timeout: 60000 } // 1분 타임아웃
    );

    console.log(`✅ LibreOffice output: ${stdout}`);
    if (stderr) {
      console.log(`⚠️ LibreOffice stderr: ${stderr}`);
    }

    // 변환된 PDF 경로
    const pdfPath = filePath.replace(/\.[^.]+$/, '.pdf');

    if (existsSync(pdfPath)) {
      console.log(`✅ PDF conversion successful: ${pdfPath}`);
      return pdfPath;
    } else {
      throw new Error('PDF file not created');
    }
  } catch (error: any) {
    console.error('❌ PDF conversion error:', error);
    throw new Error(`Failed to convert document to PDF: ${error.message}`);
  }
}

// POST: Chandra vLLM을 사용한 문서 처리
export async function POST(request: NextRequest) {
  try {
    const formData = await request.formData();
    const file = formData.get('file') as File;
    const pageRange = formData.get('pageRange') as string | null;

    if (!file) {
      return NextResponse.json(
        { success: false, error: 'No file provided' },
        { status: 400 }
      );
    }

    // 업로드 디렉토리 생성
    if (!existsSync(UPLOAD_DIR)) {
      await mkdir(UPLOAD_DIR, { recursive: true });
    }
    if (!existsSync(OUTPUT_DIR)) {
      await mkdir(OUTPUT_DIR, { recursive: true });
    }

    // 고유한 파일명 생성
    const timestamp = Date.now();
    const sanitizedFilename = file.name.replace(/[^a-zA-Z0-9.-]/g, '_');
    const inputPath = path.join(UPLOAD_DIR, `${timestamp}_${sanitizedFilename}`);
    const outputPath = path.join(OUTPUT_DIR, `${timestamp}`);

    // 파일 저장
    const bytes = await file.arrayBuffer();
    const buffer = Buffer.from(bytes);
    await writeFile(inputPath, buffer);

    console.log(`📄 File saved: ${inputPath}`);

    // 파일 확장자 확인 및 PDF 변환 필요 여부 결정
    const fileExt = path.extname(sanitizedFilename).toLowerCase();
    let processPath = inputPath;

    // 문서 파일인 경우 PDF로 변환
    const allDocFormats = [...SUPPORTED_FORMATS.document];
    if (allDocFormats.includes(fileExt)) {
      console.log(`📄 Document file detected (${fileExt}), converting to PDF...`);
      try {
        processPath = await convertToPDF(inputPath);
        console.log(`✅ Converted to PDF: ${processPath}`);
      } catch (conversionError: any) {
        console.error(`❌ PDF conversion failed:`, conversionError);
        return NextResponse.json(
          {
            success: false,
            error: '문서를 PDF로 변환할 수 없습니다.',
            details: `지원하지 않는 문서 형식이거나 변환 중 오류가 발생했습니다: ${conversionError.message}`,
          },
          { status: 400 }
        );
      }
    }

    // 페이지 범위 변환 (1-based → 0-based)
    let convertedPageRange = '';
    if (pageRange) {
      // "1,2,3" → "0,1,2" 또는 "1-3,5" → "0-2,4" 형식으로 변환
      const parts = pageRange.split(',');
      const convertedParts = parts.map(part => {
        if (part.includes('-')) {
          // 범위 형식 (예: "1-3")
          const [start, end] = part.split('-').map(Number);
          return `${start - 1}-${end - 1}`;
        } else {
          // 단일 페이지 (예: "5")
          return String(Number(part) - 1);
        }
      });
      convertedPageRange = convertedParts.join(',');
      console.log(`📄 Page range conversion: ${pageRange} (1-based) → ${convertedPageRange} (0-based)`);
    }

    // Chandra 실행 (페이지 범위 옵션 추가)
    const pageRangeOption = convertedPageRange ? `--page-range '${convertedPageRange}'` : '';
    
    const ollamaUrl = process.env.OLLAMA_PROXY_SERVER || 'https://api.hamonize.com/ollama';
    const chandraCommand = `cd ${CHANDRA_PATH} && OLLAMA_PROXY_SERVER='${ollamaUrl}' VLLM_API_BASE=http://121.78.116.30:8000/v1 VLLM_MODEL_NAME=chandra VLLM_API_KEY=EMPTY bash -c "source ~/.airun_venv/bin/activate && chandra '${processPath}' '${outputPath}' --method vllm --max-workers 1 ${pageRangeOption}"`;

    console.log(`🚀 Running Chandra: ${chandraCommand}${convertedPageRange ? ` (pages: ${convertedPageRange})` : ''}`);

    const { stdout, stderr } = await execAsync(chandraCommand, {
      timeout: 180000, // 3분 타임아웃
      maxBuffer: 10 * 1024 * 1024, // 10MB buffer
    });

    console.log(`✅ Chandra stdout:`, stdout);
    if (stderr) {
      console.log(`⚠️ Chandra stderr:`, stderr);
    }

    // 결과 파일 경로 확인 (processPath 기준으로 설정 - 변환된 PDF 파일명 사용)
    const baseFilename = path.basename(processPath, path.extname(processPath));
    const resultDir = path.join(outputPath, baseFilename);
    const markdownFile = path.join(resultDir, `${baseFilename}.md`);
    const htmlFile = path.join(resultDir, `${baseFilename}.html`);
    const jsonFile = path.join(resultDir, `${baseFilename}.json`);
    const metadataFile = path.join(resultDir, `${baseFilename}_metadata.json`);

    // 결과 파일 읽기
    const fs = require('fs');
    let markdown = '';
    let html = '';
    let structuredJson = null;
    let metadata = null;

    if (fs.existsSync(markdownFile)) {
      markdown = fs.readFileSync(markdownFile, 'utf-8');
      // 이미지 경로를 API 엔드포인트로 변환
      markdown = convertMarkdownImagePathsToAPI(markdown, timestamp, baseFilename);
    }
    if (fs.existsSync(htmlFile)) {
      html = fs.readFileSync(htmlFile, 'utf-8');
      // 이미지 경로를 API 엔드포인트로 변환
      html = convertImagePathsToAPI(html, timestamp, baseFilename);
      // 번역은 이미 Chandra CLI에서 수행됨
    }
    if (fs.existsSync(jsonFile)) {
      structuredJson = JSON.parse(fs.readFileSync(jsonFile, 'utf-8'));
    }
    if (fs.existsSync(metadataFile)) {
      metadata = JSON.parse(fs.readFileSync(metadataFile, 'utf-8'));
    }

    // VLLM 서버 연결 오류 감지
    if (stderr && stderr.includes('Connection error')) {
      return NextResponse.json(
        {
          success: false,
          error: 'VLLM API 서버에 연결할 수 없습니다.',
          details: 'VLLM 서버(http://121.78.116.30:8000)가 실행 중인지 확인해주세요.',
          stderr,
        },
        { status: 503 }
      );
    }

    // 빈 결과 파일 확인
    if (!html || html.trim().length === 0) {
      return NextResponse.json(
        {
          success: false,
          error: '문서 처리 결과가 비어있습니다.',
          details: 'Chandra가 문서를 처리했지만 HTML 결과가 생성되지 않았습니다. 로그를 확인해주세요.',
          stdout,
          stderr,
        },
        { status: 500 }
      );
    }

    // 응답 반환
    return NextResponse.json({
      success: true,
      data: {
        markdown,
        html,
        text: markdown, // 텍스트는 마크다운과 동일
        structuredJson, // 구조화된 JSON 데이터
        metadata,
        filename: file.name,
        inputPath,
        outputPath: resultDir,
      },
    });
  } catch (error: any) {
    console.error('Chandra process error:', error);
    return NextResponse.json(
      {
        success: false,
        error: 'Failed to process document with Chandra',
        details: error.message || String(error),
      },
      { status: 500 }
    );
  }
}
