import { NextRequest, NextResponse } from 'next/server';
import { writeFile, mkdir, readFile, unlink } from 'fs/promises';
import { existsSync } from 'fs';
import path from 'path';
import { exec } from 'child_process';
import { promisify } from 'util';

const execAsync = promisify(exec);

const UPLOAD_DIR = '/tmp/formbuilder-pdf-conversion';

// 지원하는 오피스 문서 포맷
const OFFICE_FORMATS = ['.doc', '.docx', '.odt', '.hwp', '.hwpx', '.ppt', '.pptx', '.xls', '.xlsx'];

/**
 * LibreOffice를 사용하여 문서를 PDF로 변환
 */
async function convertToPDF(filePath: string): Promise<string> {
  try {
    const outputDir = path.dirname(filePath);
    const fileName = path.basename(filePath);

    console.log(`📄 Converting document to PDF: ${fileName}`);

    // LibreOffice 변환 명령 실행
    const { stdout, stderr } = await execAsync(
      `cd "${outputDir}" && libreoffice --headless --convert-to pdf "${fileName}"`,
      { timeout: 60000 } // 1분 타임아웃
    );

    console.log(`✅ LibreOffice output: ${stdout}`);
    if (stderr) {
      console.log(`⚠️ LibreOffice stderr: ${stderr}`);
    }

    // 변환된 PDF 경로
    const pdfPath = filePath.replace(/\.[^.]+$/, '.pdf');

    if (existsSync(pdfPath)) {
      console.log(`✅ PDF conversion successful: ${pdfPath}`);
      return pdfPath;
    } else {
      throw new Error('PDF file not created');
    }
  } catch (error: any) {
    console.error('❌ PDF conversion error:', error);
    throw new Error(`Failed to convert document to PDF: ${error.message}`);
  }
}

// POST: 오피스 문서를 PDF로 변환
export async function POST(request: NextRequest) {
  let inputPath: string | null = null;
  let pdfPath: string | null = null;

  try {
    const formData = await request.formData();
    const file = formData.get('file') as File;

    if (!file) {
      return NextResponse.json(
        { success: false, error: 'No file provided' },
        { status: 400 }
      );
    }

    // 파일 확장자 확인
    const fileExt = path.extname(file.name).toLowerCase();
    if (!OFFICE_FORMATS.includes(fileExt)) {
      return NextResponse.json(
        { success: false, error: 'Unsupported file format. Only office documents are supported.' },
        { status: 400 }
      );
    }

    // 업로드 디렉토리 생성
    if (!existsSync(UPLOAD_DIR)) {
      await mkdir(UPLOAD_DIR, { recursive: true });
    }

    // 고유한 파일명 생성
    const timestamp = Date.now();
    const sanitizedFilename = file.name.replace(/[^a-zA-Z0-9.-]/g, '_');
    inputPath = path.join(UPLOAD_DIR, `${timestamp}_${sanitizedFilename}`);

    // 파일 저장
    const bytes = await file.arrayBuffer();
    const buffer = Buffer.from(bytes);
    await writeFile(inputPath, buffer);

    console.log(`📄 File saved: ${inputPath}`);

    // PDF로 변환
    try {
      pdfPath = await convertToPDF(inputPath);
      console.log(`✅ Converted to PDF: ${pdfPath}`);
    } catch (conversionError: any) {
      console.error(`❌ PDF conversion failed:`, conversionError);
      return NextResponse.json(
        {
          success: false,
          error: '문서를 PDF로 변환할 수 없습니다.',
          details: `지원하지 않는 문서 형식이거나 변환 중 오류가 발생했습니다: ${conversionError.message}`,
        },
        { status: 400 }
      );
    }

    // PDF 파일 읽기
    const pdfBuffer = await readFile(pdfPath);

    // 임시 파일 정리 (비동기로 실행하여 응답 속도 개선)
    Promise.all([
      inputPath ? unlink(inputPath).catch(() => {}) : Promise.resolve(),
      pdfPath ? unlink(pdfPath).catch(() => {}) : Promise.resolve(),
    ]);

    // PDF를 응답으로 반환
    return new NextResponse(pdfBuffer, {
      status: 200,
      headers: {
        'Content-Type': 'application/pdf',
        'Content-Disposition': `attachment; filename="${path.basename(pdfPath)}"`,
      },
    });
  } catch (error: any) {
    console.error('PDF conversion error:', error);

    // 에러 발생 시에도 임시 파일 정리
    if (inputPath) {
      unlink(inputPath).catch(() => {});
    }
    if (pdfPath) {
      unlink(pdfPath).catch(() => {});
    }

    return NextResponse.json(
      {
        success: false,
        error: 'Failed to convert document to PDF',
        details: error.message || String(error),
      },
      { status: 500 }
    );
  }
}
