import { NextRequest, NextResponse } from 'next/server';
import { pg } from '@/app/api/db';
import crypto from 'crypto';

// GET: 사용자의 폼 목록 조회
export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const username = searchParams.get('username');

    if (!username) {
      return NextResponse.json(
        { success: false, error: 'Username is required' },
        { status: 400 }
      );
    }

    const { rows } = await pg.query(
      `SELECT
        id, title, description, document_filename,
        share_link, view_link, response_count, status,
        created_at, updated_at
      FROM ai_forms
      WHERE username = $1
      ORDER BY created_at DESC`,
      [username]
    );

    return NextResponse.json({
      success: true,
      data: rows,
    });
  } catch (error) {
    console.error('Get forms error:', error);
    return NextResponse.json(
      { success: false, error: 'Failed to fetch forms' },
      { status: 500 }
    );
  }
}

// POST: 새 폼 생성
export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const {
      title,
      description,
      document_filename,
      analyzed_content,
      form_schema,
      user_id,
      username,
    } = body;

    if (!title || !form_schema || !user_id || !username) {
      return NextResponse.json(
        { success: false, error: 'Missing required fields' },
        { status: 400 }
      );
    }

    // 고유한 ID와 링크 생성
    const formId = `form_${Date.now()}_${crypto.randomBytes(8).toString('hex')}`;
    const shareLink = crypto.randomBytes(16).toString('hex');
    const viewLink = crypto.randomBytes(16).toString('hex');

    const { rows } = await pg.query(
      `INSERT INTO ai_forms (
        id, user_id, username, title, description,
        document_filename, analyzed_content, form_schema,
        share_link, view_link, status
      ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)
      RETURNING *`,
      [
        formId,
        user_id,
        username,
        title,
        description || '',
        document_filename || '',
        analyzed_content || '',
        JSON.stringify(form_schema),
        shareLink,
        viewLink,
        'active',
      ]
    );

    return NextResponse.json({
      success: true,
      data: rows[0],
      message: 'Form created successfully',
    });
  } catch (error) {
    console.error('Create form error:', error);
    return NextResponse.json(
      { success: false, error: 'Failed to create form' },
      { status: 500 }
    );
  }
}

// PUT: 폼 수정
export async function PUT(request: NextRequest) {
  try {
    const body = await request.json();
    const { id, title, description, form_schema, status } = body;

    if (!id) {
      return NextResponse.json(
        { success: false, error: 'Form ID is required' },
        { status: 400 }
      );
    }

    const updates: string[] = [];
    const values: any[] = [];
    let paramIndex = 1;

    if (title !== undefined) {
      updates.push(`title = $${paramIndex++}`);
      values.push(title);
    }
    if (description !== undefined) {
      updates.push(`description = $${paramIndex++}`);
      values.push(description);
    }
    if (form_schema !== undefined) {
      updates.push(`form_schema = $${paramIndex++}`);
      values.push(JSON.stringify(form_schema));
    }
    if (status !== undefined) {
      updates.push(`status = $${paramIndex++}`);
      values.push(status);
    }

    updates.push(`updated_at = CURRENT_TIMESTAMP`);
    values.push(id);

    const { rows } = await pg.query(
      `UPDATE ai_forms
       SET ${updates.join(', ')}
       WHERE id = $${paramIndex}
       RETURNING *`,
      values
    );

    if (rows.length === 0) {
      return NextResponse.json(
        { success: false, error: 'Form not found' },
        { status: 404 }
      );
    }

    return NextResponse.json({
      success: true,
      data: rows[0],
      message: 'Form updated successfully',
    });
  } catch (error) {
    console.error('Update form error:', error);
    return NextResponse.json(
      { success: false, error: 'Failed to update form' },
      { status: 500 }
    );
  }
}

// DELETE: 폼 삭제
export async function DELETE(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const id = searchParams.get('id');

    if (!id) {
      return NextResponse.json(
        { success: false, error: 'Form ID is required' },
        { status: 400 }
      );
    }

    const { rowCount } = await pg.query(
      'DELETE FROM ai_forms WHERE id = $1',
      [id]
    );

    if (rowCount === 0) {
      return NextResponse.json(
        { success: false, error: 'Form not found' },
        { status: 404 }
      );
    }

    return NextResponse.json({
      success: true,
      message: 'Form deleted successfully',
    });
  } catch (error) {
    console.error('Delete form error:', error);
    return NextResponse.json(
      { success: false, error: 'Failed to delete form' },
      { status: 500 }
    );
  }
}
