import { NextRequest, NextResponse } from 'next/server';

// POST: AI를 사용하여 분석된 문서로부터 폼 필드 생성
export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { html, markdown, text, metadata } = body;

    // 요청 헤더에서 API 키 가져오기
    const apiKeyFromHeader = request.headers.get('X-API-Key') || '';

    // 최소한 하나의 콘텐츠는 필요
    if (!html && !markdown && !text) {
      return NextResponse.json(
        { success: false, error: 'At least one content type (html, markdown, or text) is required' },
        { status: 400 }
      );
    }

    // AI API 호출하여 폼 필드 생성
    // 구조화된 데이터(HTML, Markdown)를 사용하여 문서 구조를 최대한 보존
    const fields = await generateFieldsFromStructuredContent({
      html,
      markdown,
      text,
      metadata,
      apiKey: apiKeyFromHeader,
    });

    return NextResponse.json({
      success: true,
      data: {
        title: fields.title || '자동 생성된 폼',
        description: fields.description || '',
        fields: fields.fields || [],
      },
    });
  } catch (error) {
    console.error('❌ Generate fields error:', error);
    if (error instanceof Error) {
      console.error('Error details:', {
        message: error.message,
        stack: error.stack,
      });
    }
    return NextResponse.json(
      {
        success: false,
        error: error instanceof Error ? error.message : 'Failed to generate fields',
      },
      { status: 500 }
    );
  }
}

interface StructuredContent {
  html?: string;
  markdown?: string;
  text?: string;
  metadata?: any;
  apiKey?: string;
}

async function generateFieldsFromStructuredContent(data: StructuredContent) {
  // 우선순위: HTML > Markdown > Text (스코프 문제 해결을 위해 함수 시작 부분에서 선언)
  const primaryContent = data.html || data.markdown || data.text || '';
  const contentType = data.html ? 'HTML' : (data.markdown ? 'Markdown' : 'Text');

  try {
    // Ollama 또는 OpenAI API를 사용하여 폼 필드 생성
    const apiBaseUrl = process.env.NEXT_PUBLIC_API_SERVER_URL || 'http://localhost:5500/api/v1';
    // API URL이 이미 /api/v1을 포함하고 있으면 그대로 사용, 아니면 추가
    const apiUrl = apiBaseUrl.endsWith('/api/v1') ? apiBaseUrl.replace('/api/v1', '') : apiBaseUrl;

    console.log('🔍 Form generation started');
    console.log('📊 Content type:', contentType);
    console.log('📏 Content length:', primaryContent.length);

    const systemPrompt = `당신은 문서 분석 및 폼 생성 전문가입니다.
주어진 ${contentType} 문서의 구조를 정확히 분석하여, 원본 문서에 있는 모든 입력 필드를 찾아 그대로 재현하는 입력 폼을 생성해야 합니다.

**중요 원칙:**
1. **그룹 구조 보존**: 문서의 섹션, 그룹, 카테고리 구조를 정확히 파악하여 각 필드가 어느 그룹에 속하는지 명시하세요
   - 예: "건물현황", "소유자", "수선계획" 같은 섹션 제목을 찾아 group으로 지정
   - 표의 rowspan이나 섹션 제목을 그룹명으로 사용
   - 그룹이 없는 필드는 group을 빈 문자열로 설정
2. **필드 순서 보존**: 문서에 표시된 필드 순서를 그대로 유지하세요
3. **라벨 정확성**: 필드 라벨은 문서에 표시된 텍스트를 정확히 사용하세요 (번역하거나 수정하지 마세요)
4. **필수 항목 판단**: 문서의 '*' 표시, '필수', 'required' 등의 표시를 찾아 필수 여부를 결정하세요
5. **타입 추론**: 필드 내용과 이름을 보고 적절한 입력 타입을 선택하세요
   - 이메일 주소 → email
   - 전화번호/숫자 → number
   - 날짜 → date
   - 긴 텍스트/설명 → textarea
   - 선택지가 있는 경우 → select
   - 체크박스 선택 (□) → checkbox
   - 기본 → text
6. **누락 방지**: 문서의 모든 입력 필드를 빠짐없이 포함하세요

응답은 반드시 JSON 형식으로만 작성하고, 다른 설명은 포함하지 마세요.`;

    const userPrompt = `다음 ${contentType} 문서를 분석하여 입력 폼을 생성해주세요.
문서의 모든 필드를 정확히 찾아내고, 섹션/그룹 구조와 순서, 라벨을 그대로 유지하세요.

${contentType === 'HTML' ? '--- HTML 문서 ---' : contentType === 'Markdown' ? '--- Markdown 문서 ---' : '--- 문서 내용 ---'}
${primaryContent}
---

다음 JSON 형식으로만 응답하세요:
{
  "title": "문서에서 추출한 폼 제목",
  "description": "문서의 설명 또는 안내문",
  "fields": [
    {
      "id": "field_1",
      "label": "문서에 표시된 정확한 라벨",
      "type": "text|email|number|date|textarea|select|checkbox",
      "required": true|false,
      "group": "필드가 속한 그룹명 (예: 건물현황, 소유자 등)",
      "options": ["옵션1", "옵션2"],
      "placeholder": "입력 힌트 (선택사항)"
    }
  ]
}`;

    // API 키 가져오기 (요청 헤더 > 환경변수 > 기본값)
    const apiKey = data.apiKey ||
                   process.env.AIRUN_API_KEY ||
                   process.env.API_KEY ||
                   'airun_1_fcbeea114f222187734adc16f3b31f76';

    console.log('🔑 API Key source:', data.apiKey ? 'Request Header' : 'Fallback');
    console.log('🔑 API Key configured:', apiKey ? 'Yes' : 'No');
    console.log('🔑 API Key prefix:', apiKey.substring(0, 10) + '...');
    console.log('🌐 API URL:', `${apiUrl}/api/v1/callai`);
    console.log('🤖 Using model: airun-chat');

    const response = await fetch(`${apiUrl}/api/v1/callai`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-API-Key': apiKey,
      },
      body: JSON.stringify({
        messages: [
          { role: 'system', content: systemPrompt },
          { role: 'user', content: userPrompt }
        ],
        options: {
          model: 'airun-chat',
          temperature: 0.1, // 낮은 temperature로 일관성 향상
          max_tokens: 8000
        },
        stream: false
      }),
    });

    console.log('📡 API Response status:', response.status);

    if (!response.ok) {
      const errorText = await response.text();
      console.error('❌ API Error:', errorText);
      throw new Error(`AI API request failed: ${response.status} - ${errorText}`);
    }

    const result = await response.json();
    // /callai 엔드포인트의 응답 구조: result.data.response.content
    const aiResponse = result.data?.response?.content || result.data?.response || result.response || '';

    console.log('🤖 AI Response length:', aiResponse.length);
    console.log('📝 AI Response preview:', aiResponse.substring(0, 200));
    console.log('📦 Full API response:', JSON.stringify(result).substring(0, 300));

    // JSON 파싱
    try {
      // JSON 블록 추출
      const jsonMatch = aiResponse.match(/```json\s*([\s\S]*?)\s*```/) ||
                       aiResponse.match(/\{[\s\S]*\}/);

      if (jsonMatch) {
        const jsonStr = jsonMatch[1] || jsonMatch[0];
        console.log('📋 Extracted JSON:', jsonStr.substring(0, 200));

        const parsed = JSON.parse(jsonStr);

        // ID가 없는 필드에 ID 추가
        if (parsed.fields) {
          parsed.fields = parsed.fields.map((field: any, index: number) => ({
            ...field,
            id: field.id || `field_${index + 1}`,
          }));
        }

        console.log('✅ Successfully parsed form with', parsed.fields?.length || 0, 'fields');
        return parsed;
      } else {
        console.warn('⚠️ No JSON match found in AI response');
      }
    } catch (parseError) {
      console.error('❌ Failed to parse AI response:', parseError);
      console.error('Response was:', aiResponse);
    }

    // 기본 필드 반환
    console.log('⚠️ Using default fields as fallback');
    return generateDefaultFields(primaryContent);
  } catch (error) {
    console.error('❌ AI field generation error:', error);
    if (error instanceof Error) {
      console.error('Error message:', error.message);
      console.error('Error stack:', error.stack);
    }
    console.log('🔄 Falling back to default fields');
    return generateDefaultFields(primaryContent);
  }
}

function generateDefaultFields(content: string) {
  // 문서 내용에서 키워드 추출하여 기본 필드 생성
  const fields = [];

  // 기본 필드들
  fields.push({
    id: 'field_1',
    label: '이름',
    type: 'text',
    required: true,
  });

  if (content.includes('이메일') || content.includes('email')) {
    fields.push({
      id: 'field_2',
      label: '이메일',
      type: 'email',
      required: true,
    });
  }

  if (content.includes('연락처') || content.includes('전화') || content.includes('휴대폰')) {
    fields.push({
      id: 'field_3',
      label: '연락처',
      type: 'text',
      required: true,
    });
  }

  if (content.includes('주소')) {
    fields.push({
      id: 'field_4',
      label: '주소',
      type: 'textarea',
      required: false,
    });
  }

  if (content.includes('생년월일') || content.includes('출생')) {
    fields.push({
      id: 'field_5',
      label: '생년월일',
      type: 'date',
      required: false,
    });
  }

  // 기본적으로 의견/메모 필드 추가
  fields.push({
    id: 'field_memo',
    label: '기타 의견',
    type: 'textarea',
    required: false,
  });

  return {
    title: '자동 생성된 폼',
    description: '문서 내용을 기반으로 생성된 입력 폼입니다.',
    fields,
  };
}
