import { NextResponse } from 'next/server';
import { spawn, ChildProcess } from 'child_process';
import { getGlobalMCPClient } from '@/lib/mcpClient';

// 기존 클래스들은 제거하고 전역 클라이언트 사용

// GET: 전역 MCP 클라이언트를 사용한 도구 정보 가져오기
export async function GET() {
  try {
    console.log('🔄 MCP 도구 정보 가져오기 시작...');
    
    // 전역 MCP 클라이언트 사용 (중복 초기화 방지)
    const globalClient = await getGlobalMCPClient();
    
    if (!globalClient) {
      console.warn('⚠️ 전역 MCP 클라이언트를 가져올 수 없습니다');
      return NextResponse.json({
        success: false,
        error: '전역 MCP 클라이언트를 가져올 수 없습니다. MCP 서버를 먼저 초기화해주세요.',
        implementation: 'Global MCP Client (Error)'
      }, { status: 500 });
    }
    
    console.log('🔄 기존 전역 MCP 클라이언트 재사용');
    
    // 전역 클라이언트에서 도구 정보 가져오기
    const allTools = await globalClient.getAllTools();
    
    // 서버별로 도구 정보 그룹화
    const serverToolsMap = new Map<string, any[]>();
    
    for (const tool of allTools) {
      const serverName = tool.server || tool.serverName || 'unknown';
      if (!serverToolsMap.has(serverName)) {
        serverToolsMap.set(serverName, []);
      }
      serverToolsMap.get(serverName)!.push({
        name: tool.name,
        description: tool.description || '설명 없음'
      });
    }
    
    // 서버별 도구 정보 생성
    const serverToolsInfo = Array.from(serverToolsMap.entries()).map(([serverName, tools]) => ({
      serverName,
      toolCount: tools.length,
      tools: tools,
      transport: 'stdio' // 기본값
    }));
    
    const totalTools = allTools.length;
    
    console.log(`✅ MCP 도구 정보 가져오기 완료: ${serverToolsInfo.length}개 서버, 총 ${totalTools}개 도구`);
    
    return NextResponse.json({
      success: true,
      data: {
        servers: serverToolsInfo,
        totalServerCount: serverToolsInfo.length,
        totalToolCount: totalTools,
        transportTypes: {
          stdio: serverToolsInfo.filter(s => s.transport === 'stdio').length,
          sse: serverToolsInfo.filter(s => s.transport === 'sse').length,
          websocket: serverToolsInfo.filter(s => s.transport === 'websocket').length
        },
        implementation: 'Global MCP Client (Reused)',
        features: [
          '전역 클라이언트 재사용',
          '중복 초기화 방지',
          '리소스 효율성',
          '빠른 응답 시간'
        ]
      }
    });
    
  } catch (error) {
    console.error('MCP 도구 정보 가져오기 실패:', error);
    return NextResponse.json(
      { 
        success: false, 
        error: '도구 정보를 가져오는데 실패했습니다.',
        details: (error as Error).message,
        implementation: 'Global MCP Client (Error)' 
      },
      { status: 500 }
    );
  }
  // finally 블록 제거 - 전역 클라이언트는 정리하지 않음
} 