import { NextResponse } from 'next/server';
import fs from 'fs';
import path from 'path';
import os from 'os';

interface UsageItem {
  timestamp: string;
  userId?: string;
  username?: string;
  sessionId?: string;
  feature?: string;
  prompt?: string;
  [key: string]: any;
}

interface ChatStats {
  dailyChats: number;
  weeklyChats: number;
  monthlyChats: number;
  periodStats: {
    daily: { date: string; count: number }[];
    weekly: { week: string; count: number }[];
    monthly: { month: string; count: number }[];
  };
  categoryStats: Record<string, number>;
  userStats: Array<{
    userId: string;
    username?: string;
    chatCount: number;
    messageCount: number;
    avgMessagesPerChat: number;
  }>;
  timeDistribution: {
    morning: number;    // 06:00-12:00
    afternoon: number;  // 12:00-18:00
    evening: number;    // 18:00-24:00
    night: number;      // 00:00-06:00
  };
  heatmapByHour: number[];
  weekdayStats: Array<{ day: string; count: number }>;
}

function getAppDataPath() {
  switch (process.platform) {
    case 'win32':
      return path.join(process.env.APPDATA || '', 'airun');
    case 'darwin':
      return path.join(os.homedir(), 'Library', 'Application Support', 'airun');
    default:
      return path.join(os.homedir(), '.airun');
  }
}

async function getAllUsageData(): Promise<UsageItem[]> {
  try {
    const appDataDir = getAppDataPath();
    const archiveDir = path.join(appDataDir, 'archives');
    let allData: UsageItem[] = [];

    // 아카이브 데이터
    if (fs.existsSync(archiveDir)) {
      const archiveFiles = await fs.promises.readdir(archiveDir);
      for (const file of archiveFiles) {
        if (file.startsWith('usage_') && file.endsWith('.json')) {
          const archiveContent = await fs.promises.readFile(path.join(archiveDir, file), 'utf8');
          const archiveData = JSON.parse(archiveContent) as UsageItem[];
          allData.push(...archiveData);
        }
      }
    }

    // 현재 데이터
    const usageFilePath = path.join(appDataDir, 'usage.json');
    if (fs.existsSync(usageFilePath)) {
      const fileContent = await fs.promises.readFile(usageFilePath, 'utf8');
      if (fileContent.trim()) {
        const currentData = JSON.parse(fileContent) as UsageItem[];
        allData.push(...currentData);
      }
    }

    // 날짜순 정렬
    allData.sort((a, b) => new Date(a.timestamp).getTime() - new Date(b.timestamp).getTime());
    return allData;
  } catch (error) {
    return [];
  }
}

function categorizePrompt(prompt: string): string {
  const lowerPrompt = prompt.toLowerCase();

  // 카테고리별 키워드 정의
  const categories = {
    '문서 생성': ['만들어', '작성해', '생성해', 'write', 'create', '작성', '글'],
    '요약': ['요약', 'summarize', '정리', '요약해', 'summary'],
    '번역': ['번역', 'translate', '영어', '일본어', '중국어', 'translation'],
    '코드': ['코드', 'code', '프로그래밍', 'programming', '함수', 'function'],
    '분석': ['분석', 'analyze', 'analysis', '분석해', '조사'],
    '질문': ['?', '무엇', '어떻게', '왜', 'what', 'how', 'why', '질문'],
    '수정': ['수정', 'fix', 'correct', '개선', 'improve', '바꿔'],
    '검색': ['찾아', '검색', 'search', 'find', 'lookup'],
    '설명': ['설명', 'explain', '알려줘', '가르쳐줘', 'tell me'],
    '비교': ['비교', 'compare', '차이', 'difference'],
    '계산': ['계산', 'calculate', '얼마', '수', '숫자'],
    '추천': ['추천', 'recommend', '좋은', 'best'],
    '기타': []
  };

  // 카테고리 매칭
  for (const [category, keywords] of Object.entries(categories)) {
    if (category === '기타') continue;
    for (const keyword of keywords) {
      if (lowerPrompt.includes(keyword)) {
        return category;
      }
    }
  }

  return '기타';
}

function getTimeSlot(date: Date): 'morning' | 'afternoon' | 'evening' | 'night' {
  const hour = date.getHours();
  if (hour >= 6 && hour < 12) return 'morning';
  if (hour >= 12 && hour < 18) return 'afternoon';
  if (hour >= 18 && hour < 24) return 'evening';
  return 'night';
}

export async function GET(request: NextRequest) {
  try {
    const searchParams = request.nextUrl.searchParams;
    const period = searchParams.get('period') || 'all'; // all, daily, weekly, monthly
    const userId = searchParams.get('userId');

    const allUsage = await getAllUsageData();
    const now = new Date();

    // 사용자별 필터링
    const filteredUsage = userId
      ? allUsage.filter(u => u.userId === userId)
      : allUsage;

    // 세션별 그룹화
    const sessionMap = new Map<string, UsageItem[]>();
    filteredUsage.forEach(item => {
      const sessionId = item.sessionId || 'default';
      if (!sessionMap.has(sessionId)) {
        sessionMap.set(sessionId, []);
      }
      sessionMap.get(sessionId)!.push(item);
    });

    // 기간별 통계 계산
    const startOfDay = new Date(now.getFullYear(), now.getMonth(), now.getDate());
    const startOfWeek = new Date(now);
    startOfWeek.setDate(now.getDate() - now.getDay());
    const startOfMonth = new Date(now.getFullYear(), now.getMonth(), 1);

    const dailySessions = Array.from(sessionMap.values()).filter(session => {
      const firstItem = session[0];
      return new Date(firstItem.timestamp) >= startOfDay;
    });

    const weeklySessions = Array.from(sessionMap.values()).filter(session => {
      const firstItem = session[0];
      return new Date(firstItem.timestamp) >= startOfWeek;
    });

    const monthlySessions = Array.from(sessionMap.values()).filter(session => {
      const firstItem = session[0];
      return new Date(firstItem.timestamp) >= startOfMonth;
    });

    // 기간별 추이 데이터
    const dailyStats: { date: string; count: number }[] = [];
    const weeklyStats: { week: string; count: number }[] = [];
    const monthlyStats: { month: string; count: number }[] = [];

    // 일별 통계 (최근 30일)
    for (let i = 29; i >= 0; i--) {
      const date = new Date(now);
      date.setDate(date.getDate() - i);
      const dateStr = date.toISOString().split('T')[0];
      const dayStart = new Date(date.setHours(0, 0, 0, 0));
      const dayEnd = new Date(date.setHours(23, 59, 59, 999));

      const daySessions = Array.from(sessionMap.values()).filter(session => {
        const sessionTime = new Date(session[0].timestamp);
        return sessionTime >= dayStart && sessionTime <= dayEnd;
      });

      dailyStats.push({
        date: dateStr,
        count: daySessions.length
      });
    }

    // 주별 통계 (최근 12주)
    for (let i = 11; i >= 0; i--) {
      const weekStart = new Date(now);
      weekStart.setDate(weekStart.getDate() - (weekStart.getDay() + i * 7));
      const weekEnd = new Date(weekStart);
      weekEnd.setDate(weekEnd.getDate() + 6);

      const weekStr = `${weekStart.getFullYear()}-W${Math.ceil((weekStart.getDate() + new Date(weekStart.getFullYear(), weekStart.getMonth(), 1).getDay()) / 7)}`;

      const weekSessions = Array.from(sessionMap.values()).filter(session => {
        const sessionTime = new Date(session[0].timestamp);
        return sessionTime >= weekStart && sessionTime <= weekEnd;
      });

      weeklyStats.push({
        week: weekStr,
        count: weekSessions.length
      });
    }

    // 월별 통계 (최근 12개월)
    for (let i = 11; i >= 0; i--) {
      const month = new Date(now.getFullYear(), now.getMonth() - i, 1);
      const monthStr = `${month.getFullYear()}-${String(month.getMonth() + 1).padStart(2, '0')}`;

      const monthSessions = Array.from(sessionMap.values()).filter(session => {
        const sessionTime = new Date(session[0].timestamp);
        return sessionTime.getMonth() === month.getMonth() &&
               sessionTime.getFullYear() === month.getFullYear();
      });

      monthlyStats.push({
        month: monthStr,
        count: monthSessions.length
      });
    }

    // 분류별 통계
    const categoryStats: Record<string, number> = {};
    Array.from(sessionMap.values()).forEach(session => {
      const firstPrompt = session[0]?.prompt || '';
      const category = categorizePrompt(firstPrompt);
      categoryStats[category] = (categoryStats[category] || 0) + 1;
    });

    // 사용자별 통계
    const userStatsMap = new Map<string, { chatCount: number; messageCount: number; username?: string }>();
    Array.from(sessionMap.values()).forEach(session => {
      const userId = session[0]?.userId || 'unknown';
      const username = session[0]?.username;

      if (!userStatsMap.has(userId)) {
        userStatsMap.set(userId, { chatCount: 0, messageCount: 0, username });
      }

      const stats = userStatsMap.get(userId)!;
      stats.chatCount++;
      stats.messageCount += session.length;
    });

    const userStats = Array.from(userStatsMap.entries()).map(([userId, stats]) => ({
      userId,
      username: stats.username,
      chatCount: stats.chatCount,
      messageCount: stats.messageCount,
      avgMessagesPerChat: stats.chatCount > 0 ? Math.round(stats.messageCount / stats.chatCount * 10) / 10 : 0
    })).sort((a, b) => b.chatCount - a.chatCount).slice(0, 20);

    // 시간대별 분포
    const timeDistribution = { morning: 0, afternoon: 0, evening: 0, night: 0 };
    Array.from(sessionMap.values()).forEach(session => {
      const timeSlot = getTimeSlot(new Date(session[0].timestamp));
      timeDistribution[timeSlot]++;
    });

    // 시간별 히트맵 (24시간)
    const heatmapByHour = Array(24).fill(0);
    Array.from(sessionMap.values()).forEach(session => {
      const hour = new Date(session[0].timestamp).getHours();
      heatmapByHour[hour]++;
    });

    // 요일별 통계
    const weekdayNames = ['일', '월', '화', '수', '목', '금', '토'];
    const weekdayStats = weekdayNames.map((day, index) => {
      const daySessions = Array.from(sessionMap.values()).filter(session => {
        return new Date(session[0].timestamp).getDay() === index;
      });
      return {
        day,
        count: daySessions.length
      };
    });

    const stats: ChatStats = {
      dailyChats: dailySessions.length,
      weeklyChats: weeklySessions.length,
      monthlyChats: monthlySessions.length,
      periodStats: {
        daily: dailyStats,
        weekly: weeklyStats,
        monthly: monthlyStats
      },
      categoryStats,
      userStats,
      timeDistribution,
      heatmapByHour,
      weekdayStats
    };

    return NextResponse.json({
      success: true,
      data: stats
    });

  } catch (error) {
    console.error('Error fetching chat statistics:', error);
    return NextResponse.json(
      {
        success: false,
        error: { message: error instanceof Error ? error.message : String(error) }
      },
      { status: 500 }
    );
  }
}