import { NextRequest, NextResponse } from 'next/server';

export async function GET(request: NextRequest) {
  const { searchParams } = new URL(request.url);
  const modelName = searchParams.get('model');

  if (!modelName) {
    return NextResponse.json({ error: 'Model name is required' }, { status: 400 });
  }

  // VLLM 모델 감지 (슬래시가 포함된 모델명)
  if (modelName.includes('/')) {
    console.log(`VLLM model detected: ${modelName}, returning basic metadata`);
    return NextResponse.json({
      modelName,
      baseModel: '',
      parameters: 'Unknown',
      contextLength: 2048,
      family: 'Unknown',
      architecture: '',
      quantization: '',
      size: 0,
      modifiedAt: '',
      digest: '',
      isVllmModel: true
    });
  }

  try {
    // Ollama API에서 모델 정보 가져오기 (eval-start와 동일한 방식 사용)
    const ollamaUrl = process.env.NEXT_PUBLIC_OLLAMA_PROXY_SERVER || process.env.OLLAMA_URL || "https://api.hamonize.com/ollama";
    console.log(`Fetching metadata for model: ${modelName} from ${ollamaUrl}`);
    
    const response = await fetch(`${ollamaUrl}/api/show`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ name: modelName }),
    });

    if (!response.ok) {
      const errorText = await response.text();
      console.error(`Ollama API error: ${response.status} - ${errorText}`);
      throw new Error(`Ollama API error: ${response.status} - ${errorText}`);
    }

    const data = await response.json();
    
    // 모델 정보에서 메타데이터 추출
    const modelInfo = data.details || {};
    const parameters = modelInfo.parameter_size || modelInfo.parameters || '';
    const contextLength = modelInfo.context_length || 0;
    
    // 모델명에서 베이스 모델 추출 (사용자 정의 모델의 경우)
    let baseModel = '';
    if (data.modelfile) {
      const fromMatch = data.modelfile.match(/FROM\s+([^\s\n]+)/i);
      if (fromMatch) {
        baseModel = fromMatch[1];
        // 경로에서 모델명만 추출
        const modelNameMatch = baseModel.match(/([^\/]+)(?::[^\/]*)?$/);
        if (modelNameMatch) {
          baseModel = modelNameMatch[1];
        }
      }
    }

    return NextResponse.json({
      modelName,
      baseModel,
      parameters,
      contextLength,
      family: modelInfo.family || '',
      architecture: modelInfo.architecture || '',
      quantization: modelInfo.quantization_level || '',
      size: data.size || 0,
      modifiedAt: data.modified_at || '',
      digest: data.digest || '',
    });

  } catch (error) {
    console.error('Error fetching Ollama model metadata:', error);
    return NextResponse.json(
      { error: 'Failed to fetch model metadata from Ollama' },
      { status: 500 }
    );
  }
} 