import { NextRequest, NextResponse } from 'next/server';
import ExcelJS from 'exceljs';

export async function POST(request: NextRequest) {
  try {
    const formData = await request.formData();
    const file = formData.get('file') as File;
    const sheetsData = JSON.parse(formData.get('sheets') as string);

    if (!file || !sheetsData) {
      return NextResponse.json(
        { success: false, error: 'File and sheets data required' },
        { status: 400 }
      );
    }

    // Excel 파일 분석
    const buffer = Buffer.from(await file.arrayBuffer());
    const workbook = new ExcelJS.Workbook();
    await workbook.xlsx.load(buffer);

    // AI 추천 로직
    const recommendations = await generateExcelRecommendations(sheetsData, workbook);

    // 🔍 최종 추천 결과 로그
    console.log('✅ 최종 AI 추천 결과:', {
      선택시트: recommendations.selectedSheet,
      추천열수: recommendations.columns?.length || 0,
      추천열목록: recommendations.columns,
      신뢰도: Math.round((recommendations.confidence || 0) * 100) + '%',
      AI제목: recommendations.aiGeneratedTitle,
      AI설명길이: recommendations.aiGeneratedDescription?.length || 0,
      데이터카테고리: recommendations.dataCategory,
      핵심인사이트수: recommendations.keyInsights?.length || 0
    });

    return NextResponse.json({
      success: true,
      data: recommendations
    });

  } catch (error) {
    console.error('Excel AI recommendation error:', error);
    return NextResponse.json(
      { 
        success: false, 
        error: error instanceof Error ? error.message : 'AI recommendation failed' 
      },
      { status: 500 }
    );
  }
}

async function generateExcelRecommendations(sheetsData: any[], workbook: ExcelJS.Workbook) {
  const recommendations: any = {
    selectedSheet: '',
    columns: [],
    structure: '',
    description: '',
    confidence: 0,
    aiGeneratedTitle: '',
    aiGeneratedDescription: '',
    dataCategory: '',
    keyInsights: []
  };

  // 1. 가장 적합한 시트 선택
  let bestSheet = null;
  let bestScore = 0;

  for (const sheet of sheetsData) {
    let score = 0;
    
    // 데이터 양 점수 (더 많은 데이터가 있는 시트 선호)
    score += Math.min(sheet.sampleData?.length || 0, 10) * 10;
    
    // 헤더 품질 점수 (빈 헤더가 적은 시트 선호)
    const nonEmptyHeaders = sheet.headers?.filter((h: string) => h && h.trim()) || [];
    score += nonEmptyHeaders.length * 5;
    
    // 첫 번째 시트 보너스
    if (sheetsData.indexOf(sheet) === 0) {
      score += 20;
    }

    if (score > bestScore) {
      bestScore = score;
      bestSheet = sheet;
    }
  }

  if (bestSheet) {
    recommendations.selectedSheet = bestSheet.name;
    recommendations.confidence = Math.min(bestScore / 100, 1);

    // 2. AI 기반 데이터 분석
    const headers = bestSheet.headers || [];
    const sampleData = bestSheet.sampleData || [];
    
    // 실제 전체 행/열 수 정보 추가 (analyze API에서 전달된 정보 활용)
    const actualRowCount = bestSheet.rowCount ? bestSheet.rowCount - 1 : (sampleData.length - 1); // 헤더 제외
    const actualColumnCount = bestSheet.columnCount || headers.length;
    
    console.log('🔍 실제 데이터 규모:', {
      시트명: bestSheet.name,
      전체행수: bestSheet.rowCount || '정보없음',
      데이터행수: actualRowCount,
      전체열수: actualColumnCount,
      샘플행수: sampleData.length
    });
    
    // AI 분석 수행 (실제 행 수 정보 포함)
    const aiAnalysis = await performAIAnalysis(bestSheet.name, headers, sampleData, actualRowCount, actualColumnCount);
    recommendations.aiGeneratedTitle = aiAnalysis.title;
    recommendations.aiGeneratedDescription = aiAnalysis.description;
    recommendations.dataCategory = aiAnalysis.category;
    recommendations.keyInsights = aiAnalysis.insights;
    
    // 🔍 서버 콘솔에서 AI 분석 결과 확인
    console.log('🤖 AI 분석 완료 - 시트:', bestSheet.name);
    console.log('📊 생성된 AI 데이터:', {
      제목: aiAnalysis.title,
      설명: aiAnalysis.description,
      카테고리: aiAnalysis.category,
      인사이트개수: aiAnalysis.insights.length,
      인사이트목록: aiAnalysis.insights,
      AI추천필드수: aiAnalysis.recommendedFields?.length || 0,
      AI추천필드: aiAnalysis.recommendedFields
    });

    // 3. AI가 추천한 필드만 사용 (프로그램 로직 완전 제거)
    if (aiAnalysis.recommendedFields && aiAnalysis.recommendedFields.length > 0) {
      // AI 추천 필드 중 실제 존재하는 필드만 선택
      recommendations.columns = aiAnalysis.recommendedFields.filter(fieldName => 
        headers.includes(fieldName)
      );
      
      console.log('✅ AI 추천 필드 적용:', {
        AI추천필드: aiAnalysis.recommendedFields,
        실제존재필드: recommendations.columns,
        전체필드: headers,
        AI추천중누락: aiAnalysis.recommendedFields.filter(f => !headers.includes(f))
      });
      
      // AI 추천 필드가 하나도 매치되지 않는 경우 대체 로직
      if (recommendations.columns.length === 0) {
        console.warn('⚠️ AI 추천 필드가 실제 헤더와 매치되지 않음, 상위 5개 필드로 대체');
        recommendations.columns = headers.slice(0, Math.min(5, headers.length));
      }
    } else {
      console.warn('⚠️ AI에서 추천 필드를 반환하지 않음, 상위 5개 필드로 대체');
      recommendations.columns = headers.slice(0, Math.min(5, headers.length));
    }

    // 4. 구조 추천 설명 생성 (실제 행 수 사용)
    const totalRows = actualRowCount; // 실제 데이터 행 수 (헤더 제외)
    const totalCols = actualColumnCount; // 실제 열 수
    
    recommendations.structure = `테이블 구조 (${totalRows}행 × ${totalCols}열)`;
    
    // AI 기반 설명 생성 (선택된 필드 정보 포함)
    recommendations.description = 
      `AI가 시트 '${bestSheet.name}'에서 ${recommendations.columns.length}개의 핵심 필드를 선별했습니다. ` +
      `${totalRows}행 × ${totalCols}열 규모의 ${aiAnalysis.category} 데이터에서 ` +
      `비즈니스 가치가 높고 분석에 적합한 필드들을 중심으로 구성되었습니다.`;

    // 5. 추가 처리 옵션 추천
    recommendations.processing = {
      headerNormalization: true,
      preserveFormatting: true,
      suggestedFilters: []
    };

    // AI 기반 필터 추천 (선택된 필드 기준)
    const selectedHeaders = recommendations.columns;
    const dateColumns = selectedHeaders.filter(header => 
      header.toLowerCase().includes('date') || header.toLowerCase().includes('날짜') || header.toLowerCase().includes('time')
    );
    const numericColumns = selectedHeaders.filter(header => {
      const headerLower = header.toLowerCase();
      return headerLower.includes('price') || headerLower.includes('amount') || 
             headerLower.includes('cost') || headerLower.includes('금액') || 
             headerLower.includes('가격') || headerLower.includes('수량');
    });

    if (dateColumns.length > 0) {
      recommendations.processing.suggestedFilters.push({
        type: 'date_range',
        column: dateColumns[0],
        description: `${dateColumns[0]} 기준으로 최신 데이터 우선 필터링 추천`
      });
    }

    if (numericColumns.length > 0) {
      recommendations.processing.suggestedFilters.push({
        type: 'numeric_range',
        column: numericColumns[0],
        description: `${numericColumns[0]} 유효값 범위 필터링으로 데이터 품질 향상`
      });
    }
  }

  return recommendations;
}

// AI 분석 함수 (실제 AI 프로바이더 호출)
async function performAIAnalysis(sheetName: string, headers: string[], sampleData: any[][], actualRowCount: number, actualColumnCount: number) {
  try {
    // 데이터 패턴 분석 (실제 행 수 정보 추가)
    const dataAnalysis = analyzeDataPatterns(headers, sampleData, actualRowCount);
    
    // 🔍 데이터 패턴 분석 결과 로그
    console.log('🔬 데이터 패턴 분석:', {
      개인정보포함: dataAnalysis.hasPersonalInfo,
      금융데이터포함: dataAnalysis.hasFinancialData,
      날짜정보포함: dataAnalysis.hasDateInfo,
      위치정보포함: dataAnalysis.hasLocation,
      식별자포함: dataAnalysis.hasIdentifiers,
      레코드수: dataAnalysis.recordCount,
      데이터타입분석: Object.keys(dataAnalysis.dataTypes).length + '개 필드'
    });

    // 실제 AI 호출을 위한 프롬프트 구성 (실제 행 수 정보 포함)
    const aiPrompt = createAIAnalysisPrompt(sheetName, headers, sampleData, dataAnalysis, actualRowCount, actualColumnCount);
    
    // AI 프로바이더 호출
    const aiResponse = await callAIProvider(aiPrompt);
    
    if (aiResponse.success) {
      console.log('🤖 AI 응답 성공:', aiResponse.data);
      return aiResponse.data;
    } else {
      console.warn('⚠️ AI 호출 실패, 규칙 기반 분석으로 대체');
      // AI 호출 실패 시 기존 규칙 기반 분석 사용
      return {
        title: generateAITitle(sheetName, headers, dataAnalysis),
        description: generateAIDescription(headers, sampleData, dataAnalysis, actualRowCount, actualColumnCount),
        category: categorizeData(headers, dataAnalysis),
        insights: generateKeyInsights(headers, sampleData, dataAnalysis, actualRowCount)
      };
    }
  } catch (error) {
    console.error('AI 분석 실패:', error);
    // 오류 시 규칙 기반 분석으로 대체
    return {
      title: generateAITitle(sheetName, headers, dataAnalysis),
      description: generateAIDescription(headers, sampleData, dataAnalysis, actualRowCount, actualColumnCount),
      category: categorizeData(headers, dataAnalysis),
      insights: generateKeyInsights(headers, sampleData, dataAnalysis, actualRowCount)
    };
  }
}

// AI 분석용 프롬프트 생성
function createAIAnalysisPrompt(sheetName: string, headers: string[], sampleData: any[][], patterns: any, actualRowCount?: number, actualColumnCount?: number) {
  const sampleRows = sampleData.slice(1, Math.min(6, sampleData.length)); // 최대 5개 샘플 행
  
  return `Excel 데이터 분석 전문가로서 다음 데이터를 분석하고 JSON 형태로 응답해주세요.

**데이터 정보:**
- 시트명: ${sheetName}
- 총 행수: ${actualRowCount !== undefined ? actualRowCount : patterns.recordCount}개 (헤더 제외)
- 총 열수: ${actualColumnCount !== undefined ? actualColumnCount : headers.length}개
- 전체 열 목록: ${headers.join(', ')}

**샘플 데이터:**
${sampleRows.map((row, idx) => 
  `행 ${idx + 1}: ${row.map((cell, cellIdx) => `${headers[cellIdx]}: ${cell}`).join(' | ')}`
).join('\n')}

**데이터 특성:**
- 개인정보 포함: ${patterns.hasPersonalInfo ? '예' : '아니오'}
- 금융 데이터 포함: ${patterns.hasFinancialData ? '예' : '아니오'}  
- 날짜 정보 포함: ${patterns.hasDateInfo ? '예' : '아니오'}
- 위치 정보 포함: ${patterns.hasLocation ? '예' : '아니오'}
- 식별자 포함: ${patterns.hasIdentifiers ? '예' : '아니오'}

**중요: 필드 선별 분석**
위의 전체 열 목록 중에서 데이터 분석과 검색에 가장 중요한 필드 5-8개만을 엄선해주세요.

선별 기준:
1. **비즈니스 핵심 가치**: 의사결정, 분석, 검색에 필수적인 필드
2. **데이터 품질**: 샘플 데이터를 보고 빈 값이나 무의미한 값이 적은 필드
3. **정보 밀도**: 실제로 중요한 정보를 담고 있는 필드
4. **중복 제거**: 비슷한 정보를 담은 필드가 있다면 더 중요한 하나만 선택
5. **메타데이터 배제**: 등록일시, 수정일시, 시스템 관련 필드는 제외

**제외해야 할 필드 예시:**
- 단순 순번, ID (분석에 도움이 안 되는 경우)
- created_at, updated_at 같은 메타데이터  
- 빈 값이 많거나 의미 없는 값들이 많은 필드
- 중복되는 정보 (예: 이름과 full_name이 둘 다 있으면 하나만)

샘플 데이터를 자세히 분석해서 실제로 의미 있는 데이터가 들어있는 필드들만 선별하세요.

다음 형식의 JSON으로 응답해주세요:
{
  "title": "데이터의 성격을 나타내는 한국어 제목 (예: 고객 거래 데이터, 직원 정보 데이터 등)",
  "description": "데이터의 특성, 규모, 활용 가능성을 포함한 3-4문장 설명",
  "category": "데이터 카테고리 (예: CRM/고객관리, 재무/회계, 인사/인력관리, 운영/재고관리, 마케팅/판매 등)",
  "insights": ["이 데이터에서 발견할 수 있는 핵심 인사이트 1", "비즈니스에 활용 가능한 인사이트 2", "데이터 품질이나 특성에 대한 인사이트 3"],
  "recommendedFields": ["선별된_필드명1", "선별된_필드명2", "선별된_필드명3", "선별된_필드명4", "선별된_필드명5"]
}

recommendedFields는 반드시 위에 제시된 전체 열 목록에서만 선택하고, 정확한 필드명을 사용해주세요.`;
}

// AI 프로바이더 호출 함수 (새로운 범용 AI 엔드포인트 사용)
async function callAIProvider(prompt: string) {
  try {
    // 새로 추가한 범용 AI 호출 엔드포인트 사용
    const { getApiServerUrl } = await import('@/config/serverConfig');
    const apiServerUrl = getApiServerUrl();
    
    console.log('🔄 범용 AI 엔드포인트 호출 시작:', { 
      apiServerUrl, 
      fullUrl: `${apiServerUrl}/callai`,
      promptLength: prompt.length 
    });
    
    const response = await fetch(`${apiServerUrl}/callai`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        messages: [{ role: 'user', content: prompt }],
        options: {
          // Excel 분석은 JSON 형식 요청이므로 airun-chat:latest 모델 사용
          provider: 'ollama',
          model: 'airun-chat:latest',
          temperature: 0.1, // 일관성 있는 분석을 위해 낮은 온도
          max_tokens: 4000 // JSON 응답을 위한 충분한 토큰
        }
      })
    });

    if (!response.ok) {
      throw new Error(`AI API 호출 실패: ${response.status} ${response.statusText}`);
    }

    const result = await response.json();
    console.log('🤖 /callai API 응답:', result);
    
    // /callai 엔드포인트 응답 구조에 맞춰 파싱
    if (result?.success && result?.data?.response) {
      const aiResponse = result.data.response.content || result.data.response;
      
      console.log('📝 AI 응답 내용:', aiResponse);
      
      // 먼저 직접 JSON 파싱 시도
      try {
        const parsedData = JSON.parse(aiResponse);
        console.log('✅ 직접 JSON 파싱 성공:', parsedData);
        return {
          success: true,
          data: parsedData
        };
      } catch (parseError) {
        console.log('⚠️ 직접 JSON 파싱 실패, 정규식으로 시도:', parseError.message);
        
        // JSON 부분만 추출 시도 (```json ... ``` 형태 처리)
        const jsonMatch = aiResponse.match(/```json\s*([\s\S]*?)\s*```/) || aiResponse.match(/\{[\s\S]*\}/);
        
        if (jsonMatch) {
          const jsonStr = jsonMatch[1] || jsonMatch[0];
          try {
            const parsedData = JSON.parse(jsonStr);
            console.log('✅ 정규식 JSON 파싱 성공:', parsedData);
            return {
              success: true,
              data: parsedData
            };
          } catch (regexParseError) {
            console.warn('⚠️ 정규식 JSON 파싱도 실패:', regexParseError.message);
          }
        } else {
          console.warn('⚠️ JSON 형태를 찾을 수 없음, 전체 응답:', aiResponse);
        }
      }
    }
    
    throw new Error('AI 응답에서 유효한 JSON을 찾을 수 없음');
    
  } catch (error) {
    console.error('❌ 범용 AI 호출 오류:', error);
    return {
      success: false,
      error: error instanceof Error ? error.message : 'Unknown error'
    };
  }
}

// 데이터 패턴 분석
function analyzeDataPatterns(headers: string[], sampleData: any[][], actualRowCount?: number) {
  const patterns = {
    hasPersonalInfo: false,
    hasFinancialData: false,
    hasDateInfo: false,
    hasLocation: false,
    hasIdentifiers: false,
    dataTypes: {},
    recordCount: actualRowCount !== undefined ? actualRowCount : (sampleData.length - 1) // 실제 행 수 우선 사용
  };

  headers.forEach((header, index) => {
    const headerLower = header.toLowerCase();
    const samples = sampleData.slice(1).map(row => row[index]).filter(val => val);
    
    // 개인정보 감지
    if (headerLower.includes('name') || headerLower.includes('이름') || 
        headerLower.includes('email') || headerLower.includes('phone') || 
        headerLower.includes('전화') || headerLower.includes('연락')) {
      patterns.hasPersonalInfo = true;
    }
    
    // 금융 데이터 감지
    if (headerLower.includes('price') || headerLower.includes('amount') || 
        headerLower.includes('cost') || headerLower.includes('금액') || 
        headerLower.includes('가격') || headerLower.includes('수량')) {
      patterns.hasFinancialData = true;
    }
    
    // 날짜 정보 감지
    if (headerLower.includes('date') || headerLower.includes('time') ||
        headerLower.includes('날짜') || headerLower.includes('시간')) {
      patterns.hasDateInfo = true;
    }
    
    // 위치 정보 감지
    if (headerLower.includes('address') || headerLower.includes('location') ||
        headerLower.includes('city') || headerLower.includes('region') ||
        headerLower.includes('주소') || headerLower.includes('지역')) {
      patterns.hasLocation = true;
    }
    
    // 식별자 감지
    if (headerLower.includes('id') || headerLower.includes('번호') ||
        headerLower.includes('code') || headerLower.includes('코드')) {
      patterns.hasIdentifiers = true;
    }
    
    // 데이터 타입 분석
    if (samples.length > 0) {
      const sample = samples[0];
      if (!isNaN(Number(sample)) && sample !== '') {
        patterns.dataTypes[header] = 'numeric';
      } else if (String(sample).includes('@')) {
        patterns.dataTypes[header] = 'email';
      } else if (String(sample).includes('-') && String(sample).length >= 8) {
        patterns.dataTypes[header] = 'date';
      } else {
        patterns.dataTypes[header] = 'text';
      }
    }
  });

  return patterns;
}

// AI 제목 생성
function generateAITitle(sheetName: string, headers: string[], patterns: any) {
  // 데이터 특성에 따른 제목 생성
  if (patterns.hasPersonalInfo && patterns.hasFinancialData) {
    if (patterns.hasDateInfo) {
      return '고객 거래 데이터';
    } else {
      return '고객 정보 데이터베이스';
    }
  } else if (patterns.hasPersonalInfo) {
    return '개인정보 데이터';
  } else if (patterns.hasFinancialData) {
    if (patterns.hasDateInfo) {
      return '금융 거래 기록';
    } else {
      return '재무 데이터';
    }
  } else if (patterns.hasLocation) {
    return '지역별 데이터';
  } else if (headers.some(h => h.toLowerCase().includes('product') || h.includes('상품'))) {
    return '상품 정보 데이터';
  } else if (headers.some(h => h.toLowerCase().includes('employee') || h.includes('직원') || h.includes('사원'))) {
    return '직원 데이터';
  } else if (headers.some(h => h.toLowerCase().includes('inventory') || h.includes('재고'))) {
    return '재고 관리 데이터';
  } else {
    // 기본적으로 시트명 기반으로 생성
    return sheetName || '데이터 분석 자료';
  }
}

// AI 설명 생성
function generateAIDescription(headers: string[], sampleData: any[][], patterns: any, actualRowCount?: number, actualColumnCount?: number) {
  const descriptions = [];
  
  // 데이터 규모 설명 (실제 행/열 수 사용)
  const recordCount = actualRowCount !== undefined ? actualRowCount : patterns.recordCount;
  const columnCount = actualColumnCount !== undefined ? actualColumnCount : headers.length;
  descriptions.push(`총 ${recordCount}건의 레코드로 구성된 ${columnCount}개 필드의 구조화된 데이터입니다.`);
  
  // 데이터 특성 설명
  const features = [];
  if (patterns.hasPersonalInfo) features.push('개인정보');
  if (patterns.hasFinancialData) features.push('금융/거래 정보');
  if (patterns.hasDateInfo) features.push('시간 정보');
  if (patterns.hasLocation) features.push('위치 정보');
  if (patterns.hasIdentifiers) features.push('식별자');
  
  if (features.length > 0) {
    descriptions.push(`${features.join(', ')}를 포함하여 종합적인 분석이 가능합니다.`);
  }
  
  // 활용 가능성 설명
  if (patterns.hasPersonalInfo && patterns.hasFinancialData && patterns.hasDateInfo) {
    descriptions.push('고객 행동 분석, 매출 트렌드 분석, 개인화 서비스 개발 등에 활용할 수 있습니다.');
  } else if (patterns.hasFinancialData && patterns.hasDateInfo) {
    descriptions.push('재무 분석, 수익성 평가, 시계열 분석 등의 비즈니스 인사이트 도출에 적합합니다.');
  } else if (patterns.hasPersonalInfo) {
    descriptions.push('고객 세분화, 개인화 서비스, 마케팅 타겟팅 등에 활용 가능합니다.');
  } else {
    descriptions.push('데이터 기반 의사결정 및 분석 업무에 활용할 수 있습니다.');
  }
  
  return descriptions.join(' ');
}

// 데이터 카테고리 분류
function categorizeData(headers: string[], patterns: any) {
  if (patterns.hasPersonalInfo && patterns.hasFinancialData) {
    return 'CRM/고객관리';
  } else if (patterns.hasFinancialData) {
    return '재무/회계';
  } else if (patterns.hasPersonalInfo) {
    return '인사/고객정보';
  } else if (patterns.hasLocation) {
    return '지리/위치 데이터';
  } else if (headers.some(h => h.toLowerCase().includes('product') || h.includes('상품'))) {
    return '상품/서비스';
  } else if (headers.some(h => h.toLowerCase().includes('inventory') || h.includes('재고'))) {
    return '운영/재고관리';
  } else {
    return '일반 데이터';
  }
}

// 핵심 인사이트 생성
function generateKeyInsights(headers: string[], sampleData: any[][], patterns: any, actualRowCount?: number) {
  const insights = [];
  
  // 데이터 품질 인사이트
  const completenessScore = calculateDataCompleteness(sampleData);
  if (completenessScore > 90) {
    insights.push('데이터 완성도가 높아 신뢰할 수 있는 분석 결과를 기대할 수 있습니다');
  } else if (completenessScore < 70) {
    insights.push('일부 필드에서 누락된 데이터가 있어 전처리 과정이 필요할 수 있습니다');
  }
  
  // 분석 가능성 인사이트
  if (patterns.hasDateInfo && patterns.hasFinancialData) {
    insights.push('시계열 분석을 통한 트렌드 파악이 가능합니다');
  }
  
  const recordCount = actualRowCount !== undefined ? actualRowCount : patterns.recordCount;
  if (patterns.hasPersonalInfo && recordCount > 100) {
    insights.push('고객 세분화 및 행동 패턴 분석에 충분한 데이터 볼륨을 보유하고 있습니다');
  }
  
  // 데이터 타입별 인사이트
  const numericFields = Object.values(patterns.dataTypes).filter(type => type === 'numeric').length;
  if (numericFields > 2) {
    insights.push('다양한 수치 데이터를 활용한 통계 분석 및 예측 모델링이 가능합니다');
  }
  
  return insights;
}

// 데이터 완성도 계산
function calculateDataCompleteness(sampleData: any[][]) {
  if (sampleData.length <= 1) return 0;
  
  let totalCells = 0;
  let filledCells = 0;
  
  for (let i = 1; i < sampleData.length; i++) {
    for (let j = 0; j < sampleData[i].length; j++) {
      totalCells++;
      if (sampleData[i][j] !== null && sampleData[i][j] !== undefined && sampleData[i][j] !== '') {
        filledCells++;
      }
    }
  }
  
  return totalCells > 0 ? (filledCells / totalCells) * 100 : 0;
}