import { NextRequest, NextResponse } from 'next/server';
import { withAnyAuth } from '../../middlewares';
import { getApiServerUrl } from '@/config/serverConfig';

const API_SERVER = getApiServerUrl();

// GET: 단가표 목록 조회
export async function GET(request: NextRequest) {
  return withAnyAuth(request, async (req, userData, authType) => {
    try {
      const { searchParams } = new URL(req.url);
      const params = new URLSearchParams();

      searchParams.forEach((value, key) => {
        params.append(key, value);
      });

      if (userData.username) {
        params.set('username', userData.username);
      }

      const apiUrl = `${API_SERVER}/quotations/price-catalog?${params.toString()}`;

      const headers: Record<string, string> = {
        'Content-Type': 'application/json',
      };

      const apiKey = req.headers.get('x-api-key');
      if (apiKey) {
        headers['X-API-Key'] = apiKey;
      }

      const response = await fetch(apiUrl, {
        method: 'GET',
        headers,
        cache: 'no-store'
      });

      if (!response.ok) {
        const errorText = await response.text();
        console.error('단가표 조회 실패:', response.status, errorText);
        return NextResponse.json(
          { success: false, error: { message: '단가표를 가져오는데 실패했습니다.' } },
          { status: response.status }
        );
      }

      const data = await response.json();
      return NextResponse.json(data);
    } catch (error) {
      console.error('단가표 조회 오류:', error);
      return NextResponse.json(
        { success: false, error: { message: '단가표 조회 중 오류가 발생했습니다.' } },
        { status: 500 }
      );
    }
  });
}

// POST: 단가표 항목 추가
export async function POST(request: NextRequest) {
  return withAnyAuth(request, async (req, userData, authType) => {
    try {
      const body = await req.json();

      if (userData.username) {
        body.username = userData.username;
      }

      const apiUrl = `${API_SERVER}/quotations/price-catalog`;

      const headers: Record<string, string> = {
        'Content-Type': 'application/json',
      };

      const apiKey = req.headers.get('x-api-key');
      if (apiKey) {
        headers['X-API-Key'] = apiKey;
      }

      const response = await fetch(apiUrl, {
        method: 'POST',
        headers,
        body: JSON.stringify(body)
      });

      if (!response.ok) {
        const errorText = await response.text();
        console.error('단가표 항목 추가 실패:', response.status, errorText);
        return NextResponse.json(
          { success: false, error: { message: '단가표 항목 추가에 실패했습니다.' } },
          { status: response.status }
        );
      }

      const data = await response.json();
      return NextResponse.json(data);
    } catch (error) {
      console.error('단가표 항목 추가 오류:', error);
      return NextResponse.json(
        { success: false, error: { message: '단가표 항목 추가 중 오류가 발생했습니다.' } },
        { status: 500 }
      );
    }
  });
}
