import { NextRequest, NextResponse } from 'next/server';
import { withAnyAuth } from '../middlewares';
import { getApiServerUrl } from '@/config/serverConfig';

const API_SERVER = getApiServerUrl();

// GET: 견적서 목록 조회
export async function GET(request: NextRequest) {
  return withAnyAuth(request, async (req, userData, authType) => {
    try {
      const { searchParams } = new URL(req.url);
      const params = new URLSearchParams();

      // 쿼리 파라미터 전달
      searchParams.forEach((value, key) => {
        params.append(key, value);
      });

      // username 추가
      if (userData.username) {
        params.set('username', userData.username);
      }

      const apiUrl = `${API_SERVER}/quotations?${params.toString()}`;

      const headers: Record<string, string> = {
        'Content-Type': 'application/json',
      };

      // 인증 정보 전달
      const apiKey = req.headers.get('x-api-key');
      if (apiKey) {
        headers['X-API-Key'] = apiKey;
      }

      const response = await fetch(apiUrl, {
        method: 'GET',
        headers,
        cache: 'no-store'
      });

      if (!response.ok) {
        const errorText = await response.text();
        console.error('견적서 목록 조회 실패:', response.status, errorText);
        return NextResponse.json(
          { success: false, error: { message: '견적서 목록을 가져오는데 실패했습니다.' } },
          { status: response.status }
        );
      }

      const data = await response.json();
      return NextResponse.json(data);
    } catch (error) {
      console.error('견적서 목록 조회 오류:', error);
      return NextResponse.json(
        { success: false, error: { message: '견적서 목록 조회 중 오류가 발생했습니다.' } },
        { status: 500 }
      );
    }
  });
}

// POST: 견적서 생성
export async function POST(request: NextRequest) {
  return withAnyAuth(request, async (req, userData, authType) => {
    try {
      const body = await req.json();

      // username 추가
      if (userData.username) {
        body.username = userData.username;
      }

      const apiUrl = `${API_SERVER}/quotations`;

      const headers: Record<string, string> = {
        'Content-Type': 'application/json',
      };

      const apiKey = req.headers.get('x-api-key');
      if (apiKey) {
        headers['X-API-Key'] = apiKey;
      }

      const response = await fetch(apiUrl, {
        method: 'POST',
        headers,
        body: JSON.stringify(body)
      });

      if (!response.ok) {
        const errorText = await response.text();
        console.error('견적서 생성 실패:', response.status, errorText);
        return NextResponse.json(
          { success: false, error: { message: '견적서 생성에 실패했습니다.' } },
          { status: response.status }
        );
      }

      const data = await response.json();
      return NextResponse.json(data);
    } catch (error) {
      console.error('견적서 생성 오류:', error);
      return NextResponse.json(
        { success: false, error: { message: '견적서 생성 중 오류가 발생했습니다.' } },
        { status: 500 }
      );
    }
  });
}
