import { NextRequest, NextResponse } from 'next/server';
import { withAnyAuth } from '../../middlewares';
import { writeFile, mkdir, unlink } from 'fs/promises';
import { existsSync } from 'fs';
import path from 'path';

// POST: 회사 로고 업로드
export async function POST(request: NextRequest) {
  return withAnyAuth(request, async (req, userData) => {
    try {
      const formData = await req.formData();
      const file = formData.get('logo') as File | null;

      if (!file) {
        return NextResponse.json(
          { success: false, error: { message: '파일이 제공되지 않았습니다.' } },
          { status: 400 }
        );
      }

      // 파일 타입 검증
      if (!file.type.startsWith('image/')) {
        return NextResponse.json(
          { success: false, error: { message: '이미지 파일만 업로드 가능합니다.' } },
          { status: 400 }
        );
      }

      // 파일 크기 검증 (2MB)
      if (file.size > 2 * 1024 * 1024) {
        return NextResponse.json(
          { success: false, error: { message: '파일 크기는 2MB 이하여야 합니다.' } },
          { status: 400 }
        );
      }

      // 사용자별 로고 디렉토리 생성
      const username = userData.username || 'default';
      const uploadDir = path.join(process.cwd(), 'public', 'uploads', 'logos', username);

      if (!existsSync(uploadDir)) {
        await mkdir(uploadDir, { recursive: true });
      }

      // 파일 확장자 추출
      const ext = path.extname(file.name) || '.png';
      const filename = `company-logo${ext}`;
      const filepath = path.join(uploadDir, filename);

      // 기존 로고 파일 삭제 (다른 확장자 포함)
      const extensions = ['.png', '.jpg', '.jpeg', '.gif', '.webp'];
      for (const e of extensions) {
        const oldFile = path.join(uploadDir, `company-logo${e}`);
        if (existsSync(oldFile)) {
          try {
            await unlink(oldFile);
          } catch (err) {
            console.warn('기존 로고 삭제 실패:', err);
          }
        }
      }

      // 파일 저장
      const bytes = await file.arrayBuffer();
      const buffer = Buffer.from(bytes);
      await writeFile(filepath, buffer);

      // 웹에서 접근 가능한 경로 반환
      const logoPath = `/uploads/logos/${username}/${filename}`;

      return NextResponse.json({
        success: true,
        data: {
          logoPath,
          filename,
          size: file.size,
          type: file.type
        }
      });
    } catch (error) {
      console.error('로고 업로드 오류:', error);
      return NextResponse.json(
        { success: false, error: { message: '로고 업로드 중 오류가 발생했습니다.' } },
        { status: 500 }
      );
    }
  });
}

// DELETE: 회사 로고 삭제
export async function DELETE(request: NextRequest) {
  return withAnyAuth(request, async (req, userData) => {
    try {
      const username = userData.username || 'default';
      const uploadDir = path.join(process.cwd(), 'public', 'uploads', 'logos', username);

      // 모든 로고 파일 삭제
      const extensions = ['.png', '.jpg', '.jpeg', '.gif', '.webp'];
      let deleted = false;

      for (const ext of extensions) {
        const filepath = path.join(uploadDir, `company-logo${ext}`);
        if (existsSync(filepath)) {
          try {
            await unlink(filepath);
            deleted = true;
          } catch (err) {
            console.warn('로고 삭제 실패:', err);
          }
        }
      }

      return NextResponse.json({
        success: true,
        data: { deleted }
      });
    } catch (error) {
      console.error('로고 삭제 오류:', error);
      return NextResponse.json(
        { success: false, error: { message: '로고 삭제 중 오류가 발생했습니다.' } },
        { status: 500 }
      );
    }
  });
}
