import { NextRequest, NextResponse } from 'next/server';
import { withTokenAuth } from '../middlewares';

const API_SERVER = process.env.NEXT_PUBLIC_API_SERVER_URL || 'http://localhost:5500/api/v1';

// 설정 가져오기
export async function GET(request: NextRequest) {
  return withTokenAuth(request, async (req, user) => {
    try {
      // URL에서 설정 키 추출
      const { searchParams } = new URL(req.url);
      const key = searchParams.get('key');
      
      if (!key) {
        return NextResponse.json(
          {
            success: false,
            error: {
              message: '설정 키가 필요합니다.'
            }
          },
          { status: 400 }
        );
      }
      
      // API 서버에 설정 요청
      const response = await fetch(`${API_SERVER}/config/${key}`, {
        method: 'GET',
        headers: {
          'Authorization': `Bearer ${req.cookies.get('auth_token')?.value}`
        }
      });
      
      if (!response.ok) {
        let errorMessage = '설정을 가져오는데 실패했습니다.';
        try {
          const errorData = await response.json();
          if (errorData.error?.message) {
            errorMessage = errorData.error.message;
          }
        } catch (parseError) {
          console.error('응답 파싱 오류:', parseError);
        }
        
        return NextResponse.json(
          {
            success: false,
            error: {
              message: errorMessage
            }
          },
          { status: response.status }
        );
      }
      
      const data = await response.json();
      
      return NextResponse.json({
        success: true,
        data: {
          value: data.value
        }
      });
    } catch (error) {
      console.error('설정 가져오기 중 오류:', error);
      return NextResponse.json(
        {
          success: false,
          error: {
            message: '설정을 가져오는 중 오류가 발생했습니다.'
          }
        },
        { status: 500 }
      );
    }
  });
}

// 설정 저장하기
export async function POST(request: NextRequest) {
  return withTokenAuth(request, async (req, user) => {
    try {
      const body = await req.json();
      const { key, value } = body;
      
      if (!key) {
        return NextResponse.json(
          {
            success: false,
            error: {
              message: '설정 키가 필요합니다.'
            }
          },
          { status: 400 }
        );
      }
      
      // API 서버에 설정 저장 요청
      const response = await fetch(`${API_SERVER}/config`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${req.cookies.get('auth_token')?.value}`
        },
        body: JSON.stringify({ key, value })
      });
      
      if (!response.ok) {
        let errorMessage = '설정 저장에 실패했습니다.';
        try {
          const errorData = await response.json();
          if (errorData.error?.message) {
            errorMessage = errorData.error.message;
          }
        } catch (parseError) {
          console.error('응답 파싱 오류:', parseError);
        }
        
        return NextResponse.json(
          {
            success: false,
            error: {
              message: errorMessage
            }
          },
          { status: response.status }
        );
      }
      
      const data = await response.json();
      
      return NextResponse.json({
        success: true,
        data: {
          key: data.key,
          value: data.value
        }
      });
    } catch (error) {
      console.error('설정 저장 중 오류:', error);
      return NextResponse.json(
        {
          success: false,
          error: {
            message: '설정 저장 중 오류가 발생했습니다.'
          }
        },
        { status: 500 }
      );
    }
  });
} 