import { NextRequest, NextResponse } from 'next/server';
import { withAnyAuth } from '@/app/api/middlewares';
import { getNextjsServerUrl, getRagServerUrl } from '@/config/serverConfig';
import { fetch as undiciFetch, Agent, FormData as UndiciFormData } from 'undici';
import { createClient } from 'redis';

// 백그라운드 작업 타입
interface BackgroundJobType {
  id: string;
  type: 'file_upload' | 'ticket_sync';
  status: 'pending' | 'running' | 'completed' | 'failed';
  progress: number;
  message: string;
  filename: string;
  createdAt: string;
  updatedAt: string;
  userId: string;
  error?: string;
  metadata?: {
    [key: string]: unknown;
  };
}

// Redis 클라이언트 생성
const createRedisClient = async () => {
  // serverConfig에서 Redis URL 가져오기
  const { getRedisUrl } = await import('@/config/serverConfig');
  const redisUrl = getRedisUrl();
  
  const client = createClient({
    url: redisUrl
  });
  
  await client.connect();
  return client;
};

// Redis 키 패턴
const JOB_KEY_PREFIX = 'background_job:';
const JOB_INDEX_KEY = 'background_jobs:index';


// 파일 업로드 백그라운드 처리 함수
async function processFileUploadEventBased(jobId: string, fileData: any, userId: string, userApiKey: string | null, metadata?: any) {
  // 디버그 로그 파일에 기록
  const fs = require('fs');
  const logFile = '/tmp/bg_job_debug.log';
  fs.appendFileSync(logFile, `
[${new Date().toISOString()}] processFileUploadEventBased 시작 - jobId: ${jobId}, file: ${fileData.name}, lastModified: ${fileData.lastModified || metadata?.lastModified || 'N/A'}
`);
  try {
    fs.appendFileSync(logFile, `[${new Date().toISOString()}] try 블록 진입
`);
    
    // 먼저 작업 상태를 'running'으로 업데이트
    const redis = await createRedisClient();
    try {
      const jobKey = `${JOB_KEY_PREFIX}${jobId}`;
      await redis.hSet(jobKey, {
        status: 'running',
        progress: '10',
        message: 'RAG 서버로 파일 업로드 중...',
        updatedAt: new Date().toISOString()
      });
      
      // WebSocket 알림 발행
      await redis.publish('websocket:job-status', JSON.stringify({
        jobId: jobId,
        status: 'running',
        message: 'RAG 서버로 파일 업로드 중...',
        timestamp: new Date().toISOString()
      }));
    } finally {
      await redis.quit();
    }
    
    console.log(`[processFileUploadEventBased] 파일 업로드 시작: ${fileData.name}`);
    fs.appendFileSync(logFile, `[${new Date().toISOString()}] 파일 업로드 시작: ${fileData.name}
`);
    
    // Base64 데이터를 Buffer로 변환
    fs.appendFileSync(logFile, `[${new Date().toISOString()}] Buffer 변환 시작
`);
    const buffer = Buffer.from(fileData.content, 'base64');
    fs.appendFileSync(logFile, `[${new Date().toISOString()}] Buffer 변환 완료: ${buffer.length} bytes
`);
    
    // Blob 객체 생성 (Next.js 환경에서는 전역으로 사용 가능)
    fs.appendFileSync(logFile, `[${new Date().toISOString()}] Blob 생성 시작
`);
    const blob = new Blob([buffer], { type: fileData.type });
    fs.appendFileSync(logFile, `[${new Date().toISOString()}] Blob 생성 완료: ${blob.size} bytes
`);
    
    // FormData 생성 (native FormData 사용)
    fs.appendFileSync(logFile, `[${new Date().toISOString()}] FormData 생성 시작
`);
    const formData = new FormData();
    formData.append('file', blob, fileData.name);
    fs.appendFileSync(logFile, `[${new Date().toISOString()}] FormData append 완료
`);
    formData.append('user_id', userId);
    formData.append('job_id', jobId); // job_id 전달
    formData.append('overwrite', 'true');
    formData.append('force', 'true'); // 강제 덮어쓰기

    // 파일 수정일자 추가
    const modificationTime = fileData.lastModified || (metadata && metadata.lastModified) || null;
    if (modificationTime) {
      formData.append('modification_time', modificationTime);
      fs.appendFileSync(logFile, `[${new Date().toISOString()}] modification_time 추가: ${modificationTime}
`);
    } else {
      fs.appendFileSync(logFile, `[${new Date().toISOString()}] modification_time 없음
`);
    }

    // API 서버를 통해 RAG 서버에 업로드
    // serverConfig에서 API 서버 URL 가져오기
    const { getApiServerUrl } = await import('@/config/serverConfig');
    const apiServerUrl = getApiServerUrl();
    
    console.log(`[processFileUploadEventBased] API 서버를 통한 업로드: ${apiServerUrl}`);
    console.log(`[processFileUploadEventBased] 업로드 시작 - 파일: ${fileData.name}, 사용자: ${userId}`);
    
    fs.appendFileSync(logFile, `[${new Date().toISOString()}] API 서버를 통한 업로드 시작: ${apiServerUrl}
`);
    
    // 인증 헤더 설정 - API 키가 있으면 사용, 없으면 JWT 토큰은 사용하지 않음
    const headers: HeadersInit = {};
    
    if (userApiKey) {
      headers['X-API-Key'] = userApiKey;
      console.log(`[processFileUploadEventBased] API 키 사용: ${userApiKey.substring(0, 20)}...`);
      fs.appendFileSync(logFile, `[${new Date().toISOString()}] API 키 사용: ${userApiKey.substring(0, 20)}...\n`);
    } else {
      console.log('[processFileUploadEventBased] API 키가 없어 업로드 실패');
      fs.appendFileSync(logFile, `[${new Date().toISOString()}] API 키 없음, 업로드 불가\n`);
      throw new Error('API 키가 필요합니다.');
    }
    
    console.log('[processFileUploadEventBased] 사용할 인증 헤더:', headers);
    
    const response = await fetch(`${apiServerUrl}/rag/upload`, {
      method: 'POST',
      headers,
      body: formData
    });
    
    fs.appendFileSync(logFile, `[${new Date().toISOString()}] API 서버 응답 수신: ${response.status}
`);
    
    console.log(`[processFileUploadEventBased] RAG 서버 응답 상태: ${response.status}`);
    
    const result = await response.json();
    console.log('[processFileUploadEventBased] RAG 업로드 응답:', result);
    fs.appendFileSync(logFile, `[${new Date().toISOString()}] RAG 응답 파싱 완료: ${JSON.stringify(result)}
`);
    
    // RAG 서버 처리 완료 모니터링  
    fs.appendFileSync(logFile, `[${new Date().toISOString()}] 응답 상태 확인: success=${result.success}
`);
    if (result.success === true || (result.data && result.data.status)) {
      // 작업 상태를 '임베딩 처리 중'으로 업데이트
      const redis = await createRedisClient();
      try {
        const jobKey = `${JOB_KEY_PREFIX}${jobId}`;
        await redis.hSet(jobKey, {
          status: 'running',
          progress: '50',
          message: '임베딩 처리 중...',
          updatedAt: new Date().toISOString()
        });

        // WebSocket 알림 발행
        await redis.publish('websocket:job-status', JSON.stringify({
          jobId: jobId,
          status: 'running',
          progress: 50,
          message: '임베딩 처리 중...',
          timestamp: new Date().toISOString()
        }));
      } finally {
        await redis.quit();
      }

      // 최소 2초 대기 (사용자가 상태를 확인할 수 있도록)
      await new Promise(resolve => setTimeout(resolve, 2000));

      // RAG 서버 처리 완료 확인 (폴링) - API 키도 함께 전달
      monitorRAGCompletion(jobId, fileData.name, userId, userApiKey);
    }
    
  } catch (error) {
    console.error('[processFileUploadEventBased] 업로드 실패:', error);
    
    // 실패 시 상태 업데이트
    const redis = await createRedisClient();
    try {
      const jobKey = `${JOB_KEY_PREFIX}${jobId}`;
      await redis.hSet(jobKey, {
        status: 'failed',
        progress: '0',
        message: '파일 업로드 실패',
        error: error instanceof Error ? error.message : '알 수 없는 오류',
        updatedAt: new Date().toISOString()
      });
    } finally {
      await redis.quit();
    }
  }
}

// WebSocket 기반 RAG 처리 모니터링 함수 (실시간)
async function monitorRAGCompletion(jobId: string, filename: string, userId: string, userApiKey: string | null) {
  console.log(`[monitorRAGCompletion] WebSocket 기반 실시간 모니터링 시작: ${jobId}, 파일: ${filename}`);

  // 먼저 즉시 상태 확인 (이미 완료된 문서일 수 있음)
  const initialCheck = async () => {
    try {
      const { getApiServerUrl } = await import('@/config/serverConfig');
      const apiServerUrl = getApiServerUrl();

      const headers: HeadersInit = {
        'Content-Type': 'application/json'
      };

      if (userApiKey) {
        headers['X-API-Key'] = userApiKey;
      }

      const response = await fetch(`${apiServerUrl}/rag/embedding-status?fileName=${encodeURIComponent(filename)}&userId=${userId}`, {
        method: 'GET',
        headers: headers
      });

      if (response.ok) {
        const data = await response.json();
        console.log(`[monitorRAGCompletion] 초기 상태 확인:`, data);

        if (data.success && data.data && data.data.embedded) {
          console.log(`[monitorRAGCompletion] 이미 완료된 문서 감지: ${filename}`);

          // 즉시 완료 상태로 업데이트
          const redis = await createRedisClient();
          try {
            const jobKey = `${JOB_KEY_PREFIX}${jobId}`;
            await redis.hSet(jobKey, {
              status: 'completed',
              progress: '100',
              message: '파일 업로드 및 임베딩 완료',
              updatedAt: new Date().toISOString()
            });

            await redis.publish('websocket:job-status', JSON.stringify({
              jobId: jobId,
              status: 'completed',
              progress: 100,
              message: '파일 업로드 및 임베딩 완료',
              timestamp: new Date().toISOString()
            }));
          } finally {
            await redis.quit();
          }

          return true; // 이미 완료됨
        }
      }
    } catch (error) {
      console.error(`[monitorRAGCompletion] 초기 상태 확인 중 오류:`, error);
    }

    return false; // 아직 진행 중
  };

  // 초기 상태 확인 실행
  const isAlreadyCompleted = await initialCheck();
  if (isAlreadyCompleted) {
    console.log(`[monitorRAGCompletion] 이미 완료된 문서로 모니터링 종료: ${filename}`);
    return;
  }

  // 안전장치: 최대 대기 시간 설정 (10분)
  const maxWaitTime = 10 * 60 * 1000; // 10분
  const startTime = Date.now();

  // 백업 확인 시스템 (WebSocket이 실패할 경우 대비)
  const backupCheck = async () => {
    console.log(`[monitorRAGCompletion] 백업 상태 확인 실행: ${filename}`);
    
    try {
      // API 서버의 embedding-status 엔드포인트로 최종 확인
      const { getApiServerUrl } = await import('@/config/serverConfig');
      const apiServerUrl = getApiServerUrl();
      
      const headers: HeadersInit = {
        'Content-Type': 'application/json'
      };
      
      if (userApiKey) {
        headers['X-API-Key'] = userApiKey;
      }
      
      const response = await fetch(`${apiServerUrl}/rag/embedding-status?fileName=${encodeURIComponent(filename)}&userId=${userId}`, {
        method: 'GET',
        headers: headers
      });
      
      if (response.ok) {
        const data = await response.json();
        console.log(`[monitorRAGCompletion] 백업 확인 응답:`, data);
        
        if (data.success && data.data && data.data.embedded) {
          console.log(`[monitorRAGCompletion] 백업 확인으로 완료 감지: ${filename}`);
          
          // 수동으로 완료 상태 업데이트
          const redis = await createRedisClient();
          try {
            const jobKey = `${JOB_KEY_PREFIX}${jobId}`;
            await redis.hSet(jobKey, {
              status: 'completed',
              progress: '100',
              message: '파일 업로드 및 임베딩 완료 (백업 확인)',
              updatedAt: new Date().toISOString()
            });
            
            await redis.publish('websocket:job-status', JSON.stringify({
              jobId: jobId,
              status: 'completed',
              progress: 100,
              message: '파일 업로드 및 임베딩 완료',
              timestamp: new Date().toISOString()
            }));
          } finally {
            await redis.quit();
          }
          
          return true;
        }
      }
    } catch (error) {
      console.error(`[monitorRAGCompletion] 백업 확인 중 오류:`, error);
    }
    
    return false;
  };

  // 실시간 WebSocket 모니터링이 주요 방식이므로
  // 백업 확인은 30초 후, 2분 후, 5분 후에만 실행
  const backupCheckTimeouts = [30000, 120000, 300000]; // 30초, 2분, 5분
  
  backupCheckTimeouts.forEach((delay, index) => {
    setTimeout(async () => {
      const elapsed = Date.now() - startTime;
      if (elapsed < maxWaitTime) {
        console.log(`[monitorRAGCompletion] 백업 확인 ${index + 1}/3 실행`);
        await backupCheck();
      }
    }, delay);
  });

  // 최대 대기 시간 초과 시 타임아웃 처리
  setTimeout(async () => {
    console.warn(`[monitorRAGCompletion] 최대 대기 시간 초과: ${filename}`);
    
    const redis = await createRedisClient();
    try {
      const jobKey = `${JOB_KEY_PREFIX}${jobId}`;
      const jobData = await redis.hGetAll(jobKey);
      
      // 아직 완료되지 않은 경우에만 타임아웃 처리
      if (jobData.status !== 'completed') {
        await redis.hSet(jobKey, {
          status: 'timeout',
          progress: '50',
          message: '임베딩 처리 시간 초과',
          error: '임베딩 완료 확인 실패 (시간 초과)',
          updatedAt: new Date().toISOString()
        });
      }
    } catch (error) {
      console.error(`[monitorRAGCompletion] 타임아웃 처리 중 오류:`, error);
    } finally {
      await redis.quit();
    }
  }, maxWaitTime);
  
  console.log(`[monitorRAGCompletion] 실시간 모니터링 설정 완료: ${filename} (최대 ${maxWaitTime/1000}초 대기)`);
}

// 티켓 동기화 백그라운드 처리 함수
async function processTicketSyncEventBased(jobId: string, filename: string, userId: string, userApiKey: string | null) {
  console.log(`[processTicketSyncEventBased] 티켓 동기화 시작: ${jobId}, 파일: ${filename}`);
  
  try {
    // 작업 상태를 'running'으로 업데이트
    const redis = await createRedisClient();
    try {
      const jobKey = `${JOB_KEY_PREFIX}${jobId}`;
      await redis.hSet(jobKey, {
        status: 'running',
        progress: '10',
        message: '티켓 데이터를 마크다운으로 변환 중...',
        updatedAt: new Date().toISOString()
      });
      
      // WebSocket 알림 발행
      await redis.publish('websocket:job-status', JSON.stringify({
        jobId: jobId,
        status: 'running',
        message: '티켓 데이터를 마크다운으로 변환 중...',
        timestamp: new Date().toISOString()
      }));
    } finally {
      await redis.quit();
    }

    // 티켓 데이터를 마크다운으로 내보내기
    console.log(`[processTicketSyncEventBased] 티켓 데이터 내보내기 시작`);
    
    // Next.js API 직접 호출 (내부에서 실행되므로 localhost 사용)
    const exportResponse = await fetch(`http://localhost:5700/api/support/export-tickets-for-rag`, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json'
      }
    });
    
    if (!exportResponse.ok) {
      throw new Error(`티켓 내보내기 실패: ${exportResponse.status}`);
    }
    
    const markdownContent = await exportResponse.text();
    console.log(`[processTicketSyncEventBased] 마크다운 생성 완료: ${markdownContent.length} bytes`);
    
    // 진행률 업데이트
    const redis2 = await createRedisClient();
    try {
      const jobKey = `${JOB_KEY_PREFIX}${jobId}`;
      await redis2.hSet(jobKey, {
        status: 'running',
        progress: '50',
        message: 'RAG 서버에 업로드 중...',
        updatedAt: new Date().toISOString()
      });
    } finally {
      await redis2.quit();
    }

    // 마크다운 파일을 RAG 서버에 업로드
    const buffer = Buffer.from(markdownContent, 'utf-8');
    const blob = new Blob([buffer], { type: 'text/markdown' });
    
    const formData = new FormData();
    formData.append('file', blob, filename);
    formData.append('user_id', userId);
    formData.append('job_id', jobId);
    formData.append('overwrite', 'true');
    formData.append('force', 'true');

    // 인증 헤더 설정
    const headers: HeadersInit = {};
    if (userApiKey) {
      headers['X-API-Key'] = userApiKey;
      console.log(`[processTicketSyncEventBased] API 키 사용: ${userApiKey.substring(0, 20)}...`);
    } else {
      console.log('[processTicketSyncEventBased] API 키 없음');
      throw new Error('API 키가 필요합니다.');
    }

    console.log(`[processTicketSyncEventBased] RAG 서버 업로드 시작: ${filename}`);
    
    // API 서버 URL 가져오기
    const { getApiServerUrl } = await import('@/config/serverConfig');
    const apiServerUrl = getApiServerUrl();
    
    const uploadResponse = await fetch(`${apiServerUrl}/rag/upload`, {
      method: 'POST',
      headers,
      body: formData
    });
    
    console.log(`[processTicketSyncEventBased] RAG 서버 응답 상태: ${uploadResponse.status}`);
    const result = await uploadResponse.json();
    console.log('[processTicketSyncEventBased] RAG 업로드 응답:', result);
    
    if (result.success === true || (result.data && result.data.status)) {
      // 임베딩 진행률 업데이트
      const redis3 = await createRedisClient();
      try {
        const jobKey = `${JOB_KEY_PREFIX}${jobId}`;
        await redis3.hSet(jobKey, {
          status: 'running',
          progress: '75',
          message: '임베딩 처리 중...',
          updatedAt: new Date().toISOString()
        });
        
        // WebSocket 알림 발행
        await redis3.publish('websocket:job-status', JSON.stringify({
          jobId: jobId,
          status: 'running',
          progress: 75,
          message: '임베딩 처리 중...',
          timestamp: new Date().toISOString()
        }));
      } finally {
        await redis3.quit();
      }
      
      // RAG 서버 처리 완료 확인 (폴링) - API 키도 함께 전달
      monitorRAGCompletion(jobId, filename, userId, userApiKey);
    } else {
      throw new Error(`RAG 서버 업로드 실패: ${result.message || '알 수 없는 오류'}`);
    }
    
  } catch (error) {
    console.error('[processTicketSyncEventBased] 티켓 동기화 실패:', error);
    
    // 실패 시 상태 업데이트
    const redis = await createRedisClient();
    try {
      const jobKey = `${JOB_KEY_PREFIX}${jobId}`;
      await redis.hSet(jobKey, {
        status: 'failed',
        progress: '0',
        message: '티켓 동기화 실패',
        error: error instanceof Error ? error.message : '알 수 없는 오류',
        updatedAt: new Date().toISOString()
      });
    } finally {
      await redis.quit();
    }
  }
}

// 모든 작업 로드 함수
async function loadAllJobs(): Promise<BackgroundJobType[]> {
  const redis = await createRedisClient();
  try {
    const jobIds = await redis.sMembers(JOB_INDEX_KEY);
    const jobs: BackgroundJobType[] = [];
    
    for (const jobId of jobIds) {
      const jobData = await redis.hGetAll(`${JOB_KEY_PREFIX}${jobId}`);
      if (Object.keys(jobData).length > 0) {
        jobs.push({
          ...jobData,
          progress: Number(jobData.progress || 0),
          metadata: jobData.metadata ? JSON.parse(jobData.metadata) : {}
        } as BackgroundJobType);
      }
    }
    
    return jobs;
  } finally {
    await redis.quit();
  }
}

// 작업 삭제 함수
async function deleteJob(jobId: string): Promise<void> {
  const redis = await createRedisClient();
  try {
    await redis.del(`${JOB_KEY_PREFIX}${jobId}`);
    await redis.sRem(JOB_INDEX_KEY, jobId);
  } finally {
    await redis.quit();
  }
}

// 백그라운드 작업 상태 조회
// POST - 백그라운드 작업 생성
export async function POST(req: NextRequest) {
  return withAnyAuth(req, async (req, user, authType) => {
    try {
      const body = await req.json();
      const { type, userId, filename, fileData, metadata = {} } = body;

      // 작업 ID 생성
      const jobId = `job_${Date.now()}_${Math.random().toString(36).substring(2, 9)}`;
      
      // 백그라운드 작업 생성
      const job: BackgroundJobType = {
        id: jobId,
        type,
        status: 'pending',
        progress: 0,
        message: '작업 대기 중...',
        filename,
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString(),
        userId: userId || user.id || user.username,
        metadata
      };

      // Redis에 작업 저장
      const redisClient = await createRedisClient();
      
      try {
        await redisClient.hSet(`${JOB_KEY_PREFIX}${jobId}`, {
          ...job,
          metadata: JSON.stringify(metadata)
        });
        await redisClient.sAdd(JOB_INDEX_KEY, jobId);
        
        console.log(`[백그라운드 작업 생성] 작업 ID: ${jobId}, 타입: ${type}, 사용자: ${job.userId}, 파일: ${filename}`);

        // 작업 타입별 처리
        if (type === 'file_upload' && fileData) {
          console.log(`[백그라운드 작업] 파일 업로드 작업 - 인증 타입: ${authType}, 사용자:`, user);
          
          // withAnyAuth에서 받은 사용자의 API 키 사용
          let userApiKey: string | null = null;
          
          if (authType === 'apikey') {
            // API 키 인증인 경우 헤더에서 직접 가져오기
            userApiKey = req.headers.get('x-api-key');
            console.log(`[백그라운드 작업] API 키 헤더에서 가져옴: ${userApiKey ? userApiKey.substring(0, 20) + '...' : 'None'}`);
          } else if (user && user.apiKeys && user.apiKeys.length > 0) {
            // JWT 토큰 인증인 경우 user 객체에서 활성화된 API 키 찾기
            const activeApiKey = user.apiKeys.find((apiKey: any) => apiKey.status === 'active');
            if (activeApiKey) {
              userApiKey = activeApiKey.key || activeApiKey.key_value;
              console.log(`[백그라운드 작업] 사용자 객체에서 활성 API 키 가져옴: ${userApiKey ? userApiKey.substring(0, 20) + '...' : 'None'}`);
            } else {
              console.warn(`[백그라운드 작업] 활성화된 API 키가 없습니다. 첫 번째 키 사용`);
              userApiKey = user.apiKeys[0].key || user.apiKeys[0].key_value;
              console.log(`[백그라운드 작업] 사용자 객체에서 API 키 가져옴 (비활성): ${userApiKey ? userApiKey.substring(0, 20) + '...' : 'None'}`);
            }
          }
          
          console.log(`[백그라운드 작업] 사용할 API 키: ${userApiKey ? userApiKey.substring(0, 20) + '...' : 'None'}`);
          
          // 백그라운드에서 파일 업로드 처리 시작 (non-blocking)
          processFileUploadEventBased(jobId, fileData, job.userId, userApiKey, metadata)
            .catch(error => console.error(`[백그라운드 작업 ${jobId}] 처리 실패:`, error));
        } else if (type === 'ticket_sync') {
          console.log(`[백그라운드 작업] 티켓 동기화 작업 - 인증 타입: ${authType}, 사용자:`, user);
          
          // withAnyAuth에서 받은 사용자의 API 키 사용
          let userApiKey: string | null = null;
          
          if (authType === 'apikey') {
            // API 키 인증인 경우 헤더에서 직접 가져오기
            userApiKey = req.headers.get('x-api-key');
            console.log(`[백그라운드 작업] API 키 헤더에서 가져옴: ${userApiKey ? userApiKey.substring(0, 20) + '...' : 'None'}`);
          } else if (user && user.apiKeys && user.apiKeys.length > 0) {
            // JWT 토큰 인증인 경우 user 객체에서 활성화된 API 키 찾기
            const activeApiKey = user.apiKeys.find((apiKey: any) => apiKey.status === 'active');
            if (activeApiKey) {
              userApiKey = activeApiKey.key || activeApiKey.key_value;
              console.log(`[백그라운드 작업] 사용자 객체에서 활성 API 키 가져옴: ${userApiKey ? userApiKey.substring(0, 20) + '...' : 'None'}`);
            } else {
              console.warn(`[백그라운드 작업] 활성화된 API 키가 없습니다. 첫 번째 키 사용`);
              userApiKey = user.apiKeys[0].key || user.apiKeys[0].key_value;
              console.log(`[백그라운드 작업] 사용자 객체에서 API 키 가져옴 (비활성): ${userApiKey ? userApiKey.substring(0, 20) + '...' : 'None'}`);
            }
          }
          
          console.log(`[백그라운드 작업] 사용할 API 키: ${userApiKey ? userApiKey.substring(0, 20) + '...' : 'None'}`);
          
          // 백그라운드에서 티켓 동기화 처리 시작 (non-blocking)
          processTicketSyncEventBased(jobId, filename, job.userId, userApiKey)
            .catch(error => console.error(`[백그라운드 작업 ${jobId}] 처리 실패:`, error));
        }

        return NextResponse.json({
          success: true,
          data: { jobId, status: job.status, message: job.message }
        });
        
      } finally {
        await redisClient.quit();
      }
      
    } catch (error) {
      console.error('[백그라운드 작업 생성 오류]:', error);
      return NextResponse.json(
        {
          success: false,
          error: {
            message: error instanceof Error ? error.message : '작업 생성 중 오류가 발생했습니다.'
          }
        },
        { status: 500 }
      );
    }
  });
}

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const userId = searchParams.get('userId');
    const jobId = searchParams.get('jobId');
    
    const jobs = await loadAllJobs();
    
    if (jobId) {
      const job = jobs.find(j => j.id === jobId);
      if (!job) {
        return NextResponse.json({ success: false, error: '작업을 찾을 수 없습니다' }, { status: 404 });
      }
      return NextResponse.json({ success: true, job });
    }
    
    const userJobs = jobs
      .filter(job => !userId || job.userId === userId)
      .sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime());
    
    const status = {
      jobs: userJobs,
      activeJobs: userJobs.filter(job => job.status === 'pending' || job.status === 'running').length,
      completedJobs: userJobs.filter(job => job.status === 'completed').length,
      failedJobs: userJobs.filter(job => job.status === 'failed').length
    };
    
    return NextResponse.json({ success: true, data: status });
  } catch (error) {
    console.error('백그라운드 작업 상태 조회 실패:', error);
    return NextResponse.json({ success: false, error: '상태 조회 실패' }, { status: 500 });
  }
}

// 백그라운드 작업 삭제
export async function DELETE(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const jobId = searchParams.get('jobId');
    
    if (!jobId) {
      return NextResponse.json({ success: false, error: '작업 ID가 필요합니다' }, { status: 400 });
    }
    
    await deleteJob(jobId);
    return NextResponse.json({ success: true });
  } catch (error) {
    console.error('백그라운드 작업 삭제 실패:', error);
    return NextResponse.json({ success: false, error: '작업 삭제 실패' }, { status: 500 });
  }
}