import { NextRequest, NextResponse } from 'next/server';
import { getReportServerUrl } from '@/config/serverConfig';

/**
 * 주간/월간 보고서 다운로드 API
 * POST /api/weekly-report/download
 *
 * Report Server의 /weekly-report/convert 엔드포인트를 호출하여
 * Python utils 모듈을 사용한 문서 생성을 수행합니다.
 */

// 프론트엔드에서 보내는 기존 형식
interface LegacyReportData {
  reportType: 'weekly' | 'monthly';
  reportPeriodStart: string;
  reportPeriodEnd: string;
  department?: string;
  author?: string;
  summary?: string;
  aiSummary?: string;
  completedTasks?: Array<{
    id?: string;
    taskName: string;
    priority?: string;
    progress?: number;
    timeSpent?: number;
    result?: string;
  }>;
  ongoingTasks?: Array<{
    id?: string;
    taskName: string;
    priority?: string;
    progress?: number;
    dueDate?: string;
    nextWeekPlan?: string;
  }>;
  issues?: Array<{
    id?: string;
    category: string;
    content: string;
    impact?: string;
    solution: string;
    status: string;
  }>;
  nextPeriodPlans?: Array<{
    id?: string;
    taskName: string;
    priority: string;
    estimatedTime?: number;
    targetResult?: string;
  }>;
  format: 'pdf' | 'docx' | 'hwp';
}

// 새 형식 (Report Server가 기대하는 형식)
interface NewFormatRequest {
  report: {
    summary: string;
    completedTasks: Array<{
      taskName: string;
      result: string;
      timeSpent?: string;
    }>;
    ongoingTasks: Array<{
      taskName: string;
      progress: string;
      nextPlan: string;
    }>;
    issues: Array<{
      category: string;
      content: string;
      solution: string;
      status: string;
    }>;
    nextPeriodPlans: Array<{
      taskName: string;
      priority: string;
      targetResult: string;
    }>;
  };
  metadata: {
    reportType: 'weekly' | 'monthly';
    startDate: string;
    endDate: string;
    department?: string;
    author?: string;
    documentCount?: number;
    generatedAt?: string;
  };
  format: 'pdf' | 'docx' | 'hwp';
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    console.log('[Weekly Report Download] Received body:', JSON.stringify(body, null, 2));

    // 형식 확인: 새 형식인지 기존 형식인지
    const isNewFormat = body.report && body.metadata;

    let requestBody: NewFormatRequest;

    if (isNewFormat) {
      // 새 형식 그대로 사용
      requestBody = body as NewFormatRequest;
    } else {
      // 기존 형식을 새 형식으로 변환
      const legacyData = body as LegacyReportData;

      // 입력 검증
      if (!legacyData.format) {
        return NextResponse.json(
          {
            success: false,
            error: { message: '파일 형식이 필요합니다.' }
          },
          { status: 400 }
        );
      }

      if (!['pdf', 'docx', 'hwp'].includes(legacyData.format)) {
        return NextResponse.json(
          {
            success: false,
            error: { message: '지원하지 않는 파일 형식입니다. (pdf, docx, hwp만 지원)' }
          },
          { status: 400 }
        );
      }

      // 기존 형식을 새 형식으로 변환
      requestBody = {
        report: {
          summary: legacyData.aiSummary || legacyData.summary || '',
          completedTasks: (legacyData.completedTasks || []).map(task => ({
            taskName: task.taskName || '',
            result: task.result || '',
            timeSpent: task.timeSpent ? `${task.timeSpent}시간` : undefined
          })),
          ongoingTasks: (legacyData.ongoingTasks || []).map(task => ({
            taskName: task.taskName || '',
            progress: task.progress !== undefined ? `${task.progress}%` : '',
            nextPlan: task.nextWeekPlan || ''
          })),
          issues: (legacyData.issues || []).map(issue => ({
            category: issue.category || '',
            content: issue.content || '',
            solution: issue.solution || '',
            status: issue.status || ''
          })),
          nextPeriodPlans: (legacyData.nextPeriodPlans || []).map(plan => ({
            taskName: plan.taskName || '',
            priority: plan.priority || '',
            targetResult: plan.targetResult || ''
          }))
        },
        metadata: {
          reportType: legacyData.reportType || 'weekly',
          startDate: legacyData.reportPeriodStart || new Date().toISOString().split('T')[0],
          endDate: legacyData.reportPeriodEnd || new Date().toISOString().split('T')[0],
          department: legacyData.department || undefined,
          author: legacyData.author || undefined,
          generatedAt: new Date().toISOString()
        },
        format: legacyData.format
      };
    }

    // Report Server URL
    const reportServerUrl = getReportServerUrl();
    console.log(`[Weekly Report Download] Calling Report Server: ${reportServerUrl}/weekly-report/convert`);

    // HWP는 hwpx로 변환 (Report Server에서 사용하는 형식)
    const outputFormat = requestBody.format === 'hwp' ? 'hwpx' : requestBody.format;

    const serverRequestBody = {
      report: {
        summary: requestBody.report.summary || '',
        completedTasks: (requestBody.report.completedTasks || []).map(task => ({
          taskName: task.taskName || '',
          result: task.result || '',
          timeSpent: task.timeSpent || null
        })),
        ongoingTasks: (requestBody.report.ongoingTasks || []).map(task => ({
          taskName: task.taskName || '',
          progress: task.progress || '',
          nextPlan: task.nextPlan || ''
        })),
        issues: (requestBody.report.issues || []).map(issue => ({
          category: issue.category || '',
          content: issue.content || '',
          solution: issue.solution || '',
          status: issue.status || ''
        })),
        nextPeriodPlans: (requestBody.report.nextPeriodPlans || []).map(plan => ({
          taskName: plan.taskName || '',
          priority: plan.priority || '',
          targetResult: plan.targetResult || ''
        }))
      },
      metadata: {
        reportType: requestBody.metadata?.reportType || 'weekly',
        startDate: requestBody.metadata?.startDate || new Date().toISOString().split('T')[0],
        endDate: requestBody.metadata?.endDate || new Date().toISOString().split('T')[0],
        department: requestBody.metadata?.department || null,
        author: requestBody.metadata?.author || null,
        documentCount: requestBody.metadata?.documentCount || 0,
        generatedAt: requestBody.metadata?.generatedAt || new Date().toISOString()
      },
      format: outputFormat
    };

    console.log('[Weekly Report Download] Server request body:', JSON.stringify(serverRequestBody, null, 2));

    // Report Server 호출
    const response = await fetch(`${reportServerUrl}/weekly-report/convert`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(serverRequestBody)
    });

    if (!response.ok) {
      const errorText = await response.text();
      console.error('[Weekly Report Download] Report Server error:', errorText);
      throw new Error(`Report Server 오류: ${response.status} - ${errorText}`);
    }

    // 바이너리 데이터 읽기
    const documentBuffer = await response.arrayBuffer();
    console.log(`[Weekly Report Download] Received document: ${documentBuffer.byteLength} bytes`);

    // 파일명 생성
    const periodLabel = requestBody.metadata?.reportType === 'weekly' ? '주간업무보고서' : '월간업무보고서';
    const startDateStr = (requestBody.metadata?.startDate || '').replace(/-/g, '');
    const endDateStr = (requestBody.metadata?.endDate || '').replace(/-/g, '');
    const timestamp = new Date().toISOString().slice(0, 10).replace(/-/g, '');
    const fileName = `${periodLabel}_${startDateStr}_${endDateStr || timestamp}`;

    // MIME 타입 및 확장자 설정
    const mimeTypes: Record<string, string> = {
      pdf: 'application/pdf',
      docx: 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
      hwp: 'application/hwp+zip',
      hwpx: 'application/hwp+zip'
    };

    const extensions: Record<string, string> = {
      pdf: 'pdf',
      docx: 'docx',
      hwp: 'hwpx',
      hwpx: 'hwpx'
    };

    const mimeType = mimeTypes[requestBody.format] || 'application/octet-stream';
    const extension = extensions[requestBody.format] || requestBody.format;

    // 응답 반환
    return new NextResponse(documentBuffer, {
      status: 200,
      headers: {
        'Content-Type': mimeType,
        'Content-Disposition': `attachment; filename*=UTF-8''${encodeURIComponent(fileName)}.${extension}`,
        'Content-Length': documentBuffer.byteLength.toString()
      }
    });

  } catch (error) {
    console.error('[Weekly Report Download] Error:', error);
    return NextResponse.json(
      {
        success: false,
        error: {
          message: error instanceof Error ? error.message : '보고서 다운로드 중 오류가 발생했습니다.'
        }
      },
      { status: 500 }
    );
  }
}
