import { NextRequest, NextResponse } from 'next/server';
import { getApiServerUrl } from '@/config/serverConfig';

interface DocumentInfo {
  fileName: string;
  content: string;
  modificationDate: string;
}

interface GenerateReportRequest {
  reportType: 'weekly' | 'monthly';
  startDate: string;
  endDate: string;
  documents: DocumentInfo[];
  department?: string;
  author?: string;
}

interface ReportSection {
  title: string;
  content: string;
}

interface GeneratedReport {
  summary: string;
  completedTasks: Array<{
    taskName: string;
    result: string;
    timeSpent?: string;
  }>;
  ongoingTasks: Array<{
    taskName: string;
    progress: string;
    nextPlan: string;
  }>;
  issues: Array<{
    category: string;
    content: string;
    solution: string;
    status: string;
  }>;
  nextPeriodPlans: Array<{
    taskName: string;
    priority: string;
    targetResult: string;
  }>;
}

/**
 * AI 기반 주간/월간 보고서 생성 API
 * POST /api/weekly-report/generate
 */
export async function POST(request: NextRequest) {
  try {
    const body: GenerateReportRequest = await request.json();
    const { reportType, startDate, endDate, documents, department, author } = body;

    // 입력 검증
    if (!documents || documents.length === 0) {
      return NextResponse.json(
        {
          success: false,
          error: { message: '분석할 문서가 없습니다. 먼저 문서를 검색해주세요.' }
        },
        { status: 400 }
      );
    }

    // 문서 내용 로깅
    console.log('[Weekly Report Generate] Documents received:', documents.length);
    documents.forEach((doc, idx) => {
      console.log(`[Weekly Report Generate] Doc ${idx + 1}: ${doc.fileName}, content length: ${doc.content?.length || 0}`);
    });

    // 문서 내용을 하나의 텍스트로 합치기
    const documentsText = documents.map((doc, idx) => {
      return `
### 문서 ${idx + 1}: ${doc.fileName}
(수정일: ${doc.modificationDate || '알 수 없음'})

${doc.content || '(내용 없음)'}
---
`;
    }).join('\n');

    const periodLabel = reportType === 'weekly' ? '주간' : '월간';
    const nextPeriodLabel = reportType === 'weekly' ? '차주' : '차월';

    // AI 프롬프트 생성
    const prompt = `다음은 ${startDate}부터 ${endDate}까지의 업무 관련 문서들입니다.
이 문서들을 분석하여 ${periodLabel} 업무보고서를 작성해주세요.

${department ? `부서: ${department}` : ''}
${author ? `작성자: ${author}` : ''}

## 분석할 문서들:
${documentsText}

## 요청사항:
위 문서들을 기반으로 다음 JSON 형식으로 ${periodLabel} 업무보고서를 작성해주세요.
반드시 아래 JSON 형식만 출력하고, 다른 설명이나 텍스트는 포함하지 마세요.

**중요: 행정·공공 R&D 보고서 문체 작성 지침**
- 모든 문장은 "~이다.", "~함.", "~됨.", "~있음." 등의 종결 어미로 끝낼 것
- "~습니다", "~ㅂ니다" 등의 경어체는 사용하지 말 것
- 간결하고 단정적인 표현을 사용할 것

{
  "summary": "전체 업무 요약 (행정·공공 R&D 보고서 문체, 3-5문장으로 핵심 내용 요약)",
  "completedTasks": [
    {
      "taskName": "완료된 업무 제목",
      "result": "업무 성과 및 결과 (간결하고 사실 중심으로 기술)",
      "timeSpent": "소요 시간 (예: 8시간, 2일 등)"
    }
  ],
  "ongoingTasks": [
    {
      "taskName": "진행중인 업무 제목",
      "progress": "진행률 (예: 50%, 대부분 완료 등)",
      "nextPlan": "${nextPeriodLabel} 계획"
    }
  ],
  "issues": [
    {
      "category": "이슈 분류 (기술, 인력, 일정 등)",
      "content": "이슈 내용 (객관적인 사실로 기술)",
      "solution": "해결 방안 또는 대응 계획",
      "status": "진행중/해결완료/보류"
    }
  ],
  "nextPeriodPlans": [
    {
      "taskName": "${nextPeriodLabel} 계획 업무",
      "priority": "높음/보통/낮음",
      "targetResult": "목표 성과"
    }
  ]
}`;

    // API 서버로 AI 요청 (/callai 엔드포인트 사용 - 채팅 기록 남지 않음)
    const apiKey = request.headers.get('X-API-Key');
    const token = request.headers.get('Authorization');
    const apiServerUrl = getApiServerUrl();

    // AbortController 생성하여 타임아웃 설정
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 300000); // 5분 타임아웃

    const response = await fetch(`${apiServerUrl}/callai`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        ...(apiKey && { 'X-API-Key': apiKey }),
        ...(token && { 'Authorization': token })
      },
      body: JSON.stringify({
        messages: [
          {
            role: 'user',
            content: prompt
          }
        ],
        options: {
          temperature: 0.3,
          max_tokens: 4000,
          provider: 'ollama',
          model: 'hamonize:latest',
          tools: false  // 도구 호출 비활성화
        },
        sessionId: `weekly-report-${Date.now()}`, // 고유한 세션 ID 생성
        stream: false
      }),
      signal: controller.signal
    });

    clearTimeout(timeoutId); // 응답 받으면 타임아웃 해제

    if (!response.ok) {
      const errorData = await response.json();
      throw new Error(errorData.error?.message || 'AI 보고서 생성 실패');
    }

    const data = await response.json();

    // 응답 구조 로깅
    console.log('[Weekly Report Generate] Raw API response:', JSON.stringify(data, null, 2));
    console.log('[Weekly Report Generate] Response type:', typeof data);
    console.log('[Weekly Report Generate] Response keys:', Object.keys(data || {}));

    // AI 응답에서 JSON 추출
    const reportContent = extractReportJson(data);

    if (!reportContent) {
      throw new Error('AI 응답에서 보고서 내용을 추출할 수 없습니다.');
    }

    return NextResponse.json({
      success: true,
      data: {
        report: reportContent,
        metadata: {
          reportType,
          startDate,
          endDate,
          department,
          author,
          documentCount: documents.length,
          generatedAt: new Date().toISOString()
        }
      },
      timestamp: new Date().toISOString()
    });

  } catch (error) {
    console.error('[Weekly Report Generate] Error:', error);
    return NextResponse.json(
      {
        success: false,
        error: {
          message: error instanceof Error ? error.message : '보고서 생성 중 오류가 발생했습니다.'
        }
      },
      { status: 500 }
    );
  }
}

/**
 * AI 응답에서 보고서 JSON 추출
 */
function extractReportJson(data: unknown): GeneratedReport | null {
  let jsonStr = '';  // try 블록 밖에서 변수 선언하여 블록 스코프 문제 해결

  try {
    let content = '';

    // /callai 엔드포인트 응답 형식 처리
    // 실제 응답 구조: { success: true, data: { response: { content: "..." } } }
    if (typeof data === 'string') {
      content = data;
    } else if (data && typeof data === 'object') {
      const obj = data as Record<string, unknown>;

      // /callai 응답 형식: { success: true, data: { response: { content: "..." } } }
      if (obj.success === true && obj.data && typeof obj.data === 'object') {
        const dataObj = obj.data as Record<string, unknown>;

        // data.response가 객체이고 content 필드가 있는 경우
        if (dataObj.response && typeof dataObj.response === 'object') {
          const responseObj = dataObj.response as Record<string, unknown>;
          content = (responseObj.content || responseObj.response || responseObj.message || '') as string;
        } else {
          // data.response가 문자열인 경우
          content = (dataObj.response || dataObj.content || dataObj.message || '') as string;
        }
      }
      // 이전 /chat/sync 응답 형식도 지원 (호환성)
      else if (obj.data && typeof obj.data === 'object') {
        const outerData = obj.data as Record<string, unknown>;

        // 한 단계 더 중첩된 경우 (data.data.response)
        if (outerData.data && typeof outerData.data === 'object') {
          const innerData = outerData.data as Record<string, unknown>;
          content = (innerData.response || innerData.content || innerData.message || '') as string;
        } else {
          // 한 단계만 중첩된 경우 (data.response)
          content = (outerData.response || outerData.content || outerData.message || '') as string;
        }
      } else {
        // 중첩되지 않은 경우
        content = (obj.content || obj.message || obj.response || '') as string;
      }
    }

    // content가 문자열인지 확인
    if (typeof content !== 'string') {
      console.error('[extractReportJson] Content is not a string:', typeof content);
      return null;
    }

    console.log('[extractReportJson] Extracted content length:', content?.length || 0);
    console.log('[extractReportJson] Content preview:', content?.substring(0, 500));

    if (!content || content.length === 0) {
      console.error('[extractReportJson] No content found in response');
      return null;
    }

    // JSON 블록 추출 (```json ... ``` 또는 { ... })
    jsonStr = content;

    // 마크다운 코드 블록에서 JSON 추출
    const jsonBlockMatch = content.match(/```(?:json)?\s*([\s\S]*?)```/);
    if (jsonBlockMatch) {
      jsonStr = jsonBlockMatch[1].trim();
    } else {
      // 순수 JSON 객체 추출
      const jsonMatch = content.match(/\{[\s\S]*\}/);
      if (jsonMatch) {
        jsonStr = jsonMatch[0];
      }
    }

    const parsed = JSON.parse(jsonStr);
    console.log('[extractReportJson] Parsed summary:', parsed.summary?.substring(0, 200));
    console.log('[extractReportJson] completedTasks count:', parsed.completedTasks?.length || 0);

    // 기본 구조 확인 및 보완
    const result = {
      summary: parsed.summary || '',
      completedTasks: Array.isArray(parsed.completedTasks) ? parsed.completedTasks : [],
      ongoingTasks: Array.isArray(parsed.ongoingTasks) ? parsed.ongoingTasks : [],
      issues: Array.isArray(parsed.issues) ? parsed.issues : [],
      nextPeriodPlans: Array.isArray(parsed.nextPeriodPlans) ? parsed.nextPeriodPlans : []
    };
    console.log('[extractReportJson] Result summary length:', result.summary.length);
    return result;
  } catch (e) {
    console.error('[extractReportJson] Parse error:', e);
    console.error('[extractReportJson] JSON string preview:', jsonStr?.substring(0, 300));
    return null;
  }
}
