import { NextRequest, NextResponse } from 'next/server';
import { convertToLocaleDateTime, formatLocalDate } from '@/lib/dateUtils';
import { getRagServerUrl, getApiServerUrl } from '@/config/serverConfig';
import fs from 'fs';
import path from 'path';

interface DocumentMetadata {
  source: string;
  filename: string;
  modification_date: string;
  timestamp: string;
  user_id: string;
  [key: string]: unknown;
}

interface RagDocument {
  file_name: string;
  source: string;
  chunk_count: number;
  last_modified: string;
  metadata: DocumentMetadata;
  user_id: string;
  file_exists: boolean;
  embedding_status: string;
  created_at?: string;
  updated_at?: string;
}

interface DbDocument {
  filename: string;
  filepath: string;
  created_at: string;
  updated_at: string;
  embedding_status: string;
  metadata: Record<string, unknown>;
}

interface SearchDocsRequest {
  startDate: string; // YYYY-MM-DD
  endDate: string;   // YYYY-MM-DD
  userId: string;
}

interface DocumentResult {
  fileName: string;
  source: string;
  modificationDate: string;
  chunkCount: number;
  content: string;
  embeddingStatus: string;
}

/**
 * RAG 문서 날짜 기반 검색 API
 * POST /api/weekly-report/search-docs
 *
 * 지정된 날짜 범위 내에 수정된 문서들을 검색합니다.
 * 임베딩 완료 문서는 RAG 서버에서, 대기 중인 문서는 파일 시스템에서 직접 읽어옵니다.
 */
export async function POST(request: NextRequest) {
  try {
    const body: SearchDocsRequest = await request.json();
    const { startDate, endDate, userId } = body;

    // 입력 검증
    if (!startDate || !endDate) {
      return NextResponse.json(
        {
          success: false,
          error: { message: '시작 날짜와 종료 날짜를 입력해주세요.' }
        },
        { status: 400 }
      );
    }

    if (!userId) {
      return NextResponse.json(
        {
          success: false,
          error: { message: '사용자 ID가 필요합니다.' }
        },
        { status: 400 }
      );
    }

    // 날짜 파싱
    const startDateTime = new Date(startDate);
    const endDateTime = new Date(endDate);
    endDateTime.setHours(23, 59, 59, 999); // 종료일의 끝까지 포함

    if (isNaN(startDateTime.getTime()) || isNaN(endDateTime.getTime())) {
      return NextResponse.json(
        {
          success: false,
          error: { message: '유효하지 않은 날짜 형식입니다.' }
        },
        { status: 400 }
      );
    }

    if (startDateTime > endDateTime) {
      return NextResponse.json(
        {
          success: false,
          error: { message: '시작 날짜가 종료 날짜보다 클 수 없습니다.' }
        },
        { status: 400 }
      );
    }

    const ragServerUrl = getRagServerUrl();
    const apiServerUrl = getApiServerUrl();
    const allDocuments: DocumentResult[] = [];
    const processedFileNames = new Set<string>();

    // 1. RAG 서버에서 임베딩 완료된 문서 목록 조회
    try {
      const listResponse = await fetch(`${ragServerUrl}/list?user_id=${encodeURIComponent(userId)}`);
      console.log("[Weekly Report Search Docs] Fetching RAG documents list:", `${ragServerUrl}/list?user_id=${encodeURIComponent(userId)}`);

      if (listResponse.ok) {
        const listData = await listResponse.json();
        const documents: RagDocument[] = listData.documents || [];

        // 날짜 범위로 문서 필터링
        const filteredDocuments = documents.filter((doc) => {
          const metadata = doc.metadata || {};
          let docDate: Date | null = null;

          if (metadata.modification_date) {
            docDate = new Date(metadata.modification_date);
          } else if (metadata.timestamp) {
            docDate = new Date(metadata.timestamp);
          } else if (doc.last_modified) {
            docDate = new Date(doc.last_modified);
          } else if (doc.updated_at) {
            docDate = new Date(doc.updated_at);
          } else if (doc.created_at) {
            docDate = new Date(doc.created_at);
          }

          if (!docDate || isNaN(docDate.getTime())) {
            return false;
          }

          return docDate >= startDateTime && docDate <= endDateTime;
        });

        // RAG 청크에서 문서 내용 가져오기 (이미 추출된 텍스트 사용)
        console.log(`[Weekly Report Search Docs] Processing ${filteredDocuments.length} filtered documents from RAG`);
        for (const doc of filteredDocuments) {
          try {
            let content = '';

            // RAG 청크 엔드포인트에서 이미 추출된 텍스트 가져오기
            const chunksUrl = `${ragServerUrl}/chunks?filename=${encodeURIComponent(doc.file_name)}&user_id=${encodeURIComponent(userId)}`;
            console.log(`[Weekly Report Search Docs] Fetching chunks: ${chunksUrl}`);

            const chunksResponse = await fetch(chunksUrl);
            if (chunksResponse.ok) {
              const chunksData = await chunksResponse.json();
              if (chunksData.success && chunksData.chunks && chunksData.chunks.length > 0) {
                // 청크들을 결합하여 내용 구성 (최대 3개 청크, 2000자 제한)
                const chunkContents = chunksData.chunks
                  .slice(0, 3)
                  .map((c: { content: string }) => c.content)
                  .join('\n\n');
                content = chunkContents.substring(0, 2000);
                console.log(`[Weekly Report Search Docs] Got ${chunksData.chunks.length} chunks for ${doc.file_name}, using ${Math.min(3, chunksData.chunks.length)} chunks, content: ${content.length} chars`);
              } else {
                console.log(`[Weekly Report Search Docs] No chunks found for ${doc.file_name}`);
              }
            } else {
              console.log(`[Weekly Report Search Docs] Chunks fetch failed for ${doc.file_name}: ${chunksResponse.status}`);
            }

            allDocuments.push({
              fileName: doc.file_name,
              source: doc.source,
              modificationDate: doc.metadata?.modification_date || doc.last_modified || doc.updated_at || doc.created_at,
              chunkCount: doc.chunk_count,
              content: content,
              embeddingStatus: doc.embedding_status
            });
            processedFileNames.add(doc.file_name);
          } catch (err) {
            console.error(`[Weekly Report Search Docs] Error processing ${doc.file_name}:`, err);
            // 오류 발생 시에도 문서 정보는 포함
            allDocuments.push({
              fileName: doc.file_name,
              source: doc.source,
              modificationDate: doc.metadata?.modification_date || doc.last_modified || doc.updated_at || doc.created_at,
              chunkCount: doc.chunk_count,
              content: '',
              embeddingStatus: doc.embedding_status
            });
            processedFileNames.add(doc.file_name);
          }
        }
      }
    } catch (ragError) {
      console.warn('[Weekly Report Search Docs] RAG server error:', ragError);
    }

    // 2. API 서버에서 pending/processing 상태의 문서 조회
    try {
      const apiKey = request.headers.get('X-API-Key');
      const token = request.headers.get('Authorization');

      const dbDocsResponse = await fetch(`${apiServerUrl}/rag/documents?user_id=${encodeURIComponent(userId)}`, {
        headers: {
          'Content-Type': 'application/json',
          ...(apiKey && { 'X-API-Key': apiKey }),
          ...(token && { 'Authorization': token })
        }
      });

      if (dbDocsResponse.ok) {
        const dbDocsData = await dbDocsResponse.json();
        const dbDocuments: DbDocument[] = dbDocsData.data?.documents || [];

        for (const doc of dbDocuments) {
          // 이미 RAG에서 처리한 문서는 건너뛰기
          if (processedFileNames.has(doc.filename)) {
            continue;
          }

          // 날짜 필터링
          const docDate = new Date(doc.updated_at || doc.created_at);
          if (isNaN(docDate.getTime()) || docDate < startDateTime || docDate > endDateTime) {
            continue;
          }

          // 임베딩 대기 중인 문서는 청크가 없으므로 상태 메시지 표시
          let content = '';
          if (doc.embedding_status === 'pending' || doc.embedding_status === 'processing') {
            const ext = doc.filepath ? path.extname(doc.filepath).toLowerCase() : '';
            content = `[${ext ? ext.toUpperCase().slice(1) + ' ' : ''}문서 임베딩 ${doc.embedding_status === 'processing' ? '처리 중' : '대기 중'}...]`;
          } else if (doc.filepath) {
            // 텍스트 파일만 직접 읽기 (임베딩 완료되었지만 청크를 못 가져온 경우)
            const ext = path.extname(doc.filepath).toLowerCase();
            if (['.txt', '.md', '.csv', '.json', '.xml', '.html', '.log'].includes(ext) && fs.existsSync(doc.filepath)) {
              try {
                const fileContent = fs.readFileSync(doc.filepath, 'utf-8');
                content = fileContent.substring(0, 2000);
              } catch (fileError) {
                console.warn(`[Weekly Report Search Docs] Failed to read text file: ${doc.filepath}`, fileError);
              }
            }
          }

          allDocuments.push({
            fileName: doc.filename,
            source: doc.filepath,
            modificationDate: doc.updated_at || doc.created_at,
            chunkCount: 0,
            content,
            embeddingStatus: doc.embedding_status
          });
          processedFileNames.add(doc.filename);
        }
      }
    } catch (dbError) {
      console.warn('[Weekly Report Search Docs] DB documents fetch error:', dbError);
    }

    // 3. 파일 시스템에서 직접 검색 (RAG docs 디렉토리)
    const ragDocsPath = path.join(process.env.HOME || '/home/hamonikr', '문서', 'rag-docs', userId);
    try {
      if (fs.existsSync(ragDocsPath)) {
        const files = fs.readdirSync(ragDocsPath);

        for (const fileName of files) {
          // 이미 처리한 파일은 건너뛰기
          if (processedFileNames.has(fileName)) {
            continue;
          }

          const filePath = path.join(ragDocsPath, fileName);
          const stats = fs.statSync(filePath);

          // 디렉토리는 건너뛰기
          if (stats.isDirectory()) {
            continue;
          }

          // 날짜 필터링 (파일 수정 시간 사용)
          const fileModTime = stats.mtime;
          if (fileModTime < startDateTime || fileModTime > endDateTime) {
            continue;
          }

          // 텍스트 파일만 직접 읽기, 바이너리 파일은 임베딩 후 확인 가능 메시지 표시
          let content = '';
          const ext = path.extname(fileName).toLowerCase();
          if (['.txt', '.md', '.json', '.csv', '.xml', '.html', '.log'].includes(ext)) {
            try {
              const fileContent = fs.readFileSync(filePath, 'utf-8');
              content = fileContent.substring(0, 2000);
            } catch {
              // 읽기 실패 시 내용 없이 진행
            }
          } else if (['.pdf', '.doc', '.docx', '.hwp', '.hwpx', '.pptx', '.xlsx'].includes(ext)) {
            content = `[${ext.toUpperCase().slice(1)} 문서 - RAG 임베딩 후 내용 확인 가능]`;
          }

          allDocuments.push({
            fileName,
            source: filePath,
            modificationDate: fileModTime.toISOString(),
            chunkCount: 0,
            content,
            embeddingStatus: 'pending'
          });
        }
      }
    } catch (fsError) {
      console.warn('[Weekly Report Search Docs] Filesystem search error:', fsError);
    }

    // 날짜순으로 정렬 (최신순)
    allDocuments.sort((a, b) => {
      const dateA = new Date(a.modificationDate || 0);
      const dateB = new Date(b.modificationDate || 0);
      return dateB.getTime() - dateA.getTime();
    });

    // 최종 결과 로깅
    console.log(`[Weekly Report Search Docs] Total documents found: ${allDocuments.length}`);
    allDocuments.forEach((doc, idx) => {
      console.log(`[Weekly Report Search Docs] Final doc ${idx + 1}: ${doc.fileName}, content: ${doc.content?.length || 0} chars`);
    });

    return NextResponse.json({
      success: true,
      data: {
        documents: allDocuments,
        count: allDocuments.length,
        dateRange: {
          start: startDate,
          end: endDate
        }
      },
      timestamp: new Date().toISOString()
    });

  } catch (error) {
    console.error('[Weekly Report Search Docs] Error:', error);
    return NextResponse.json(
      {
        success: false,
        error: {
          message: error instanceof Error ? error.message : '문서 검색 중 오류가 발생했습니다.'
        }
      },
      { status: 500 }
    );
  }
}
