import { NextRequest, NextResponse } from 'next/server';
import { getApiServerUrl } from '@/config/serverConfig';

// 타입 정의
interface TaskItem {
  id: string;
  taskName: string;
  priority: '높음' | '보통' | '낮음';
  progress: number;
  timeSpent?: number;
  result?: string;
  assignee?: string;
  dueDate?: string;
  nextWeekPlan?: string;
  remarks?: string;
}

interface IssueItem {
  id: string;
  category: string;
  content: string;
  impact: string;
  solution: string;
  status: string;
}

interface PlanItem {
  id: string;
  taskName: string;
  priority: string;
  estimatedTime?: number;
  targetResult?: string;
  assignee?: string;
}

interface ReportData {
  reportType: 'weekly' | 'monthly';
  reportPeriodStart: string;
  reportPeriodEnd: string;
  department: string;
  author: string;
  summary: string;
  completedTasks: TaskItem[];
  ongoingTasks: TaskItem[];
  issues: IssueItem[];
  nextPeriodPlans: PlanItem[];
  aiSummary?: string;
}

/**
 * AI 요약 생성 API
 * POST /api/weekly-report/summary
 */
export async function POST(request: NextRequest) {
  try {
    const reportData: ReportData = await request.json();

    // 입력 데이터 검증
    if (!reportData.completedTasks?.length && !reportData.ongoingTasks?.length) {
      return NextResponse.json(
        {
          success: false,
          error: { message: '완료 업무 또는 진행중 업무를 최소 1개 이상 입력해주세요.' }
        },
        { status: 400 }
      );
    }

    // 보고서 데이터를 텍스트로 변환
    const reportText = formatReportForAI(reportData);

    // API 서버로 요약 요청
    const apiKey = request.headers.get('X-API-Key');
    const token = request.headers.get('Authorization');
    const apiServerUrl = getApiServerUrl();

    const response = await fetch(`${apiServerUrl}/api/v1/chat/sync`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        ...(apiKey && { 'X-API-Key': apiKey }),
        ...(token && { 'Authorization': token })
      },
      body: JSON.stringify({
        prompt: createSummaryPrompt(reportText, reportData.reportType),
        options: {
          temperature: 0.7,
          max_tokens: 1000,
          rag: false,
          web: false,
          tools: false
        }
      })
    });

    if (!response.ok) {
      const errorData = await response.json();
      throw new Error(errorData.error?.message || 'AI 요약 생성 실패');
    }

    const data = await response.json();

    // AI 응답에서 요약 추출
    const summary = extractSummary(data);

    return NextResponse.json({
      success: true,
      data: { summary },
      timestamp: new Date().toISOString()
    });

  } catch (error) {
    console.error('[Weekly Report Summary] Error:', error);
    return NextResponse.json(
      {
        success: false,
        error: {
          message: error instanceof Error ? error.message : 'AI 요약 생성 중 오류가 발생했습니다.'
        }
      },
      { status: 500 }
    );
  }
}

/**
 * 보고서 데이터를 AI 요약용 텍스트로 변환
 */
function formatReportForAI(data: ReportData): string {
  const periodLabel = data.reportType === 'weekly' ? '주간' : '월간';
  let text = `# ${periodLabel} 업무보고서\n`;
  text += `- 보고 기간: ${data.reportPeriodStart} ~ ${data.reportPeriodEnd}\n`;

  if (data.department) text += `- 부서: ${data.department}\n`;
  if (data.author) text += `- 작성자: ${data.author}\n`;
  if (data.summary) text += `- 개요: ${data.summary}\n`;

  text += '\n';

  // 완료 업무
  if (data.completedTasks.length > 0) {
    text += '## 완료 업무\n';
    data.completedTasks.forEach((task, i) => {
      text += `${i + 1}. ${task.taskName}`;
      if (task.priority) text += ` (우선순위: ${task.priority})`;
      if (task.timeSpent) text += ` - 소요시간: ${task.timeSpent}시간`;
      if (task.result) text += `\n   성과: ${task.result}`;
      text += '\n';
    });
    text += '\n';
  }

  // 진행중 업무
  if (data.ongoingTasks.length > 0) {
    text += '## 진행중 업무\n';
    data.ongoingTasks.forEach((task, i) => {
      text += `${i + 1}. ${task.taskName} (진행률: ${task.progress}%)`;
      if (task.dueDate) text += ` - 예상완료: ${task.dueDate}`;
      if (task.nextWeekPlan) text += `\n   다음 계획: ${task.nextWeekPlan}`;
      text += '\n';
    });
    text += '\n';
  }

  // 이슈 사항
  if (data.issues.length > 0) {
    text += '## 이슈 및 건의사항\n';
    data.issues.forEach((issue, i) => {
      text += `${i + 1}. [${issue.category}] ${issue.content}`;
      text += ` (영향도: ${issue.impact}, 상태: ${issue.status})`;
      if (issue.solution) text += `\n   대응: ${issue.solution}`;
      text += '\n';
    });
    text += '\n';
  }

  // 차주/차월 계획
  if (data.nextPeriodPlans.length > 0) {
    const nextPeriod = data.reportType === 'weekly' ? '차주' : '차월';
    text += `## ${nextPeriod} 업무 계획\n`;
    data.nextPeriodPlans.forEach((plan, i) => {
      text += `${i + 1}. ${plan.taskName} (${plan.priority})`;
      if (plan.estimatedTime) text += ` - 예상 ${plan.estimatedTime}시간`;
      if (plan.targetResult) text += `\n   목표: ${plan.targetResult}`;
      text += '\n';
    });
  }

  return text;
}

/**
 * AI 요약 생성 프롬프트 생성
 */
function createSummaryPrompt(reportText: string, reportType: 'weekly' | 'monthly'): string {
  const periodLabel = reportType === 'weekly' ? '주간' : '월간';

  return `다음 ${periodLabel} 업무보고서 내용을 바탕으로 임원에게 보고할 핵심 요약을 작성해주세요.

요약 작성 지침:
1. 3-5문장으로 간결하게 작성
2. 주요 성과와 진행 상황을 먼저 언급
3. 중요한 이슈가 있다면 언급
4. 차주/차월 중점 추진 사항 포함
5. 전문적이고 명확한 어조 사용

---

${reportText}

---

위 내용을 바탕으로 핵심 요약을 작성해주세요:`;
}

/**
 * AI 응답에서 요약 텍스트 추출
 */
function extractSummary(data: unknown): string {
  if (typeof data === 'string') return data;

  if (data && typeof data === 'object') {
    const obj = data as Record<string, unknown>;

    // 다양한 응답 형식 처리
    if (obj.data && typeof obj.data === 'object') {
      const dataObj = obj.data as Record<string, unknown>;
      if (typeof dataObj.content === 'string') return dataObj.content;
      if (typeof dataObj.message === 'string') return dataObj.message;
      if (typeof dataObj.response === 'string') return dataObj.response;
    }

    if (typeof obj.content === 'string') return obj.content;
    if (typeof obj.message === 'string') return obj.message;
    if (typeof obj.response === 'string') return obj.response;
  }

  return '요약을 생성할 수 없습니다. 다시 시도해주세요.';
}
