'use client';

import { useState } from 'react';
import { useRouter } from 'next/navigation';
import Link from 'next/link';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Label } from '@/components/ui/label';
import { 
  ArrowLeft, 
  Building2, 
  Save, 
  Check, 
  X,
  User,
  Phone,
  Mail,
  MapPin,
  Users,
  Calendar,
  Briefcase
} from 'lucide-react';

interface CompanyFormData {
  business_number: string;
  company_name: string;
  company_size: string;
  industry_code: string;
  industry_name: string;
  address: string;
  region: string;
  employee_count: string;
  annual_revenue: string;
  establishment_date: string;
  technologies: string[];
  certifications: string[];
  contact_person: string;
  contact_phone: string;
  contact_email: string;
}

interface ValidationResult {
  valid: boolean;
  message: string;
}

export default function NewCompanyPage() {
  const router = useRouter();
  const [loading, setLoading] = useState(false);
  const [validationResults, setValidationResults] = useState<{
    business_number?: ValidationResult;
    contact_email?: ValidationResult;
    contact_phone?: ValidationResult;
  }>({});

  const [formData, setFormData] = useState<CompanyFormData>({
    business_number: '',
    company_name: '',
    company_size: '',
    industry_code: '',
    industry_name: '',
    address: '',
    region: '',
    employee_count: '',
    annual_revenue: '',
    establishment_date: '',
    technologies: [],
    certifications: [],
    contact_person: '',
    contact_phone: '',
    contact_email: ''
  });

  // 입력값 변경 처리
  const handleInputChange = (field: keyof CompanyFormData, value: string) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }));

    // 실시간 유효성 검증
    if (field === 'business_number' && value.length >= 10) {
      validateBusinessNumber(value);
    } else if (field === 'contact_email' && value) {
      validateEmail(value);
    } else if (field === 'contact_phone' && value) {
      validatePhone(value);
    }
  };

  // 배열 필드 처리 (기술, 인증)
  const handleArrayFieldChange = (field: 'technologies' | 'certifications', value: string) => {
    const items = value.split(',').map(item => item.trim()).filter(item => item);
    setFormData(prev => ({
      ...prev,
      [field]: items
    }));
  };

  // 사업자등록번호 유효성 검증
  const validateBusinessNumber = async (businessNumber: string) => {
    try {
      const response = await fetch('/api/companies/validate/business-number', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ business_number: businessNumber })
      });

      if (response.ok) {
        const data = await response.json();
        setValidationResults(prev => ({
          ...prev,
          business_number: data.data
        }));
      }
    } catch (error) {
      console.error('사업자등록번호 검증 오류:', error);
    }
  };

  // 이메일 유효성 검증
  const validateEmail = async (email: string) => {
    try {
      const response = await fetch('/api/companies/validate/contact', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ email })
      });

      if (response.ok) {
        const data = await response.json();
        setValidationResults(prev => ({
          ...prev,
          contact_email: data.data.email
        }));
      }
    } catch (error) {
      console.error('이메일 검증 오류:', error);
    }
  };

  // 전화번호 유효성 검증
  const validatePhone = async (phone: string) => {
    try {
      const response = await fetch('/api/companies/validate/contact', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ phone })
      });

      if (response.ok) {
        const data = await response.json();
        setValidationResults(prev => ({
          ...prev,
          contact_phone: data.data.phone
        }));
      }
    } catch (error) {
      console.error('전화번호 검증 오류:', error);
    }
  };

  // 폼 제출
  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (loading) return;

    // 필수 필드 검증
    if (!formData.business_number.trim()) {
      alert('사업자등록번호를 입력해주세요.');
      return;
    }

    if (!formData.company_name.trim()) {
      alert('회사명을 입력해주세요.');
      return;
    }

    // 사업자등록번호 유효성 검증
    if (!validationResults.business_number?.valid) {
      alert('유효한 사업자등록번호를 입력해주세요.');
      return;
    }

    try {
      setLoading(true);

      // 데이터 정제
      const submitData = {
        ...formData,
        employee_count: formData.employee_count ? parseInt(formData.employee_count) : undefined,
        annual_revenue: formData.annual_revenue ? parseInt(formData.annual_revenue.replace(/,/g, '')) : undefined,
        establishment_date: formData.establishment_date || undefined
      };

      const response = await fetch('/api/companies', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(submitData)
      });

      if (response.ok) {
        const data = await response.json();
        alert('기업이 성공적으로 등록되었습니다.');
        router.push(`/companies/${data.data.id}`);
      } else {
        const errorData = await response.json();
        alert(`등록 실패: ${errorData.message || '알 수 없는 오류'}`);
      }
    } catch (error) {
      console.error('기업 등록 오류:', error);
      alert('기업 등록 중 오류가 발생했습니다.');
    } finally {
      setLoading(false);
    }
  };

  // 유효성 검증 아이콘
  const ValidationIcon = ({ result }: { result?: ValidationResult }) => {
    if (!result) return null;
    
    return result.valid ? (
      <Check className="w-4 h-4 text-green-500" />
    ) : (
      <X className="w-4 h-4 text-red-500" />
    );
  };

  return (
    <div className="container mx-auto p-6 max-w-4xl">
      {/* 헤더 */}
      <div className="flex items-center gap-4 mb-6">
        <Link href="/companies">
          <Button variant="ghost" size="sm">
            <ArrowLeft className="w-4 h-4" />
            목록으로
          </Button>
        </Link>
        <div>
          <h1 className="text-3xl font-bold">신규 기업 등록</h1>
          <p className="text-gray-600 mt-2">새로운 기업 정보를 등록합니다.</p>
        </div>
      </div>

      <form onSubmit={handleSubmit} className="space-y-6">
        {/* 기본 정보 */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Building2 className="w-5 h-5" />
              기본 정보
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <Label htmlFor="business_number">사업자등록번호 *</Label>
                <div className="flex items-center gap-2">
                  <Input
                    id="business_number"
                    placeholder="123-45-67890"
                    value={formData.business_number}
                    onChange={(e) => handleInputChange('business_number', e.target.value)}
                    required
                  />
                  <ValidationIcon result={validationResults.business_number} />
                </div>
                {validationResults.business_number && (
                  <p className={`text-xs mt-1 ${validationResults.business_number.valid ? 'text-green-600' : 'text-red-600'}`}>
                    {validationResults.business_number.message}
                  </p>
                )}
              </div>

              <div>
                <Label htmlFor="company_name">회사명 *</Label>
                <Input
                  id="company_name"
                  placeholder="회사명을 입력하세요"
                  value={formData.company_name}
                  onChange={(e) => handleInputChange('company_name', e.target.value)}
                  required
                />
              </div>

              <div>
                <Label htmlFor="company_size">기업규모</Label>
                <Select
                  value={formData.company_size}
                  onValueChange={(value) => handleInputChange('company_size', value)}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="기업규모 선택" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="대기업">대기업</SelectItem>
                    <SelectItem value="중견기업">중견기업</SelectItem>
                    <SelectItem value="중소기업">중소기업</SelectItem>
                    <SelectItem value="스타트업">스타트업</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label htmlFor="industry_name">업종명</Label>
                <Input
                  id="industry_name"
                  placeholder="소프트웨어 개발업"
                  value={formData.industry_name}
                  onChange={(e) => handleInputChange('industry_name', e.target.value)}
                />
              </div>

              <div className="md:col-span-2">
                <Label htmlFor="address">주소</Label>
                <Input
                  id="address"
                  placeholder="전체 주소를 입력하세요"
                  value={formData.address}
                  onChange={(e) => handleInputChange('address', e.target.value)}
                />
              </div>

              <div>
                <Label htmlFor="region">지역</Label>
                <Input
                  id="region"
                  placeholder="서울, 경기, 부산 등"
                  value={formData.region}
                  onChange={(e) => handleInputChange('region', e.target.value)}
                />
              </div>

              <div>
                <Label htmlFor="establishment_date">설립일</Label>
                <Input
                  id="establishment_date"
                  type="date"
                  value={formData.establishment_date}
                  onChange={(e) => handleInputChange('establishment_date', e.target.value)}
                />
              </div>
            </div>
          </CardContent>
        </Card>

        {/* 기업 현황 */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Briefcase className="w-5 h-5" />
              기업 현황
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <Label htmlFor="employee_count">직원 수</Label>
                <div className="flex items-center">
                  <Input
                    id="employee_count"
                    type="number"
                    placeholder="50"
                    value={formData.employee_count}
                    onChange={(e) => handleInputChange('employee_count', e.target.value)}
                  />
                  <span className="ml-2 text-sm text-gray-500">명</span>
                </div>
              </div>

              <div>
                <Label htmlFor="annual_revenue">연매출</Label>
                <div className="flex items-center">
                  <Input
                    id="annual_revenue"
                    placeholder="1,000,000,000"
                    value={formData.annual_revenue}
                    onChange={(e) => {
                      // 숫자만 허용하고 콤마 형식화
                      const value = e.target.value.replace(/[^0-9]/g, '');
                      const formatted = value ? parseInt(value).toLocaleString() : '';
                      handleInputChange('annual_revenue', formatted);
                    }}
                  />
                  <span className="ml-2 text-sm text-gray-500">원</span>
                </div>
              </div>

              <div className="md:col-span-2">
                <Label htmlFor="technologies">보유 기술</Label>
                <Input
                  id="technologies"
                  placeholder="React, Node.js, Python, AI (쉼표로 구분)"
                  value={formData.technologies.join(', ')}
                  onChange={(e) => handleArrayFieldChange('technologies', e.target.value)}
                />
                <p className="text-xs text-gray-500 mt-1">기술명을 쉼표(,)로 구분하여 입력하세요.</p>
              </div>

              <div className="md:col-span-2">
                <Label htmlFor="certifications">보유 인증</Label>
                <Input
                  id="certifications"
                  placeholder="ISO27001, CMMI Level 3 (쉼표로 구분)"
                  value={formData.certifications.join(', ')}
                  onChange={(e) => handleArrayFieldChange('certifications', e.target.value)}
                />
                <p className="text-xs text-gray-500 mt-1">인증명을 쉼표(,)로 구분하여 입력하세요.</p>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* 담당자 정보 */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <User className="w-5 h-5" />
              담당자 정보
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              <div>
                <Label htmlFor="contact_person">담당자명</Label>
                <Input
                  id="contact_person"
                  placeholder="홍길동"
                  value={formData.contact_person}
                  onChange={(e) => handleInputChange('contact_person', e.target.value)}
                />
              </div>

              <div>
                <Label htmlFor="contact_phone">연락처</Label>
                <div className="flex items-center gap-2">
                  <Input
                    id="contact_phone"
                    placeholder="02-123-4567"
                    value={formData.contact_phone}
                    onChange={(e) => handleInputChange('contact_phone', e.target.value)}
                  />
                  <ValidationIcon result={validationResults.contact_phone} />
                </div>
                {validationResults.contact_phone && (
                  <p className={`text-xs mt-1 ${validationResults.contact_phone.valid ? 'text-green-600' : 'text-red-600'}`}>
                    {validationResults.contact_phone.message}
                  </p>
                )}
              </div>

              <div>
                <Label htmlFor="contact_email">이메일</Label>
                <div className="flex items-center gap-2">
                  <Input
                    id="contact_email"
                    type="email"
                    placeholder="contact@company.com"
                    value={formData.contact_email}
                    onChange={(e) => handleInputChange('contact_email', e.target.value)}
                  />
                  <ValidationIcon result={validationResults.contact_email} />
                </div>
                {validationResults.contact_email && (
                  <p className={`text-xs mt-1 ${validationResults.contact_email.valid ? 'text-green-600' : 'text-red-600'}`}>
                    {validationResults.contact_email.message}
                  </p>
                )}
              </div>
            </div>
          </CardContent>
        </Card>

        {/* 제출 버튼 */}
        <div className="flex justify-end gap-4">
          <Link href="/companies">
            <Button type="button" variant="outline">
              취소
            </Button>
          </Link>
          <Button type="submit" disabled={loading} className="flex items-center gap-2">
            {loading ? (
              <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white"></div>
            ) : (
              <Save className="w-4 h-4" />
            )}
            {loading ? '등록 중...' : '기업 등록'}
          </Button>
        </div>
      </form>
    </div>
  );
}