'use client'

import Link from 'next/link'

import { useEffect, useState, Suspense } from 'react'
import { useRouter, useSearchParams } from 'next/navigation'
import { Metadata } from 'next'



interface User {
  id: string
  username: string
  email: string
  role: string
}

// 핵심 가치 제안
const coreValues = [
  {
    icon: '🏢',
    title: 'Enterprise-Grade 보안',
    description: '기업 데이터를 안전하게 보호하는 엔터프라이즈급 보안 시스템'
  },
  {
    icon: '🔗',
    title: '통합 AI 플랫폼',
    description: 'RAG, 웹검색, 리포트, 채팅을 하나로 연결한 완전한 AI 솔루션'
  },
  {
    icon: '⚙️',
    title: '맞춤형 커스터마이징',
    description: '각 기업의 워크플로우에 최적화된 AI 오케스트레이션 구현'
  }
]

// 주요 기능
const mainFeatures = [
  {
    icon: '🧠',
    title: '지능형 문서 검색',
    description: 'RAG 기반 하이브리드 검색으로 기업 문서에서 정확한 정보 추출',
    features: ['4-Score 평가 시스템', '의미 기반 검색', '한국어 최적화'],
    color: 'from-blue-500 to-indigo-600',
    gradient: 'bg-gradient-to-br from-blue-50 to-indigo-50 dark:from-blue-900/20 dark:to-indigo-900/20'
  },
  {
    icon: '🌐',
    title: '실시간 웹 검색',
    description: 'AI가 실시간 웹 정보를 수집하여 최신 동향과 데이터 제공',
    features: ['실시간 정보 수집', '신뢰성 검증', '자동 요약'],
    color: 'from-green-500 to-teal-600',
    gradient: 'bg-gradient-to-br from-green-50 to-teal-50 dark:from-green-900/20 dark:to-teal-900/20'
  },
  {
    icon: '📊',
    title: 'AI 보고서 생성',
    description: '수집된 정보를 바탕으로 전문적인 보고서를 자동 생성',
    features: ['다양한 템플릿', 'PDF/Word 출력', '차트 자동 생성'],
    color: 'from-purple-500 to-pink-600',
    gradient: 'bg-gradient-to-br from-purple-50 to-pink-50 dark:from-purple-900/20 dark:to-pink-900/20'
  },
  {
    icon: '💬',
    title: '대화형 AI 어시스턴트',
    description: '다중 AI 모델을 활용한 지능적인 대화형 업무 지원',
    features: ['멀티모달 AI', '컨텍스트 유지', '업무 자동화'],
    color: 'from-orange-500 to-red-600',
    gradient: 'bg-gradient-to-br from-orange-50 to-red-50 dark:from-orange-900/20 dark:to-red-900/20'
  }
]

// 특장점
const advantages = [
  {
    icon: '⚡',
    title: '하이브리드 검색 시스템',
    description: '키워드와 의미 검색을 결합한 혁신적인 정보 검색 기술로 95% 이상의 정확도 달성'
  },
  {
    icon: '🛡️',
    title: '온프레미스 배포',
    description: '클라우드 의존성 없이 기업 내부에서 완전히 독립적으로 운영되는 소버린 AI'
  },
  {
    icon: '🔄',
    title: 'API 기반 확장성',
    description: '기존 시스템과의 완벽한 연동을 위한 RESTful API와 유연한 아키텍처 제공'
  }
]

function HomePageContent() {
  const router = useRouter()
  const searchParams = useSearchParams()
  const [user, setUser] = useState<User | null>(null)
  const [isLoading, setIsLoading] = useState(false)
  
  // URL에서 슬로건 파라미터 가져오기
  const slogan = searchParams?.get('slogan') || 'Build Your Sovereign AI - Private & Secure'

  // 페이지 타이틀에 슬로건 설정
  useEffect(() => {
    document.title = slogan
  }, [slogan])

  useEffect(() => {
    // 로컬 스토리지에서 사용자 정보 확인
    const userId = localStorage.getItem('userId')
    const username = localStorage.getItem('username')
    const userEmail = localStorage.getItem('userEmail')
    const userRole = localStorage.getItem('userRole')

    if (userId && username) {
      setUser({
        id: userId,
        username,
        email: userEmail || '',
        role: userRole || 'user'
      })
    }
  }, [])

  const handleNavigation = (path: string) => {
    router.push(path)
  }

  const handleDemoLogin = async () => {
    setIsLoading(true)
    
    try {
      const response = await fetch('/api/auth/login', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ username: 'demo', password: 'demo' }),
      })
      
      const data = await response.json()
      
      if (data.success) {
        // 로컬 스토리지에 사용자 정보와 토큰 저장
        if (data.data.token) {
          localStorage.setItem('userToken', data.data.token)
        }
        
        // API 키 저장
        if (data.data.apiKeys && Array.isArray(data.data.apiKeys) && data.data.apiKeys.length > 0) {
          const apiKey = data.data.apiKeys[0].key
          localStorage.setItem('apiKey', apiKey)
        }
        
        if (data.data.user) {
          localStorage.setItem('userId', data.data.user.id)
          localStorage.setItem('username', data.data.user.username)
          localStorage.setItem('userEmail', data.data.user.email || '')
          localStorage.setItem('userRole', data.data.user.role || 'user')
        }
        
        // 세션 스토리지에도 정보 저장
        sessionStorage.setItem('isLoggedIn', 'true')
        sessionStorage.setItem('username', localStorage.getItem('username') || '')
        
        // 로그인 이벤트 발생
        window.dispatchEvent(new Event('login'))
        
        // 페이지 새로고침하여 상태 업데이트
        window.location.reload()
      } else {
        throw new Error('체험 모드 시작에 실패했습니다.')
      }
    } catch (err) {
      console.error('Demo login error:', err)
      alert('체험 모드 시작에 실패했습니다. 다시 시도해주세요.')
    } finally {
      setIsLoading(false)
    }
  }

  return (
    <div className="min-h-screen" style={{ backgroundColor: 'var(--background)' }}>
      {/* URL Slogan Banner */}
      <div className="w-full py-3 px-4 text-center" style={{ backgroundColor: '#1e40af' }}>
        <span className="text-white font-semibold text-lg">{slogan}</span>
      </div>
      {/* Hero Section */}
      <div className="relative overflow-hidden py-20 px-4">
        <div className="absolute inset-0 bg-gradient-to-br from-blue-50 via-indigo-50 to-purple-50 dark:from-gray-900 dark:via-blue-950 dark:to-indigo-950"></div>
        <div className="relative z-10 max-w-7xl mx-auto text-center">
          {/* 사용자 환영 메시지 */}
          {user && (
            <div className="mb-8">
              <div className="inline-flex items-center px-6 py-3 rounded-full" style={{ backgroundColor: 'var(--card-bg)' }}>
                <span className="text-lg text-gray-700 dark:text-gray-300">
                  안녕하세요, <span className="font-semibold text-blue-600 dark:text-blue-400">{user.username}</span>님! 👋
                </span>
              </div>
            </div>
          )}
          
          <h2 className="text-3xl md:text-4xl font-bold text-transparent bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text mb-8 leading-tight">
            소버린 AI 오케스트레이션 플랫폼
          </h2>
          <p className="text-xl md:text-2xl text-gray-600 dark:text-gray-300 mb-12 max-w-4xl mx-auto leading-relaxed">
            기업 데이터를 안전하게 보호하면서 AI의 모든 가능성을 실현하는
            <span className="font-semibold text-blue-600 dark:text-blue-400"> 완전한 AI 솔루션</span>
          </p>
          
          {!user && (
            <div className="flex flex-col sm:flex-row gap-6 justify-center">
              <button
                onClick={handleDemoLogin}
                disabled={isLoading}
                className="group px-12 py-4 rounded-xl font-semibold text-lg transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-xl disabled:opacity-50 disabled:cursor-not-allowed min-w-[200px]"
                style={{ backgroundColor: 'var(--accent-blue)', color: 'white' }}
              >
                {isLoading ? (
                  <div className="flex items-center justify-center whitespace-nowrap">
                    <div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin mr-3"></div>
                    시작 중...
                  </div>
                ) : (
                  <div className="flex items-center justify-center whitespace-nowrap">
                    <span className="text-2xl mr-3">🚀</span>
                    무료 체험 시작
                    <svg className="w-5 h-5 ml-3 transform group-hover:translate-x-1 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
                    </svg>
                  </div>
                )}
              </button>
              
              <button
                onClick={() => handleNavigation('/support')}
                className="group px-12 py-4 rounded-xl font-semibold text-lg transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-xl min-w-[200px]"
                style={{ backgroundColor: 'var(--accent-green)', color: 'white' }}
              >
                <div className="flex items-center justify-center whitespace-nowrap">
                  <span className="text-2xl mr-3">💬</span>
                  도입 문의
                  <svg className="w-5 h-5 ml-3 transform group-hover:translate-x-1 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
                  </svg>
                </div>
              </button>
            </div>
          )}
        </div>
      </div>

      {/* 핵심 가치 제안 섹션 */}
      <div className="py-20 px-4" style={{ backgroundColor: 'var(--card-bg)' }}>
        <div className="max-w-7xl mx-auto">
          
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            {coreValues.map((value, index) => (
              <div key={index} className="text-center p-8 rounded-2xl" style={{ backgroundColor: 'var(--background)' }}>
                <div className="text-6xl mb-6">{value.icon}</div>
                <h3 className="text-2xl font-bold text-gray-900 dark:text-white mb-4">
                  {value.title}
                </h3>
                <p className="text-gray-600 dark:text-gray-300 leading-relaxed">
                  {value.description}
                </p>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* 주요 기능 섹션 */}
      <div className="py-20 px-4">
        <div className="max-w-7xl mx-auto">
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
            {mainFeatures.map((feature, index) => (
              <div
                key={index}
                className={`group relative overflow-hidden rounded-2xl ${feature.gradient} border border-gray-200 dark:border-gray-700 shadow-lg hover:shadow-2xl transition-all duration-300 cursor-pointer transform hover:scale-105`}
                onClick={() => user ? handleNavigation(index === 0 ? '/admin/rag' : index === 2 ? '/document' : '/chat') : handleDemoLogin()}
              >
                <div className="absolute inset-0 bg-gradient-to-br from-white/20 to-transparent dark:from-white/5"></div>
                <div className="relative p-8">
                  <div className="flex items-center mb-6">
                    <div className="text-3xl mr-6">{feature.icon}</div>
                    <h3 className="text-2xl font-bold text-gray-900 dark:text-white">
                      {feature.title}
                    </h3>
                  </div>
                  
                  <p className="text-lg text-gray-700 dark:text-gray-200 mb-8 leading-relaxed">
                    {feature.description}
                  </p>
                  
                  <div className="grid grid-cols-1 gap-3 mb-8">
                    {feature.features.map((item, idx) => (
                      <div key={idx} className="flex items-center text-gray-600 dark:text-gray-300">
                        <div className="w-2 h-2 bg-current rounded-full mr-4 opacity-70"></div>
                        <span className="font-medium">{item}</span>
                      </div>
                    ))}
                  </div>
                  
                  <div className="flex items-center text-blue-600 dark:text-blue-400 font-semibold group-hover:text-blue-700 dark:group-hover:text-blue-300 transition-colors text-lg">
                    {user ? '사용하기' : '체험하기'}
                    <svg className="w-5 h-5 ml-2 transform group-hover:translate-x-1 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
                    </svg>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* 특장점 섹션 */}
      <div className="py-20 px-4" style={{ backgroundColor: 'var(--card-bg)' }}>
        <div className="max-w-7xl mx-auto">
          <div className="text-center mb-16">
            <h2 className="text-4xl font-bold text-gray-900 dark:text-white mb-6">
              기술적 차별점
            </h2>
            <p className="text-xl text-gray-600 dark:text-gray-300 max-w-3xl mx-auto">
              검증된 기술력과 엔터프라이즈급 성능을 제공합니다
            </p>
          </div>
          
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
            {advantages.map((advantage, index) => (
              <div key={index} className="p-8 rounded-2xl" style={{ backgroundColor: 'var(--background)' }}>
                <div className="text-5xl mb-6">{advantage.icon}</div>
                <h3 className="text-2xl font-bold text-gray-900 dark:text-white mb-4">
                  {advantage.title}
                </h3>
                <p className="text-gray-600 dark:text-gray-300 leading-relaxed text-lg">
                  {advantage.description}
                </p>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* CTA 섹션 */}
      {!user && (
        <div className="py-20 px-4">
          <div className="max-w-4xl mx-auto text-center">
            <div className="p-12 rounded-3xl" style={{ backgroundColor: 'var(--card-bg)' }}>
              <h2 className="text-4xl font-bold text-gray-900 dark:text-white mb-6">
                소버린 AI로 시작하세요
              </h2>
              <p className="text-xl text-gray-600 dark:text-gray-300 mb-10 max-w-3xl mx-auto leading-relaxed">
                데이터 보안을 유지하면서 AI의 모든 가능성을 실현해보세요
              </p>
              
              <div className="max-w-md mx-auto p-6 rounded-2xl border-2 border-dashed border-blue-300 dark:border-blue-600 mb-8" style={{ backgroundColor: 'var(--background)' }}>
                <div className="text-center">
                  <div className="text-lg font-semibold text-blue-600 dark:text-blue-400 mb-4">
                    🎯 Demo 계정으로 체험하기
                  </div>
                  <div className="space-y-2 text-gray-700 dark:text-gray-300">
                    <div className="flex justify-between items-center">
                      <span>아이디:</span>
                      <span className="font-mono font-bold text-yellow-600 dark:text-yellow-400">demo</span>
                    </div>
                    <div className="flex justify-between items-center">
                      <span>비밀번호:</span>
                      <span className="font-mono font-bold text-yellow-600 dark:text-yellow-400">demo</span>
                    </div>
                  </div>
                </div>
              </div>
              
              <div className="flex flex-col sm:flex-row gap-6 justify-center">
                <button
                  onClick={handleDemoLogin}
                  disabled={isLoading}
                  className="group px-12 py-4 rounded-xl font-semibold text-lg transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-xl disabled:opacity-50 disabled:cursor-not-allowed min-w-[200px]"
                  style={{ backgroundColor: 'var(--accent-blue)', color: 'white' }}
                >
                  {isLoading ? (
                    <div className="flex items-center justify-center whitespace-nowrap">
                      <div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin mr-3"></div>
                      시작 중...
                    </div>
                  ) : (
                    <div className="flex items-center justify-center whitespace-nowrap">
                      <span className="text-2xl mr-3">🚀</span>
                      무료 체험 시작
                      <svg className="w-5 h-5 ml-3 transform group-hover:translate-x-1 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
                      </svg>
                    </div>
                  )}
                </button>
                
                <button
                  onClick={() => handleNavigation('/support')}
                  className="group px-12 py-4 rounded-xl font-semibold text-lg transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-xl min-w-[200px]"
                  style={{ backgroundColor: 'var(--accent-green)', color: 'white' }}
                >
                  <div className="flex items-center justify-center whitespace-nowrap">
                    <span className="text-2xl mr-3">💬</span>
                    도입 문의하기
                    <svg className="w-5 h-5 ml-3 transform group-hover:translate-x-1 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
                    </svg>
                  </div>
                </button>
              </div>
              
              <p className="text-gray-500 dark:text-gray-400 mt-6">
                전문 상담원이 맞춤형 도입 방안을 제안해드립니다
              </p>
            </div>
          </div>
        </div>
      )}

      {/* Demo 사용자를 위한 안내 */}
      {user?.username === 'demo' && (
        <div className="py-20 px-4">
          <div className="max-w-4xl mx-auto text-center">
            <div className="p-12 rounded-3xl" style={{ backgroundColor: 'var(--card-bg)' }}>
              <div className="text-6xl mb-6">🎉</div>
              <h2 className="text-4xl font-bold text-gray-900 dark:text-white mb-6">
                체험 모드로 이용 중입니다
              </h2>
              <p className="text-xl text-gray-600 dark:text-gray-300 mb-10 max-w-3xl mx-auto leading-relaxed">
                모든 기능을 자유롭게 체험해보시고, 도입을 원하시면 언제든 문의해주세요!
              </p>
              
              <div className="flex flex-col sm:flex-row gap-6 justify-center">
                <button
                  onClick={() => handleNavigation('/support')}
                  className="group px-12 py-4 rounded-xl font-semibold text-lg transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-xl min-w-[200px]"
                  style={{ backgroundColor: 'var(--accent-green)', color: 'white' }}
                >
                  <div className="flex items-center justify-center whitespace-nowrap">
                    <span className="text-2xl mr-3">💬</span>
                    도입 문의하기
                    <svg className="w-5 h-5 ml-3 transform group-hover:translate-x-1 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
                    </svg>
                  </div>
                </button>
                
                <button
                  onClick={() => window.open('mailto:sales@invesume.com', '_blank')}
                  className="group px-12 py-4 rounded-xl font-semibold text-lg border-2 transition-all duration-300 transform hover:scale-105 shadow-lg hover:shadow-xl min-w-[200px]"
                  style={{ 
                    borderColor: 'var(--accent-green)', 
                    backgroundColor: 'var(--background)',
                    color: 'var(--accent-green)'
                  }}
                  onMouseEnter={(e) => {
                    e.currentTarget.style.backgroundColor = 'var(--accent-green)'
                    e.currentTarget.style.color = 'white'
                  }}
                  onMouseLeave={(e) => {
                    e.currentTarget.style.backgroundColor = 'var(--background)'
                    e.currentTarget.style.color = 'var(--accent-green)'
                  }}
                >
                  <div className="flex items-center justify-center whitespace-nowrap">
                    <span className="text-2xl mr-3">📧</span>
                    이메일 상담
                    <svg className="w-5 h-5 ml-3 transform group-hover:translate-x-1 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10 6H6a2 2 0 00-2 2v10a2 2 0 002 2h10a2 2 0 002-2v-2M14 4h6m0 0v6m0-6L10 14" />
                    </svg>
                  </div>
                </button>
              </div>
            </div>
          </div>
        </div>
      )}

    </div>
  )
}

export default function HomePage() {
  return (
    <Suspense fallback={<div>Loading...</div>}>
      <HomePageContent />
    </Suspense>
  )
}
