'use client';

import { useState, useEffect } from 'react';
import { useRouter } from 'next/navigation';
import Link from 'next/link';
import { API_SERVER, getAuthHeaders, apiRequest } from '@/utils/api';
import { internalApiClient, externalApiClient } from '@/lib/apiClient';

interface ApiKey {
  id: number;
  key: string;
  name: string;
  status: string;
  createdAt: string;
}

interface ContextMemory {
  id?: number;
  user_id?: string;
  session_id?: string;
  type: string;
  content: string;
  summary?: string;
  entities?: any[];
  keywords?: string[];
  importance: number;
  confidence: number;
  source_context?: string;
  reference_count?: number;
  createdAt?: string;
  updatedAt: string;
}

interface SessionSummary {
  id?: number;
  user_id?: string;
  sessionId: string;
  summary: string;
  keyTopics?: string[];
  mentionedEntities?: string[];
  messageCount: number;
  importance: number;
  createdAt: string;
  updatedAt?: string;
}

interface ContextStats {
  totalMemories: number;
  totalSessions: number;
  memoryTypes: { [key: string]: number };
  entityCount: number;
  averageImportance: number;
}

interface UserProfile {
  id: number;
  username: string;
  email: string;
  name: string;
  role: string;
  status: string;
  created_at?: string;
  last_login?: string;
  apiKeys?: ApiKey[];
  profileImageUrl?: string;
  language?: string;
}

export default function ProfilePage() {
  const [user, setUser] = useState<UserProfile | null>(null);
  const [name, setName] = useState('');
  const [email, setEmail] = useState('');
  const [password, setPassword] = useState('');
  const [language, setLanguage] = useState('ko');
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [success, setSuccess] = useState<string | null>(null);
  const [error, setError] = useState<string | null>(null);
  const [copied, setCopied] = useState(false);
  
  // ContextAI 메모리 관련 상태
  const [contextMemories, setContextMemories] = useState<ContextMemory[]>([]);
  const [sessionSummaries, setSessionSummaries] = useState<SessionSummary[]>([]);
  const [contextStats, setContextStats] = useState<ContextStats | null>(null);
  const [memoriesLoading, setMemoriesLoading] = useState(false);
  const [selectedMemoryType, setSelectedMemoryType] = useState<string>('all');
  const [activeTab, setActiveTab] = useState<'memories' | 'sessions'>('memories');
  const [expandedMemory, setExpandedMemory] = useState<number | null>(null);
  
  // 프로필 이미지 관련 상태
  const [profileImage, setProfileImage] = useState<string>('/images/user.png');
  const [uploadingImage, setUploadingImage] = useState(false);
  const [isDragOver, setIsDragOver] = useState(false);
  
  // 모달 상태
  const [showConfirmModal, setShowConfirmModal] = useState(false);
  const [showDeleteAllConfirmModal, setShowDeleteAllConfirmModal] = useState(false);
  const [memoryToDelete, setMemoryToDelete] = useState<ContextMemory | null>(null);

  // API 키 삭제 모달 상태
  const [showApiKeyDeleteModal, setShowApiKeyDeleteModal] = useState(false);
  const [apiKeyToDelete, setApiKeyToDelete] = useState<{ id: number; name: string } | null>(null);
  const [deletingApiKey, setDeletingApiKey] = useState(false);
  
  const router = useRouter();

  // 사용자 정보 불러오기
  useEffect(() => {
    async function fetchProfile() {
      setLoading(true);
      setError(null);
      try {
        const res = await externalApiClient.get('/auth/me');
        if (res.success && res.data) {
          setUser(res.data);
          setName(res.data.name || '');
          setEmail(res.data.email || '');
          setPassword('');
          setLanguage(res.data.language || 'ko');
          // ContextAI 메모리와 통계 불러오기
          fetchContextData();
          // 프로필 이미지 불러오기
          fetchProfileImage();
        } else {
          setError('사용자 정보를 불러올 수 없습니다.');
        }
      } catch (e) {
        setError('사용자 정보를 불러올 수 없습니다.');
      } finally {
        setLoading(false);
      }
    }
    fetchProfile();
  }, []);

  // 프로필 이미지 불러오기
  const fetchProfileImage = async () => {
    try {
      const userId = localStorage.getItem('userId');
      if (!userId) return;

      const response = await fetch(`/api/profile/image?userId=${userId}`, {
        method: 'GET',
        credentials: 'include'
      });

      if (response.ok) {
        const data = await response.json();
        if (data.success && data.data.imageUrl) {
          setProfileImage(data.data.imageUrl);
        }
      }
    } catch (error) {
      console.error('프로필 이미지 불러오기 실패:', error);
    }
  };

  // 프로필 이미지 업로드
  const handleImageUpload = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (!file) return;
    
    uploadImageFile(file);
    
    // 파일 입력 초기화
    event.target.value = '';
  };

  // 프로필 이미지 삭제
  const handleImageDelete = async () => {
    try {
      const userId = localStorage.getItem('userId');
      if (!userId) {
        setError('로그인이 필요합니다.');
        return;
      }

      const response = await fetch(`/api/profile/image?userId=${userId}`, {
        method: 'DELETE',
        credentials: 'include'
      });

      const data = await response.json();
      
      if (response.ok && data.success) {
        setProfileImage('/images/user.png');
        setSuccess('프로필 이미지가 삭제되었습니다.');
        window.dispatchEvent(new Event('profileImageUpdated'));
        setTimeout(() => setSuccess(null), 3000);
      } else {
        setError(data.error?.message || '이미지 삭제에 실패했습니다.');
      }
    } catch (error) {
      console.error('이미지 삭제 오류:', error);
      setError('이미지 삭제 중 오류가 발생했습니다.');
    }
  };

  // 드래그앤드롭 이벤트 핸들러
  const handleDragEnter = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragOver(true);
  };

  const handleDragLeave = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    if (e.currentTarget === e.target) {
      setIsDragOver(false);
    }
  };

  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
  };

  const handleDrop = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragOver(false);

    const files = Array.from(e.dataTransfer.files);
    const imageFile = files.find(file => file.type.startsWith('image/'));

    if (imageFile) {
      uploadImageFile(imageFile);
    } else {
      setError('이미지 파일만 업로드 가능합니다.');
    }
  };

  // 공통 이미지 업로드 함수
  const uploadImageFile = async (file: File) => {
    // 파일 크기 체크 (5MB)
    if (file.size > 5 * 1024 * 1024) {
      setError('이미지 파일은 5MB 이하여야 합니다.');
      return;
    }

    // 이미지 파일 타입 체크
    if (!file.type.startsWith('image/')) {
      setError('올바른 이미지 파일을 선택해주세요.');
      return;
    }

    const userId = localStorage.getItem('userId');
    if (!userId) {
      setError('로그인이 필요합니다.');
      return;
    }

    setUploadingImage(true);
    setError(null);

    try {
      const formData = new FormData();
      formData.append('image', file);
      formData.append('userId', userId);

      const response = await fetch('/api/profile/image', {
        method: 'POST',
        body: formData,
        credentials: 'include'
      });

      const data = await response.json();
      
      if (response.ok && data.success) {
        setProfileImage(data.data.imageUrl);
        setSuccess('프로필 이미지가 성공적으로 업로드되었습니다.');
        window.dispatchEvent(new Event('profileImageUpdated'));
        setTimeout(() => setSuccess(null), 3000);
      } else {
        setError(data.error?.message || '이미지 업로드에 실패했습니다.');
      }
    } catch (error) {
      console.error('이미지 업로드 오류:', error);
      setError('이미지 업로드 중 오류가 발생했습니다.');
    } finally {
      setUploadingImage(false);
    }
  };

  // ContextAI 데이터 불러오기
  const fetchContextData = async () => {
    setMemoriesLoading(true);
    try {
      // 컨텍스트 메모리, 세션 요약, 통계를 병렬로 불러오기
      const [memoriesRes, sessionsRes, statsRes] = await Promise.all([
        externalApiClient.get('/context/memories'),
        externalApiClient.get('/context/sessions'),
        externalApiClient.get('/context/stats')
      ]);

      if (memoriesRes.success) {
        const memoriesData = memoriesRes.data.memories || [];
        // 선택된 메모리 타입으로 필터링
        const filteredMemories = selectedMemoryType === 'all'
          ? memoriesData
          : memoriesData.filter((m: ContextMemory) => m.type === selectedMemoryType);

        // 중요도와 날짜순으로 정렬
        filteredMemories.sort((a: ContextMemory, b: ContextMemory) => {
          const importanceDiff = (b.importance || 0) - (a.importance || 0);
          if (importanceDiff !== 0) return importanceDiff;
          return new Date(b.updatedAt).getTime() - new Date(a.updatedAt).getTime();
        });

        setContextMemories(filteredMemories);
      }

      if (sessionsRes.success) {
        const sessionsData = sessionsRes.data.sessions || [];
        // 날짜순으로 정렬
        sessionsData.sort((a: SessionSummary, b: SessionSummary) => {
          return new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime();
        });
        setSessionSummaries(sessionsData);
      }

      if (statsRes.success) {
        setContextStats(statsRes.data);
      }
    } catch (error) {
      console.error('ContextAI 데이터 로딩 실패:', error);
      setError('메모리 데이터를 불러올 수 없습니다.');
    } finally {
      setMemoriesLoading(false);
    }
  };

  // 메모리 타입 변경 시 데이터 다시 불러오기
  useEffect(() => {
    if (user) {
      fetchContextData();
    }
  }, [selectedMemoryType]);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setSaving(true);
    setError(null);
    setSuccess(null);
    try {
      const patch: any = { name, email, language };
      if (password) patch.password = password;
      const res = await externalApiClient.put('/auth/me', patch);
      if (res.success) {
        setSuccess('프로필이 성공적으로 저장되었습니다.');
        setPassword('');
        setUser((prev) => prev ? { ...prev, name, email, language } : prev);
        
        // 언어 변경 이벤트 발생
        window.dispatchEvent(new Event('languageChanged'));
      } else {
        setError(res.error?.message || '저장에 실패했습니다.');
      }
    } catch (e: any) {
      setError(e.message || '저장에 실패했습니다.');
    } finally {
      setSaving(false);
    }
  };

  const handleCopy = (key: string) => {
    navigator.clipboard.writeText(key);
    setCopied(true);
    setTimeout(() => setCopied(false), 1200);
  };

  // 개별 메모리 삭제 확인
  const handleDeleteMemory = (memory: ContextMemory) => {
    setMemoryToDelete(memory);
    setShowConfirmModal(true);
  };

  // 단기 메모리 초기화 확인
  const handleClearShortTermMemory = () => {
    setShowDeleteAllConfirmModal(true);
  };

  // 개별 메모리 삭제 실행
  const confirmDeleteMemory = async () => {
    if (!memoryToDelete) return;

    try {
      // 메모리 타입과 내용으로 삭제 요청
      const res = await externalApiClient.delete(`/context/memories/${memoryToDelete.type}/${encodeURIComponent(memoryToDelete.content)}`);
      if (res.success) {
        await fetchContextData();
        setSuccess('메모리가 삭제되었습니다.');
        setTimeout(() => setSuccess(null), 3000);
      } else {
        setError('메모리 삭제에 실패했습니다.');
      }
    } catch (error) {
      console.error('메모리 삭제 오류:', error);
      setError('메모리 삭제 중 오류가 발생했습니다.');
    } finally {
      setShowConfirmModal(false);
      setMemoryToDelete(null);
    }
  };

  // 단기 메모리 초기화 실행
  const confirmDeleteAllMemories = async () => {
    try {
      console.log('단기 메모리 초기화 API 호출...');
      const res = await externalApiClient.delete('/context/memories/short-term');
      console.log('API 응답:', res);
      if (res.success) {
        console.log('단기 메모리 초기화 성공, 데이터 다시 불러오는 중...');
        await fetchContextData();
        console.log('데이터 새로고침 완료');
        setSuccess('단기 메모리가 초기화되었습니다.');
        setTimeout(() => setSuccess(null), 3000);
      } else {
        console.error('단기 메모리 초기화 실패:', res.error);
        setError('단기 메모리 초기화에 실패했습니다.');
      }
    } catch (error) {
      console.error('단기 메모리 초기화 오류:', error);
      setError('단기 메모리 초기화 중 오류가 발생했습니다.');
    } finally {
      setShowDeleteAllConfirmModal(false);
    }
  };

  // 장기 메모리 전체 초기화
  const handleClearLongTermMemory = async () => {
    try {
      const res = await externalApiClient.delete('/context/memories/long-term');
      if (res.success) {
        await fetchContextData();
        setSuccess('장기 메모리가 초기화되었습니다.');
        setTimeout(() => setSuccess(null), 3000);
      } else {
        setError('장기 메모리 초기화에 실패했습니다.');
      }
    } catch (error) {
      console.error('장기 메모리 초기화 오류:', error);
      setError('장기 메모리 초기화 중 오류가 발생했습니다.');
    }
  };

  // API 키 삭제 확인 모달 열기
  const handleDeleteApiKey = (apiKey: ApiKey) => {
    setApiKeyToDelete({ id: apiKey.id, name: apiKey.name });
    setShowApiKeyDeleteModal(true);
  };

  // API 키 삭제 실행
  const confirmDeleteApiKey = async () => {
    if (!apiKeyToDelete) return;

    setDeletingApiKey(true);
    setError(null);

    try {
      const response = await fetch(`/api/profile/api-keys/${apiKeyToDelete.id}`, {
        method: 'DELETE',
        credentials: 'include'
      });

      const data = await response.json();

      if (response.ok && data.success) {
        setSuccess('API 키가 성공적으로 삭제되었습니다.');
        setTimeout(() => setSuccess(null), 3000);

        // 사용자 정보 다시 불러오기
        const res = await externalApiClient.get('/auth/me');
        if (res.success && res.data) {
          setUser(res.data);
        }
      } else {
        setError(data.error?.message || 'API 키 삭제에 실패했습니다.');
      }
    } catch (error) {
      console.error('API 키 삭제 오류:', error);
      setError('API 키 삭제 중 오류가 발생했습니다.');
    } finally {
      setDeletingApiKey(false);
      setShowApiKeyDeleteModal(false);
      setApiKeyToDelete(null);
    }
  };


  // 중요도에 따른 색상 반환
  const getImportanceColor = (level: number) => {
    switch (level) {
      case 5: return 'bg-red-100 text-red-800 dark:bg-red-900/40 dark:text-red-300';
      case 4: return 'bg-orange-100 text-orange-800 dark:bg-orange-900/40 dark:text-orange-300';
      case 3: return 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/40 dark:text-yellow-300';
      case 2: return 'bg-blue-100 text-blue-800 dark:bg-blue-900/40 dark:text-blue-300';
      case 1: return 'bg-gray-100 text-gray-800 dark:bg-gray-900/40 dark:text-gray-300';
      default: return 'bg-gray-100 text-gray-800 dark:bg-gray-900/40 dark:text-gray-300';
    }
  };

  // 메모리 타입별 개수 (ContextAI용)
  const memoryCounts = contextMemories.reduce((acc, memory) => {
    acc[memory.type] = (acc[memory.type] || 0) + 1;
    return acc;
  }, {} as Record<string, number>);

  if (loading) return <div className="p-8 text-center">로딩 중...</div>;
  if (error) return <div className="p-8 text-center text-red-600">{error}</div>;
  if (!user) return null;

  return (
    <div className="max-w-6xl mx-auto py-4 md:py-10 px-2 md:px-4 h-full overflow-y-auto bg-white dark:bg-gray-900">
      <h2 className="text-xl md:text-2xl font-bold mb-4 md:mb-8">내 프로필</h2>
      
      {/* 전역 알림 메시지 */}
      {success && (
        <div className="mb-6 p-4 bg-green-100 border border-green-400 text-green-700 rounded">
          {success}
        </div>
      )}
      {error && (
        <div className="mb-6 p-4 bg-red-100 border border-red-400 text-red-700 rounded">
          {error}
        </div>
      )}

      {/* 상단 정보 카드 영역 */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
        {/* 기본 정보 */}
        <div className="bg-gray-50 dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg shadow-sm p-4">
          <h3 className="text-base font-bold mb-3">기본 정보</h3>
          <div className="flex flex-col sm:flex-row items-center sm:items-start gap-4">
            {/* 프로필 이미지 */}
            <div className="flex-shrink-0">
              <div 
                className={`relative group ${
                  isDragOver 
                    ? 'ring-2 ring-purple-500 ring-offset-2' 
                    : ''
                }`}
                onDragEnter={handleDragEnter}
                onDragLeave={handleDragLeave}
                onDragOver={handleDragOver}
                onDrop={handleDrop}
              >
                <img 
                  src={profileImage} 
                  alt="프로필 이미지" 
                  className="w-16 h-16 rounded-full object-cover border-2 border-gray-200 dark:border-gray-600"
                />
                {/* 드래그 오버레이 */}
                {isDragOver && (
                  <div className="absolute inset-0 bg-purple-500 bg-opacity-50 rounded-full flex items-center justify-center">
                    <svg className="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12" />
                    </svg>
                  </div>
                )}
                <label className="absolute bottom-0 right-0 bg-purple-600 hover:bg-purple-700 text-white rounded-full p-1 cursor-pointer transition-colors">
                  <svg className="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M12 4v16m8-8H4" />
                  </svg>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={handleImageUpload}
                    className="hidden"
                    disabled={uploadingImage}
                  />
                </label>
              </div>
              <div className="mt-1 text-center">
                {profileImage !== '/images/user.png' && (
                  <button
                    onClick={handleImageDelete}
                    className="text-xs text-red-600 hover:text-red-800 dark:text-red-400 dark:hover:text-red-300"
                  >
                    삭제
                  </button>
                )}
                {uploadingImage && (
                  <div className="text-xs text-gray-500">업로드 중...</div>
                )}
              </div>
            </div>
            
            {/* 사용자 정보 */}
            <div className="flex-1 space-y-2 text-sm">
              <p><span className="text-gray-500">이름:</span> {user.name}</p>
              <p><span className="text-gray-500">이메일:</span> {user.email}</p>
              <p><span className="text-gray-500">역할:</span> {user.role}</p>
              {/* <p><span className="text-gray-500">언어:</span> {
                user.language === 'ko' ? '한국어' :
                user.language === 'en' ? 'English' :
                user.language === 'ja' ? '日本語' :
                user.language === 'zh' ? '中文' :
                user.language === 'es' ? 'Español' :
                user.language === 'fr' ? 'Français' :
                '한국어'
              }</p> */}
            </div>
          </div>
        </div>

        {/* 로그인 정보 */}
        <div className="bg-gray-50 dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg shadow-sm p-4">
          <h3 className="text-base font-bold mb-3">로그인 정보</h3>
          <div className="space-y-2 text-sm">
            <p><span className="text-gray-500">이메일:</span> {user.email}</p>
            <p><span className="text-gray-500">가입일:</span> {user.created_at?.slice(0, 10) || ''}</p>
            {user.last_login && (
              <p><span className="text-gray-500">마지막 로그인:</span> {user.last_login?.replace('T', ' ').slice(0, 19)}</p>
            )}
          </div>
        </div>

        {/* API 키 */}
        <div className="bg-gray-50 dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg shadow-sm p-4">
          <h3 className="text-base font-bold mb-3">API 키</h3>
          <div className="space-y-2 text-sm">
            {user.apiKeys && user.apiKeys.length > 0 ? (
              <>
                {user.apiKeys.map((k) => (
                  <div key={k.id} className="flex items-center gap-2">
                    <span className="flex-1 px-2 py-0.5 bg-gray-100 dark:bg-gray-700 rounded text-xs truncate" title={k.key}>
                      {k.key}
                    </span>
                    <button
                      onClick={() => handleCopy(k.key)}
                      className="shrink-0 px-2 py-0.5 text-xs text-white bg-purple-600 hover:bg-purple-700 rounded transition-colors"
                    >
                      복사
                    </button>
                    <button
                      onClick={() => handleDeleteApiKey(k)}
                      className="shrink-0 px-2 py-0.5 text-xs text-white bg-red-600 hover:bg-red-700 rounded transition-colors"
                      title="API 키 삭제"
                    >
                      삭제
                    </button>
                  </div>
                ))}
                {copied && <div className="text-green-600 text-xs">API 키가 복사되었습니다!</div>}
              </>
            ) : (
              <div className="text-gray-500 text-xs">API 키가 없습니다.</div>
            )}
          </div>
        </div>
      </div>

      {/* 하단 영역 */}
      <div className="flex flex-col lg:flex-row gap-6">
        {/* 내 정보 수정 */}
        <div className="w-full lg:w-[32%] lg:min-w-[280px] bg-gray-50 dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg shadow-sm p-4">
          <h3 className="text-base font-bold mb-3">내 정보 수정</h3>
          <form onSubmit={handleSubmit} className="space-y-2">
            <div className="flex flex-col sm:grid sm:grid-cols-[4rem,1fr] sm:items-center gap-2">
              <label className="text-sm text-gray-500 dark:text-gray-400">이름</label>
              <input
                type="text"
                value={name}
                onChange={(e) => setName(e.target.value)}
                className="w-full min-w-0 px-2 py-1 text-sm border rounded bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700 text-gray-900 dark:text-gray-100 focus:outline-none focus:ring-1 focus:ring-purple-500"
                placeholder="이름"
              />
            </div>
            <div className="flex flex-col sm:grid sm:grid-cols-[4rem,1fr] sm:items-center gap-2">
              <label className="text-sm text-gray-500 dark:text-gray-400">이메일</label>
              <input
                type="email"
                value={email}
                onChange={(e) => setEmail(e.target.value)}
                className="w-full min-w-0 px-2 py-1 text-sm border rounded bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700 text-gray-900 dark:text-gray-100 focus:outline-none focus:ring-1 focus:ring-purple-500"
                placeholder="이메일"
              />
            </div>
            {/* <div className="grid grid-cols-[4rem,1fr] items-center gap-2">
              <label className="text-sm text-gray-500 dark:text-gray-400">언어</label>
              <select
                value={language}
                onChange={(e) => setLanguage(e.target.value)}
                className="w-full min-w-0 px-2 py-1 text-sm border rounded bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700 text-gray-900 dark:text-gray-100 focus:outline-none focus:ring-1 focus:ring-purple-500"
              >
                <option value="ko">한국어</option>
                <option value="en">English</option>
                <option value="ja">日本語</option>
                <option value="zh">中文</option>
                <option value="es">Español</option>
                <option value="fr">Français</option>
              </select>
            </div> */}
            <div className="flex flex-col sm:grid sm:grid-cols-[4rem,1fr] sm:items-center gap-2">
              <label className="text-sm text-gray-500 dark:text-gray-400">비밀번호</label>
              <input
                type="password"
                value={password}
                onChange={(e) => setPassword(e.target.value)}
                className="w-full min-w-0 px-2 py-1 text-sm border rounded bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700 text-gray-900 dark:text-gray-100 focus:outline-none focus:ring-1 focus:ring-purple-500"
                placeholder="비밀번호"
              />
            </div>
            <div className="pt-2">
              <button
                type="submit"
                disabled={saving}
                className="w-full px-3 py-1.5 text-xs text-white bg-purple-600 hover:bg-purple-700 rounded transition-colors disabled:opacity-50"
              >
                {saving ? '저장 중...' : '저장'}
              </button>
            </div>
          </form>
        </div>

        {/* ContextAI 메모리 관리 */}
        <div className="flex-1 bg-gray-50 dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg shadow-sm p-6">
          <div className="mb-4">
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <h3 className="text-base font-bold text-gray-700 dark:text-gray-300">ContextAI 메모리</h3>
              </div>
              <div className="flex items-center gap-2">
                <button
                  onClick={handleClearShortTermMemory}
                  className="px-2 py-1 text-xs text-white bg-yellow-600 hover:bg-yellow-700 rounded transition-colors"
                  disabled={memoriesLoading}
                >
                  단기 메모리 초기화
                </button>
                <button
                  onClick={handleClearLongTermMemory}
                  className="px-2 py-1 text-xs text-white bg-red-600 hover:bg-red-700 rounded transition-colors"
                  disabled={memoriesLoading}
                >
                  장기 메모리 초기화
                </button>
              </div>
            </div>

            {/* 탭 네비게이션 */}
            <div className="flex border-b border-gray-200 dark:border-gray-700 mt-4">
              <button
                onClick={() => setActiveTab('memories')}
                className={`px-3 py-2 text-sm font-medium ${
                  activeTab === 'memories'
                    ? 'text-purple-600 border-b-2 border-purple-600'
                    : 'text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-300'
                }`}
              >
                장기 메모리 ({contextMemories.length})
              </button>
              <button
                onClick={() => setActiveTab('sessions')}
                className={`px-3 py-2 text-sm font-medium ${
                  activeTab === 'sessions'
                    ? 'text-purple-600 border-b-2 border-purple-600'
                    : 'text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-300'
                }`}
              >
                단기 메모리 ({sessionSummaries.length})
              </button>
            </div>

            {/* 메모리 타입 필터 (장기 메모리 탭일 때만) */}
            {activeTab === 'memories' && (
              <div className="flex gap-2 mt-3 flex-wrap">
                {['all', 'entity', 'preference', 'skill', 'project', 'fact', 'pattern', 'relationship'].map((type) => (
                  <button
                    key={type}
                    onClick={() => setSelectedMemoryType(type)}
                    className={`px-2 py-1 text-xs rounded transition-colors ${
                      selectedMemoryType === type
                        ? 'bg-purple-600 text-white'
                        : 'bg-gray-200 text-gray-700 hover:bg-gray-300 dark:bg-gray-700 dark:text-gray-300 dark:hover:bg-gray-600'
                    }`}
                  >
                    {type === 'all' ? '전체' : type}
                  </button>
                ))}
              </div>
            )}
          </div>

          <div className="space-y-0.5">
            {memoriesLoading ? (
              <div className="text-center text-xs text-gray-500 py-4">메모리를 불러오는 중...</div>
            ) : activeTab === 'memories' ? (
              contextMemories.length === 0 ? (
                <div className="text-center text-xs text-gray-500 py-4">
                  저장된 장기 메모리가 없습니다.
                </div>
              ) : (
                <div className="divide-y divide-gray-200/50 dark:divide-gray-700/50">
                  {contextMemories.map((memory, idx) => (
                    <div key={memory.id || idx} className="py-2 px-2 hover:bg-gray-50 dark:hover:bg-gray-800">
                      <div className="flex items-start justify-between">
                        <div className="flex-1 min-w-0 pr-3">
                          <div className="flex items-center gap-2 mb-1">
                            <span className="px-1.5 py-0.5 text-xs font-medium bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200 rounded">
                              {memory.type}
                            </span>
                            <span className="text-xs text-gray-500">
                              중요도: {(memory.importance || 0).toFixed(1)}
                            </span>
                            <span className="text-xs text-gray-500">
                              신뢰도: {(memory.confidence || 0).toFixed(1)}
                            </span>
                          </div>
                          <div className="text-sm text-gray-900 dark:text-gray-100 mb-1">
                            {memory.content}
                          </div>
                          {memory.summary && (
                            <div className="text-xs text-gray-600 dark:text-gray-400 mb-1">
                              {memory.summary}
                            </div>
                          )}
                          {memory.entities && memory.entities.length > 0 && (
                            <div className="flex flex-wrap gap-1 mb-1">
                              {memory.entities.map((entity, idx) => (
                                <span key={idx} className="px-1 py-0.5 text-xs bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200 rounded">
                                  {typeof entity === 'string' ? entity : (entity as any).name || JSON.stringify(entity)}
                                </span>
                              ))}
                            </div>
                          )}
                          <div className="text-xs text-gray-500">
                            {new Date(memory.updatedAt).toLocaleDateString('ko-KR', {
                              year: 'numeric',
                              month: 'numeric',
                              day: 'numeric',
                              hour: '2-digit',
                              minute: '2-digit'
                            })}
                          </div>
                        </div>
                        <button
                          onClick={() => handleDeleteMemory(memory)}
                          className="flex-shrink-0 p-1.5 text-gray-400 hover:text-red-500 rounded-full hover:bg-gray-100 dark:hover:bg-gray-700"
                        >
                          <svg className="w-3.5 h-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                          </svg>
                        </button>
                      </div>
                    </div>
                  ))}
                </div>
              )
            ) : (
              sessionSummaries.length === 0 ? (
                <div className="text-center text-xs text-gray-500 py-4">
                  저장된 단기 메모리가 없습니다.
                </div>
              ) : (
                <div className="divide-y divide-gray-200/50 dark:divide-gray-700/50">
                  {sessionSummaries.map((session, idx) => (
                    <div key={session.sessionId || idx} className="py-2 px-2 hover:bg-gray-50 dark:hover:bg-gray-800">
                      <div className="flex items-start justify-between">
                        <div className="flex-1 min-w-0">
                          <div className="flex items-center gap-2 mb-1">
                            <span className="text-xs font-medium text-gray-700 dark:text-gray-300">
                              세션 {session.sessionId.substring(0, 8)}...
                            </span>
                            <span className="text-xs text-gray-500">
                              메시지: {session.messageCount}개
                            </span>
                            <span className="text-xs text-gray-500">
                              중요도: {(session.importance || 0).toFixed(1)}
                            </span>
                          </div>
                          <div className="text-sm text-gray-900 dark:text-gray-100 mb-1">
                            {session.summary}
                          </div>
                          {session.keyTopics && session.keyTopics.length > 0 && (
                            <div className="flex flex-wrap gap-1 mb-1">
                              {session.keyTopics.map((topic, idx) => (
                                <span key={idx} className="px-1 py-0.5 text-xs bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-200 rounded">
                                  {topic}
                                </span>
                              ))}
                            </div>
                          )}
                          {session.mentionedEntities && session.mentionedEntities.length > 0 && (
                            <div className="flex flex-wrap gap-1 mb-1">
                              {session.mentionedEntities.map((entity, idx) => (
                                <span key={idx} className="px-1 py-0.5 text-xs bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200 rounded">
                                  {typeof entity === 'string' ? entity : (entity as any).name || JSON.stringify(entity)}
                                </span>
                              ))}
                            </div>
                          )}
                          <div className="text-xs text-gray-500">
                            {new Date(session.createdAt).toLocaleDateString('ko-KR', {
                              year: 'numeric',
                              month: 'numeric',
                              day: 'numeric',
                              hour: '2-digit',
                              minute: '2-digit'
                            })}
                          </div>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              )
            )}
          </div>
        </div>
      </div>

      {/* 단기 메모리 초기화 확인 모달 */}
      {showDeleteAllConfirmModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white dark:bg-gray-800 rounded-lg p-6 max-w-md mx-4">
            <h3 className="text-lg font-semibold mb-4 text-gray-900 dark:text-gray-100">
              단기 메모리 초기화 확인
            </h3>
            <p className="text-gray-700 dark:text-gray-300 mb-6">
              정말로 단기 메모리를 초기화하시겠습니까? 현재 세션과 관련된 메모리가 삭제됩니다.
            </p>
            <div className="flex justify-end gap-3">
              <button
                onClick={() => setShowDeleteAllConfirmModal(false)}
                className="px-4 py-2 text-gray-600 hover:text-gray-800 dark:text-gray-400 dark:hover:text-gray-200"
              >
                취소
              </button>
              <button
                onClick={confirmDeleteAllMemories}
                className="px-4 py-2 bg-yellow-600 text-white rounded hover:bg-yellow-700"
              >
                초기화
              </button>
            </div>
          </div>
        </div>
      )}

      {/* 삭제 확인 모달 */}
      {showConfirmModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white dark:bg-gray-800 rounded-lg p-6 max-w-md mx-4">
            <h3 className="text-lg font-semibold mb-4 text-gray-900 dark:text-gray-100">
              메모리 삭제 확인
            </h3>
            <p className="text-gray-700 dark:text-gray-300 mb-6">
              이 메모리를 삭제하시겠습니까? 삭제된 메모리는 복구할 수 없습니다.
            </p>
            <div className="flex justify-end gap-3">
              <button
                onClick={() => {
                  setShowConfirmModal(false);
                  setMemoryToDelete(null);
                }}
                className="px-4 py-2 text-gray-600 hover:text-gray-800 dark:text-gray-400 dark:hover:text-gray-200"
              >
                취소
              </button>
              <button
                onClick={confirmDeleteMemory}
                className="px-4 py-2 bg-red-600 text-white rounded hover:bg-red-700"
              >
                삭제
              </button>
            </div>
          </div>
        </div>
      )}

      {/* API 키 삭제 확인 모달 */}
      {showApiKeyDeleteModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white dark:bg-gray-800 rounded-lg p-6 max-w-md mx-4">
            <h3 className="text-lg font-semibold mb-4 text-gray-900 dark:text-gray-100">
              API 키 삭제 확인
            </h3>
            <p className="text-gray-700 dark:text-gray-300 mb-6">
              {apiKeyToDelete?.name ? (
                <>
                  API 키 <strong>{apiKeyToDelete.name}</strong>를 삭제하시겠습니까? 삭제된 API 키는 복구할 수 없으며, 이 키를 사용하는 모든 서비스가 중단됩니다.
                </>
              ) : (
                '이 API 키를 삭제하시겠습니까? 삭제된 API 키는 복구할 수 없습니다.'
              )}
            </p>
            <div className="flex justify-end gap-3">
              <button
                onClick={() => {
                  setShowApiKeyDeleteModal(false);
                  setApiKeyToDelete(null);
                }}
                className="px-4 py-2 text-gray-600 hover:text-gray-800 dark:text-gray-400 dark:hover:text-gray-200"
                disabled={deletingApiKey}
              >
                취소
              </button>
              <button
                onClick={confirmDeleteApiKey}
                className="px-4 py-2 bg-red-600 text-white rounded hover:bg-red-700 disabled:opacity-50 disabled:cursor-not-allowed"
                disabled={deletingApiKey}
              >
                {deletingApiKey ? '삭제 중...' : '삭제'}
              </button>
            </div>
          </div>
        </div>
      )}

    </div>
  );
} 