'use client';

import { useState, useEffect, useRef } from 'react';
import { useRouter } from 'next/navigation';
import { Eye, Trash2, FileText, Upload, X } from 'lucide-react';

// 타입 정의
interface QuotationItem {
  id?: number;
  itemOrder: number;
  itemName: string;
  description?: string;
  specification?: string;
  unit: string;
  quantity: number;
  unitPrice: number;
  discountRate?: number;
  discountAmount?: number;
  amount: number;
  remarks?: string;
  isOptional?: boolean;
}

interface Quotation {
  id: number;
  quotation_number: string;
  quotation_date: string;
  validity_date: string;
  title: string;
  client_company: string;
  client_contact?: string;
  client_phone?: string;
  client_email?: string;
  subtotal: number;
  tax_rate: number;
  tax_amount: number;
  total_amount: number;
  currency: string;
  status: 'draft' | 'sent' | 'accepted' | 'rejected' | 'expired' | 'cancelled';
  generated_by_ai: boolean;
  ai_prompt?: string;  // 사용자가 요청한 원본 프롬프트
  notes?: string;
  items?: QuotationItem[];
  created_at: string;
  updated_at: string;
  // 회사 정보
  company_name?: string;
  logo_path?: string;
  seal_path?: string;
}

interface CompanyProfile {
  id?: number;
  companyName: string;
  businessNumber?: string;
  ceoName?: string;
  address?: string;
  phone?: string;
  fax?: string;
  email?: string;
  website?: string;
  logoPath?: string;
  sealPath?: string;
  bankName?: string;
  bankAccount?: string;
  bankHolder?: string;
  defaultValidityDays?: number;
  defaultPaymentTerms?: string;
  defaultNotes?: string;
}

interface PriceCatalogItem {
  id: number;
  category: string;
  item_code: string;
  item_name: string;
  description?: string;
  unit: string;
  unit_price: number;
  currency: string;
  tags?: string[];
}

// 검증 결과 타입
interface VerificationCheck {
  pass: boolean;
  message: string;
}

interface VerificationResult {
  verified: boolean;
  grade: string;
  score: number;
  checks: {
    itemComposition?: VerificationCheck;
    priceAppropriateness?: VerificationCheck;
    calculationAccuracy?: VerificationCheck;
    projectRelevance?: VerificationCheck;
    overallQuality?: VerificationCheck;
  };
  summary: string;
  recommendations: string[];
  attemptCount?: number;
  maxAttempts?: number;
}

// 탭 타입
type TabType = 'generate' | 'list' | 'settings';

// 상태 배지 색상
const statusColors: Record<string, string> = {
  draft: 'bg-gray-200 text-gray-800',
  sent: 'bg-blue-200 text-blue-800',
  accepted: 'bg-green-200 text-green-800',
  rejected: 'bg-red-200 text-red-800',
  expired: 'bg-yellow-200 text-yellow-800',
  cancelled: 'bg-gray-400 text-gray-800'
};

const statusLabels: Record<string, string> = {
  draft: '작성중',
  sent: '발송됨',
  accepted: '수락됨',
  rejected: '거절됨',
  expired: '만료됨',
  cancelled: '취소됨'
};

export default function QuotationPage() {
  const router = useRouter();
  const [activeTab, setActiveTab] = useState<TabType>('generate');
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [success, setSuccess] = useState<string | null>(null);

  // AI 생성 관련 상태
  const [aiPrompt, setAiPrompt] = useState('');
  const [generatingProgress, setGeneratingProgress] = useState(0);
  const [generatedQuotation, setGeneratedQuotation] = useState<Quotation | null>(null);
  const [verificationResult, setVerificationResult] = useState<VerificationResult | null>(null);

  // 견적서 목록 관련 상태
  const [quotations, setQuotations] = useState<Quotation[]>([]);
  const [selectedQuotation, setSelectedQuotation] = useState<Quotation | null>(null);
  const [pagination, setPagination] = useState({ page: 1, limit: 10, total: 0, totalPages: 0 });
  const [statusFilter, setStatusFilter] = useState<string>('');

  // 설정 관련 상태
  const [companyProfile, setCompanyProfile] = useState<CompanyProfile>({
    companyName: '',
    businessNumber: '',
    ceoName: '',
    address: '',
    phone: '',
    email: '',
    bankName: '',
    bankAccount: '',
    bankHolder: '',
    defaultValidityDays: 30
  });
  const [priceCatalog, setPriceCatalog] = useState<PriceCatalogItem[]>([]);
  const [newPriceItem, setNewPriceItem] = useState({
    category: '',
    itemCode: '',
    itemName: '',
    description: '',
    unit: '식',
    unitPrice: 0
  });

  // API 호출 헬퍼
  const apiCall = async (endpoint: string, options: RequestInit = {}) => {
    const apiKey = typeof window !== 'undefined' ? localStorage.getItem('apiKey') : null;
    const token = typeof window !== 'undefined' ? localStorage.getItem('userToken') : null;

    const headers: Record<string, string> = {
      'Content-Type': 'application/json',
      ...(apiKey && { 'X-API-Key': apiKey }),
      ...(token && { 'Authorization': `Bearer ${token}` })
    };

    const response = await fetch(`/api/quotation${endpoint}`, {
      ...options,
      headers: { ...headers, ...options.headers }
    });

    const data = await response.json();

    if (!response.ok || !data.success) {
      throw new Error(data.error?.message || data.message || '요청 실패');
    }

    return data;
  };

  // 견적서 목록 로드
  const loadQuotations = async () => {
    try {
      setIsLoading(true);
      const params = new URLSearchParams({
        page: pagination.page.toString(),
        limit: pagination.limit.toString(),
        ...(statusFilter && { status: statusFilter })
      });

      const data = await apiCall(`?${params.toString()}`);
      setQuotations(data.data.quotations || []);
      setPagination(prev => ({
        ...prev,
        total: data.data.pagination?.total || 0,
        totalPages: data.data.pagination?.totalPages || 0
      }));
    } catch (err: any) {
      setError(err.message);
    } finally {
      setIsLoading(false);
    }
  };

  // 회사 프로필 로드
  const loadCompanyProfile = async () => {
    try {
      const data = await apiCall('/company/profile');
      if (data.data) {
        setCompanyProfile({
          companyName: data.data.company_name || '',
          businessNumber: data.data.business_number || '',
          ceoName: data.data.ceo_name || '',
          address: data.data.address || '',
          phone: data.data.phone || '',
          fax: data.data.fax || '',
          email: data.data.email || '',
          website: data.data.website || '',
          logoPath: data.data.logo_path || '',
          sealPath: data.data.seal_path || '',
          bankName: data.data.bank_name || '',
          bankAccount: data.data.bank_account || '',
          bankHolder: data.data.bank_holder || '',
          defaultValidityDays: data.data.default_validity_days || 30,
          defaultPaymentTerms: data.data.default_payment_terms || '',
          defaultNotes: data.data.default_notes || ''
        });
      }
    } catch (err: any) {
      // 회사 정보가 없는 경우 무시
      console.log('회사 정보가 없습니다.');
    }
  };

  // 단가 카탈로그 로드
  const loadPriceCatalog = async () => {
    try {
      const data = await apiCall('/price-catalog');
      setPriceCatalog(data.data?.items || []);
    } catch (err: any) {
      console.log('단가 카탈로그 로드 실패:', err.message);
    }
  };

  // 초기 로드
  useEffect(() => {
    loadQuotations();
    loadCompanyProfile();
    loadPriceCatalog();
  }, []);

  // 탭 변경 시 데이터 로드
  useEffect(() => {
    if (activeTab === 'list') {
      loadQuotations();
    }
  }, [activeTab, pagination.page, statusFilter]);

  // AI 견적서 생성
  const generateQuotation = async () => {
    if (!aiPrompt.trim()) {
      setError('견적서 생성 요청을 입력해주세요.');
      return;
    }

    try {
      setIsLoading(true);
      setError(null);
      setGeneratingProgress(10);
      setGeneratedQuotation(null);
      setVerificationResult(null);

      const progressInterval = setInterval(() => {
        setGeneratingProgress(prev => Math.min(prev + 10, 90));
      }, 500);

      const data = await apiCall('/generate', {
        method: 'POST',
        body: JSON.stringify({ prompt: aiPrompt })
      });

      clearInterval(progressInterval);
      setGeneratingProgress(100);

      if (data.success && data.data) {
        // API 응답을 Quotation 형식으로 변환
        const quotationData: Quotation = {
          id: data.data.quotationId,
          quotation_number: data.data.quotationNumber,
          quotation_date: data.data.quotationDate,
          validity_date: data.data.validityDate,
          title: data.data.title,
          client_company: data.data.clientCompany,
          client_contact: data.data.clientContact,
          subtotal: data.data.subtotal,
          tax_rate: data.data.taxRate,
          tax_amount: data.data.taxAmount,
          total_amount: data.data.totalAmount,
          currency: data.data.currency || 'KRW',
          status: data.data.status || 'draft',
          generated_by_ai: true,
          notes: data.data.notes,
          items: data.data.items?.map((item: any, index: number) => ({
            itemOrder: item.itemOrder || index + 1,
            itemName: item.itemName,
            description: item.description || '',
            unit: item.unit || '식',
            quantity: item.quantity,
            unitPrice: item.unitPrice,
            amount: item.amount
          })),
          created_at: new Date().toISOString(),
          updated_at: new Date().toISOString()
        };
        setGeneratedQuotation(quotationData);

        // 검증 결과 저장
        if (data.data.verification) {
          setVerificationResult(data.data.verification);
        } else {
          setVerificationResult(null);
        }

        setSuccess('견적서가 성공적으로 생성되었습니다!');
        setAiPrompt('');

        // 목록 새로고침
        loadQuotations();
      }
    } catch (err: any) {
      setError(err.message);
    } finally {
      setIsLoading(false);
      setTimeout(() => setGeneratingProgress(0), 1000);
    }
  };

  // 회사 프로필 저장
  const saveCompanyProfile = async () => {
    try {
      setIsLoading(true);
      await apiCall('/company/profile', {
        method: 'POST',
        body: JSON.stringify(companyProfile)
      });
      setSuccess('회사 정보가 저장되었습니다.');
    } catch (err: any) {
      setError(err.message);
    } finally {
      setIsLoading(false);
    }
  };

  // 로고 업로드 핸들러
  const handleLogoUpload = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;

    // 파일 타입 검증
    if (!file.type.startsWith('image/')) {
      setError('이미지 파일만 업로드 가능합니다.');
      return;
    }

    // 파일 크기 검증 (2MB)
    if (file.size > 2 * 1024 * 1024) {
      setError('파일 크기는 2MB 이하여야 합니다.');
      return;
    }

    try {
      setIsLoading(true);
      const formData = new FormData();
      formData.append('logo', file);

      // 인증 헤더 준비
      const apiKey = typeof window !== 'undefined' ? localStorage.getItem('apiKey') : null;
      const token = typeof window !== 'undefined' ? localStorage.getItem('userToken') : null;
      const headers: Record<string, string> = {};
      if (apiKey) headers['X-API-Key'] = apiKey;
      if (token) headers['Authorization'] = `Bearer ${token}`;

      const response = await fetch('/api/quotation/upload-logo', {
        method: 'POST',
        body: formData,
        headers,
        credentials: 'include'
      });

      const result = await response.json();
      if (result.success) {
        setCompanyProfile(prev => ({ ...prev, logoPath: result.data.logoPath }));
        setSuccess('로고가 업로드되었습니다.');
      } else {
        setError(result.error?.message || '로고 업로드에 실패했습니다.');
      }
    } catch (err: any) {
      setError(err.message || '로고 업로드 중 오류가 발생했습니다.');
    } finally {
      setIsLoading(false);
    }
  };

  // 로고 삭제 핸들러
  const handleLogoDelete = async () => {
    try {
      // 인증 헤더 준비
      const apiKey = typeof window !== 'undefined' ? localStorage.getItem('apiKey') : null;
      const token = typeof window !== 'undefined' ? localStorage.getItem('userToken') : null;
      const headers: Record<string, string> = {};
      if (apiKey) headers['X-API-Key'] = apiKey;
      if (token) headers['Authorization'] = `Bearer ${token}`;

      await fetch('/api/quotation/upload-logo', {
        method: 'DELETE',
        headers,
        credentials: 'include'
      });

      setCompanyProfile(prev => ({ ...prev, logoPath: '' }));
      setSuccess('로고가 삭제되었습니다.');
    } catch (err: any) {
      console.error('로고 삭제 오류:', err);
      // 에러가 발생해도 UI에서는 삭제
      setCompanyProfile(prev => ({ ...prev, logoPath: '' }));
    }
  };

  // 단가 항목 추가
  const addPriceItem = async () => {
    if (!newPriceItem.itemName || newPriceItem.unitPrice <= 0) {
      setError('품목명과 단가를 입력해주세요.');
      return;
    }

    try {
      setIsLoading(true);
      await apiCall('/price-catalog', {
        method: 'POST',
        body: JSON.stringify(newPriceItem)
      });
      setSuccess('단가 항목이 추가되었습니다.');
      setNewPriceItem({
        category: '',
        itemCode: '',
        itemName: '',
        description: '',
        unit: '식',
        unitPrice: 0
      });
      loadPriceCatalog();
    } catch (err: any) {
      setError(err.message);
    } finally {
      setIsLoading(false);
    }
  };

  // 견적서 상세 조회
  const viewQuotationDetail = async (id: number) => {
    try {
      setIsLoading(true);
      const data = await apiCall(`/${id}`);
      setSelectedQuotation(data.data);
    } catch (err: any) {
      setError(err.message);
    } finally {
      setIsLoading(false);
    }
  };

  // 견적서 삭제
  const deleteQuotation = async (id: number) => {
    if (!confirm('이 견적서를 삭제하시겠습니까?')) return;

    try {
      setIsLoading(true);
      await apiCall(`/${id}`, { method: 'DELETE' });
      setSuccess('견적서가 삭제되었습니다.');
      setSelectedQuotation(null);
      loadQuotations();
    } catch (err: any) {
      setError(err.message);
    } finally {
      setIsLoading(false);
    }
  };

  // 견적서 상태 변경
  const updateQuotationStatus = async (id: number, status: string) => {
    try {
      setIsLoading(true);
      await apiCall(`/${id}/status`, {
        method: 'PATCH',
        body: JSON.stringify({ status })
      });
      setSuccess('견적서 상태가 변경되었습니다.');
      loadQuotations();
      if (selectedQuotation?.id === id) {
        viewQuotationDetail(id);
      }
    } catch (err: any) {
      setError(err.message);
    } finally {
      setIsLoading(false);
    }
  };

  // 금액 포맷팅
  const formatCurrency = (amount: number | string | null | undefined, currency: string = 'KRW') => {
    // null, undefined, NaN 처리
    if (amount === null || amount === undefined) return '₩0';

    // 문자열인 경우 숫자로 변환
    const numAmount = typeof amount === 'string' ? parseFloat(amount) : amount;

    // NaN 체크
    if (isNaN(numAmount)) return '₩0';

    return new Intl.NumberFormat('ko-KR', {
      style: 'currency',
      currency
    }).format(numAmount);
  };

  // 날짜 포맷팅
  const formatDate = (dateStr: string) => {
    return new Date(dateStr).toLocaleDateString('ko-KR', {
      year: 'numeric',
      month: '2-digit',
      day: '2-digit'
    });
  };

  // 알림 메시지 자동 제거
  useEffect(() => {
    if (error) {
      const timer = setTimeout(() => setError(null), 5000);
      return () => clearTimeout(timer);
    }
  }, [error]);

  useEffect(() => {
    if (success) {
      const timer = setTimeout(() => setSuccess(null), 3000);
      return () => clearTimeout(timer);
    }
  }, [success]);

  return (
    <div className="min-h-screen" style={{ backgroundColor: 'var(--body-bg)', color: 'var(--text-primary)' }}>
      {/* 알림 메시지 */}
      {error && (
        <div className="fixed top-4 right-4 z-50 bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded shadow-lg">
          <span>{error}</span>
          <button onClick={() => setError(null)} className="ml-4 font-bold">&times;</button>
        </div>
      )}
      {success && (
        <div className="fixed top-4 right-4 z-50 bg-emerald-600 border border-emerald-700 text-white px-4 py-3 rounded shadow-xl">
          <span className="font-semibold">{success}</span>
          <button onClick={() => setSuccess(null)} className="ml-4 font-bold text-white">&times;</button>
        </div>
      )}

      <div className="container mx-auto px-4 py-6 max-w-7xl">
        {/* 헤더 */}
        <div className="mb-6">
          <h1 className="text-2xl font-bold mb-2">견적서 AI 에이전트</h1>
          <p className="text-sm opacity-70">
            자연어로 견적서를 자동 생성하고 관리합니다. 과거 견적서와 단가 정보를 참조하여 최적의 견적서를 작성합니다.
          </p>
        </div>

        {/* 탭 네비게이션 */}
        <div className="flex border-b mb-6" style={{ borderColor: 'var(--border-color)' }}>
          <button
            onClick={() => setActiveTab('generate')}
            className={`px-4 py-2 font-medium ${
              activeTab === 'generate'
                ? 'border-b-2 border-blue-500 text-blue-500'
                : 'opacity-60 hover:opacity-100'
            }`}
          >
            AI 견적서 생성
          </button>
          <button
            onClick={() => setActiveTab('list')}
            className={`px-4 py-2 font-medium ${
              activeTab === 'list'
                ? 'border-b-2 border-blue-500 text-blue-500'
                : 'opacity-60 hover:opacity-100'
            }`}
          >
            견적서 목록
          </button>
          <button
            onClick={() => setActiveTab('settings')}
            className={`px-4 py-2 font-medium ${
              activeTab === 'settings'
                ? 'border-b-2 border-blue-500 text-blue-500'
                : 'opacity-60 hover:opacity-100'
            }`}
          >
            설정
          </button>
        </div>

        {/* AI 견적서 생성 탭 */}
        {activeTab === 'generate' && (
          <div className="space-y-6">
            {/* 입력 영역 */}
            <div className="p-6 rounded-lg shadow" style={{ backgroundColor: 'var(--card-bg)' }}>
              <h2 className="text-lg font-semibold mb-4">견적서 생성 요청</h2>
              <p className="text-sm opacity-70 mb-4">
                자연어로 견적서 생성을 요청하세요. 예: "혜안 견적서 만들어줘. AI Agent 개발 견적으로 총 500만원, 고객은 삼성전자, 담당자 이재용"
              </p>
              <textarea
                value={aiPrompt}
                onChange={(e) => setAiPrompt(e.target.value)}
                placeholder="견적서 생성 요청을 입력하세요..."
                className="w-full h-32 p-3 border rounded-lg resize-none"
                style={{
                  backgroundColor: 'var(--input-bg)',
                  borderColor: 'var(--border-color)',
                  color: 'var(--text-primary)'
                }}
                disabled={isLoading}
              />
              <div className="flex justify-between items-center mt-4">
                <div className="text-sm opacity-70">
                  {generatingProgress > 0 && generatingProgress < 100 && (
                    <span>생성 중... {generatingProgress}%</span>
                  )}
                </div>
                <button
                  onClick={generateQuotation}
                  disabled={isLoading || !aiPrompt.trim()}
                  className="px-6 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  {isLoading ? '생성 중...' : '견적서 생성'}
                </button>
              </div>
              {generatingProgress > 0 && (
                <div className="mt-2 w-full bg-gray-200 rounded-full h-2">
                  <div
                    className="bg-blue-500 h-2 rounded-full transition-all duration-300"
                    style={{ width: `${generatingProgress}%` }}
                  />
                </div>
              )}
            </div>

            {/* 생성된 견적서 미리보기 */}
            {generatedQuotation && (
              <div className="p-6 rounded-lg shadow" style={{ backgroundColor: 'var(--card-bg)' }}>
                <div className="flex justify-between items-center mb-4">
                  <h2 className="text-lg font-semibold">생성된 견적서</h2>
                  <span className={`px-2 py-1 rounded text-sm ${statusColors[generatedQuotation.status]}`}>
                    {statusLabels[generatedQuotation.status]}
                  </span>
                </div>

                {/* 견적서 헤더 정보 */}
                <div className="grid grid-cols-2 gap-4 mb-6 pb-4 border-b" style={{ borderColor: 'var(--border-color)' }}>
                  <div>
                    <p className="text-sm opacity-70">견적번호</p>
                    <p className="font-medium">{generatedQuotation.quotation_number}</p>
                  </div>
                  <div>
                    <p className="text-sm opacity-70">견적일자</p>
                    <p className="font-medium">{formatDate(generatedQuotation.quotation_date)}</p>
                  </div>
                  <div>
                    <p className="text-sm opacity-70">고객사</p>
                    <p className="font-medium">{generatedQuotation.client_company}</p>
                  </div>
                  <div>
                    <p className="text-sm opacity-70">담당자</p>
                    <p className="font-medium">{generatedQuotation.client_contact || '-'}</p>
                  </div>
                  <div className="col-span-2">
                    <p className="text-sm opacity-70">제목</p>
                    <p className="font-medium">{generatedQuotation.title}</p>
                  </div>
                </div>

                {/* 품목 테이블 */}
                {generatedQuotation.items && generatedQuotation.items.length > 0 && (
                  <div className="mb-6">
                    <h3 className="font-medium mb-2">품목 내역</h3>
                    <div className="overflow-x-auto">
                      <table className="w-full text-sm">
                        <thead>
                          <tr className="border-b" style={{ borderColor: 'var(--border-color)' }}>
                            <th className="py-2 px-3 text-left">No.</th>
                            <th className="py-2 px-3 text-left">품목명</th>
                            <th className="py-2 px-3 text-left">규격/설명</th>
                            <th className="py-2 px-3 text-center">단위</th>
                            <th className="py-2 px-3 text-right">수량</th>
                            <th className="py-2 px-3 text-right">단가</th>
                            <th className="py-2 px-3 text-right">금액</th>
                          </tr>
                        </thead>
                        <tbody>
                          {generatedQuotation.items.map((item, index) => (
                            <tr key={index} className="border-b" style={{ borderColor: 'var(--border-color)' }}>
                              <td className="py-2 px-3">{item.itemOrder}</td>
                              <td className="py-2 px-3 font-medium">{item.itemName}</td>
                              <td className="py-2 px-3 text-sm opacity-70">{item.description || '-'}</td>
                              <td className="py-2 px-3 text-center">{item.unit}</td>
                              <td className="py-2 px-3 text-right">{item.quantity}</td>
                              <td className="py-2 px-3 text-right">{formatCurrency(item.unitPrice)}</td>
                              <td className="py-2 px-3 text-right font-medium">{formatCurrency(item.amount)}</td>
                            </tr>
                          ))}
                        </tbody>
                      </table>
                    </div>
                  </div>
                )}

                {/* 금액 요약 */}
                <div className="flex justify-end">
                  <div className="w-64 space-y-2">
                    <div className="flex justify-between">
                      <span className="opacity-70">공급가액</span>
                      <span>{formatCurrency(generatedQuotation.subtotal)}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="opacity-70">부가세 ({generatedQuotation.tax_rate}%)</span>
                      <span>{formatCurrency(generatedQuotation.tax_amount)}</span>
                    </div>
                    <div className="flex justify-between font-bold text-lg pt-2 border-t" style={{ borderColor: 'var(--border-color)' }}>
                      <span>합계</span>
                      <span className="text-blue-500">{formatCurrency(generatedQuotation.total_amount)}</span>
                    </div>
                  </div>
                </div>

                {/* AI 검증 결과 */}
                {verificationResult && (
                  <div className={`mt-6 p-4 rounded-lg border-2 ${
                    verificationResult.verified
                      ? 'border-green-500 bg-green-50 dark:bg-green-900/30'
                      : 'border-red-500 bg-red-50 dark:bg-red-900/30'
                  }`}>
                    <div className="flex items-center justify-between mb-3">
                      <h4 className="font-medium flex items-center gap-2">
                        {verificationResult.verified ? '✅' : '⚠️'} AI 검증 결과
                        {verificationResult.attemptCount && verificationResult.attemptCount > 1 && (
                          <span className="text-xs bg-gray-200 dark:bg-gray-600 px-2 py-0.5 rounded">
                            {verificationResult.attemptCount}회 시도
                          </span>
                        )}
                      </h4>
                      <div className="flex items-center gap-3">
                        <span className={`px-3 py-1 rounded-full text-sm font-medium ${
                          verificationResult.grade === 'A' ? 'bg-green-500 text-white' :
                          verificationResult.grade === 'B' ? 'bg-blue-500 text-white' :
                          verificationResult.grade === 'C' ? 'bg-yellow-500 text-white' :
                          'bg-red-500 text-white'
                        }`}>
                          등급: {verificationResult.grade}
                        </span>
                        <span className="text-sm font-medium">
                          점수: {verificationResult.score}점
                        </span>
                      </div>
                    </div>

                    <p className="text-sm mb-3 opacity-80">{verificationResult.summary}</p>

                    {/* 검증 항목 */}
                    {Object.keys(verificationResult.checks || {}).length > 0 && (
                      <div className="mb-3">
                        <h5 className="text-sm font-medium mb-2">검증 항목:</h5>
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-2">
                          {Object.entries(verificationResult.checks).map(([key, check]) => (
                            <div key={key} className="flex items-start gap-2 text-sm">
                              <span>{check?.pass ? '✅' : '❌'}</span>
                              <div>
                                <span className="font-medium">
                                  {key === 'itemComposition' && '품목 구성'}
                                  {key === 'priceAppropriateness' && '단가 적정성'}
                                  {key === 'calculationAccuracy' && '금액 계산'}
                                  {key === 'projectRelevance' && '프로젝트 연관성'}
                                  {key === 'overallQuality' && '전체 품질'}
                                </span>
                                {check?.message && (
                                  <p className="text-xs opacity-70 mt-0.5">{check.message}</p>
                                )}
                              </div>
                            </div>
                          ))}
                        </div>
                      </div>
                    )}

                    {/* 권고사항 */}
                    {verificationResult.recommendations && verificationResult.recommendations.length > 0 && (
                      <div>
                        <h5 className="text-sm font-medium mb-2">개선 권고사항:</h5>
                        <ul className="list-disc list-inside text-sm space-y-1">
                          {verificationResult.recommendations.map((rec, index) => (
                            <li key={index} className="opacity-80">{rec}</li>
                          ))}
                        </ul>
                      </div>
                    )}
                  </div>
                )}

                {/* 액션 버튼 */}
                <div className="flex justify-end gap-2 mt-6 pt-4 border-t" style={{ borderColor: 'var(--border-color)' }}>
                  <button
                    onClick={() => router.push(`/quotation/view?id=${generatedQuotation.id}`)}
                    className="px-4 py-2 border rounded hover:bg-gray-100 dark:hover:bg-gray-700 flex items-center gap-2"
                    style={{ borderColor: 'var(--border-color)' }}
                  >
                    <FileText className="w-4 h-4" />
                    HTML 보기
                  </button>
                  <button
                    onClick={() => viewQuotationDetail(generatedQuotation.id)}
                    className="px-4 py-2 border rounded hover:bg-gray-100 dark:hover:bg-gray-700"
                    style={{ borderColor: 'var(--border-color)' }}
                  >
                    상세보기
                  </button>
                  <button
                    onClick={() => updateQuotationStatus(generatedQuotation.id, 'sent')}
                    className="px-4 py-2 bg-blue-500 text-white rounded hover:bg-blue-600"
                  >
                    발송 처리
                  </button>
                </div>
              </div>
            )}

            {/* 예시 프롬프트 */}
            <div className="p-6 rounded-lg shadow" style={{ backgroundColor: 'var(--card-bg)' }}>
              <h3 className="font-medium mb-3">예시 요청</h3>
              <div className="grid gap-2">
                {[
                  '혜안 견적서 만들어줘. 기관 맞춤형 문서 및 견적서 AI Agent 개발에 소요되는 여러 품목을 고려해서 총 견적금액은 부가세 포함 500만원으로 해줘. 고객은 삼성전자이고 담당자 이름은 이재용 회장이다.',
                  'AI 챗봇 개발 견적서 작성해줘. 고객사 LG전자, 담당자 구광모, 예산 3000만원',
                  '웹사이트 리뉴얼 프로젝트 견적서. 네이버, 예산 미정'
                ].map((example, index) => (
                  <button
                    key={index}
                    onClick={() => setAiPrompt(example)}
                    className="text-left p-3 rounded border hover:bg-gray-50 dark:hover:bg-gray-700 text-sm"
                    style={{ borderColor: 'var(--border-color)' }}
                  >
                    {example}
                  </button>
                ))}
              </div>
            </div>
          </div>
        )}

        {/* 견적서 목록 탭 */}
        {activeTab === 'list' && (
          <div className="space-y-6">
            {/* 필터 */}
            <div className="flex gap-4 items-center">
              <select
                value={statusFilter}
                onChange={(e) => setStatusFilter(e.target.value)}
                className="px-3 py-2 border rounded"
                style={{
                  backgroundColor: 'var(--input-bg)',
                  borderColor: 'var(--border-color)',
                  color: 'var(--text-primary)'
                }}
              >
                <option value="">전체 상태</option>
                <option value="draft">작성중</option>
                <option value="sent">발송됨</option>
                <option value="accepted">수락됨</option>
                <option value="rejected">거절됨</option>
                <option value="expired">만료됨</option>
              </select>
              <button
                onClick={loadQuotations}
                className="px-4 py-2 border rounded hover:bg-gray-100 dark:hover:bg-gray-700"
                style={{ borderColor: 'var(--border-color)' }}
              >
                새로고침
              </button>
            </div>

            {/* 목록 테이블 */}
            <div className="rounded-lg shadow overflow-hidden" style={{ backgroundColor: 'var(--card-bg)' }}>
              {isLoading ? (
                <div className="p-8 text-center">로딩 중...</div>
              ) : quotations.length === 0 ? (
                <div className="p-8 text-center opacity-70">견적서가 없습니다.</div>
              ) : (
                <table className="w-full">
                  <thead>
                    <tr className="border-b" style={{ borderColor: 'var(--border-color)', backgroundColor: 'var(--sidebar-bg)' }}>
                      <th className="py-3 px-4 text-left font-medium">견적번호</th>
                      <th className="py-3 px-4 text-left font-medium">제목</th>
                      <th className="py-3 px-4 text-left font-medium">고객사</th>
                      <th className="py-3 px-4 text-right font-medium">총액</th>
                      <th className="py-3 px-4 text-center font-medium">상태</th>
                      <th className="py-3 px-4 text-center font-medium">생성일</th>
                      <th className="py-3 px-4 text-center font-medium">작업</th>
                    </tr>
                  </thead>
                  <tbody>
                    {quotations.map((q) => (
                      <tr
                        key={q.id}
                        className="border-b hover:bg-gray-50 dark:hover:bg-gray-700 cursor-pointer"
                        style={{ borderColor: 'var(--border-color)' }}
                        onClick={() => viewQuotationDetail(q.id)}
                      >
                        <td className="py-3 px-4 font-mono text-sm">{q.quotation_number}</td>
                        <td className="py-3 px-4">
                          {q.title}
                          {q.generated_by_ai && (
                            <span className="ml-2 text-xs bg-purple-100 text-purple-700 px-1.5 py-0.5 rounded">AI</span>
                          )}
                        </td>
                        <td className="py-3 px-4">{q.client_company}</td>
                        <td className="py-3 px-4 text-right font-medium">{formatCurrency(q.total_amount)}</td>
                        <td className="py-3 px-4 text-center">
                          <span className={`px-2 py-1 rounded text-xs ${statusColors[q.status]}`}>
                            {statusLabels[q.status]}
                          </span>
                        </td>
                        <td className="py-3 px-4 text-center text-sm">{formatDate(q.created_at)}</td>
                        <td className="py-3 px-4 text-center">
                          <div className="flex items-center justify-center gap-2">
                            <button
                              onClick={(e) => {
                                e.stopPropagation();
                                router.push(`/quotation/view?id=${q.id}`);
                              }}
                              className="text-blue-500 hover:text-blue-700 p-1"
                              title="보기"
                            >
                              <Eye className="w-4 h-4" />
                            </button>
                            <button
                              onClick={(e) => {
                                e.stopPropagation();
                                deleteQuotation(q.id);
                              }}
                              className="text-red-500 hover:text-red-700 p-1"
                              title="삭제"
                            >
                              <Trash2 className="w-4 h-4" />
                            </button>
                          </div>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              )}
            </div>

            {/* 페이지네이션 */}
            {pagination.totalPages > 1 && (
              <div className="flex justify-center gap-2">
                <button
                  onClick={() => setPagination(prev => ({ ...prev, page: prev.page - 1 }))}
                  disabled={pagination.page === 1}
                  className="px-3 py-1 border rounded disabled:opacity-50"
                  style={{ borderColor: 'var(--border-color)' }}
                >
                  이전
                </button>
                <span className="px-3 py-1">
                  {pagination.page} / {pagination.totalPages}
                </span>
                <button
                  onClick={() => setPagination(prev => ({ ...prev, page: prev.page + 1 }))}
                  disabled={pagination.page === pagination.totalPages}
                  className="px-3 py-1 border rounded disabled:opacity-50"
                  style={{ borderColor: 'var(--border-color)' }}
                >
                  다음
                </button>
              </div>
            )}

            {/* 상세보기 모달 */}
            {selectedQuotation && (
              <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50" onClick={() => setSelectedQuotation(null)}>
                <div
                  className="bg-white rounded-lg shadow-xl max-w-4xl w-full max-h-[90vh] overflow-y-auto m-4"
                  style={{ backgroundColor: 'var(--card-bg)', color: 'var(--text-primary)' }}
                  onClick={(e) => e.stopPropagation()}
                >
                  <div className="p-6">
                    <div className="flex justify-between items-center mb-4">
                      <h2 className="text-xl font-bold">{selectedQuotation.title}</h2>
                      <button onClick={() => setSelectedQuotation(null)} className="text-2xl">&times;</button>
                    </div>

                    {/* 견적서 상세 내용 - 위와 유사한 구조 */}
                    <div className="space-y-4">
                      <div className="grid grid-cols-2 gap-4">
                        <div>
                          <p className="text-sm opacity-70">견적번호</p>
                          <p className="font-medium">{selectedQuotation.quotation_number}</p>
                        </div>
                        <div>
                          <p className="text-sm opacity-70">상태</p>
                          <span className={`px-2 py-1 rounded text-sm ${statusColors[selectedQuotation.status]}`}>
                            {statusLabels[selectedQuotation.status]}
                          </span>
                        </div>
                        <div>
                          <p className="text-sm opacity-70">고객사</p>
                          <p className="font-medium">{selectedQuotation.client_company}</p>
                        </div>
                        <div>
                          <p className="text-sm opacity-70">담당자</p>
                          <p className="font-medium">{selectedQuotation.client_contact || '-'}</p>
                        </div>
                      </div>

                      {/* AI 생성 프롬프트 표시 */}
                      {selectedQuotation.ai_prompt && (
                        <div className="bg-blue-50 dark:bg-blue-900/30 border border-blue-200 dark:border-blue-700 rounded-lg p-4">
                          <div className="flex items-center gap-2 mb-2">
                            <span className="text-blue-600 dark:text-blue-400">🤖</span>
                            <h3 className="font-medium text-blue-800 dark:text-blue-300">AI 생성 요청 내용</h3>
                          </div>
                          <p className="text-sm text-blue-700 dark:text-blue-200 whitespace-pre-wrap">{selectedQuotation.ai_prompt}</p>
                        </div>
                      )}

                      {/* AI 검증 결과 (DB에서 조회) */}
                      {selectedQuotation.metadata?.aiVerification && (
                        <div className={`p-4 rounded-lg border-2 ${
                          selectedQuotation.metadata.aiVerification.verified
                            ? 'border-green-500 bg-green-50 dark:bg-green-900/30'
                            : 'border-red-500 bg-red-50 dark:bg-red-900/30'
                        }`}>
                          <div className="flex items-center justify-between mb-3">
                            <h4 className="font-medium flex items-center gap-2">
                              {selectedQuotation.metadata.aiVerification.verified ? '✅' : '⚠️'} AI 검증 결과
                              {selectedQuotation.metadata.aiVerification.attemptCount && selectedQuotation.metadata.aiVerification.attemptCount > 1 && (
                                <span className="text-xs bg-gray-200 dark:bg-gray-600 px-2 py-0.5 rounded">
                                  {selectedQuotation.metadata.aiVerification.attemptCount}회 시도
                                </span>
                              )}
                            </h4>
                            <div className="flex items-center gap-3">
                              <span className={`px-3 py-1 rounded-full text-sm font-bold ${
                                selectedQuotation.metadata.aiVerification.grade === 'A' ? 'bg-green-500 text-white' :
                                selectedQuotation.metadata.aiVerification.grade === 'B' ? 'bg-blue-500 text-white' :
                                selectedQuotation.metadata.aiVerification.grade === 'C' ? 'bg-yellow-500 text-white' :
                                'bg-red-500 text-white'
                              }`}>
                                등급 {selectedQuotation.metadata.aiVerification.grade}
                              </span>
                              <span className="text-sm text-gray-600 dark:text-gray-300">
                                점수: {selectedQuotation.metadata.aiVerification.score}점
                              </span>
                            </div>
                          </div>

                          <p className="text-sm mb-3 text-gray-700 dark:text-gray-200">{selectedQuotation.metadata.aiVerification.summary}</p>

                          {/* 검증 항목 */}
                          {selectedQuotation.metadata.aiVerification.checks && Object.keys(selectedQuotation.metadata.aiVerification.checks).length > 0 && (
                            <div className="mb-3">
                              <h5 className="text-sm font-medium mb-2 text-gray-700 dark:text-gray-200">검증 항목</h5>
                              <div className="grid grid-cols-2 gap-2 text-sm">
                                {Object.entries(selectedQuotation.metadata.aiVerification.checks).map(([key, check]: [string, any]) => (
                                  <div key={key} className="flex items-center gap-1">
                                    <span>{check.pass ? '✓' : '✗'}</span>
                                    <span className={check.pass ? 'text-green-600 dark:text-green-400' : 'text-red-600 dark:text-red-400'}>
                                      {key.replace(/([A-Z])/g, ' $1').trim()}
                                    </span>
                                  </div>
                                ))}
                              </div>
                            </div>
                          )}

                          {/* 권고사항 */}
                          {selectedQuotation.metadata.aiVerification.recommendations && selectedQuotation.metadata.aiVerification.recommendations.length > 0 && (
                            <div>
                              <h5 className="text-sm font-medium mb-1 text-gray-700 dark:text-gray-200">권고사항</h5>
                              <ul className="text-sm text-gray-600 dark:text-gray-300 list-disc list-inside">
                                {selectedQuotation.metadata.aiVerification.recommendations.map((rec: string, idx: number) => (
                                  <li key={idx}>{rec}</li>
                                ))}
                              </ul>
                            </div>
                          )}
                        </div>
                      )}

                      {/* 품목 테이블 */}
                      {selectedQuotation.items && selectedQuotation.items.length > 0 && (
                        <div>
                          <h3 className="font-medium mb-2">품목 내역</h3>
                          <table className="w-full text-sm border" style={{ borderColor: 'var(--border-color)' }}>
                            <thead>
                              <tr className="bg-gray-50 dark:bg-gray-700">
                                <th className="py-2 px-3 text-left border-b" style={{ color: 'var(--text-primary)' }}>품목명</th>
                                <th className="py-2 px-3 text-center border-b">단위</th>
                                <th className="py-2 px-3 text-right border-b">수량</th>
                                <th className="py-2 px-3 text-right border-b">단가</th>
                                <th className="py-2 px-3 text-right border-b">금액</th>
                              </tr>
                            </thead>
                            <tbody>
                              {selectedQuotation.items.map((item: any, index) => (
                                <tr key={index} className="border-b">
                                  <td className="py-2 px-3">{item.item_name || item.itemName}</td>
                                  <td className="py-2 px-3 text-center">{item.unit}</td>
                                  <td className="py-2 px-3 text-right">{item.quantity}</td>
                                  <td className="py-2 px-3 text-right">{formatCurrency(item.unit_price || item.unitPrice)}</td>
                                  <td className="py-2 px-3 text-right font-medium">{formatCurrency(item.amount)}</td>
                                </tr>
                              ))}
                            </tbody>
                          </table>
                        </div>
                      )}

                      {/* 금액 요약 */}
                      <div className="flex justify-end">
                        <div className="w-64 space-y-2">
                          <div className="flex justify-between">
                            <span>공급가액</span>
                            <span>{formatCurrency(selectedQuotation.subtotal)}</span>
                          </div>
                          <div className="flex justify-between">
                            <span>부가세</span>
                            <span>{formatCurrency(selectedQuotation.tax_amount)}</span>
                          </div>
                          <div className="flex justify-between font-bold text-lg border-t pt-2">
                            <span>합계</span>
                            <span className="text-blue-500">{formatCurrency(selectedQuotation.total_amount)}</span>
                          </div>
                        </div>
                      </div>

                      {/* 상태 변경 버튼 */}
                      <div className="flex gap-2 pt-4 border-t flex-wrap">
                        <button
                          onClick={() => {
                            setSelectedQuotation(null);
                            router.push(`/quotation/view?id=${selectedQuotation.id}`);
                          }}
                          className="px-4 py-2 bg-gray-700 text-white rounded hover:bg-gray-800 flex items-center gap-2"
                        >
                          <FileText className="w-4 h-4" />
                          HTML 보기
                        </button>
                        {selectedQuotation.status === 'draft' && (
                          <button
                            onClick={() => updateQuotationStatus(selectedQuotation.id, 'sent')}
                            className="px-4 py-2 bg-blue-500 text-white rounded hover:bg-blue-600"
                          >
                            발송 처리
                          </button>
                        )}
                        {selectedQuotation.status === 'sent' && (
                          <>
                            <button
                              onClick={() => updateQuotationStatus(selectedQuotation.id, 'accepted')}
                              className="px-4 py-2 bg-green-500 text-white rounded hover:bg-green-600"
                            >
                              수락됨
                            </button>
                            <button
                              onClick={() => updateQuotationStatus(selectedQuotation.id, 'rejected')}
                              className="px-4 py-2 bg-red-500 text-white rounded hover:bg-red-600"
                            >
                              거절됨
                            </button>
                          </>
                        )}
                        <button
                          onClick={() => deleteQuotation(selectedQuotation.id)}
                          className="px-4 py-2 border border-red-500 text-red-500 rounded hover:bg-red-50 dark:hover:bg-red-900/30"
                        >
                          삭제
                        </button>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            )}
          </div>
        )}

        {/* 설정 탭 */}
        {activeTab === 'settings' && (
          <div className="space-y-6">
            {/* 회사 정보 설정 */}
            <div className="p-6 rounded-lg shadow" style={{ backgroundColor: 'var(--card-bg)' }}>
              <h2 className="text-lg font-semibold mb-4">회사 정보</h2>
              <p className="text-sm opacity-70 mb-4">견적서에 표시될 회사 정보를 설정합니다.</p>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium mb-1">회사명 *</label>
                  <input
                    type="text"
                    value={companyProfile.companyName}
                    onChange={(e) => setCompanyProfile(prev => ({ ...prev, companyName: e.target.value }))}
                    className="w-full px-3 py-2 border rounded"
                    style={{ backgroundColor: 'var(--input-bg)', borderColor: 'var(--border-color)' }}
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium mb-1">사업자등록번호</label>
                  <input
                    type="text"
                    value={companyProfile.businessNumber}
                    onChange={(e) => setCompanyProfile(prev => ({ ...prev, businessNumber: e.target.value }))}
                    className="w-full px-3 py-2 border rounded"
                    style={{ backgroundColor: 'var(--input-bg)', borderColor: 'var(--border-color)' }}
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium mb-1">대표자명</label>
                  <input
                    type="text"
                    value={companyProfile.ceoName}
                    onChange={(e) => setCompanyProfile(prev => ({ ...prev, ceoName: e.target.value }))}
                    className="w-full px-3 py-2 border rounded"
                    style={{ backgroundColor: 'var(--input-bg)', borderColor: 'var(--border-color)' }}
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium mb-1">전화번호</label>
                  <input
                    type="text"
                    value={companyProfile.phone}
                    onChange={(e) => setCompanyProfile(prev => ({ ...prev, phone: e.target.value }))}
                    className="w-full px-3 py-2 border rounded"
                    style={{ backgroundColor: 'var(--input-bg)', borderColor: 'var(--border-color)' }}
                  />
                </div>
                <div className="col-span-2">
                  <label className="block text-sm font-medium mb-1">주소</label>
                  <input
                    type="text"
                    value={companyProfile.address}
                    onChange={(e) => setCompanyProfile(prev => ({ ...prev, address: e.target.value }))}
                    className="w-full px-3 py-2 border rounded"
                    style={{ backgroundColor: 'var(--input-bg)', borderColor: 'var(--border-color)' }}
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium mb-1">이메일</label>
                  <input
                    type="email"
                    value={companyProfile.email}
                    onChange={(e) => setCompanyProfile(prev => ({ ...prev, email: e.target.value }))}
                    className="w-full px-3 py-2 border rounded"
                    style={{ backgroundColor: 'var(--input-bg)', borderColor: 'var(--border-color)' }}
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium mb-1">팩스</label>
                  <input
                    type="text"
                    value={companyProfile.fax}
                    onChange={(e) => setCompanyProfile(prev => ({ ...prev, fax: e.target.value }))}
                    className="w-full px-3 py-2 border rounded"
                    style={{ backgroundColor: 'var(--input-bg)', borderColor: 'var(--border-color)' }}
                  />
                </div>
              </div>

              <h3 className="font-medium mt-6 mb-3">계좌 정보</h3>
              <div className="grid grid-cols-3 gap-4">
                <div>
                  <label className="block text-sm font-medium mb-1">은행명</label>
                  <input
                    type="text"
                    value={companyProfile.bankName}
                    onChange={(e) => setCompanyProfile(prev => ({ ...prev, bankName: e.target.value }))}
                    className="w-full px-3 py-2 border rounded"
                    style={{ backgroundColor: 'var(--input-bg)', borderColor: 'var(--border-color)' }}
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium mb-1">계좌번호</label>
                  <input
                    type="text"
                    value={companyProfile.bankAccount}
                    onChange={(e) => setCompanyProfile(prev => ({ ...prev, bankAccount: e.target.value }))}
                    className="w-full px-3 py-2 border rounded"
                    style={{ backgroundColor: 'var(--input-bg)', borderColor: 'var(--border-color)' }}
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium mb-1">예금주</label>
                  <input
                    type="text"
                    value={companyProfile.bankHolder}
                    onChange={(e) => setCompanyProfile(prev => ({ ...prev, bankHolder: e.target.value }))}
                    className="w-full px-3 py-2 border rounded"
                    style={{ backgroundColor: 'var(--input-bg)', borderColor: 'var(--border-color)' }}
                  />
                </div>
              </div>

              {/* 로고 업로드 섹션 */}
              <h3 className="font-medium mt-6 mb-3">회사 로고</h3>
              <div className="flex items-start gap-4">
                {companyProfile.logoPath ? (
                  <div className="relative">
                    <img
                      src={companyProfile.logoPath}
                      alt="회사 로고"
                      className="w-32 h-32 object-contain border rounded"
                      style={{ borderColor: 'var(--border-color)' }}
                    />
                    <button
                      onClick={handleLogoDelete}
                      className="absolute -top-2 -right-2 bg-red-500 text-white rounded-full p-1 hover:bg-red-600"
                      title="로고 삭제"
                    >
                      <X className="w-4 h-4" />
                    </button>
                  </div>
                ) : (
                  <label className="flex flex-col items-center justify-center w-32 h-32 border-2 border-dashed rounded cursor-pointer hover:bg-gray-50 dark:hover:bg-gray-700" style={{ borderColor: 'var(--border-color)' }}>
                    <Upload className="w-8 h-8 opacity-50 mb-2" />
                    <span className="text-xs opacity-70">로고 업로드</span>
                    <input
                      type="file"
                      accept="image/*"
                      onChange={handleLogoUpload}
                      className="hidden"
                    />
                  </label>
                )}
                <div className="text-sm opacity-70">
                  <p>견적서 상단에 표시될 회사 로고입니다.</p>
                  <p className="mt-1">권장 크기: 200x80 픽셀</p>
                  <p>지원 형식: PNG, JPG, GIF (최대 2MB)</p>
                </div>
              </div>

              <div className="mt-6">
                <button
                  onClick={saveCompanyProfile}
                  disabled={isLoading || !companyProfile.companyName}
                  className="px-6 py-2 bg-blue-500 text-white rounded hover:bg-blue-600 disabled:opacity-50"
                >
                  {isLoading ? '저장 중...' : '저장'}
                </button>
              </div>
            </div>

            {/* 단가 카탈로그 */}
            <div className="p-6 rounded-lg shadow" style={{ backgroundColor: 'var(--card-bg)' }}>
              <h2 className="text-lg font-semibold mb-4">단가 카탈로그</h2>
              <p className="text-sm opacity-70 mb-4">
                자주 사용하는 품목과 단가를 등록해두면 AI가 견적서 작성 시 참조합니다.
              </p>

              {/* 새 항목 추가 */}
              <div className="grid grid-cols-6 gap-2 mb-4 pb-4 border-b" style={{ borderColor: 'var(--border-color)' }}>
                <input
                  type="text"
                  placeholder="카테고리"
                  value={newPriceItem.category}
                  onChange={(e) => setNewPriceItem(prev => ({ ...prev, category: e.target.value }))}
                  className="px-2 py-1 border rounded text-sm"
                  style={{ backgroundColor: 'var(--input-bg)', borderColor: 'var(--border-color)' }}
                />
                <input
                  type="text"
                  placeholder="품목명 *"
                  value={newPriceItem.itemName}
                  onChange={(e) => setNewPriceItem(prev => ({ ...prev, itemName: e.target.value }))}
                  className="px-2 py-1 border rounded text-sm col-span-2"
                  style={{ backgroundColor: 'var(--input-bg)', borderColor: 'var(--border-color)' }}
                />
                <input
                  type="text"
                  placeholder="단위"
                  value={newPriceItem.unit}
                  onChange={(e) => setNewPriceItem(prev => ({ ...prev, unit: e.target.value }))}
                  className="px-2 py-1 border rounded text-sm"
                  style={{ backgroundColor: 'var(--input-bg)', borderColor: 'var(--border-color)' }}
                />
                <input
                  type="number"
                  placeholder="단가 *"
                  value={newPriceItem.unitPrice || ''}
                  onChange={(e) => setNewPriceItem(prev => ({ ...prev, unitPrice: parseFloat(e.target.value) || 0 }))}
                  className="px-2 py-1 border rounded text-sm"
                  style={{ backgroundColor: 'var(--input-bg)', borderColor: 'var(--border-color)' }}
                />
                <button
                  onClick={addPriceItem}
                  disabled={isLoading}
                  className="px-3 py-1 bg-green-500 text-white rounded text-sm hover:bg-green-600 disabled:opacity-50"
                >
                  추가
                </button>
              </div>

              {/* 기존 항목 목록 */}
              {priceCatalog.length === 0 ? (
                <p className="text-center opacity-70 py-4">등록된 단가 항목이 없습니다.</p>
              ) : (
                <div className="overflow-x-auto">
                  <table className="w-full text-sm">
                    <thead>
                      <tr className="border-b" style={{ borderColor: 'var(--border-color)' }}>
                        <th className="py-2 px-3 text-left">카테고리</th>
                        <th className="py-2 px-3 text-left">품목명</th>
                        <th className="py-2 px-3 text-left">설명</th>
                        <th className="py-2 px-3 text-center">단위</th>
                        <th className="py-2 px-3 text-right">단가</th>
                      </tr>
                    </thead>
                    <tbody>
                      {priceCatalog.map((item) => (
                        <tr key={item.id} className="border-b" style={{ borderColor: 'var(--border-color)' }}>
                          <td className="py-2 px-3">{item.category || '-'}</td>
                          <td className="py-2 px-3 font-medium">{item.item_name}</td>
                          <td className="py-2 px-3 opacity-70">{item.description || '-'}</td>
                          <td className="py-2 px-3 text-center">{item.unit}</td>
                          <td className="py-2 px-3 text-right">{formatCurrency(item.unit_price)}</td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
              )}
            </div>
          </div>
        )}
      </div>
    </div>
  );
}
