'use client';

import { useState, useEffect, useRef, Suspense } from 'react';
import { useRouter, useSearchParams } from 'next/navigation';
import { Printer, Download, ArrowLeft, CheckCircle, XCircle, AlertTriangle } from 'lucide-react';

interface QuotationItem {
  id?: number;
  itemOrder: number;
  itemName: string;
  description?: string;
  specification?: string;
  unit: string;
  quantity: number;
  unitPrice: number;
  amount: number;
  notes?: string;
}

interface Quotation {
  id: number;
  quotation_number: string;
  quotation_date: string;
  validity_date: string;
  title: string;
  client_company: string;
  client_contact?: string;
  client_phone?: string;
  client_email?: string;
  client_address?: string;
  subtotal: number;
  supply_amount?: number;
  tax_rate: number;
  tax_amount: number;
  vat_amount?: number;
  total_amount: number;
  currency: string;
  status: string;
  generated_by_ai: boolean;
  notes?: string;
  items?: QuotationItem[];
  created_at: string;
  updated_at: string;
  project_name?: string;
  validity_period?: number;
  // 공급자 정보
  company_name?: string;
  business_number?: string;
  ceo_name?: string;
  company_address?: string;
  company_phone?: string;
  company_fax?: string;
  company_email?: string;
  logo_path?: string;
  seal_path?: string;
  bank_name?: string;
  bank_account?: string;
  bank_holder?: string;
}

interface CompanyProfile {
  company_name?: string;
  business_number?: string;
  ceo_name?: string;
  address?: string;
  phone?: string;
  fax?: string;
  email?: string;
  website?: string;
  logo_path?: string;
  seal_path?: string;
  bank_name?: string;
  bank_account?: string;
  bank_holder?: string;
}

// 로딩 컴포넌트
function LoadingFallback() {
  return (
    <div className="min-h-screen flex items-center justify-center" style={{ backgroundColor: 'var(--body-bg)' }}>
      <div className="text-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500 mx-auto mb-4"></div>
        <p>견적서를 불러오는 중...</p>
      </div>
    </div>
  );
}

// 메인 페이지 컴포넌트 (Suspense로 감싸서 export)
export default function QuotationViewPage() {
  return (
    <Suspense fallback={<LoadingFallback />}>
      <QuotationViewContent />
    </Suspense>
  );
}

// 실제 컨텐츠 컴포넌트
function QuotationViewContent() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const quotationId = searchParams.get('id');

  const [quotation, setQuotation] = useState<Quotation | null>(null);
  const [companyProfile, setCompanyProfile] = useState<CompanyProfile | null>(null);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const printRef = useRef<HTMLDivElement>(null);

  // API 호출 헬퍼
  const apiCall = async (endpoint: string) => {
    const apiKey = typeof window !== 'undefined' ? localStorage.getItem('apiKey') : null;
    const token = typeof window !== 'undefined' ? localStorage.getItem('userToken') : null;

    const headers: Record<string, string> = {
      'Content-Type': 'application/json',
      ...(apiKey && { 'X-API-Key': apiKey }),
      ...(token && { 'Authorization': `Bearer ${token}` })
    };

    const response = await fetch(`/api/quotation${endpoint}`, { headers });
    const data = await response.json();

    if (!response.ok || !data.success) {
      throw new Error(data.error?.message || '요청 실패');
    }

    return data;
  };

  // 데이터 로드
  useEffect(() => {
    const loadData = async () => {
      if (!quotationId) {
        setError('견적서 ID가 필요합니다.');
        setIsLoading(false);
        return;
      }

      try {
        setIsLoading(true);

        // 견적서와 회사 정보 동시 로드
        const [quotationData, companyData] = await Promise.all([
          apiCall(`/${quotationId}`),
          apiCall('/company/profile').catch(() => ({ data: null }))
        ]);

        setQuotation(quotationData.data);
        setCompanyProfile(companyData.data);
      } catch (err: any) {
        setError(err.message);
      } finally {
        setIsLoading(false);
      }
    };

    loadData();
  }, [quotationId]);

  // 금액 포맷팅
  const formatCurrency = (amount: number | string | null | undefined) => {
    // null, undefined, NaN 처리
    if (amount === null || amount === undefined) return '0';

    // 문자열인 경우 숫자로 변환 (API가 "29250000.00" 형태의 문자열로 반환)
    const numAmount = typeof amount === 'string' ? parseFloat(amount) : amount;

    // NaN 체크
    if (isNaN(numAmount)) return '0';

    return new Intl.NumberFormat('ko-KR').format(numAmount);
  };

  // 날짜 포맷팅
  const formatDate = (dateStr: string) => {
    if (!dateStr) return '-';
    const date = new Date(dateStr);
    return `${date.getFullYear()}년 ${date.getMonth() + 1}월 ${date.getDate()}일`;
  };

  // 인쇄
  const handlePrint = () => {
    window.print();
  };

  // PDF 다운로드 (인쇄 대화상자 이용)
  const handleDownloadPDF = () => {
    window.print();
  };

  if (isLoading) {
    return (
      <div className="min-h-screen flex items-center justify-center" style={{ backgroundColor: 'var(--body-bg)' }}>
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500 mx-auto mb-4"></div>
          <p>견적서를 불러오는 중...</p>
        </div>
      </div>
    );
  }

  if (error || !quotation) {
    return (
      <div className="min-h-screen flex items-center justify-center" style={{ backgroundColor: 'var(--body-bg)' }}>
        <div className="text-center">
          <XCircle className="w-16 h-16 text-red-500 mx-auto mb-4" />
          <p className="text-xl mb-4">{error || '견적서를 찾을 수 없습니다.'}</p>
          <button
            onClick={() => router.push('/quotation')}
            className="px-4 py-2 bg-blue-500 text-white rounded hover:bg-blue-600"
          >
            목록으로 돌아가기
          </button>
        </div>
      </div>
    );
  }

  // 공급가액과 부가세 계산
  const supplyAmount = quotation.supply_amount || quotation.subtotal || 0;
  const vatAmount = quotation.vat_amount || quotation.tax_amount || Math.round(supplyAmount * 0.1);
  const totalAmount = quotation.total_amount || (supplyAmount + vatAmount);

  return (
    <>
      {/* 인쇄용 스타일 */}
      <style jsx global>{`
        @media print {
          body * {
            visibility: hidden;
          }
          .print-area, .print-area * {
            visibility: visible;
          }
          .print-area {
            position: absolute;
            left: 0;
            top: 0;
            width: 210mm;
            padding: 15mm;
            background: white !important;
            -webkit-print-color-adjust: exact;
            print-color-adjust: exact;
          }
          .no-print {
            display: none !important;
          }
          @page {
            size: A4;
            margin: 10mm;
          }
        }
      `}</style>

      <div className="min-h-screen" style={{ backgroundColor: 'var(--body-bg)', color: 'var(--text-primary)' }}>
        {/* 상단 툴바 (인쇄 시 숨김) */}
        <div className="no-print sticky top-0 z-50 border-b" style={{ backgroundColor: 'var(--card-bg)', borderColor: 'var(--border-color)' }}>
          <div className="max-w-5xl mx-auto px-4 py-3 flex items-center justify-between">
            <button
              onClick={() => router.push('/quotation')}
              className="flex items-center gap-2 text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white"
            >
              <ArrowLeft className="w-5 h-5" />
              <span>목록으로</span>
            </button>
            <div className="flex items-center gap-3">
              <button
                onClick={handlePrint}
                className="flex items-center gap-2 px-4 py-2 border rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700"
                style={{ borderColor: 'var(--border-color)' }}
              >
                <Printer className="w-4 h-4" />
                <span>인쇄</span>
              </button>
              <button
                onClick={handleDownloadPDF}
                className="flex items-center gap-2 px-4 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600"
              >
                <Download className="w-4 h-4" />
                <span>PDF 저장</span>
              </button>
            </div>
          </div>
        </div>

        {/* 견적서 본문 */}
        <div className="max-w-5xl mx-auto py-8 px-4">
          <div ref={printRef} className="print-area bg-white shadow-xl rounded-lg overflow-hidden" style={{ color: '#1a1a1a' }}>
            {/* 견적서 헤더 */}
            <div className="bg-gradient-to-r from-blue-600 to-blue-800 text-white px-8 py-6">
              <div className="flex justify-between items-start">
                <div className="flex items-center gap-4">
                  {/* 회사 로고 */}
                  {(quotation.logo_path || companyProfile?.logo_path) && (
                    <div className="bg-white rounded-lg p-2 shadow-md">
                      <img
                        src={quotation.logo_path || companyProfile?.logo_path}
                        alt="회사 로고"
                        className="h-16 w-auto object-contain"
                        style={{ maxWidth: '120px' }}
                      />
                    </div>
                  )}
                  <div>
                    <h1 className="text-3xl font-bold tracking-wide">견 적 서</h1>
                    <p className="text-blue-200 mt-1">QUOTATION</p>
                  </div>
                </div>
                <div className="text-right">
                  <p className="text-sm text-blue-200">견적번호</p>
                  <p className="text-xl font-mono font-bold">{quotation.quotation_number}</p>
                </div>
              </div>
            </div>

            {/* 견적 기본 정보 */}
            <div className="px-8 py-6 border-b" style={{ borderColor: '#e5e7eb' }}>
              <div className="grid grid-cols-2 gap-8">
                <div className="space-y-3">
                  <div className="flex">
                    <span className="w-24 text-gray-500 text-sm">견적일자</span>
                    <span className="font-medium">{formatDate(quotation.quotation_date || quotation.created_at)}</span>
                  </div>
                  <div className="flex">
                    <span className="w-24 text-gray-500 text-sm">유효기간</span>
                    <span className="font-medium">견적일로부터 {quotation.validity_period || 30}일</span>
                  </div>
                </div>
                <div className="text-right">
                  <div className="inline-block bg-blue-50 px-4 py-3 rounded-lg">
                    <p className="text-sm text-gray-500 mb-1">견적 총액</p>
                    <p className="text-2xl font-bold text-blue-600">
                      {formatCurrency(totalAmount)}
                      <span className="text-lg ml-1">원</span>
                    </p>
                    <p className="text-xs text-gray-400 mt-1">(VAT 포함)</p>
                  </div>
                </div>
              </div>
            </div>

            {/* 프로젝트명 */}
            <div className="px-8 py-4 bg-gray-50 border-b" style={{ borderColor: '#e5e7eb' }}>
              <div className="flex items-center">
                <span className="text-gray-500 text-sm mr-4">프로젝트명</span>
                <span className="text-lg font-semibold text-gray-800">
                  {quotation.project_name || quotation.title || '-'}
                </span>
              </div>
            </div>

            {/* 거래 당사자 정보 */}
            <div className="px-8 py-6 grid grid-cols-2 gap-8 border-b" style={{ borderColor: '#e5e7eb' }}>
              {/* 공급자 */}
              <div className="bg-gray-50 rounded-lg p-5">
                <h3 className="text-sm font-semibold text-gray-500 mb-4 uppercase tracking-wider">공급자</h3>
                <div className="space-y-2">
                  <p className="text-lg font-bold text-gray-800">
                    {quotation.company_name || companyProfile?.company_name || '인베슘 주식회사'}
                  </p>
                  <div className="text-sm space-y-1 text-gray-600">
                    <p>
                      <span className="text-gray-400 mr-2">대표</span>
                      {quotation.ceo_name || companyProfile?.ceo_name || '김인베'}
                    </p>
                    <p>
                      <span className="text-gray-400 mr-2">사업자번호</span>
                      {quotation.business_number || companyProfile?.business_number || '123-45-67890'}
                    </p>
                    <p>
                      <span className="text-gray-400 mr-2">주소</span>
                      {quotation.company_address || companyProfile?.address || '서울시 강남구 테헤란로 123'}
                    </p>
                    <p>
                      <span className="text-gray-400 mr-2">연락처</span>
                      {quotation.company_phone || companyProfile?.phone || '02-1234-5678'}
                    </p>
                    {(quotation.company_email || companyProfile?.email) && (
                      <p>
                        <span className="text-gray-400 mr-2">이메일</span>
                        {quotation.company_email || companyProfile?.email}
                      </p>
                    )}
                  </div>
                </div>
              </div>

              {/* 수요자 */}
              <div className="bg-blue-50 rounded-lg p-5">
                <h3 className="text-sm font-semibold text-blue-600 mb-4 uppercase tracking-wider">수요자 (귀하)</h3>
                <div className="space-y-2">
                  <p className="text-lg font-bold text-gray-800">{quotation.client_company}</p>
                  <div className="text-sm space-y-1 text-gray-600">
                    {quotation.client_contact && (
                      <p>
                        <span className="text-gray-400 mr-2">담당자</span>
                        {quotation.client_contact}
                      </p>
                    )}
                    {quotation.client_phone && (
                      <p>
                        <span className="text-gray-400 mr-2">연락처</span>
                        {quotation.client_phone}
                      </p>
                    )}
                    {quotation.client_email && (
                      <p>
                        <span className="text-gray-400 mr-2">이메일</span>
                        {quotation.client_email}
                      </p>
                    )}
                    {quotation.client_address && (
                      <p>
                        <span className="text-gray-400 mr-2">주소</span>
                        {quotation.client_address}
                      </p>
                    )}
                  </div>
                </div>
              </div>
            </div>

            {/* 견적 품목 테이블 */}
            <div className="px-8 py-6">
              <h3 className="text-sm font-semibold text-gray-500 mb-4 uppercase tracking-wider">견적 품목</h3>
              <div className="overflow-hidden rounded-lg border" style={{ borderColor: '#e5e7eb' }}>
                <table className="w-full">
                  <thead>
                    <tr className="bg-gray-800 text-white text-sm">
                      <th className="py-3 px-4 text-center w-12">No.</th>
                      <th className="py-3 px-4 text-left">품목명</th>
                      <th className="py-3 px-4 text-left">규격/사양</th>
                      <th className="py-3 px-4 text-center w-16">단위</th>
                      <th className="py-3 px-4 text-center w-16">수량</th>
                      <th className="py-3 px-4 text-right w-28">단가</th>
                      <th className="py-3 px-4 text-right w-32">금액</th>
                    </tr>
                  </thead>
                  <tbody>
                    {quotation.items && quotation.items.length > 0 ? (
                      quotation.items.map((item: any, index) => (
                        <tr key={index} className={`border-b ${index % 2 === 0 ? 'bg-white' : 'bg-gray-50'}`} style={{ borderColor: '#e5e7eb' }}>
                          <td className="py-3 px-4 text-center text-gray-500">{item.item_order || item.itemOrder || index + 1}</td>
                          <td className="py-3 px-4 font-medium text-gray-800">{item.item_name || item.itemName}</td>
                          <td className="py-3 px-4 text-sm text-gray-600">{item.specification || item.description || item.notes || '-'}</td>
                          <td className="py-3 px-4 text-center text-gray-600">{item.unit}</td>
                          <td className="py-3 px-4 text-center text-gray-600">{item.quantity}</td>
                          <td className="py-3 px-4 text-right text-gray-600">{formatCurrency(item.unit_price || item.unitPrice)}</td>
                          <td className="py-3 px-4 text-right font-semibold text-gray-800">{formatCurrency(item.amount)}</td>
                        </tr>
                      ))
                    ) : (
                      <tr>
                        <td colSpan={7} className="py-8 text-center text-gray-400">
                          품목 정보가 없습니다.
                        </td>
                      </tr>
                    )}
                  </tbody>
                </table>
              </div>
            </div>

            {/* 금액 요약 */}
            <div className="px-8 py-6 bg-gray-50 border-t" style={{ borderColor: '#e5e7eb' }}>
              <div className="flex justify-end">
                <div className="w-80">
                  <div className="space-y-3">
                    <div className="flex justify-between text-gray-600">
                      <span>공급가액</span>
                      <span className="font-medium">{formatCurrency(supplyAmount)}원</span>
                    </div>
                    <div className="flex justify-between text-gray-600">
                      <span>부가세 (10%)</span>
                      <span className="font-medium">{formatCurrency(vatAmount)}원</span>
                    </div>
                    <div className="border-t pt-3" style={{ borderColor: '#d1d5db' }}>
                      <div className="flex justify-between items-center">
                        <span className="text-lg font-bold text-gray-800">합계</span>
                        <span className="text-2xl font-bold text-blue-600">{formatCurrency(totalAmount)}원</span>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </div>

            {/* 특기사항 */}
            {quotation.notes && (
              <div className="px-8 py-6 border-t" style={{ borderColor: '#e5e7eb' }}>
                <h3 className="text-sm font-semibold text-gray-500 mb-3 uppercase tracking-wider">특기사항</h3>
                <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                  <div className="text-sm text-gray-700 whitespace-pre-line">{quotation.notes}</div>
                </div>
              </div>
            )}

            {/* 계좌 정보 */}
            {(companyProfile?.bank_name || quotation.bank_name) && (
              <div className="px-8 py-6 border-t" style={{ borderColor: '#e5e7eb' }}>
                <h3 className="text-sm font-semibold text-gray-500 mb-3 uppercase tracking-wider">입금 계좌</h3>
                <div className="bg-gray-100 rounded-lg p-4 inline-block">
                  <p className="font-medium text-gray-800">
                    {quotation.bank_name || companyProfile?.bank_name} {quotation.bank_account || companyProfile?.bank_account}
                  </p>
                  <p className="text-sm text-gray-600">
                    예금주: {quotation.bank_holder || companyProfile?.bank_holder || quotation.company_name || companyProfile?.company_name}
                  </p>
                </div>
              </div>
            )}

            {/* 푸터 */}
            <div className="px-8 py-6 bg-gray-800 text-white">
              <div className="flex justify-between items-center">
                <div>
                  <p className="text-lg font-bold">{quotation.company_name || companyProfile?.company_name || '인베슘 주식회사'}</p>
                  <p className="text-sm text-gray-400 mt-1">
                    본 견적서는 상기 유효기간 내에 한하여 유효합니다.
                  </p>
                </div>
                <div className="text-right">
                  {quotation.generated_by_ai && (
                    <div className="inline-flex items-center gap-2 bg-purple-600 px-3 py-1 rounded-full text-sm mb-2">
                      <span>AI Generated</span>
                    </div>
                  )}
                  <p className="text-sm text-gray-400">
                    발행일: {formatDate(quotation.created_at)}
                  </p>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </>
  );
}
