'use client';

import { useState, useEffect, useMemo } from 'react';
import { externalApiClient as apiClient } from '@/lib/apiClient';
import { setConfig, getConfig } from '@/lib/apiClient';

// 그룹/프로바이더/항목 구조 명시
const configGroups = [
  {
    name: 'provider',
    title: '프로바이더',
    providers: [
      {
        name: 'openai',
        title: 'OpenAI',
        keys: ['OPENAI_API_KEY', 'OPENAI_MODEL'],
      },
      {
        name: 'anthropic',
        title: 'Anthropic',
        keys: ['ANTHROPIC_API_KEY', 'ANTHROPIC_MODEL'],
      },
      {
        name: 'gemini',
        title: 'Gemini',
        keys: ['GEMINI_API_KEY', 'GEMINI_MODEL'],
      },
      {
        name: 'groq',
        title: 'Groq',
        keys: ['GROQ_API_KEY', 'GROQ_MODEL'],
      },
      {
        name: 'ollama',
        title: 'Ollama',
        keys: ['OLLAMA_MODEL', 'NEXT_PUBLIC_OLLAMA_PROXY_SERVER'],
      },
      {
        name: 'vllm',
        title: 'VLLM',
        keys: ['VLLM_MODEL', 'NEXT_PUBLIC_VLLM_SERVER'],
      },
    ],
  },
  {
    name: 'ai',
    title: 'AI/모델',
    keys: ['USE_LLM', 'USE_LLM_MODEL'],
  },
  {
    name: 'rag',
    title: 'RAG',
    keys: [
      'RAG_CHUNK_OVERLAP',
      'RAG_CHUNK_SIZE',
      'RAG_DETECT_COLUMN_TYPE',
      'RAG_DOCS_PATH',
      'RAG_EMBEDDING_MODEL',
      'RAG_IMAGE_EMBEDDING_MODEL',
      'RAG_KEYWORD_PRECISION_THRESHOLD',
      'RAG_SEMANTIC_RELEVANCE_THRESHOLD', 
      'RAG_CONTENT_QUALITY_THRESHOLD',
      'RAG_RECENCY_THRESHOLD',
      'RAG_FORCE_CPU',
      'RAG_FORCE_PDF_CONVERSION',
      'RAG_LARGE_CONTEXT_CHUNKING',
      'USE_ENHANCED_EMBEDDING',
      'ENHANCED_EMBEDDING_STRATEGY',
      'RAG_GENERATE_IMAGE_EMBEDDINGS',
      'RAG_ENABLE_GRAPHRAG',
      'RAG_SEARCH_MODE',
      'RAG_GRAPHRAG_ENTITY_TYPES',
      'RAG_GRAPHRAG_MAX_GLEANINGS',
      'RAG_GRAPHRAG_COMMUNITY_DETECTION',
      'RAG_MODE',
      'RAG_CHUNKING_STRATEGY',
      'RAG_PDF_BACKEND',
      'RAG_PROCESS_TEXT_ONLY',
      'RAG_SEMANTIC_CHUNKER',
      'RAG_SENTENCE_TRANSFORMER_MODEL',
      'RAG_TOP_K',
    ],
  },
  {
    name: 'email',
    title: '이메일/알림',
    keys: [
      'SMTP_HOST',
      'SMTP_PASSWORD',
      'SMTP_PORT',
      'SMTP_SECURE',
      'SMTP_USERNAME',
      'USE_SMTP',
    ],
  },
  {
    name: 'database',
    title: '데이터베이스',
    keys: [
      'DB_HOST',
      'DB_PORT',
      'DB_USER',
      'DB_PASSWORD',
      'DB_NAME',
      'DB_POOL_MAX_SIZE',
      'DB_POOL_MIN_SIZE',
    ],
  },
  {
    name: 'path',
    title: '시스템',
    keys: [
      'AIRUN_API_KEY',
      'AIRUN_HOME_PATH',
      'AIRUN_PATH',
      'PYTHON_VENV_PATH',
      'SERVER_URL',
      'TRUST_PROXY',
    ],
  },
  {
    name: 'feature',
    title: '서비스 설정',
    keys: [
      'AUTO_ANALYZE',
      'AUTO_EXECUTE',
      'AUTO_PROVIDER_SELECTION',
      'DUAL_MODEL',
      'USE_AI_KEYWORD',
      'USE_COT',
      'USE_LLM',
      'USE_REVIEW',
      'USE_VLLM',
      'UPDATE_CHECK',
      'AUTO_UPDATE',
      'LANGUAGE',
    ],
  },
  {
    name: 'license',
    title: '라이선스 정보',
    keys: [],
  },
];

// 항목별 타입/설명 매핑 (실제 conf에 있는 값만)
const configMeta: Record<string, {
  label: string;
  type: 'string' | 'number' | 'boolean' | 'password';
  readonly?: boolean;
  description?: string;
}> = {
  OPENAI_API_KEY: { label: 'OpenAI API 키', type: 'password' },
  OPENAI_MODEL: { label: 'OpenAI 모델', type: 'string' },
  ANTHROPIC_API_KEY: { label: 'Anthropic API 키', type: 'password' },
  ANTHROPIC_MODEL: { label: 'Anthropic 모델', type: 'string' },
  GEMINI_API_KEY: { label: 'Gemini API 키', type: 'password' },
  GEMINI_MODEL: { label: 'Gemini 모델', type: 'string' },
  GROQ_API_KEY: { label: 'Groq API 키', type: 'password' },
  GROQ_MODEL: { label: 'Groq 모델', type: 'string' },
  OLLAMA_MODEL: { label: 'Ollama 모델', type: 'string' },
  // 데이터베이스 설정
  DB_HOST: { label: '데이터베이스 호스트', type: 'string', readonly: true, description: '시스템 설정으로 변경 불가' },
  DB_PORT: { label: '데이터베이스 포트', type: 'number', readonly: true, description: '시스템 설정으로 변경 불가' },
  DB_USER: { label: '데이터베이스 사용자', type: 'string', description: '새 사용자명을 입력하면 PostgreSQL에 자동 생성됩니다' },
  DB_PASSWORD: { label: '데이터베이스 비밀번호', type: 'password', description: '실제 PostgreSQL 사용자 비밀번호가 변경됩니다' },
  DB_NAME: { label: '데이터베이스 이름', type: 'string', readonly: true, description: '시스템 설정으로 변경 불가' },
  DB_POOL_MAX_SIZE: { label: '최대 연결 수', type: 'number', readonly: true, description: '시스템 설정으로 변경 불가' },
  DB_POOL_MIN_SIZE: { label: '최소 연결 수', type: 'number', readonly: true, description: '시스템 설정으로 변경 불가' },
  VLLM_MODEL: { label: 'VLLM 모델', type: 'string' },
  NEXT_PUBLIC_VLLM_SERVER: { label: 'VLLM 서버', type: 'string' },
  RAG_CHUNK_OVERLAP: { label: '청크 오버랩', type: 'number' },
  RAG_CHUNK_SIZE: { label: '청크 크기', type: 'number' },
  RAG_DETECT_COLUMN_TYPE: { label: '컬럼타입 감지', type: 'boolean' },
  RAG_DOCS_PATH: { label: '문서 경로', type: 'string' },
  RAG_EMBEDDING_MODEL: { label: '문서 임베딩 모델', type: 'string' },
  RAG_IMAGE_EMBEDDING_MODEL: { label: '이미지 임베딩 모델', type: 'string' },
  RAG_KEYWORD_PRECISION_THRESHOLD: { label: '키워드 정확도 임계값', type: 'number' },
  RAG_SEMANTIC_RELEVANCE_THRESHOLD: { label: '의미 관련성 임계값', type: 'number' },
  RAG_CONTENT_QUALITY_THRESHOLD: { label: '내용 품질 임계값', type: 'number' },
  RAG_RECENCY_THRESHOLD: { label: '최신성 임계값', type: 'number' },
  RAG_KEYWORD_PRECISION_WEIGHT: { label: '키워드 정확도 가중치', type: 'number' },
  RAG_SEMANTIC_RELEVANCE_WEIGHT: { label: '의미 관련성 가중치', type: 'number' },
  RAG_CONTENT_QUALITY_WEIGHT: { label: '내용 품질 가중치', type: 'number' },
  RAG_RECENCY_WEIGHT: { label: '최신성 가중치', type: 'number' },
  RAG_FORCE_CPU: { label: 'CPU 강제 사용', type: 'boolean' },
  RAG_FORCE_PDF_CONVERSION: { label: 'PDF 변환 강제', type: 'boolean' },
  RAG_LARGE_CONTEXT_CHUNKING: { label: '파일단위 청크', type: 'boolean' },
  USE_ENHANCED_EMBEDDING: { label: '향상된 임베딩 사용', type: 'boolean' },
  ENHANCED_EMBEDDING_STRATEGY: { label: '임베딩 전략', type: 'string' },
  RAG_GENERATE_IMAGE_EMBEDDINGS: { label: '이미지 임베딩 생성', type: 'boolean' },
  RAG_MODE: { label: '문서처리 모드', type: 'string' },
  RAG_CHUNKING_STRATEGY: { label: '청크 전략', type: 'string' },
  RAG_PDF_BACKEND: { label: 'PDF 백엔드', type: 'string' },
  RAG_PROCESS_TEXT_ONLY: { label: '텍스트만 처리', type: 'boolean' },
  RAG_SEMANTIC_CHUNKER: { label: '시맨틱 청킹', type: 'boolean' },
  RAG_SENTENCE_TRANSFORMER_MODEL: { label: '시맨틱 청킹 모델', type: 'string' },
  RAG_TOP_K: { label: '최종 반환 결과 수', type: 'number' },
  // GraphRAG 관련 설정
  RAG_ENABLE_GRAPHRAG: { label: 'Graph RAG 활성화', type: 'boolean' },
  RAG_SEARCH_MODE: { label: '검색 모드', type: 'string' },
  RAG_GRAPHRAG_ENTITY_TYPES: { label: '엔티티 타입', type: 'string' },
  RAG_GRAPHRAG_MAX_GLEANINGS: { label: '최대 정보 수집 횟수', type: 'number' },
  RAG_GRAPHRAG_COMMUNITY_DETECTION: { label: '커뮤니티 감지', type: 'boolean' },
  SMTP_HOST: { label: 'SMTP Host', type: 'string' },
  SMTP_PASSWORD: { label: 'SMTP 비밀번호', type: 'password' },
  SMTP_PORT: { label: 'SMTP Port', type: 'number' },
  SMTP_SECURE: { label: 'SMTP 보안', type: 'boolean' },
  SMTP_USERNAME: { label: 'SMTP 사용자', type: 'string' },
  USE_SMTP: { label: 'SMTP 사용', type: 'boolean' },
      AIRUN_API_KEY: { label: 'AI API 키', type: 'password' },
    AIRUN_HOME_PATH: { label: 'AI 홈 경로', type: 'string' },
    AIRUN_PATH: { label: 'AI 경로', type: 'string' },
  PYTHON_VENV_PATH: { label: 'Python venv 경로', type: 'string' },
  SERVER_URL: { label: 'API 서버 URL', type: 'string' },
  UPDATE_CHECK: { label: '업데이트 확인', type: 'boolean' },
  AIMODE: { label: 'AI 모드', type: 'string' },
  AUTO_ANALYZE: { label: '자동 분석', type: 'boolean' },
  AUTO_EXECUTE: { label: '자동 실행', type: 'boolean' },
  AUTO_PROVIDER_SELECTION: { label: '프로바이더 자동선택', type: 'boolean' },
  AUTO_UPDATE: { label: '자동 업데이트', type: 'boolean' },
  DUAL_MODEL: { label: '듀얼 모델', type: 'boolean' },
  HIDE_CODE: { label: '코드 숨김', type: 'boolean' },
  USE_AI_KEYWORD: { label: 'AI 키워드 사용', type: 'boolean' },
  USE_COT: { label: 'COT 사용', type: 'boolean' },
  USE_RAG: { label: 'RAG 사용', type: 'boolean' },
  USE_REVIEW: { label: '리뷰 사용', type: 'boolean' },
  USE_VLLM: { label: 'VLLM 사용', type: 'boolean' },
  USE_WEB_SEARCH: { label: '웹검색 사용', type: 'boolean' },
  NEXT_PUBLIC_OLLAMA_PROXY_SERVER: { label: 'Ollama 서버', type: 'string' },
  LANGUAGE: { label: '언어', type: 'string' },
  // 서버 보안 설정
  TRUST_PROXY: { label: '프록시 신뢰 설정', type: 'string' },
  // 추가 설정들 - API 서버 기본값과 매핑
  USE_LLM: { label: 'LLM 사용', type: 'string' },
};

// 문자열 conf 값을 boolean으로 변환하는 함수
function parseBool(val: string | undefined): boolean {
  return /^(yes|true|1)$/i.test(String(val).trim());
}

// 탭별 저장 버튼 스타일 함수
function getSaveButtonStyle(groupName: string) {
  switch (groupName) {
    case 'provider':
      return { text: '프로바이더 설정 저장', bg: '#3B82F6', textColor: 'white' }; // blue-500
    case 'ai':
      return { text: 'AI 설정 저장', bg: '#3B82F6', textColor: 'white' }; // blue-500
    case 'rag':
      return { text: 'RAG 설정 저장', bg: '#3B82F6', textColor: 'white' }; // blue-500
    case 'email':
      return { text: '이메일 설정 저장', bg: '#3B82F6', textColor: 'white' }; // blue-500
    case 'path':
      return { text: '시스템 설정 저장', bg: '#3B82F6', textColor: 'white' }; // blue-500
    case 'feature':
      return { text: '서비스 설정 저장', bg: '#3B82F6', textColor: 'white' }; // blue-500
    case 'database':
      return { text: '데이터베이스 설정 저장', bg: '#059669', textColor: 'white' }; // emerald-600
    default:
      return { text: '설정 저장', bg: '#3B82F6', textColor: 'white' }; // blue-500
  }
}

const retrieverGroups = {
  thresholds: {
    title: '임계값 설정',
    description: '검색 결과 필터링을 위한 임계값입니다. 각각 0~1 사이의 값을 설정합니다.',
    keys: [
      'RAG_KEYWORD_PRECISION_THRESHOLD',
      'RAG_SEMANTIC_RELEVANCE_THRESHOLD', 
      'RAG_CONTENT_QUALITY_THRESHOLD',
      'RAG_RECENCY_THRESHOLD',
    ]
  },
  topk: {
    title: '검색 결과 수 설정',
    description: '실제 검색에서는 설정값의 3배수만큼 검색한 후, 가중치를 적용하여 최종 순위를 결정합니다. 예: 3으로 설정하면 9개를 검색하여 상위 3개 반환',
    keys: ['RAG_TOP_K']
  },  
  scoreWeights: {
    title: '가중치 설정',
    description: '검색 결과 순위 결정에 사용되는 가중치입니다. 합계가 1.0이 되어야 합니다.',
    keys: [
      'RAG_KEYWORD_PRECISION_WEIGHT',    // 키워드 정확도 가중치
      'RAG_SEMANTIC_RELEVANCE_WEIGHT',   // 의미 관련성 가중치
      'RAG_CONTENT_QUALITY_WEIGHT',      // 내용 품질 가중치
      'RAG_RECENCY_WEIGHT',               // 최신성 가중치
    ]
  }
};

export default function SettingsPage() {
  const [configs, setConfigs] = useState<Record<string, string>>({});
  const [isLoading, setIsLoading] = useState(true);
  const [isSaving, setIsSaving] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [success, setSuccess] = useState<string | null>(null);
  const [activeGroup, setActiveGroup] = useState<string>('ai');
  const [editedConfigs, setEditedConfigs] = useState<Record<string, string>>({});
  // 사용가능한 프로바이더/모델 상태
  const [providerData, setProviderData] = useState<any>({});
  const [selectedProvider, setSelectedProvider] = useState<string>('');
  const [selectedModel, setSelectedModel] = useState<string>('');
  const [providerApiKey, setProviderApiKey] = useState<string>('');
  const [apiKeySaved, setApiKeySaved] = useState<boolean>(false);
  const [showAdvanced, setShowAdvanced] = useState<boolean>(false);
  // API키 입력란 보이기/숨기기 상태
  const [showApiKey, setShowApiKey] = useState<Record<string, boolean>>({});
  // 1) 프로바이더별 입력값 상태 추가
  const [providerInputs, setProviderInputs] = useState<Record<string, Record<string, string>>>({});
  const [providerSaveStatus, setProviderSaveStatus] = useState<Record<string, string>>({});
  const [ollamaModels, setOllamaModels] = useState<string[]>([]);
  const [initJobId, setInitJobId] = useState<string | null>(null);
  const [initStatus, setInitStatus] = useState<string | null>(null);
  const [initProgress, setInitProgress] = useState<string | null>(null);
  const [isPolling, setIsPolling] = useState(false);
  const [licenseInfo, setLicenseInfo] = useState<any>(null);
  const [licenseLoading, setLicenseLoading] = useState(false);
  const [licenseError, setLicenseError] = useState<string | null>(null);
  const [showResetAllConfirm, setShowResetAllConfirm] = useState(false);
  const [refreshingModels, setRefreshingModels] = useState(false);

  // 로고 업로드 관련 상태 추가
  const [logoFile, setLogoFile] = useState<File | null>(null);
  const [logoPreview, setLogoPreview] = useState<string | null>(null);
  const [logoUploading, setLogoUploading] = useState(false);
  const [currentLogo, setCurrentLogo] = useState<string | null>(null);

  // 모델 목록 새로고침 함수
  const refreshModels = async () => {
    setRefreshingModels(true);
    try {
      console.log(`${currentProviderKey} 모델 목록 새로고침 시도...`);
      const res = await apiClient.getProviderModels(currentProviderKey);
      if (res.success && res.data) {
        const modelNames = res.data.map((m: { id?: string; name?: string }) => m.id || m.name);
        setModelOptions(prev => ({ ...prev, [currentProviderKey]: modelNames }));
        setSuccess(`${currentProviderKey} 모델 목록을 새로고침했습니다! (${modelNames.length}개 모델)`);
        console.log(`${currentProviderKey} 모델 목록:`, res.data);
      } else {
        console.error(`${currentProviderKey} 모델 목록 가져오기 실패:`, res);
        setError(`${currentProviderKey} 모델 목록을 가져올 수 없습니다: ${res.error?.message || '알 수 없는 오류'}`);
      }
    } catch (error: any) {
      console.error('모델 새로고침 실패:', error);
      setError(`모델 새로고침 실패: ${error.message}`);
    } finally {
      setRefreshingModels(false);
    }
  };

  // 1. providerList, currentProviderKey, currentModelKey 계산
  const providerList = useMemo(() => Object.entries(providerData).map(([key, v]) => ({
    key,
    title: (v as any).displayName || key,
  })), [providerData]);
  
  // 유효한 프로바이더 키 확인 및 기본값 설정
  const rawProviderKey = configs['USE_LLM'] ?? '';
  const validProviders = Object.keys(providerData);
  const currentProviderKey = validProviders.includes(rawProviderKey) ? rawProviderKey : (validProviders[0] || 'ollama');
  const currentModelKey = configs[`${currentProviderKey?.toUpperCase()}_MODEL`] ?? '';

  // 2. 모델 리스트 상태 (provider별로 따로 관리)
  const [modelOptions, setModelOptions] = useState<Record<string, string[]>>({});

  // 3. 프로바이더 변경 시 모델 리스트 fetch
  useEffect(() => {
    if (currentProviderKey) {
      apiClient.getProviderModels(currentProviderKey).then(res => {
        if (res.success && res.data) {
          setModelOptions(prev => ({ ...prev, [currentProviderKey]: res.data.map((m: { id?: string; name?: string }) => m.id || m.name) }));
        }
      });
    }
  }, [currentProviderKey]);

  // 디버그
  useEffect(() => {
    // console.log('currentModel', currentModelKey, typeof currentModelKey);
    // console.log('modelOptions', modelOptions);
  }, [currentModelKey, modelOptions]);

  useEffect(() => {
    loadConfigs();
    loadProviders();
    loadCurrentLogo();
  }, []);

  // 디버그: API 서버에서 받아온 값 출력
  useEffect(() => {
    // console.log('providerData', providerData);
    // console.log('configs', configs);
    // console.log('editedConfigs', editedConfigs);
    // console.log('selectedProvider', selectedProvider);
    // console.log('selectedModel', selectedModel);
  }, [providerData, configs, editedConfigs, selectedProvider, selectedModel]);

  useEffect(() => {
    if (activeGroup === 'license') {
      setLicenseLoading(true);
      apiClient.get('/license/status')
        .then(res => {
          if (res.success) {
            setLicenseInfo(res.data);
            setLicenseError(null);
          } else {
            setLicenseError('라이선스 정보를 불러올 수 없습니다: ' + (res.error?.message || ''));
          }
        })
        .catch(err => {
          console.error('라이선스 정보 로드 오류:', err);
          setLicenseError('라이선스 정보를 불러올 수 없습니다.');
        })
        .finally(() => setLicenseLoading(false));
    }
  }, [activeGroup]);

  const loadConfigs = async () => {
    try {
      setIsLoading(true);
      setError(null);
      // 전체 설정을 한 번에 받아옴
      const res = await apiClient.get('/config');
      if (res.success && res.data && res.data.configs) {
        // console.log('🔄 설정 로드 완료:', res.data.configs);
        // console.log('🔍 키워드 매칭 임계값들 로드 확인:');
        // 4-Score System configuration loaded
        
        setConfigs(res.data.configs);
        setEditedConfigs(res.data.configs);
        
        // providerInputs 상태도 동기화 - API 키 값들을 프로바이더별로 정리
        const newProviderInputs: Record<string, Record<string, string>> = {};
        const providerKeys = ['openai', 'anthropic', 'gemini', 'groq', 'ollama', 'vllm'];
        
        providerKeys.forEach(provider => {
          const providerData: Record<string, string> = {};
          Object.entries(res.data.configs).forEach(([key, value]) => {
            if (key.toLowerCase().startsWith(provider.toUpperCase())) {
              providerData[key] = String(value || '');
            }
          });
          if (Object.keys(providerData).length > 0) {
            newProviderInputs[provider] = providerData;
          }
        });
        
        setProviderInputs(newProviderInputs);
        await loadProviders();
      } else {
        setError('설정 로드 실패: ' + (res.error?.message || ''));
      }
    } catch (e) {
      setError('설정 로드 중 오류가 발생했습니다.');
    } finally {
      setIsLoading(false);
    }
  };

  const handleConfigChange = (key: string, value: string) => {
    setEditedConfigs((prev) => ({ ...prev, [key]: value }));
  };

  const saveConfigs = async () => {
    try {
      setIsSaving(true);
      setError(null);
      setSuccess(null);
      // 변경된 값만 추출
      const changed: Record<string, any> = {};
      Object.entries(editedConfigs).forEach(([key, value]) => {
        if (configs[key] !== value) {
          const meta = configMeta[key];
          if (meta?.type === 'boolean') changed[key] = parseBool(value) ? 'yes' : 'no';
          else if (meta?.type === 'number') changed[key] = Number(value);
          else changed[key] = value;
        }
      });
      if (Object.keys(changed).length === 0) {
        setSuccess('변경된 설정이 없습니다.');
        return;
      }
      // 여러 설정을 개별적으로 저장
      const results = await Promise.all(
        Object.entries(changed).map(([key, value]) => setConfig(key, value))
      );
      if (results.every((r) => r)) {
        setSuccess('설정이 성공적으로 저장되었습니다.');
        // 기존 설정과 변경사항만 병합 (전체 재로드 대신)
        const updatedConfigs = { ...configs, ...changed };
        setConfigs(updatedConfigs);
        setEditedConfigs(updatedConfigs);
        
        // API 키 관련 설정이 변경된 경우 프로바이더 정보 다시 로드
        const apiKeyChanged = Object.keys(changed).some(key => 
          key.includes('_API_KEY') || key === 'USE_LLM' || key.includes('_MODEL')
        );
        if (apiKeyChanged) {
          await loadProviders();
        }
      } else {
        setError('설정 저장 실패: 일부 항목 저장에 실패했습니다.');
      }
    } catch (e) {
      setError('설정 저장 중 오류가 발생했습니다.');
    } finally {
      setIsSaving(false);
    }
  };

  // 사용가능한 프로바이더/모델 불러오기
  const loadProviders = async () => {
    try {
      const res = await apiClient.getProviders();
      if (res.success && res.data) {
        // API 키 상태를 현재 설정과 동기화
        const updatedProviderData = { ...res.data };
        
        // 각 프로바이더의 API 키 상태를 현재 설정에서 확인하여 업데이트
        Object.keys(updatedProviderData).forEach(providerKey => {
          const provider = updatedProviderData[providerKey];
          if (provider.requiresApiKey) {
            // 현재 설정에서 API 키 확인
            const apiKeyKey = `${providerKey.toUpperCase()}_API_KEY`;
            const hasApiKey = configs[apiKeyKey] && configs[apiKeyKey].trim() !== '';
            provider.apiKeyConfigured = hasApiKey;
          }
        });
        
        setProviderData(updatedProviderData);
        
        // configs의 USE_LLM/USE_LLM_MODEL 값을 기준으로 동기화
        const confProvider = configs['USE_LLM'] || Object.keys(updatedProviderData)[0] || '';
        const modelList = updatedProviderData[confProvider]?.models || [];
        // conf에 저장된 모델이 실제 모델 리스트에 없으면 첫 번째 모델로 대체
        let confModel = configs['USE_LLM_MODEL'];
        if (!modelList.find((m: any) => (typeof m === 'string' ? m : m.id) === confModel)) {
          confModel = modelList[0]?.id || modelList[0] || '';
        }
        setSelectedProvider(confProvider);
        setSelectedModel(confModel);
        setProviderApiKey(configs[`${confProvider}_api_key`] || '');
      }
    } catch (e) {
      console.error('프로바이더 로드 실패:', e);
    }
  };

  // 현재 로고 로드 함수
  const loadCurrentLogo = async () => {
    try {
      const response = await fetch('/api/config/logo', {
        method: 'GET',
        credentials: 'include',
      });
      
      if (response.ok) {
        const data = await response.json();
        if (data.success && data.data?.logoUrl) {
          setCurrentLogo(data.data.logoUrl);
        }
      }
    } catch (e) {
      // 무시
    }
  };

  // 로고 파일 선택 핸들러
  const handleLogoFileChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (file) {
      // 파일 크기 체크 (5MB 제한)
      if (file.size > 5 * 1024 * 1024) {
        setError('로고 파일은 5MB 이하여야 합니다.');
        return;
      }
      
      // 파일 타입 체크
      if (!file.type.startsWith('image/')) {
        setError('이미지 파일만 업로드할 수 있습니다.');
        return;
      }
      
      setLogoFile(file);
      
      // 미리보기 생성
      const reader = new FileReader();
      reader.onload = (e) => {
        setLogoPreview(e.target?.result as string);
      };
      reader.readAsDataURL(file);
    }
  };

  // 로고 업로드 핸들러
  const handleLogoUpload = async () => {
    if (!logoFile) return;
    
    setLogoUploading(true);
    setError(null);
    
    try {
      console.log('🔄 로고 업로드 시작:', {
        fileName: logoFile.name,
        fileSize: logoFile.size,
        fileType: logoFile.type
      });
      
      const res = await apiClient.uploadLogo(logoFile);
      
      console.log('📥 로고 업로드 응답:', res);
      
      if (res.success && res.data) {
        setCurrentLogo(res.data.logoUrl);
        setLogoFile(null);
        setLogoPreview(null);
        setSuccess('로고가 성공적으로 업로드되었습니다.');
        
        // 헤더에 로고 변경 알림
        window.dispatchEvent(new CustomEvent('logoUpdated', { detail: { logoUrl: res.data.logoUrl } }));
      } else {
        console.error('❌ 로고 업로드 실패 응답:', res);
        setError('로고 업로드에 실패했습니다: ' + (res.error?.message || JSON.stringify(res.error) || '알 수 없는 오류'));
      }
    } catch (e: any) {
      console.error('❌ 로고 업로드 예외:', e);
      console.error('❌ 스택 트레이스:', e.stack);
      
      let errorMessage = '로고 업로드 중 오류가 발생했습니다';
      
      if (e.message) {
        errorMessage += ': ' + e.message;
      } else if (typeof e === 'string') {
        errorMessage += ': ' + e;
      } else if (e.toString) {
        errorMessage += ': ' + e.toString();
      }
      
      setError(errorMessage);
    } finally {
      setLogoUploading(false);
    }
  };

  // 로고 삭제 핸들러
  const handleLogoDelete = async () => {
    try {
      const res = await apiClient.deleteLogo();
      if (res.success) {
        setCurrentLogo(null);
        setSuccess('로고가 삭제되었습니다.');
        
        // 헤더에 로고 삭제 알림
        window.dispatchEvent(new CustomEvent('logoUpdated', { detail: { logoUrl: null } }));
      } else {
        setError('로고 삭제에 실패했습니다: ' + (res.error?.message || ''));
      }
    } catch (e: any) {
      setError('로고 삭제 중 오류가 발생했습니다: ' + e.message);
    }
  };

  // 2) 프로바이더 정보 입력값 변경 핸들러
  const handleProviderInputChange = (provider: string, key: string, value: string) => {
    // 프로바이더별 입력값 상태 업데이트 (기존 값 보존)
    setProviderInputs((prev) => ({
      ...prev,
      [provider]: { 
        ...(prev[provider] || {}), 
        [key]: value 
      },
    }));
    
    // 전체 설정 상태 업데이트 (기존 값 보존)
    setEditedConfigs((prev) => ({
      ...prev,
      [key]: value,
    }));
  };

  // 3) 프로바이더 정보 저장 핸들러
  const handleProviderSave = async (provider: string, keys: string[]) => {
    setProviderSaveStatus((prev) => ({ ...prev, [provider]: '...' }));
    try {
      const savedValues: Record<string, string> = {};
      
      for (const key of keys) {
        let value = providerInputs[provider]?.[key];
        if (value === undefined) value = editedConfigs[key] ?? configs[key] ?? '';
        
        // 빈 값도 저장하여 API 키를 제거할 수 있도록 함
        await setConfig(key, value);
        savedValues[key] = value;
      }
      
      // 상태 업데이트 - 기존 값들을 보존하면서 새로운 값만 업데이트
      setProviderInputs((prev) => ({
        ...prev,
        [provider]: {
          ...prev[provider],
          ...savedValues
        }
      }));
      
      setEditedConfigs((prev) => ({
        ...prev,
        ...savedValues
      }));
      
      setConfigs((prev) => ({
        ...prev,
        ...savedValues
      }));
      
      // 프로바이더 정보 다시 로드하여 API 키 상태 업데이트
      await loadProviders();
      
      setProviderSaveStatus((prev) => ({ ...prev, [provider]: '저장됨' }));
      setTimeout(() => {
        setProviderSaveStatus((prev) => ({ ...prev, [provider]: '' }));
      }, 2000);
      
    } catch (e) {
      setProviderSaveStatus((prev) => ({ ...prev, [provider]: 'failed' }));
      setTimeout(() => {
        setProviderSaveStatus((prev) => ({ ...prev, [provider]: '' }));
      }, 2000);
    }
  };

  // 4) 프로바이더 정보 카드 리팩터링
  const renderProviderInfoCards = () => {
    const aiProviders = configGroups.find((g) => g.name === 'provider')?.providers || [];
    const vllmProvider = aiProviders.find((p) => p.name === 'vllm');
    const otherProviders = aiProviders.filter((p) => p.name !== 'vllm');
    

    
    return (
      <div className="mt-2">
        <h3 className="text-lg font-bold mb-4">프로바이더 정보</h3>
        <div className="space-y-8">
          {otherProviders.map((provider) => {
            const providerInfo = providerData[provider.name] as any;
            const requiresApiKey = providerInfo?.requiresApiKey;
            // 현재 설정에서 직접 API 키 확인 (providerInputs도 포함)
            const apiKeyName = `${provider.name.toUpperCase()}_API_KEY`;
            const currentApiKey = providerInputs[provider.name]?.[apiKeyName] || editedConfigs[apiKeyName] || configs[apiKeyName] || '';
            const apiKeyConfigured = requiresApiKey ? (currentApiKey && currentApiKey.trim() !== '') : true;
            const models = providerInfo?.models || [];
            const keys = provider.keys || [];
            

            return (
              <div key={provider.name} className="rounded-lg p-4 border" style={{ backgroundColor: 'var(--card-bg)', borderColor: 'var(--card-border)' }}>
                <div className="flex justify-between items-center mb-4">
                  <h4 className="text-base font-semibold">{provider.title} 정보</h4>
                  {/* 프로바이더별 저장 버튼 */}
                  <button
                    type="button"
                    className="px-4 py-2 bg-blue-500 text-white rounded hover:bg-blue-600 text-sm font-medium"
                    onClick={() => handleProviderSave(provider.name, keys)}
                  >
                    {providerSaveStatus[provider.name] || '저장'}
                  </button>
                </div>
                
                {requiresApiKey && !apiKeyConfigured && (
                  <div className="mb-3 text-base text-red-600 font-semibold">API 키가 필요합니다.</div>
                )}
                
                <div className="space-y-3">
                  {keys.map((key) => {
                    const meta = configMeta[key];
                    if (!meta) return null;
                    // 모델 입력란(모델이 포함된 label)은 VLLM이 아닌 경우 숨김
                    if (meta.label.includes('모델')) return null;
                    const isApiKey = meta.type === 'password';
                    
                    return (
                      <div key={key} className="flex flex-col gap-2">
                        <label className="text-sm font-medium text-gray-700 dark:text-gray-200">
                          {meta.label}
                        </label>
                        <div className="flex items-center gap-2">
                          <input
                            type={meta.type === 'password' ? (showApiKey[provider.name] ? 'text' : 'password') : meta.type === 'number' ? 'number' : 'text'}
                            className="flex-1 border rounded px-3 py-2 transition"
                            style={{ 
                              borderColor: 'var(--card-border)', 
                              backgroundColor: 'var(--card-bg)', 
                              color: 'var(--text-primary)' 
                            }}
                            value={providerInputs[provider.name]?.[key] ?? editedConfigs[key] ?? ''}
                            onChange={(e) => handleProviderInputChange(provider.name, key, e.target.value)}
                            placeholder={meta.label}
                          />
                          {isApiKey && (
                            <button
                              type="button"
                              aria-label={showApiKey[provider.name] ? 'API 키 숨기기' : 'API 키 보기'}
                              className="text-gray-400 hover:text-gray-700 dark:hover:text-gray-200 p-2"
                              onClick={() => setShowApiKey((prev) => ({ ...prev, [provider.name]: !prev[provider.name] }))}
                            >
                              {showApiKey[provider.name] ? (
                                <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13.875 18.825A10.05 10.05 0 0112 19c-5.523 0-10-4.477-10-10 0-1.657.403-3.22 1.125-4.575m1.664-2.13A9.956 9.956 0 0112 3c5.523 0 10 4.477 10 10 0 2.21-.715 4.254-1.925 5.925M15 12a3 3 0 11-6 0 3 3 0 016 0z" /></svg>
                              ) : (
                                <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" /><path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M2.458 12C3.732 7.943 7.523 5 12 5c4.477 0 8.268 2.943 9.542 7-.274.832-.64 1.624-1.09 2.357" /></svg>
                              )}
                            </button>
                          )}
                        </div>
                      </div>
                    );
                  })}
                </div>
                {/* API 키 상태 및 연결 확인 */}
                <div className="mt-4">
                  <div className="text-base font-semibold mb-2">연결 상태</div>
                  {requiresApiKey ? (
                    apiKeyConfigured ? (
                      <div className="flex items-center text-sm text-green-600 dark:text-green-400">
                        <svg className="w-4 h-4 mr-2" fill="currentColor" viewBox="0 0 20 20">
                          <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                        </svg>
                        API 키가 설정되었습니다
                      </div>
                    ) : (
                      <div className="flex items-center text-sm text-red-600 dark:text-red-400">
                        <svg className="w-4 h-4 mr-2" fill="currentColor" viewBox="0 0 20 20">
                          <path fillRule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                        </svg>
                        API 키가 필요합니다
                      </div>
                    )
                  ) : (
                    <div className="flex items-center text-sm text-blue-600 dark:text-blue-400">
                      <svg className="w-4 h-4 mr-2" fill="currentColor" viewBox="0 0 20 20">
                        <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clipRule="evenodd" />
                      </svg>
                      API 키가 필요하지 않습니다
                    </div>
                  )}
                </div>
              </div>
            );
          })}
          {/* VLLM 프로바이더 카드 별도 표시 */}
          {vllmProvider && (
            <div className="bg-yellow-50 dark:bg-yellow-900 rounded-lg p-4 border-2 border-gray-400 dark:border-gray-600">
              <h4 className="text-base font-semibold mb-3 text-gray-800 dark:text-gray-200">VLLM 설정</h4>
              <div className="space-y-2">
                {vllmProvider.keys?.map((key) => {
                  const meta = configMeta[key];
                  if (!meta) return null;
                  return (
                    <div key={key} className="flex flex-row items-center gap-2 border-none py-2">
                      <label className="block text-base font-medium min-w-[120px]">{meta.label}</label>
                      <input
                        type={meta.type === 'password' ? 'password' : meta.type === 'number' ? 'number' : 'text'}
                        className={
                          (key === 'NEXT_PUBLIC_OLLAMA_PROXY_SERVER' || meta.type === 'password'
                            ? 'flex-1 min-w-[320px] w-[400px] '
                            : 'flex-1 ') +
                          'border rounded px-3 py-2 transition'
                        }
                        style={{ 
                          borderColor: 'var(--card-border)', 
                          backgroundColor: 'var(--card-bg)', 
                          color: 'var(--text-primary)' 
                        }}
                        value={providerInputs['vllm']?.[key] ?? editedConfigs[key] ?? ''}
                        onChange={(e) => handleProviderInputChange('vllm', key, e.target.value)}
                        placeholder={meta.label}
                      />
                    </div>
                  );
                })}
              </div>
            </div>
          )}
        </div>
      </div>
    );
  };

  // 4. renderAiProviderSelector 리팩터링
  const renderAiProviderSelector = () => {
    if (!providerList.length) return null;
    const models = modelOptions[currentProviderKey] || [];
    return (
      <div className="mb-8 p-6 rounded-lg flex flex-col md:flex-row md:items-center gap-4 border" style={{ backgroundColor: 'var(--card-bg)', borderColor: 'var(--card-border)' }}>
        <div className="flex flex-col md:flex-row md:items-center gap-2">
          <label className="font-semibold mr-2">시스템에서 사용할 프로바이더</label>
          <select
            className="border rounded px-3 py-2 min-w-[120px] transition"
            style={{ 
              borderColor: 'var(--card-border)', 
              backgroundColor: 'var(--card-bg)', 
              color: 'var(--text-primary)' 
            }}
            value={currentProviderKey}
            onChange={e => {
              const provider = e.target.value;
              apiClient.getProviderModels(provider).then(res => {
                if (res.success && res.data) {
                  setModelOptions(prev => ({ ...prev, [provider]: res.data.map((m: { id?: string; name?: string }) => m.id || m.name) }));
                  const firstModel = res.data[0]?.id || res.data[0]?.name || '';
                  setConfig('USE_LLM', provider).then(() => {
                    setEditedConfigs(prev => ({ ...prev, USE_LLM: provider }));
                    setConfigs(prev => ({ ...prev, USE_LLM: provider }));
                  });
                  setConfig(`${provider.toUpperCase()}_MODEL`, firstModel).then(() => {
                    setEditedConfigs(prev => ({ ...prev, [`${provider.toUpperCase()}_MODEL`]: firstModel }));
                    setConfigs(prev => ({ ...prev, [`${provider.toUpperCase()}_MODEL`]: firstModel }));
                  });
                }
              });
            }}
          >
            {providerList.map((p: { key: string; title: string }) => (
              <option key={p.key} value={p.key}>{p.title}</option>
            ))}
          </select>
        </div>
        <div className="flex flex-col md:flex-row md:items-center gap-2">
          <label className="font-semibold mr-2">모델</label>
          <div className="flex items-center gap-2">
            <select
              className="border rounded px-3 py-2 min-w-[120px] transition"
            style={{ 
              borderColor: 'var(--card-border)', 
              backgroundColor: 'var(--card-bg)', 
              color: 'var(--text-primary)' 
            }}
              value={currentModelKey}
              onChange={e => {
                const model = e.target.value;
                setConfig(`${currentProviderKey.toUpperCase()}_MODEL`, model).then(() => {
                  setEditedConfigs(prev => ({
                    ...prev,
                    [`${currentProviderKey.toUpperCase()}_MODEL`]: model,
                  }));
                  setConfigs(prev => ({
                    ...prev,
                    [`${currentProviderKey.toUpperCase()}_MODEL`]: model,
                  }));
                });
              }}
            >
              {models.map((m: string) => (
                <option key={m} value={m}>{m}</option>
              ))}
            </select>
            <button
              type="button"
              onClick={refreshModels}
              disabled={refreshingModels}
              className="px-3 py-2 bg-green-500 hover:bg-green-600 text-white rounded-lg transition-colors disabled:opacity-50 disabled:cursor-not-allowed text-sm"
              title="모델 목록 새로고침"
            >
              {refreshingModels ? '...' : '🔄'}
            </button>
          </div>
        </div>
        {models.length === 0 && currentProviderKey && (
          <div className="text-sm text-orange-600 dark:text-orange-400 mt-2">
            💡 {currentProviderKey === 'ollama' ? 'Ollama 서버에 연결할 수 없거나 모델이 없습니다.' : `${currentProviderKey} 모델을 찾을 수 없습니다.`}
            {currentProviderKey === 'ollama' && (
              <div className="mt-1 text-xs">
                프로바이더 섹션에서 "Ollama 서버" 설정을 확인해주세요.
              </div>
            )}
          </div>
        )}
      </div>
    );
  };

  // 그룹별 본문 렌더링
  const renderGroupContent = () => {
    const group = configGroups.find((g) => g.name === activeGroup);
    if (!group) return null;
    if (group.name === 'provider') {
      // 프로바이더 카드만 렌더링
      const aiProviders = group.providers || [];
      const vllmProvider = aiProviders.find((p) => p.name === 'vllm');
      const otherProviders = aiProviders.filter((p) => p.name !== 'vllm');
      return (
        <div className="space-y-8">
          {otherProviders.map((provider) => {
            const providerInfo = providerData[provider.name] as any;
            const requiresApiKey = providerInfo?.requiresApiKey;
            // 현재 설정에서 직접 API 키 확인 (providerInputs도 포함)
            const apiKeyName = `${provider.name.toUpperCase()}_API_KEY`;
            const currentApiKey = providerInputs[provider.name]?.[apiKeyName] || editedConfigs[apiKeyName] || configs[apiKeyName] || '';
            const apiKeyConfigured = requiresApiKey ? (currentApiKey && currentApiKey.trim() !== '') : true;
            const models = providerInfo?.models || [];
            const keys = provider.keys || [];
            return (
              <div key={provider.name} className="rounded-lg p-4 border" style={{ backgroundColor: 'var(--card-bg)', borderColor: 'var(--card-border)' }}>
                <div className="flex justify-between items-center mb-4">
                  <h4 className="text-base font-semibold">{provider.title} 정보</h4>
                  {/* 프로바이더별 저장 버튼 */}
                  <button
                    type="button"
                    className="px-4 py-2 bg-blue-500 text-white rounded hover:bg-blue-600 text-sm font-medium"
                    onClick={() => handleProviderSave(provider.name, keys)}
                  >
                    {providerSaveStatus[provider.name] || '저장'}
                  </button>
                </div>
                
                {requiresApiKey && !apiKeyConfigured && (
                  <div className="mb-3 text-base text-red-600 font-semibold">API 키가 필요합니다.</div>
                )}
                
                <div className="space-y-3">
                  {keys.map((key) => {
                    const meta = configMeta[key];
                    if (!meta) return null;
                    // 모델 입력란(모델이 포함된 label)은 VLLM이 아닌 경우 숨김
                    if (meta.label.includes('모델')) return null;
                    const isApiKey = meta.type === 'password';
                    
                    return (
                      <div key={key} className="flex flex-col gap-2">
                        <label className="text-sm font-medium text-gray-700 dark:text-gray-200">
                          {meta.label}
                        </label>
                        <div className="flex items-center gap-2">
                          <input
                            type={meta.type === 'password' ? (showApiKey[provider.name] ? 'text' : 'password') : meta.type === 'number' ? 'number' : 'text'}
                            className="flex-1 border rounded px-3 py-2 transition"
                            style={{ 
                              borderColor: 'var(--card-border)', 
                              backgroundColor: 'var(--card-bg)', 
                              color: 'var(--text-primary)' 
                            }}
                            value={providerInputs[provider.name]?.[key] ?? editedConfigs[key] ?? ''}
                            onChange={(e) => handleProviderInputChange(provider.name, key, e.target.value)}
                            placeholder={meta.label}
                          />
                          {isApiKey && (
                            <button
                              type="button"
                              aria-label={showApiKey[provider.name] ? 'API 키 숨기기' : 'API 키 보기'}
                              className="text-gray-400 hover:text-gray-700 dark:hover:text-gray-200 p-2"
                              onClick={() => setShowApiKey((prev) => ({ ...prev, [provider.name]: !prev[provider.name] }))}
                            >
                              {showApiKey[provider.name] ? (
                                <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13.875 18.825A10.05 10.05 0 0112 19c-5.523 0-10-4.477-10-10 0-1.657.403-3.22 1.125-4.575m1.664-2.13A9.956 9.956 0 0112 3c5.523 0 10 4.477 10 10 0 2.21-.715 4.254-1.925 5.925M15 12a3 3 0 11-6 0 3 3 0 016 0z" /></svg>
                              ) : (
                                <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" /><path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M2.458 12C3.732 7.943 7.523 5 12 5c4.477 0 8.268 2.943 9.542 7-.274.832-.64 1.624-1.09 2.357" /></svg>
                              )}
                            </button>
                          )}
                        </div>
                      </div>
                    );
                  })}
                </div>
                {/* API 키 상태 및 연결 확인 */}
                <div className="mt-4">
                  <div className="text-base font-semibold mb-2">연결 상태</div>
                  {requiresApiKey ? (
                    apiKeyConfigured ? (
                      <div className="flex items-center text-sm text-green-600 dark:text-green-400">
                        <svg className="w-4 h-4 mr-2" fill="currentColor" viewBox="0 0 20 20">
                          <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                        </svg>
                        API 키가 설정되었습니다
                      </div>
                    ) : (
                      <div className="flex items-center text-sm text-red-600 dark:text-red-400">
                        <svg className="w-4 h-4 mr-2" fill="currentColor" viewBox="0 0 20 20">
                          <path fillRule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                        </svg>
                        API 키가 필요합니다
                      </div>
                    )
                  ) : (
                    <div className="flex items-center text-sm text-blue-600 dark:text-blue-400">
                      <svg className="w-4 h-4 mr-2" fill="currentColor" viewBox="0 0 20 20">
                        <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clipRule="evenodd" />
                      </svg>
                      API 키가 필요하지 않습니다
                    </div>
                  )}
                </div>
              </div>
            );
          })}
          {/* VLLM 프로바이더 카드 별도 표시 */}
          {vllmProvider && (
            <div className="rounded-lg p-4 border-2 border-gray-400 dark:border-gray-600">
              <h4 className="text-base font-semibold mb-3 text-gray-800 dark:text-gray-200">VLLM 설정</h4>
              <div className="space-y-2">
                {vllmProvider.keys?.map((key) => {
                  const meta = configMeta[key];
                  if (!meta) return null;
                  return (
                    <div key={key} className="flex flex-row items-center gap-2 border-none py-2">
                      <label className="block text-base font-medium min-w-[120px]">{meta.label}</label>
                      <input
                        type={meta.type === 'password' ? 'password' : meta.type === 'number' ? 'number' : 'text'}
                        className={
                          (key === 'NEXT_PUBLIC_OLLAMA_PROXY_SERVER' || meta.type === 'password'
                            ? 'flex-1 min-w-[320px] w-[400px] '
                            : 'flex-1 ') +
                          'border rounded px-3 py-2 transition'
                        }
                        style={{ 
                          borderColor: 'var(--card-border)', 
                          backgroundColor: 'var(--card-bg)', 
                          color: 'var(--text-primary)' 
                        }}
                        value={providerInputs['vllm']?.[key] ?? editedConfigs[key] ?? ''}
                        onChange={(e) => handleProviderInputChange('vllm', key, e.target.value)}
                        placeholder={meta.label}
                      />
                    </div>
                  );
                })}
              </div>
            </div>
          )}
        </div>
      );
    }
    if (group.name === 'ai') {
      // 시스템 전체 프로바이더/모델 드롭다운만 렌더링
      return renderAiProviderSelector();
    }
    if (group.name === 'feature') {
      // 서비스 설정 그룹: 모든 항목을 한 줄(제목-값)로 표시
      return (
        <div className="space-y-2">
          {group.keys?.map((key) => {
            const meta = configMeta[key];
            if (!meta) return null;
            return (
              <div key={key} className="flex flex-row items-center justify-between gap-4 py-2 border-b border-gray-100">
                <span className="text-base font-medium text-gray-700 dark:text-gray-200 min-w-[180px]">{meta.label}</span>
                <div className="flex-1">{renderInput(key, editedConfigs[key], meta)}</div>
              </div>
            );
          })}
        </div>
      );
    }
    if (group.name === 'rag') {
      // 기존의 quickKeys, processingKeys, chunkingKeys, embeddingKeys는 유지
      const quickKeys = ['RAG_DOCS_PATH', 'RAG_MODE', 'RAG_CHUNKING_STRATEGY', 'RAG_GENERATE_IMAGE_EMBEDDINGS'];
      const processingKeys = ['RAG_FORCE_PDF_CONVERSION', 'RAG_PDF_BACKEND', 'RAG_PROCESS_TEXT_ONLY', 'RAG_DETECT_COLUMN_TYPE'];
      const chunkingKeys = ['RAG_CHUNK_OVERLAP', 'RAG_CHUNK_SIZE', 'RAG_LARGE_CONTEXT_CHUNKING', 'RAG_SEMANTIC_CHUNKER'];
      const embeddingKeys = ['RAG_EMBEDDING_MODEL', 'RAG_SENTENCE_TRANSFORMER_MODEL', 'RAG_IMAGE_EMBEDDING_MODEL', 'USE_ENHANCED_EMBEDDING', 'ENHANCED_EMBEDDING_STRATEGY'];
      const graphragKeys = ['RAG_ENABLE_GRAPHRAG', 'RAG_SEARCH_MODE', 'RAG_GRAPHRAG_ENTITY_TYPES', 'RAG_GRAPHRAG_MAX_GLEANINGS', 'RAG_GRAPHRAG_COMMUNITY_DETECTION'];

      // 가중치 합계 계산 함수
      const calculateWeightSum = (keys: string[]) => {
        return keys.reduce((sum, key) => {
          const value = Number(editedConfigs[key]) || 0;
          return sum + value;
        }, 0);
      };

      return (
        <div className="space-y-6">
          {/* 빠른 설정 섹션 */}
          <div className="rounded-lg p-4 border" style={{ backgroundColor: 'var(--card-bg)', borderColor: 'var(--card-border)' }}>
            <div className="font-semibold mb-2 text-gray-700 dark:text-gray-200">빠른 설정</div>
            <div className="space-y-2">
              {quickKeys.map((key, idx) => (
                key === 'RAG_DOCS_PATH' ? (
                  <div key={key} className="flex flex-col gap-1 py-2">
                    <span className="text-base font-medium text-gray-700 dark:text-gray-200 min-w-[180px] mb-1">{configMeta[key].label}</span>
                    <div className="flex flex-row items-center gap-2 w-full">
                      <div className="flex-1">
                        {renderInput(key, editedConfigs[key], configMeta[key])}
                      </div>
                      <button
                        className="px-3 py-2 bg-blue-500 text-white rounded hover:bg-blue-600 text-base w-fit"
                        onClick={handleInitCollection}
                        disabled={isSaving}
                      >
                        {isSaving ? '초기화 중...' : '컬렉션 초기화'}
                      </button>
                      {!isSaving && !isPolling && initProgress && !initProgress.includes('성공적으로') && (
                        <button
                          className="px-3 py-2 bg-green-500 text-white rounded hover:bg-green-600 text-base w-fit"
                          onClick={restartStatusCheck}
                        >
                          상태 확인 재시작
                        </button>
                      )}
                    </div>
                    {(initProgress || isPolling) && (
                      <div className="mt-2 w-full">
                        {(() => {
                          const match = initProgress && initProgress.match(/(\d+)%/);
                          if (match) {
                            return (
                              <div className="w-full mb-1">
                                <div className="relative h-3 bg-gray-200 rounded-full overflow-hidden">
                                  <div
                                    className="absolute left-0 top-0 h-3 bg-blue-500 transition-all duration-500"
                                    style={{ width: `${parseInt(match[1])}%` }}
                                  ></div>
                                </div>
                                <div className="text-xs text-gray-700 dark:text-gray-200 mt-1 text-right">
                                  {match[1]}%
                                </div>
                              </div>
                            );
                          }
                          return null;
                        })()}
                        <div className={`text-base mt-1 ${
                          !isSaving && !isPolling && initProgress && !initProgress.includes('성공적으로') ? 
                          'text-orange-600 dark:text-orange-400 font-medium' : 
                          'text-gray-700 dark:text-gray-200'
                        }`}>
                          {initProgress}
                          {!isSaving && !isPolling && initProgress && !initProgress.includes('성공적으로') && (
                            <div className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                              ⚠️ 상태 모니터링이 중단되었습니다. 작업은 백그라운드에서 계속 진행됩니다.
                            </div>
                          )}
                        </div>
                      </div>
                    )}
                  </div>
                ) : key === 'RAG_CHUNKING_STRATEGY' ? (
                  <div key={key} className="flex flex-row items-center justify-between gap-4 py-2">
                    <span className="text-base font-medium text-gray-700 dark:text-gray-200 min-w-[180px]">{configMeta[key].label}</span>
                    <div className="flex-1">
                      <select
                        value={editedConfigs[key] ?? ''}
                        onChange={(e) => handleConfigChange(key, e.target.value)}
                        className="w-full border rounded px-3 py-2 transition"
                        style={{ 
                          borderColor: 'var(--card-border)', 
                          backgroundColor: 'var(--card-bg)', 
                          color: 'var(--text-primary)' 
                        }}
                      >
                        {['default', 'semantic', 'page'].map(opt => (
                          <option key={opt} value={opt}>{opt}</option>
                        ))}
                      </select>
                    </div>
                  </div>
                ) : (
                  <div key={key} className="flex flex-row items-center justify-between gap-4 py-2">
                    <span className="text-base font-medium text-gray-700 dark:text-gray-200 min-w-[180px]">{configMeta[key].label}</span>
                    <div className="flex-1">{renderInput(key, editedConfigs[key], configMeta[key])}</div>
                  </div>
                )
              ))}
            </div>
          </div>
          {/* Workflow */}
          <div className="rounded-lg p-4 border" style={{ backgroundColor: 'var(--card-bg)', borderColor: 'var(--card-border)' }}>
            <div className="font-semibold mb-2 text-gray-700 dark:text-gray-200">Workflow</div>
            <div className="space-y-2">
              {processingKeys.map((key) => {
                const meta = configMeta[key];
                if (!meta) return null;
                return (
                  <div key={key} className="flex flex-row items-center justify-between gap-4 border-b border-gray-100 py-2">
                    <span className="text-base font-medium text-gray-700 dark:text-gray-200 min-w-[180px]">{meta.label}</span>
                    <div className="flex-1">{renderInput(key, editedConfigs[key], meta)}</div>
                  </div>
                );
              })}
            </div>
          </div>
          {/* Chunking */}
          <div className="rounded-lg p-4 border" style={{ backgroundColor: 'var(--card-bg)', borderColor: 'var(--card-border)' }}>
            <div className="font-semibold mb-2 text-gray-700 dark:text-gray-200">Chunking</div>
            <div className="space-y-2">
              {chunkingKeys.map((key) => {
                const meta = configMeta[key];
                if (!meta) return null;
                return (
                  <div key={key} className="flex flex-row items-center justify-between gap-4 border-b border-gray-100 py-2">
                    <span className="text-base font-medium text-gray-700 dark:text-gray-200 min-w-[180px]">{meta.label}</span>
                    <div className="flex-1">{renderInput(key, editedConfigs[key], meta)}</div>
                  </div>
                );
              })}
            </div>
          </div>
          {/* Embedding */}
          <div className="rounded-lg p-4 border" style={{ backgroundColor: 'var(--card-bg)', borderColor: 'var(--card-border)' }}>
            <div className="font-semibold mb-2 text-gray-700 dark:text-gray-200">Embedding</div>
            <div className="space-y-2">
              {embeddingKeys.map((key) => {
                const meta = configMeta[key];
                if (!meta) return null;
                return (
                  <div key={key} className="flex flex-row items-center justify-between gap-4 border-b border-gray-100 py-2">
                    <span className="text-base font-medium text-gray-700 dark:text-gray-200 min-w-[180px]">{meta.label}</span>
                    <div className="flex-1">{renderInput(key, editedConfigs[key], meta)}</div>
                  </div>
                );
              })}
            </div>
          </div>
          {/* GraphRAG */}
          <div className="rounded-lg p-4 border" style={{ backgroundColor: 'var(--card-bg)', borderColor: 'var(--card-border)' }}>
            <div className="font-semibold mb-2 text-gray-700 dark:text-gray-200">GraphRAG</div>
            <div className="space-y-2">
              {graphragKeys.map((key) => {
                const meta = configMeta[key];
                if (!meta) return null;
                return (
                  <div key={key} className="flex flex-row items-center justify-between gap-4 border-b border-gray-100 py-2">
                    <span className="text-base font-medium text-gray-700 dark:text-gray-200 min-w-[180px]">{meta.label}</span>
                    <div className="flex-1">{renderInput(key, editedConfigs[key], meta)}</div>
                  </div>
                );
              })}
            </div>
          </div>
          {/* Retriever */}
          <div className="rounded-lg p-4 border" style={{ backgroundColor: 'var(--card-bg)', borderColor: 'var(--card-border)' }}>
            <div className="font-semibold mb-4 text-gray-700 dark:text-gray-200">Retriever</div>
            <div className="space-y-8">
              {Object.entries(retrieverGroups).map(([groupKey, group]) => {
                const weightSum = calculateWeightSum(group.keys);
                const isWeightGroup = groupKey.toLowerCase().includes('weight');
                const isInvalidSum = isWeightGroup && Math.abs(weightSum - 1.0) > 0.001;

                return (
                  <div key={groupKey} className="space-y-4">
                    <div>
                      <h3 className="text-base font-medium mb-1">{group.title}</h3>
                      <p className="text-sm text-gray-500 dark:text-gray-400">{group.description}</p>
                      {isWeightGroup && (
                        <div className={`text-sm mt-1 ${isInvalidSum ? 'text-red-500' : 'text-green-500'}`}>
                          현재 가중치 합계: {weightSum.toFixed(3)}
                          {isInvalidSum && ' (합계가 1.0이 되어야 합니다)'}
                        </div>
                      )}
                    </div>
                    <div className="grid gap-4">
                      {group.keys.map((key) => {
                        const meta = configMeta[key];
                        if (!meta) return null;
                        return (
                          <div key={key} className="flex items-center gap-4">
                            <label className="text-base font-medium text-gray-700 dark:text-gray-200 min-w-[180px]">
                              {meta.label}
                            </label>
                            <input
                              type="number"
                              step="0.1"
                              min="0"
                              max={groupKey === 'thresholds' ? "1" : undefined}
                              className={`flex-1 border rounded px-3 py-2 transition
                                ${isInvalidSum ? 'border-red-300 dark:border-red-700' : ''}`}
                              style={{ 
                                borderColor: isInvalidSum ? undefined : 'var(--card-border)',
                                backgroundColor: 'var(--card-bg)', 
                                color: 'var(--text-primary)' 
                              }}
                              value={editedConfigs[key] || ''}
                              onChange={(e) => handleConfigChange(key, e.target.value)}
                            />
                          </div>
                        );
                      })}
                    </div>
                  </div>
                );
              })}
            </div>
          </div>
        </div>
      );
    }
    if (group.name === 'license') {
      // 라이선스 정보 카드 (읽기 전용, 저장 버튼 없음, '보다' 제거)
      return (
        <div>
          {licenseLoading ? (
            <div className="text-base text-gray-500">로딩 중...</div>
          ) : licenseError ? (
            <div className="text-base text-red-500">{licenseError}</div>
          ) : licenseInfo ? (
            <div className="space-y-2">
              <div><b>사용자</b>: {licenseInfo.user || '-'}</div>
              <div><b>회사</b>: {licenseInfo.company || '-'}</div>
              <div><b>플랜</b>: {licenseInfo.apiPlan || '-'}</div>
              <div><b>만료일</b>: {licenseInfo.expiresAt ? new Date(licenseInfo.expiresAt).toLocaleDateString() : '-'}</div>
              <div><b>최대 디바이스</b>: {licenseInfo.maxDevices || '-'}</div>
              <div><b>오프라인 인증</b>: {licenseInfo.offline ? '예' : '아니오'}</div>
            </div>
          ) : (
            <div className="text-base text-gray-500">라이선스 정보가 없습니다.</div>
          )}
        </div>
      );
    }
    // 일반 그룹
    return (
      <div className="space-y-6">
        {/* 시스템 그룹에 로고 업로드 기능 추가 */}
        {group.name === 'path' && (
          <div className="mb-6">
            <div className="rounded-lg p-4 border border-gray-200 dark:border-gray-700">
              <h4 className="text-base font-semibold mb-3">웹사이트 로고</h4>
              
              {/* 현재 로고 표시 */}
              {currentLogo && (
                <div className="mb-4">
                  <label className="block text-sm font-medium mb-2">현재 로고</label>
                  <div className="flex items-center gap-4">
                    <img 
                      src={currentLogo} 
                      alt="현재 로고" 
                      className="h-12 w-auto object-contain border border-gray-200 dark:border-gray-600 rounded p-2"
                    />
                    <button
                      onClick={handleLogoDelete}
                      className="px-3 py-1 text-sm bg-red-500 text-white rounded hover:bg-red-600"
                    >
                      삭제
                    </button>
                  </div>
                </div>
              )}
              
              {/* 로고 업로드 */}
              <div className="space-y-3">
                <label className="block text-sm font-medium">새 로고 업로드</label>
                <input
                  type="file"
                  accept="image/*"
                  onChange={handleLogoFileChange}
                  className="block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded file:border-0 file:text-sm file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100"
                />
                
                {/* 미리보기 */}
                {logoPreview && (
                  <div className="mt-3">
                    <label className="block text-sm font-medium mb-2">미리보기</label>
                    <img 
                      src={logoPreview} 
                      alt="로고 미리보기" 
                      className="h-12 w-auto object-contain border border-gray-200 dark:border-gray-600 rounded p-2"
                    />
                  </div>
                )}
                
                {/* 업로드 버튼 */}
                {logoFile && (
                  <button
                    onClick={handleLogoUpload}
                    disabled={logoUploading}
                    className="px-4 py-2 bg-blue-500 text-white rounded hover:bg-blue-600 disabled:opacity-50"
                  >
                    {logoUploading ? '업로드 중...' : '로고 업로드'}
                  </button>
                )}
                
                <p className="text-xs text-gray-500">
                  * 권장 크기: 높이 56-80px, 가로세로 비율 3:2 또는 2:1<br/>
                  * 최대 크기: 높이 80px, 폭 200px<br/>
                  * 지원 형식: PNG, JPG, GIF, SVG<br/>
                  * 최대 파일 크기: 5MB
                </p>
              </div>
            </div>
          </div>
        )}
        
        {group.keys?.map((key) => {
          const meta = configMeta[key];
          if (!meta) return null;
          return (
            <div key={key} className="mb-2">
              <label className="block text-base font-medium mb-1">{meta.label}</label>
              {renderInput(key, editedConfigs[key], meta)}
              {meta.description && (
                <p className="text-xs text-gray-500 mt-1">{meta.description}</p>
              )}
            </div>
          );
        })}
      </div>
    );
  };

  // --- renderInput 함수 복원 ---
  const renderInput = (key: string, value: string, meta: { label: string; type: string }) => {
    if (key === 'USE_SMTP') {
      // YES/yes/1/true 모두 활성화로 처리
      const isEnabled = /^(yes|true|1)$/i.test(String(value));
      return (
        <select
          value={isEnabled ? 'true' : 'false'}
          onChange={e => handleConfigChange(key, e.target.value === 'true' ? 'YES' : 'NO')}
          className="w-full border rounded px-3 py-2 transition"
          style={{ 
            borderColor: 'var(--card-border)', 
            backgroundColor: 'var(--card-bg)', 
            color: 'var(--text-primary)' 
          }}
        >
          <option value="true">활성화</option>
          <option value="false">비활성화</option>
        </select>
      );
    }
    switch (meta.type) {
      case 'boolean':
        const boolVal = parseBool(value);
        return (
          <select
            value={boolVal ? 'true' : 'false'}
            onChange={e => handleConfigChange(key, e.target.value === 'true' ? 'YES' : 'NO')}
            className="w-full border rounded px-3 py-2 transition"
            style={{ 
              borderColor: 'var(--card-border)', 
              backgroundColor: 'var(--card-bg)', 
              color: 'var(--text-primary)' 
            }}
          >
            <option value="true">활성화</option>
            <option value="false">비활성화</option>
          </select>
        );
      case 'number':
        return (
          <input
            type="number"
            value={value ?? ''}
            onChange={(e) => handleConfigChange(key, e.target.value)}
            readOnly={configMeta[key]?.readonly}
            disabled={configMeta[key]?.readonly}
            className={`w-full border rounded px-3 py-2 transition ${configMeta[key]?.readonly ? 'cursor-not-allowed opacity-60' : ''}`}
            style={{
              borderColor: 'var(--card-border)',
              backgroundColor: configMeta[key]?.readonly ? 'var(--bg-secondary)' : 'var(--card-bg)',
              color: configMeta[key]?.readonly ? 'var(--text-muted)' : 'var(--text-primary)'
            }}
            title={configMeta[key]?.readonly ? configMeta[key]?.description : undefined}
          />
        );
      case 'password':
        return (
          <div className="flex items-center gap-2">
            <input
              type={showApiKey[key] ? 'text' : 'password'}
              value={value ?? ''}
              onChange={(e) => handleConfigChange(key, e.target.value)}
              readOnly={configMeta[key]?.readonly}
              disabled={configMeta[key]?.readonly}
              className={`flex-1 border rounded px-3 py-2 transition ${configMeta[key]?.readonly ? 'cursor-not-allowed opacity-60' : ''}`}
              style={{
                borderColor: 'var(--card-border)',
                backgroundColor: configMeta[key]?.readonly ? 'var(--bg-secondary)' : 'var(--card-bg)',
                color: configMeta[key]?.readonly ? 'var(--text-muted)' : 'var(--text-primary)'
              }}
              placeholder="••••••••"
              title={configMeta[key]?.readonly ? configMeta[key]?.description : undefined}
            />
            {!configMeta[key]?.readonly && (
              <button
                type="button"
                aria-label={showApiKey[key] ? 'API 키 숨기기' : 'API 키 보기'}
                className="hover:opacity-80 transition-opacity"
                style={{ background: 'none', border: 'none', padding: 0, color: 'var(--text-muted)' }}
                tabIndex={0}
                onClick={() => setShowApiKey((prev) => ({ ...prev, [key]: !prev[key] }))}
                onMouseEnter={(e) => e.currentTarget.style.color = 'var(--text-primary)'}
                onMouseLeave={(e) => e.currentTarget.style.color = 'var(--text-muted)'}
              >
                {showApiKey[key] ? (
                  <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13.875 18.825A10.05 10.05 0 0112 19c-5.523 0-10-4.477-10-10 0-1.657.403-3.22 1.125-4.575m1.664-2.13A9.956 9.956 0 0112 3c5.523 0 10 4.477 10 10 0 2.21-.715 4.254-1.925 5.925M15 12a3 3 0 11-6 0 3 3 0 016 0z" /></svg>
                ) : (
                  <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" /><path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M2.458 12C3.732 7.943 7.523 5 12 5c4.477 0 8.268 2.943 9.542 7-.274.832-.64 1.624-1.09 2.357" /></svg>
                )}
              </button>
            )}
          </div>
        );
      case 'string':
        if (key === 'RAG_PDF_BACKEND') {
          const options = ['pypdf2', 'pdfplumber', 'docling', 'pymupdf', 'unstructured'];
          return (
            <select
              value={value ?? ''}
              onChange={(e) => handleConfigChange(key, e.target.value)}
              className="w-full border rounded px-3 py-2 transition"
              style={{ 
                borderColor: 'var(--card-border)', 
                backgroundColor: 'var(--card-bg)', 
                color: 'var(--text-primary)' 
              }}
            >
              {options.map(opt => (
                <option key={opt} value={opt}>{opt}</option>
              ))}
            </select>
          );
        }
        if (key === 'RAG_MODE') {
          const options = ['fast', 'normal', 'rich', 'intelligence'];
          return (
            <select
              value={value ?? ''}
              onChange={(e) => handleConfigChange(key, e.target.value)}
              className="w-full border rounded px-3 py-2 transition"
              style={{ 
                borderColor: 'var(--card-border)', 
                backgroundColor: 'var(--card-bg)', 
                color: 'var(--text-primary)' 
              }}
            >
              {options.map(opt => (
                <option key={opt} value={opt}>{opt}</option>
              ))}
            </select>
          );
        }
        if (key === 'LANGUAGE') {
          const options = [
            { value: 'ko', label: '한국어' },
            { value: 'en', label: 'English' },
            { value: 'ja', label: '日本語' },
            { value: 'zh', label: '中文' },
            { value: 'es', label: 'Español' },
            { value: 'fr', label: 'Français' },
          ];
          return (
            <select
              value={value ?? 'ko'}
              onChange={(e) => handleConfigChange(key, e.target.value)}
              className="w-full border rounded px-3 py-2 transition"
              style={{ 
                borderColor: 'var(--card-border)', 
                backgroundColor: 'var(--card-bg)', 
                color: 'var(--text-primary)' 
              }}
            >
              {options.map(opt => (
                <option key={opt.value} value={opt.value}>{opt.label}</option>
              ))}
            </select>
          );
        }
        if (key === 'ENHANCED_EMBEDDING_STRATEGY') {
          const options = ['original', 'questions', 'hybrid'];
          return (
            <select
              value={value ?? 'questions'}
              onChange={(e) => handleConfigChange(key, e.target.value)}
              className="w-full border rounded px-3 py-2 transition"
              style={{ 
                borderColor: 'var(--card-border)', 
                backgroundColor: 'var(--card-bg)', 
                color: 'var(--text-primary)' 
              }}
            >
              {options.map(opt => (
                <option key={opt} value={opt}>{opt}</option>
              ))}
            </select>
          );
        }
        if (key === 'RAG_SEARCH_MODE') {
          const options = ['vector', 'graph', 'hybrid'];
          return (
            <select
              value={value ?? 'vector'}
              onChange={(e) => handleConfigChange(key, e.target.value)}
              className="w-full border rounded px-3 py-2 transition"
              style={{
                borderColor: 'var(--card-border)',
                backgroundColor: 'var(--card-bg)',
                color: 'var(--text-primary)'
              }}
            >
              {options.map(opt => (
                <option key={opt} value={opt}>{opt}</option>
              ))}
            </select>
          );
        }
        return (
          <input
            type="text"
            value={value ?? ''}
            onChange={(e) => handleConfigChange(key, e.target.value)}
            readOnly={configMeta[key]?.readonly}
            disabled={configMeta[key]?.readonly}
            className={`w-full border rounded px-3 py-2 transition ${configMeta[key]?.readonly ? 'cursor-not-allowed opacity-60' : ''}`}
            style={{
              borderColor: 'var(--card-border)',
              backgroundColor: configMeta[key]?.readonly ? 'var(--bg-secondary)' : 'var(--card-bg)',
              color: configMeta[key]?.readonly ? 'var(--text-muted)' : 'var(--text-primary)'
            }}
            title={configMeta[key]?.readonly ? configMeta[key]?.description : undefined}
          />
        );
      default:
        return null;
    }
  };

  const handleInitCollection = async () => {
    setIsSaving(true);
    setError(null);
    setSuccess(null);
    setInitProgress(null);
    setIsPolling(false);
    try {
      const apiKey = localStorage.getItem('apiKey') || '';
      const { getApiServerUrl } = await import('@/config/serverConfig');
      const apiServerUrl = getApiServerUrl();
      const res = await fetch(`${apiServerUrl}/api/v1/rag/init`, {
        method: 'POST',
        headers: {
          'accept': 'application/json',
          'X-API-Key': apiKey,
        },
        body: '', // 빈 바디
      });
      const data = await res.json();
      if (data?.status === 'success' || data?.message?.includes('시작')) {
        setInitProgress('초기화 대기 중...');
        setIsPolling(true);
        pollInitStatus();
      } else {
        setError(data?.message || '컬렉션 초기화 요청 실패');
      }
    } catch (e) {
      setError('컬렉션 초기화 중 오류가 발생했습니다.');
    } finally {
      setIsSaving(false);
    }
  };

  // 상태 확인 재시작 함수
  const restartStatusCheck = () => {
    setIsPolling(true);
    setInitProgress('상태 확인을 재시작합니다...');
    pollInitStatus();
  };

  const pollInitStatus = () => {
    let tries = 0;
    const maxTries = 600; // 20분으로 증가 (600 * 2초)
    let consecutiveErrors = 0;
    const maxConsecutiveErrors = 5;
    
    const apiKey = localStorage.getItem('apiKey') || '';
    const { getApiServerUrl } = require('@/config/serverConfig');
    const apiServerUrl = getApiServerUrl();
    const interval = setInterval(async () => {
      tries++;
      try {
        const resp = await fetch(`${apiServerUrl}/api/v1/rag/init-status`, {
          headers: {
            'accept': 'application/json',
            'X-API-Key': apiKey,
          },
        });
        const data = await resp.json();
        const qs = data.queue_status || {};
        const totalTime = data.total_time?.total_time?.formatted || null;
        
        // 에러 카운터 리셋
        consecutiveErrors = 0;
        
        if (data.status === 'success' && qs.remaining === 0) {
          setInitProgress(`컬렉션이 성공적으로 초기화되었습니다.${totalTime ? ` (총 소요시간: ${totalTime})` : ''}`);
          setSuccess(null);
          setIsPolling(false);
          clearInterval(interval);
        } else if (qs.total_documents) {
          const elapsed = tries * 2; // 경과 시간 (초)
          const minutes = Math.floor(elapsed / 60);
          const seconds = elapsed % 60;
          setInitProgress(
            `진행 중... (${qs.completed || 0}/${qs.total_documents} 문서 완료, ${qs.progress_percentage || 0}%)`
            + (totalTime ? `, 총 경과: ${totalTime}` : '')
            + `, 모니터링: ${minutes}분 ${seconds}초`
          );
        } else {
          const elapsed = tries * 2;
          const minutes = Math.floor(elapsed / 60);
          const seconds = elapsed % 60;
          setInitProgress(
            (data.message || '진행 상태 확인 중...') + 
            ` (모니터링: ${minutes}분 ${seconds}초)`
          );
        }
        
        // 최대 시도 횟수 도달 시
        if (tries > maxTries) {
          setInitProgress(
            `장시간 진행 중입니다. (${Math.floor(maxTries * 2 / 60)}분 모니터링 완료) ` +
            '아래 "상태 확인 재시작" 버튼으로 계속 확인할 수 있습니다.'
          );
          setIsPolling(false);
          clearInterval(interval);
        }
      } catch (e) {
        consecutiveErrors++;
        const elapsed = tries * 2;
        const minutes = Math.floor(elapsed / 60);
        const seconds = elapsed % 60;
        
        if (consecutiveErrors >= maxConsecutiveErrors) {
          setInitProgress(
            `연결 오류가 지속되고 있습니다. (${consecutiveErrors}회 연속 실패, 모니터링: ${minutes}분 ${seconds}초) ` +
            '아래 "상태 확인 재시작" 버튼으로 다시 시도해보세요.'
          );
          setIsPolling(false);
          clearInterval(interval);
        } else {
          setInitProgress(
            `일시적 연결 오류 (${consecutiveErrors}/${maxConsecutiveErrors}, 모니터링: ${minutes}분 ${seconds}초) - 재시도 중...`
          );
        }
      }
    }, 2000);
  };

  const handleResetAllDefaults = async () => {
    setShowResetAllConfirm(false);
    try {
      // API 서버에서 기본값 가져오기
      const defaultsResponse = await apiClient.get('/config/defaults');
      
      console.log('API 응답 전체:', defaultsResponse);
      
      if (!defaultsResponse.success) {
        throw new Error('기본값을 가져오지 못했습니다.');
      }
      
      // standardResponse로 래핑된 응답에서 data 추출
      let defaultSettings = defaultsResponse.data as Record<string, string>;
      
      console.log('API에서 받은 기본값:', defaultSettings);
      console.log('기본값 타입:', typeof defaultSettings);
      console.log('기본값 키들:', Object.keys(defaultSettings || {}));
      console.log('configMeta에 정의된 키들:', Object.keys(configMeta));

      // configMeta에 정의된 키들만 필터링 (빈 값은 제외, API 키는 포함)
      console.log('🔍 필터링 전 기본값들:');
      console.log('4-Score System thresholds loaded from defaults');
      console.log('- SMTP_HOST:', defaultSettings['SMTP_HOST']);
      console.log('- SMTP_PORT:', defaultSettings['SMTP_PORT']);
      console.log('- SMTP_SECURE:', defaultSettings['SMTP_SECURE']);
      console.log('- SMTP_USERNAME:', defaultSettings['SMTP_USERNAME']);
      console.log('- SMTP_PASSWORD:', defaultSettings['SMTP_PASSWORD']);
      console.log('- USE_SMTP:', defaultSettings['USE_SMTP']);
      
      const filteredDefaults = Object.entries(defaultSettings)
        .filter(([key, value]) => {
          const hasConfigMeta = configMeta[key];
          const hasValue = value !== '' && value !== null && value !== undefined;
          
          // 4-Score System threshold validation
          if (key === 'RAG_KEYWORD_PRECISION_THRESHOLD' || key === 'RAG_SEMANTIC_RELEVANCE_THRESHOLD' || key === 'RAG_CONTENT_QUALITY_THRESHOLD') {
            console.log(`🔍 특별히 확인: ${key} = ${value} (타입: ${typeof value})`);
            console.log(`   - configMeta 존재: ${hasConfigMeta ? '✅' : '❌'}`);
            console.log(`   - 값 존재: ${hasValue ? '✅' : '❌'}`);
          }
          
          if (!hasConfigMeta) {
            console.log(`❌ configMeta에 없음: ${key} = ${value}`);
          } else if (!hasValue) {
            console.log(`❌ 빈 값: ${key} = ${value}`);
          } else {
            console.log(`✅ 포함: ${key} = ${value}`);
          }
          
          return hasConfigMeta && hasValue;
        })
        .reduce((acc, [key, value]) => {
          acc[key] = value;
          return acc;
        }, {} as Record<string, string>);
        
      console.log('최종 필터링된 기본값:', filteredDefaults);
      console.log('🔍 필터링 후 확인:');
      console.log('4-Score System filtered defaults applied');
      console.log('- SMTP_HOST:', filteredDefaults['SMTP_HOST']);
      console.log('- SMTP_PORT:', filteredDefaults['SMTP_PORT']);
      console.log('- SMTP_SECURE:', filteredDefaults['SMTP_SECURE']);
      console.log('- SMTP_USERNAME:', filteredDefaults['SMTP_USERNAME']);
      console.log('- SMTP_PASSWORD:', filteredDefaults['SMTP_PASSWORD']);
      console.log('- USE_SMTP:', filteredDefaults['USE_SMTP']);

      // 1. 로컬 상태 업데이트
      setEditedConfigs((prev) => ({ ...prev, ...filteredDefaults }));
      
      // 2. 서버에 자동 저장
      const savePromises = Object.entries(filteredDefaults).map(async ([key, value]) => {
        try {
          const meta = configMeta[key];
          let processedValue: string | number = value;
          
          console.log(`🔄 저장 시도: ${key} = ${value} (타입: ${meta?.type})`);
          
          if (meta?.type === 'boolean') {
            processedValue = parseBool(value) ? 'yes' : 'no';
            console.log(`   Boolean 변환: ${value} → ${processedValue}`);
          } else if (meta?.type === 'number') {
            processedValue = Number(value);
            console.log(`   Number 변환: ${value} → ${processedValue} (isNaN: ${isNaN(processedValue)})`);
            
            // 특별히 키워드 매칭 임계값들 확인
            // 4-Score System threshold validation
          if (key === 'RAG_KEYWORD_PRECISION_THRESHOLD' || key === 'RAG_SEMANTIC_RELEVANCE_THRESHOLD' || key === 'RAG_CONTENT_QUALITY_THRESHOLD') {
              console.log(`🔍 특별 디버깅 - ${key}:`);
              console.log(`   원본값: ${value} (타입: ${typeof value})`);
              console.log(`   Number 변환: ${processedValue} (타입: ${typeof processedValue})`);
              console.log(`   isNaN: ${isNaN(processedValue)}`);
              console.log(`   String()으로 변환: ${String(processedValue)}`);
            }
          }
          
          const result = await setConfig(key, processedValue);
          console.log(`✅ 저장 성공: ${key} = ${processedValue}`, '결과:', result);
          
          // 특별히 키워드 매칭 임계값들 저장 후 확인
          // 4-Score System threshold validation
          if (key === 'RAG_KEYWORD_PRECISION_THRESHOLD' || key === 'RAG_SEMANTIC_RELEVANCE_THRESHOLD' || key === 'RAG_CONTENT_QUALITY_THRESHOLD') {
            console.log(`🔍 ${key} 저장 후 즉시 확인:`);
            console.log(`   저장 함수 반환값:`, result);
            try {
              // 약간의 지연을 주고 다시 확인
              await new Promise(resolve => setTimeout(resolve, 100));
              const checkResult = await getConfig(key);
              console.log(`   저장 직후 값: ${checkResult} (타입: ${typeof checkResult})`);
            } catch (checkError) {
              console.error(`   저장 직후 확인 실패:`, checkError);
            }
          }
          
          return { key, success: result, error: null };
        } catch (error: any) {
          console.error(`❌ 저장 실패: ${key} = ${value}`, error);
          return { key, success: false, error: error.message };
        }
      });
      
      const results = await Promise.all(savePromises);
      const successfulResults = results.filter(r => r.success);
      const failedResults = results.filter(r => !r.success);
      
      if (successfulResults.length > 0) {
        // 성공한 설정들만 상태에 반영
        const successfulDefaults = Object.entries(filteredDefaults)
          .filter(([key]) => successfulResults.some(r => r.key === key))
          .reduce((acc, [key, value]) => {
            acc[key] = value;
            return acc;
          }, {} as Record<string, string>);
        
        // 저장 완료 후 전체 설정을 다시 로드
        console.log('🔄 설정 다시 로드 중...');
        await loadConfigs();
        
        if (failedResults.length === 0) {
          setSuccess(`모든 설정을 기본값으로 초기화했습니다! (${successfulResults.length}개 설정, API 키 포함)`);
        } else {
          setSuccess(`${successfulResults.length}개 설정을 초기화했습니다. ${failedResults.length}개 설정은 실패했습니다.`);
          console.error('Failed configs:', failedResults);
        }
      } else {
        setError('모든 설정 초기화에 실패했습니다.');
        console.error('All failed:', failedResults);
      }
    } catch (e: any) {
      setError(`설정 초기화 중 오류: ${e.message}`);
    }
  };

  if (isLoading) {
    return (
      <div className="container mx-auto px-4 py-8">
        <div className="flex justify-center items-center h-64">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500"></div>
        </div>
      </div>
    );
  }

  return (
    <div className="container mx-auto px-4 py-8 max-w-7xl h-full min-h-screen overflow-y-auto pb-40" style={{ backgroundColor: 'var(--body-bg)' }}>
      {/* 알림 메시지 */}
      {error && !error?.includes('컬렉션') && (
        <div className="mb-6 p-4 rounded-lg" style={{ backgroundColor: 'var(--error)', color: 'white' }}>
          {error}
          <button onClick={() => setError(null)} className="ml-4 text-base underline">닫기</button>
        </div>
      )}
      {success && !success?.includes('컬렉션') && (
        <div className="mb-6 p-4 rounded-lg" style={{ backgroundColor: 'var(--success)', color: 'white' }}>
          {success}
          <button onClick={() => setSuccess(null)} className="ml-4 text-base underline">닫기</button>
        </div>
      )}
      {showResetAllConfirm && (
        <div className="fixed inset-0 z-50 flex items-center justify-center backdrop-blur-sm" style={{ backgroundColor: 'rgba(0, 0, 0, 0.5)' }}>
          <div className="rounded-lg shadow-xl p-6 w-full max-w-md mx-4" style={{ backgroundColor: 'var(--card-bg)', border: '1px solid var(--card-border)' }}>
            <h3 className="text-lg font-bold mb-4" style={{ color: 'var(--text-primary)' }}>전체 설정 초기화</h3>
            <p className="mb-6" style={{ color: 'var(--text-secondary)' }}>
              정말로 모든 설정을 기본값으로 되돌리시겠습니까?
              <br />
              <strong style={{ color: 'var(--error)' }}>⚠️ 주의: API 키들도 기본값으로 설정됩니다!</strong>
              <br />
              이 작업은 즉시 적용되며 되돌릴 수 없습니다.
            </p>
            <div className="flex justify-end space-x-4">
              <button
                onClick={() => setShowResetAllConfirm(false)}
                className="px-4 py-2 rounded-lg hover:opacity-80 transition-opacity"
                style={{ backgroundColor: 'var(--card-border)', color: 'var(--text-primary)' }}
              >
                취소
              </button>
              <button
                onClick={handleResetAllDefaults}
                className="px-4 py-2 rounded-lg hover:opacity-80 transition-opacity"
                style={{ backgroundColor: 'var(--error)', color: 'white' }}
              >
                초기화
              </button>
            </div>
          </div>
        </div>
      )}
      <div className="grid grid-cols-1 lg:grid-cols-4 gap-8">
        {/* 사이드바 */}
        <div className="lg:col-span-1">
          <div className="rounded-lg shadow-md p-4" style={{ backgroundColor: 'var(--sidebar-bg)', border: '1px solid var(--card-border)' }}>
            <h2 className="font-semibold mb-4" style={{ color: 'var(--text-primary)' }}>설정 카테고리</h2>
            <nav className="space-y-2">
              {configGroups.map((group) => (
                <button
                  key={group.name}
                  onClick={() => setActiveGroup(group.name)}
                  className="w-full text-left px-3 py-2 rounded-lg transition-colors"
                  style={{
                    backgroundColor: activeGroup === group.name ? 'var(--accent-blue)' : 'transparent',
                    color: activeGroup === group.name ? 'white' : 'var(--text-primary)',
                    border: activeGroup === group.name ? 'none' : '1px solid transparent'
                  }}
                  onMouseEnter={(e) => {
                    if (activeGroup !== group.name) {
                      e.currentTarget.style.backgroundColor = 'var(--card-bg)';
                    }
                  }}
                  onMouseLeave={(e) => {
                    if (activeGroup !== group.name) {
                      e.currentTarget.style.backgroundColor = 'transparent';
                    }
                  }}
                >
                  <div className="font-medium">{group.title}</div>
                </button>
              ))}
            </nav>
            <div className="mt-6 pt-4 border-t" style={{ borderColor: 'var(--card-border)' }}>
              <button
                onClick={() => setShowResetAllConfirm(true)}
                className="w-full px-3 py-2 rounded-lg transition-colors text-sm font-medium hover:opacity-80"
                style={{ backgroundColor: 'var(--error)', color: 'white' }}
              >
                기본값 초기화
              </button>
            </div>
          </div>
        </div>
        {/* 본문 */}
        <div className="lg:col-span-3">
          <div className="rounded-lg p-6 pb-12" style={{ backgroundColor: 'var(--card-bg)', border: '1px solid var(--card-border)' }}>
            <div className="flex justify-between items-center mb-6">
              <div>
                <h2 className="text-xl font-semibold" style={{ color: 'var(--text-primary)' }}>{configGroups.find((g) => g.name === activeGroup)?.title}</h2>
              </div>
              {/* 저장 버튼: 라이선스 그룹이 아닐 때만 노출 */}
              {activeGroup !== 'license' && (
                <div className="flex space-x-2">
                  <button
                    onClick={saveConfigs}
                    disabled={isSaving}
                    className="px-4 py-2 rounded-lg hover:opacity-80 disabled:opacity-50 transition-opacity font-medium"
                    style={{ 
                      backgroundColor: getSaveButtonStyle(activeGroup).bg, 
                      color: getSaveButtonStyle(activeGroup).textColor 
                    }}
                  >
                    {isSaving ? '...' : getSaveButtonStyle(activeGroup).text}
                  </button>
                </div>
              )}
            </div>
            {renderGroupContent()}
          </div>
        </div>
      </div>
    </div>
  );
} 