'use client';

import { useState, useEffect, Suspense } from 'react';
import { useRouter, useSearchParams } from 'next/navigation';
import dynamic from 'next/dynamic';
import {
  Receipt,
  Activity,
  Leaf,
  ChevronLeft,
  ChevronRight,
  Menu,
  Wrench,
  ClipboardList
} from 'lucide-react';

const QuotationContent = dynamic(
  () => import('@/components/agent-features/QuotationContent'),
  {
    loading: () => <LoadingSpinner message="견적서 로딩 중..." />,
    ssr: false
  }
);

const HealthBatchContent = dynamic(
  () => import('@/components/agent-features/HealthBatchContent'),
  {
    loading: () => <LoadingSpinner message="헬스 로딩 중..." />,
    ssr: false
  }
);

const ESGAnalysisContent = dynamic(
  () => import('@/components/agent-features/ESGAnalysisContent'),
  {
    loading: () => <LoadingSpinner message="ESG 로딩 중..." />,
    ssr: false
  }
);

const WeeklyReportContent = dynamic(
  () => import('@/components/agent-features/WeeklyReportContent'),
  {
    loading: () => <LoadingSpinner message="주간보고서 로딩 중..." />,
    ssr: false
  }
);

// 로딩 스피너 컴포넌트
function LoadingSpinner({ message }: { message: string }) {
  return (
    <div className="flex flex-col items-center justify-center h-64">
      <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mb-4"></div>
      <p className="text-gray-600 dark:text-gray-400">{message}</p>
    </div>
  );
}

// 메뉴 아이템 정의
const menuItems = [
  {
    id: 'quotation',
    name: '견적서',
    icon: Receipt,
    description: 'AI 견적서 생성',
    color: 'green',
    adminOnly: false
  },
  {
    id: 'weekly-report',
    name: '주간보고서',
    icon: ClipboardList,
    description: '주간/월간 업무보고서',
    color: 'purple',
    adminOnly: false
  },
  {
    id: 'health',
    name: '헬스',
    icon: Activity,
    description: '건강검진 분석',
    color: 'red',
    adminOnly: true
  },
  {
    id: 'esg',
    name: 'ESG',
    icon: Leaf,
    description: 'ESG 탄소배출 분석',
    color: 'emerald',
    adminOnly: true
  }
];

// 메인 도구 페이지 컴포넌트
function ToolsPageContent() {
  const router = useRouter();
  const searchParams = useSearchParams();

  // URL에서 현재 탭 가져오기, 기본값은 'quotation'
  const currentTab = searchParams.get('tab') || 'quotation';
  const [activeTab, setActiveTab] = useState(currentTab);
  const [isSidebarCollapsed, setIsSidebarCollapsed] = useState(false);
  const [isMobileSidebarOpen, setIsMobileSidebarOpen] = useState(false);
  const [user, setUser] = useState<{ role: string } | null>(null);

  // URL 파라미터 변경 시 탭 업데이트
  useEffect(() => {
    const tab = searchParams.get('tab');
    if (tab && menuItems.some(item => item.id === tab)) {
      setActiveTab(tab);
    }
  }, [searchParams]);

  // 사용자 역할 확인
  useEffect(() => {
    const userRole = localStorage.getItem('userRole');
    setUser({ role: userRole || 'user' });
  }, []);

  // 탭 변경 핸들러
  const handleTabChange = (tabId: string) => {
    setActiveTab(tabId);
    router.push(`/tools?tab=${tabId}`, { scroll: false });
    setIsMobileSidebarOpen(false);
  };

  // 현재 탭에 따른 콘텐츠 렌더링
  const renderContent = () => {
    switch (activeTab) {
      case 'quotation':
        return <QuotationContent />;
      case 'weekly-report':
        return <WeeklyReportContent />;
      case 'health':
        // 헬스는 admin만 접근 가능
        if (user?.role !== 'admin') {
          return (
            <div className="flex flex-col items-center justify-center h-64">
              <Activity className="w-16 h-16 text-gray-400 mb-4" />
              <p className="text-gray-600 dark:text-gray-400">관리자만 접근 가능합니다.</p>
            </div>
          );
        }
        return <HealthBatchContent />;
      case 'esg':
        // ESG는 admin만 접근 가능
        if (user?.role !== 'admin') {
          return (
            <div className="flex flex-col items-center justify-center h-64">
              <Leaf className="w-16 h-16 text-gray-400 mb-4" />
              <p className="text-gray-600 dark:text-gray-400">관리자만 접근 가능합니다.</p>
            </div>
          );
        }
        return <ESGAnalysisContent />;
      default:
        return <QuotationContent />;
    }
  };

  // 필터된 메뉴 (admin이 아니면 헬스, ESG 숨김)
  const filteredMenuItems = menuItems.filter(item => {
    if (item.adminOnly && user?.role !== 'admin') {
      return false;
    }
    return true;
  });

  const getColorClasses = (color: string, isActive: boolean) => {
    const colors: Record<string, { active: string; inactive: string }> = {
        green: {
        active: 'tools-menu-active-green',
        inactive: 'tools-menu-inactive'
      },
      purple: {
        active: 'tools-menu-active-purple',
        inactive: 'tools-menu-inactive'
      },
      red: {
        active: 'tools-menu-active-red',
        inactive: 'tools-menu-inactive'
      },
      emerald: {
        active: 'tools-menu-active-emerald',
        inactive: 'tools-menu-inactive'
      }
    };
    return isActive ? colors[color].active : colors[color].inactive;
  };

  return (
    <div className="flex h-[calc(100vh-120px)] overflow-hidden" style={{ backgroundColor: 'var(--body-bg)' }}>
      {/* 모바일 사이드바 오버레이 */}
      {isMobileSidebarOpen && (
        <div
          className="fixed inset-0 bg-black/50 z-40 lg:hidden"
          onClick={() => setIsMobileSidebarOpen(false)}
        />
      )}

      {/* 사이드바 */}
      <aside
        className={`
          tools-sidebar
          fixed lg:relative z-50 h-full
          transition-all duration-300 ease-in-out
          ${isSidebarCollapsed ? 'w-16' : 'w-64'}
          ${isMobileSidebarOpen ? 'translate-x-0' : '-translate-x-full lg:translate-x-0'}
        `}
      >
        {/* 사이드바 헤더 */}
        <div className="flex items-center justify-between p-4 border-b border-gray-200 dark:border-gray-700">
          {!isSidebarCollapsed && (
            <div className="flex items-center gap-2">
              <Wrench className="w-5 h-5 text-gray-600 dark:text-gray-400" />
              <h2 className="text-lg font-semibold text-gray-800 dark:text-gray-200">
                AI 도구
              </h2>
            </div>
          )}
          <button
            onClick={() => setIsSidebarCollapsed(!isSidebarCollapsed)}
            className="p-2 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors hidden lg:block"
          >
            {isSidebarCollapsed ? (
              <ChevronRight className="w-5 h-5 text-gray-600 dark:text-gray-400" />
            ) : (
              <ChevronLeft className="w-5 h-5 text-gray-600 dark:text-gray-400" />
            )}
          </button>
        </div>

        {/* 메뉴 리스트 */}
        <nav className="p-2 space-y-1">
          {filteredMenuItems.map((item) => {
            const isActive = activeTab === item.id;
            const IconComponent = item.icon;

            return (
              <button
                key={item.id}
                onClick={() => handleTabChange(item.id)}
                className={`
                  w-full flex items-center gap-3 px-3 py-3 rounded-lg
                  transition-all duration-200
                  ${getColorClasses(item.color, isActive)}
                `}
                title={isSidebarCollapsed ? item.name : undefined}
              >
                <IconComponent className="w-5 h-5 flex-shrink-0" />
                {!isSidebarCollapsed && (
                  <div className="flex flex-col items-start min-w-0">
                    <span className="font-medium truncate">{item.name}</span>
                    <span className="text-xs truncate tools-menu-description">
                      {item.description}
                    </span>
                  </div>
                )}
              </button>
            );
          })}
        </nav>
      </aside>

      {/* 메인 콘텐츠 영역 */}
      <main className="flex-1 overflow-auto">
        {/* 모바일 헤더 */}
        <div className="lg:hidden flex items-center gap-2 p-4 border-b border-gray-200 dark:border-gray-700" style={{ backgroundColor: 'var(--card-bg)' }}>
          <button
            onClick={() => setIsMobileSidebarOpen(true)}
            className="p-2 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700"
          >
            <Menu className="w-5 h-5 text-gray-600 dark:text-gray-400" />
          </button>
          <span className="font-medium text-gray-800 dark:text-gray-200">
            {menuItems.find(item => item.id === activeTab)?.name || '도구'}
          </span>
        </div>

        {/* 콘텐츠 */}
        <div className="p-4 lg:p-6">
          {renderContent()}
        </div>
      </main>
    </div>
  );
}

// 메인 export - Suspense로 감싸기
export default function ToolsPage() {
  return (
    <Suspense fallback={<LoadingSpinner message="페이지 로딩 중..." />}>
      <ToolsPageContent />
    </Suspense>
  );
}
