import React, { useState, useMemo, memo } from 'react';
import ReactMarkdown from 'react-markdown';
import remarkGfm from 'remark-gfm';
import Prism from 'prismjs';
import 'prismjs/components/prism-python';
import 'prismjs/components/prism-jsx';
import 'prismjs/components/prism-tsx';
import 'prismjs/components/prism-typescript';
import 'prismjs/components/prism-json';
import 'prismjs/components/prism-bash';
import 'prismjs/components/prism-markdown';
import { useRef, useEffect } from 'react';
import { useTheme } from 'next-themes';
import remarkBreaks from 'remark-breaks';
import remarkBoldBox from '@/utils/remarkBoldBox';
import { getApiServerUrl } from '@/config/serverConfig';

interface CodeRendererProps {
  content: string;
  className?: string;
  messageId?: string;
  onExecuteCode?: (messageId: string, code: string, language: string) => Promise<void>;
  onDownloadCode?: (messageId: string, code: string, language: string) => Promise<void>;
  onRegenerateCode?: (messageId: string) => Promise<void>;
  isExecuting?: boolean;
  executionResult?: string;
}

// VSCode 스타일 코드블록 컴포넌트 (완전히 정적으로 변경)
const CodeBlock = memo(function CodeBlock({ code, language = 'plaintext' }: { code: string; language?: string }) {
  // Prism 하이라이트 - 동적 스타일 제거하여 깜빡임 방지
  const highlighted = Prism.highlight(code, Prism.languages[language] || Prism.languages.plaintext, language);
  
  const highlightedLines = highlighted.split('\n');
  const codeLines = code.split('\n');

  return (
    <div className="relative group rounded-lg my-4 border overflow-hidden border-gray-200 bg-gray-50 dark:border-gray-700 dark:bg-gray-900">
      {/* 언어 표시 */}
      {language && language !== 'plaintext' && (
        <div className="px-3 py-1 text-xs font-medium border-b bg-gray-100 text-gray-600 border-gray-200 dark:bg-gray-800 dark:text-gray-300 dark:border-gray-700">
          {language}
        </div>
      )}
      
      {/* 복사 버튼 */}
      <button
        className="absolute top-2 right-2 z-10 px-2 py-1 text-xs rounded transition-colors bg-gray-200 text-gray-600 hover:bg-blue-600 hover:text-white dark:bg-gray-800 dark:text-gray-300"
        onClick={() => {
          navigator.clipboard.writeText(code);
        }}
        title="코드 복사"
      >
        복사
      </button>
      
      <div className="bg-gray-50 dark:bg-gray-900">
        <pre 
          className="m-0 p-0 font-mono overflow-x-auto bg-gray-50 dark:bg-gray-900"
          style={{ 
            fontSize: '13px', 
            lineHeight: '1.6'
          }}
        >
          {codeLines.map((line, i) => (
            <div 
              key={i} 
              className="flex transition-colors bg-gray-50 hover:bg-gray-100 dark:bg-gray-900 dark:hover:bg-gray-800"
            >
              <span
                className="select-none text-right pr-3 pl-3 text-gray-400 bg-gray-100 border-r border-gray-300 dark:text-gray-500 dark:bg-gray-800 dark:border-r dark:border-gray-600"
                style={{ 
                  minWidth: '3em', 
                  fontSize: '13px', 
                  lineHeight: '1.6'
                }}
              >
                {i + 1}
              </span>
              <code
                className="pl-3 pr-4 flex-1 block token-code text-gray-900 bg-gray-50 dark:text-gray-100 dark:bg-gray-900"
                style={{ 
                  fontSize: '13px', 
                  lineHeight: '1.6',
                  minHeight: '1.6em',
                  fontFamily: 'D2Coding, Consolas, Monaco, monospace'
                }}
                dangerouslySetInnerHTML={{ 
                  __html: highlightedLines[i] || '<span>&nbsp;</span>'
                }}
              />
            </div>
          ))}
        </pre>
      </div>
    </div>
  );
});

export const CodeRenderer: React.FC<CodeRendererProps> = ({ 
  content, 
  className = '', 
  messageId,
  onExecuteCode,
  onDownloadCode,
  onRegenerateCode,
  isExecuting,
  executionResult
}) => {
  // 참조 정보 추출 및 content 분리
  const referenceInfoMatch = content.match(/<!-- REFERENCE_INFO:([\s\S]*?) -->/);
  const downloadInfoMatch = content.match(/<!-- DOWNLOAD_INFO:([\s\S]*?) -->/);
  let downloadInfo: any[] = [];
  let cleanContent = content;
  
  // 새로운 REFERENCE_INFO 형식 처리
  if (referenceInfoMatch) {
    try {
      const referenceInfo = JSON.parse(referenceInfoMatch[1]);
      downloadInfo = referenceInfo.downloadInfo || [];
      cleanContent = content.replace(/<!-- REFERENCE_INFO:[\s\S]*? -->/, '').trim();
    } catch (error) {
      console.error('referenceInfo 파싱 오류:', error);
    }
  }
  // 기존 DOWNLOAD_INFO 형식 호환성 유지
  else if (downloadInfoMatch) {
    try {
      downloadInfo = JSON.parse(downloadInfoMatch[1]);
      cleanContent = content.replace(/<!-- DOWNLOAD_INFO:[\s\S]*? -->/, '').trim();
    } catch (error) {
      console.error('downloadInfo 파싱 오류:', error);
    }
  }

  // 출처 섹션이 있는지 확인
  const hasReferencesSection = cleanContent.includes('# 출처(References)') || cleanContent.includes('## 출처');

  return (
    <div className={`markdown-content text-base ${className}`} style={{ fontFamily: 'Spoqa Han Sans Neo' }}>
      <ReactMarkdown
        remarkPlugins={[remarkGfm, remarkBreaks, remarkBoldBox]}
        unwrapDisallowed={true}
        components={{
          // 강조 텍스트 (**text**)
          strong: ({ children }) => (
            <strong className="font-bold text-orange-600 dark:text-orange-400 bg-orange-50 dark:bg-orange-900/20 px-1 py-0.5 rounded">
              {children}
            </strong>
          ),
          // 기울임 텍스트 (*text*)
          em: ({ children }) => (
            <span className="italic text-purple-600 dark:text-purple-400">
              {children}
            </span>
          ),
          // 코드 블록
          code: ({ className, children, ...props }: any) => {
            const inline = !className?.includes('language-');
            if (inline) {
              return (
                <code
                  className="px-1.5 py-0.5 bg-gray-100 dark:bg-gray-800 text-red-600 dark:text-red-400 rounded text-sm font-mono"
                  {...props}
                >
                  {children}
                </code>
              );
            }
            // language 추출
            const match = /language-(\w+)/.exec(className || '');
            return (
              <CodeBlock code={String(children).replace(/\n$/, '')} language={match?.[1] || 'plaintext'} />
            );
          },
          // 링크
          a: ({ href, children }) => {
            if (!href) return <span>{children}</span>;
            
            // rag/download 관련 링크는 일반 텍스트로 처리 (별도 다운로드 섹션이 있으므로)
            if (href.includes('rag/download') || href.includes('/download?filename=') || href.includes('filename=')) {
              // 중첩된 링크에서 텍스트만 추출
              const textContent = React.Children.toArray(children).map(child => {
                if (typeof child === 'string') return child;
                if (React.isValidElement(child) && (child as any).props?.children) {
                  return React.Children.toArray((child as any).props.children).join('');
                }
                return '';
              }).join('');
              
              return <span className="text-gray-600 dark:text-gray-400">{textContent}</span>;
            }
            
            // 출처 섹션의 링크인지 확인 (링크 텍스트나 URL로 판단)
            const linkText = React.Children.toArray(children).join('').toLowerCase();
            const isReferenceLink = hasReferencesSection && (
              href?.includes('http') || 
              linkText.includes('pdf') ||
              linkText.includes('문서') ||
              linkText.includes('페이지') ||
              href?.includes('github') ||
              href?.includes('docs') ||
              linkText.length > 30 // 긴 제목의 링크는 출처일 가능성이 높음
            );
            
            if (isReferenceLink) {
              return (
                <a
                  href={href}
                  target="_blank"
                  rel="noopener noreferrer"
                  className="text-blue-600 dark:text-blue-400 hover:underline text-sm break-words break-all"
                  style={{ wordBreak: 'break-all', overflowWrap: 'anywhere' }}
                >
                  {children}
                </a>
              );
            }
            
            return (
              <a
                href={href}
                target="_blank"
                rel="noopener noreferrer"
                className="text-blue-600 dark:text-blue-400 hover:underline break-words break-all"
                style={{ wordBreak: 'break-all', overflowWrap: 'anywhere' }}
              >
                {children}
              </a>
            );
          },
          // 리스트
          ul: ({ children }) => (
            <ul className="list-disc list-inside my-2 space-y-1">
              {children}
            </ul>
          ),
          ol: ({ children }) => (
            <ol className="list-decimal list-outside my-2 space-y-1 ml-6">
              {children}
            </ol>
          ),
          li: ({ children }) => {
            // 출처 섹션의 리스트 항목인지 확인 (더 정확한 감지)
            const childrenText = React.Children.toArray(children).join('').toLowerCase();
            const isReferenceItem = hasReferencesSection && (
              childrenText.includes('.pdf') || 
              childrenText.includes('.md') || 
              childrenText.includes('.doc') ||
              childrenText.includes('.txt') ||
              childrenText.includes('페이지') ||
              childrenText.includes('page') ||
              childrenText.includes('http') ||
              /^\d+\.\s/.test(childrenText) // 숫자로 시작하는 리스트 (1. 2. 등)
            );
            
            if (isReferenceItem) {
              // 파일명 추출 함수
              const extractFileName = (text: string): string | null => {
                // 다양한 패턴으로 파일명 추출
                const patterns = [
                  /([^/\\]+\.(?:pdf|doc|docx|txt|md|hwp|ppt|pptx|xls|xlsx))/gi,
                  /관련 문서:\s*([^(\n]+)/gi,
                  /문서:\s*([^(\n]+)/gi,
                  /출처:\s*([^(\n]+)/gi
                ];
                
                for (const pattern of patterns) {
                  const match = text.match(pattern);
                  if (match) {
                    return match[1]?.trim() || match[0]?.trim();
                  }
                }
                return null;
              };

              const fileName = extractFileName(childrenText);
              const hasDownloadableFile = fileName && !childrenText.includes('http');

              if (hasDownloadableFile) {
                return (
                  <li className="text-gray-600 dark:text-gray-400 pl-2 text-sm leading-relaxed flex items-center justify-between group">
                    <span className="flex-1">{children}</span>
                    <button
                      onClick={async () => {
                        try {
                          // API 키를 포함한 헤더 생성
                          const headers: Record<string, string> = {};
                          
                          if (typeof window !== 'undefined') {
                            const apiKey = localStorage.getItem('apiKey');
                            const token = localStorage.getItem('userToken');
                            
                            if (apiKey) {
                              headers['X-API-Key'] = apiKey;
                            }
                            if (token) {
                              headers['Authorization'] = `Bearer ${token}`;
                            }
                          }
                          
                          // API 서버 URL 구성
                          const apiServerUrl = getApiServerUrl();
                          const downloadUrl = `${apiServerUrl}/rag/download?filename=${encodeURIComponent(fileName)}`;
                          
                          // 인증된 fetch 요청으로 파일 다운로드
                          const response = await fetch(downloadUrl, {
                            method: 'GET',
                            headers
                          });
                          
                          if (!response.ok) {
                            const errorData = await response.json();
                            throw new Error(errorData.error?.message || '파일 다운로드에 실패했습니다.');
                          }
                          
                          // Blob으로 변환하여 다운로드
                          const blob = await response.blob();
                          const url = window.URL.createObjectURL(blob);
                          const a = document.createElement('a');
                          a.href = url;
                          a.download = fileName.split('/').pop() || 'download';
                          document.body.appendChild(a);
                          a.click();
                          window.URL.revokeObjectURL(url);
                          document.body.removeChild(a);
                        } catch (error) {
                          console.error('다운로드 오류:', error);
                          alert(`파일 다운로드 중 오류가 발생했습니다: ${error instanceof Error ? error.message : '알 수 없는 오류'}`);
                        }
                      }}
                      className="ml-2 opacity-0 group-hover:opacity-100 transition-opacity duration-200 inline-flex items-center px-2 py-1 text-xs font-medium text-white bg-blue-600 hover:bg-blue-700 rounded-md"
                      title={`${fileName} 다운로드`}
                    >
                      <svg className="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                      </svg>
                      다운로드
                    </button>
                  </li>
                );
              }
              
              return (
                <li className="text-gray-600 dark:text-gray-400 pl-2 text-sm leading-relaxed">
                  {children}
                </li>
              );
            }
            
            return (
              <li className="text-gray-700 dark:text-gray-300 pl-2">
                {children}
              </li>
            );
          },
          // 제목
          h1: ({ children }) => {
            // 출처(References) 제목인지 확인
            const childrenText = React.Children.toArray(children).join('').toLowerCase();
            const isReferences = childrenText.includes('출처') || childrenText.includes('references');
            
            if (isReferences) {
              return (
                <h3 className="text-lg font-bold text-gray-900 dark:text-white my-3 flex items-center space-x-2 pt-10">
                  <svg className="w-5 h-5 text-blue-600 dark:text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                  </svg>
                  <span>출처</span>
                </h3>
              );
            }
            
            return (
              <h1 className="text-xl font-bold text-gray-900 dark:text-white my-3">
                {children}
              </h1>
            );
          },
          h2: ({ children }) => (
            <h2 className="text-lg font-bold text-gray-900 dark:text-white mt-8 my-2">
              {children}
            </h2>
          ),
          h3: ({ children }) => (
            <h4 className="text-sm font-bold text-gray-900 dark:text-white mt-8 my-2">
              {children}
            </h4>
          ),
          // 인용문
          blockquote: ({ children }) => (
            <blockquote className="border-l-4 border-gray-300 dark:border-gray-600 pl-4 my-2 italic text-gray-600 dark:text-gray-400">
              {children}
            </blockquote>
          ),
          // 구분선
          hr: () => (
            <hr className="border-gray-300 dark:border-gray-600 my-4" />
          ),

          // 표
          table: ({ children }) => (
            <div className="overflow-x-auto my-2">
              <table className="min-w-full border border-gray-300 dark:border-gray-600">
                {children}
              </table>
            </div>
          ),
          thead: ({ children }) => (
            <thead className="bg-gray-50 dark:bg-gray-800">
              {children}
            </thead>
          ),
          tbody: ({ children }) => (
            <tbody className="bg-white dark:bg-gray-900">
              {children}
            </tbody>
          ),
          tr: ({ children }) => (
            <tr className="border-b border-gray-300 dark:border-gray-600">
              {children}
            </tr>
          ),
          th: ({ children }) => (
            <th className="px-4 py-2 text-left font-medium text-gray-900 dark:text-white">
              {children}
            </th>
          ),
          td: ({ children }) => (
            <td className="px-4 py-2 text-gray-700 dark:text-gray-300">
              {children}
            </td>
          ),
          // 이미지 처리 - PNG 이미지만 처리 (SVG는 제외)
          img: ({ src, alt, title }) => {
            if (!src) return null;
            
            // src가 문자열인지 확인
            if (typeof src !== 'string') return null;
            
            // PNG 이미지만 처리하고 SVG는 제외
            const isPngImage = src.toLowerCase().includes('.png');
            const isSvgImage = src.toLowerCase().includes('.svg');
            
            // SVG 이미지는 렌더링하지 않음 (PNG 버전이 있다고 가정)
            if (isSvgImage) {
              return null;
            }
            
            // PNG 이미지가 아닌 경우에도 일반적인 이미지 처리 (jpg, jpeg, gif 등)
            return (
              <img
                src={src}
                alt={alt || '이미지'}
                title={title || alt || '이미지'}
                className="block max-w-full h-auto mx-auto my-4 rounded-lg border border-gray-200 dark:border-gray-700 shadow-sm hover:shadow-md transition-shadow"
                onError={(e) => {
                  console.error('[이미지 로딩 오류]', {
                    src,
                    alt,
                    error: e
                  });
                  // 오류 발생 시 대체 텍스트 표시
                  const target = e.target as HTMLImageElement;
                  target.style.display = 'none';
                  
                  // 새로운 오류 표시 요소 생성
                  const errorDiv = document.createElement('div');
                  errorDiv.className = 'flex items-center justify-center p-4 bg-gray-100 dark:bg-gray-800 rounded-lg border border-gray-300 dark:border-gray-600 my-4';
                  errorDiv.innerHTML = `
                    <div class="text-center">
                      <svg class="w-8 h-8 mx-auto mb-2 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"></path>
                      </svg>
                      <p class="text-sm text-gray-500 dark:text-gray-400">이미지를 불러올 수 없습니다</p>
                      <p class="text-xs text-gray-400 dark:text-gray-500 mt-1 break-all">${src}</p>
                    </div>
                  `;
                  
                  // 이미지 요소를 오류 표시 요소로 교체
                  target.parentNode?.replaceChild(errorDiv, target);
                }}
              />
            );
          },
          // 문단 처리 - 이미지가 포함된 경우 별도 처리
          p: ({ children }) => {
            // 문단 안에 이미지만 있는 경우 처리
            const hasOnlyImage = React.Children.count(children) === 1 && 
              React.Children.toArray(children).some(child => 
                React.isValidElement(child) && child.type === 'img'
              );
            
            if (hasOnlyImage) {
              // 이미지만 있는 문단은 div로 감싸서 블록 레벨로 처리
              return <div className="my-4">{children}</div>;
            }
            
            return (
              <p className="my-2 leading-loose">
                {children}
              </p>
            );
          },
        }}
      >
        {cleanContent}
      </ReactMarkdown>
      
      {/* 다운로드 섹션 렌더링 */}
      {downloadInfo.length > 0 && (
        <div className="mt-4 pt-3 border-t border-gray-200 dark:border-gray-700">
          <div className="space-y-2">
            {downloadInfo.map((item, index) => {
              const apiServerUrl = getApiServerUrl();
              const downloadUrl = `${apiServerUrl}/rag/download?filename=${encodeURIComponent(item.filename)}`;
              
              return (
                <div key={index} className="flex items-center justify-between p-2 bg-gray-50 dark:bg-gray-800 rounded border border-gray-200 dark:border-gray-600">
                  <div className="flex items-center space-x-2">
                    <svg className="w-4 h-4 text-blue-600 dark:text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                    </svg>
                    <div>
                      <div className="text-xs font-medium text-gray-900 dark:text-gray-100">
                        {item.filename}
                      </div>
                      {item.pages && item.pages.length > 0 && (
                        <div className="text-xs text-gray-500 dark:text-gray-400">
                          페이지: {item.pages.join(', ')}
                        </div>
                      )}
                    </div>
                  </div>
                  <button
                    onClick={async () => {
                      try {
                        console.log('[다운로드] 시작:', item.filename);
                        
                        const response = await fetch(downloadUrl, {
                          method: 'GET',
                          headers: {
                            'X-API-Key': localStorage.getItem('apiKey') || 'airun_1_3d85009c98964579c622eded69997b16',
                            'Authorization': `Bearer ${localStorage.getItem('userToken') || ''}`
                          }
                        });
                        
                        if (!response.ok) {
                          const errorData = await response.json().catch(() => ({ error: { message: '파일을 찾을 수 없습니다.' } }));
                          throw new Error(errorData.error?.message || '파일을 찾을 수 없습니다.');
                        }
                        
                        const blob = await response.blob();
                        const url = window.URL.createObjectURL(blob);
                        const a = document.createElement('a');
                        a.href = url;
                        a.download = item.filename;
                        document.body.appendChild(a);
                        a.click();
                        window.URL.revokeObjectURL(url);
                        document.body.removeChild(a);
                        
                        console.log('[다운로드] 완료:', item.filename);
                      } catch (error) {
                        console.error('다운로드 오류:', error);
                        alert('다운로드에 실패했습니다: ' + (error instanceof Error ? error.message : String(error)));
                      }
                    }}
                    className="inline-flex items-center px-2 py-1 text-xs font-medium text-white bg-blue-600 hover:bg-blue-700 rounded transition-colors duration-200"
                    title={`${item.filename} 다운로드`}
                  >
                    <svg className="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                    </svg>
                    다운로드
                  </button>
                </div>
              );
            })}
          </div>
        </div>
      )}


    </div>
  );
}; 