import React from 'react';
import ReactMarkdown from 'react-markdown';
import remarkGfm from 'remark-gfm';
import remarkBreaks from 'remark-breaks';
import { getApiServerUrl } from '@/config/serverConfig';

interface CustomMarkdownRendererProps {
  content: string;
  className?: string;
  ragMode?: string; // RAG 모드 추가 (intelligence, normal, rich, fast)
}

export const CustomMarkdownRenderer: React.FC<CustomMarkdownRendererProps> = ({ content, className = '', ragMode }) => {
  // downloadInfo 추출 및 content 분리
  const downloadInfoMatch = content.match(/<!-- DOWNLOAD_INFO:([\s\S]*?) -->/);
  let downloadInfo: any[] = [];
  let cleanContent = content;
  
  if (downloadInfoMatch) {
    try {
      downloadInfo = JSON.parse(downloadInfoMatch[1]);
      cleanContent = content.replace(/<!-- DOWNLOAD_INFO:[\s\S]*? -->/, '').trim();
    } catch (error) {
      console.error('downloadInfo 파싱 오류:', error);
    }
  }

  // 출처 섹션이 있는지 확인
  const hasReferencesSection = cleanContent.includes('# 출처(References)') || cleanContent.includes('## 출처');

  return (
    <div className={`custom-markdown ${className}`}>
      <ReactMarkdown
        remarkPlugins={[remarkGfm, remarkBreaks]}
        components={{
        p: ({ children }) => (
          <div className={`text-sm ${hasReferencesSection ? 'my-0 leading-[1.2]' : 'my-1 leading-[1.6]'}`}>{children}</div>
        ),
        li: ({ children }) => {
          // 출처 섹션의 리스트 항목인지 확인
          const childrenText = React.Children.toArray(children).join('').toLowerCase();
          const isReferenceItem = hasReferencesSection && (
            childrenText.includes('.pdf') || 
            childrenText.includes('.md') || 
            childrenText.includes('.doc') ||
            childrenText.includes('.txt') ||
            childrenText.includes('페이지') ||
            childrenText.includes('page') ||
            childrenText.includes('http') ||
            /^\d+\.\s/.test(childrenText) // 숫자로 시작하는 리스트 (1. 2. 등)
          );
          
          if (isReferenceItem) {
            // 파일명 추출 함수
            const extractFileName = (text: string): string | null => {
              // 다양한 패턴으로 파일명 추출
              const patterns = [
                /([^/\\]+\.(?:pdf|doc|docx|txt|md|hwp|ppt|pptx|xls|xlsx))/gi,
                /관련 문서:\s*([^(\n]+)/gi,
                /문서:\s*([^(\n]+)/gi,
                /출처:\s*([^(\n]+)/gi
              ];
              
              for (const pattern of patterns) {
                const match = text.match(pattern);
                if (match) {
                  return match[1]?.trim() || match[0]?.trim();
                }
              }
              return null;
            };

            const fileName = extractFileName(childrenText);
            const hasDownloadableFile = fileName && !childrenText.includes('http');

            if (hasDownloadableFile) {
              return (
                <li className="text-gray-600 dark:text-gray-400 leading-[1.6] my-0 text-sm py-0 mb-0">
                  {children}
                </li>
              );
            }
            
            return (
              <li className="text-gray-600 dark:text-gray-400 leading-[1.6] my-0 text-sm py-0 mb-0">{children}</li>
            );
          }
          
          return (
            <li className="leading-[1.3] my-0 text-sm">{children}</li>
          );
        },
        pre: ({ children }) => (
          <pre className="bg-gray-100 dark:bg-gray-800 rounded px-3 py-2 my-2 text-sm leading-[1.5] overflow-x-auto">{children}</pre>
        ),
        code: ({ className, children }) => (
          <code className="bg-gray-100 dark:bg-gray-800 rounded px-1.5 py-0.5 text-sm">{children}</code>
        ),
        h1: ({ children }) => {
          // 출처(References) 제목인지 확인
          const childrenText = React.Children.toArray(children).join('').toLowerCase();
          const isReferences = childrenText.includes('출처') || childrenText.includes('references');
          
          if (isReferences) {
            return (
              <h3 className="text-sm font-bold text-gray-900 dark:text-white mt-3 mb-1 flex items-center space-x-1">
                <svg className="w-4 h-4 text-blue-600 dark:text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                </svg>
                <span>출처</span>
              </h3>
            );
          }
          
          return (
            <h3 className="text-base font-bold my-2">{children}</h3>
          );
        },
        h2: ({ children }) => {
          // 출처(References) 제목인지 확인
          const childrenText = React.Children.toArray(children).join('').toLowerCase();
          const isReferences = childrenText.includes('출처') || childrenText.includes('references');
          
          if (isReferences) {
            return (
              <h3 className="text-sm font-bold text-gray-900 dark:text-white mt-3 mb-1 flex items-center space-x-1">
                <svg className="w-4 h-4 text-blue-600 dark:text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                </svg>
                <span>출처</span>
              </h3>
            );
          }
          
          return (
            <h3 className="text-base font-bold my-2">{children}</h3>
          );
        },
        h3: ({ children }) => (
          <h3 className="text-base font-semibold my-2">{children}</h3>
        ),
        blockquote: ({ children }) => (
          <blockquote className="border-l-4 border-gray-300 dark:border-gray-600 pl-3 my-1 italic text-gray-600 dark:text-gray-400 text-sm">{children}</blockquote>
        ),
        ul: ({ children }) => (
          <ul className={`list-disc list-inside space-y-0 ${hasReferencesSection ? 'my-0 leading-none' : 'my-0'}`}>{children}</ul>
        ),
        ol: ({ children }) => (
          <ol className={`list-decimal list-inside space-y-0 ${hasReferencesSection ? 'my-0 leading-none' : 'my-0'}`}>{children}</ol>
        ),
        table: ({ children }) => (
          <div className="overflow-x-auto my-2"><table className="min-w-full border border-gray-300 dark:border-gray-600 text-sm">{children}</table></div>
        ),
        th: ({ children }) => (
          <th className="px-2 py-1 text-left whitespace-wrap text-gray-900 dark:text-white">{children}</th>
        ),
        td: ({ children }) => (
          <td className="px-2 py-1 text-left text-gray-700 dark:text-gray-300">{children}</td>
        ),
        a: ({ href, children }) => {
          if (!href) return <span>{children}</span>;
          
          // rag/download 관련 링크는 일반 텍스트로 처리 (별도 다운로드 섹션이 있으므로)
          if (href.includes('rag/download') || href.includes('/download?filename=') || href.includes('filename=')) {
            // 중첩된 링크에서 텍스트만 추출
            const textContent = React.Children.toArray(children).map(child => {
              if (typeof child === 'string') return child;
              if (React.isValidElement(child) && (child as any).props?.children) {
                return React.Children.toArray((child as any).props.children).join('');
              }
              return '';
            }).join('');
            
            return <span className="text-gray-600 dark:text-gray-400">{textContent}</span>;
          }
          
          // 내부 링크 처리
          if (href.startsWith('#')) {
            return (
              <a
                href={href}
                className="text-blue-600 dark:text-blue-400 hover:underline"
              >
                {children}
              </a>
            );
          }
          
          // 출처 섹션의 링크인지 확인
          const linkText = React.Children.toArray(children).join('').toLowerCase();
          const isReferenceLink = hasReferencesSection && (
            href?.includes('http') || 
            linkText.includes('pdf') ||
            linkText.includes('문서') ||
            linkText.includes('페이지') ||
            href?.includes('github') ||
            href?.includes('docs') ||
            linkText.length > 30 // 긴 제목의 링크는 출처일 가능성이 높음
          );
          
          if (isReferenceLink) {
            return (
              <a
                href={href}
                target="_blank"
                rel="noopener noreferrer"
                className="text-blue-600 dark:text-blue-400 hover:underline text-sm break-words break-all"
                style={{ wordBreak: 'break-all', overflowWrap: 'anywhere' }}
              >
                {children}
              </a>
            );
          }
          
          return (
            <a
              href={href}
              target="_blank"
              rel="noopener noreferrer"
              className="text-blue-600 dark:text-blue-400 hover:underline break-words break-all"
              style={{ wordBreak: 'break-all', overflowWrap: 'anywhere' }}
            >
              {children}
            </a>
          );
        },
        // 이미지 처리 - RAG 모드에 따라 제어
        img: ({ src, alt, title }) => {
          if (!src) return null;

          // src가 문자열인지 확인
          if (typeof src !== 'string') return null;

          // RAG 모드가 intelligence가 아닌 경우 이미지 미리보기 제한
          if (ragMode !== 'intelligence') {
            return (
              <div className="flex items-center justify-center p-4 bg-gray-100 dark:bg-gray-800 rounded-lg border border-gray-300 dark:border-gray-600 my-4">
                <div className="text-center">
                  <svg className="w-8 h-8 mx-auto mb-2 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"></path>
                  </svg>
                  <div className="text-sm text-gray-500 dark:text-gray-400">이미지 미리보기</div>
                  <div className="text-xs text-gray-400 dark:text-gray-500 mt-1">
                    {ragMode ? `${ragMode.toUpperCase()} 모드에서는 지원되지 않습니다` : 'Intelligence 모드에서만 지원됩니다'}
                  </div>
                  {alt && (
                    <div className="text-xs text-gray-400 dark:text-gray-500 mt-1 italic">
                      이미지 설명: {alt}
                    </div>
                  )}
                </div>
              </div>
            );
          }

          // intelligence 모드인 경우 정상적인 이미지 렌더링
          return (
            <img
              src={src}
              alt={alt || '이미지'}
              title={title || alt || '이미지'}
              className="block max-w-full h-auto my-4 rounded-lg border border-gray-200 dark:border-gray-700 shadow-sm hover:shadow-md transition-shadow"
              onError={(e) => {
                console.error('[이미지 로딩 오류]', {
                  src,
                  alt,
                  error: e
                });

                const target = e.target as HTMLImageElement;

                // 한 번 더 재시도
                if (!target.dataset.retried) {
                  target.dataset.retried = 'true';
                  setTimeout(() => {
                    target.src = src + '&retry=' + Date.now();
                  }, 1000);
                  return;
                }

                // 재시도 후에도 실패하면 오류 표시
                target.style.display = 'none';

                const errorDiv = document.createElement('div');
                errorDiv.className = 'flex items-center justify-center p-4 bg-gray-100 dark:bg-gray-800 rounded-lg border border-gray-300 dark:border-gray-600 my-4';
                errorDiv.innerHTML = `
                  <div class="text-center">
                    <svg class="w-8 h-8 mx-auto mb-2 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"></path>
                    </svg>
                    <p class="text-sm text-gray-500 dark:text-gray-400">이미지를 불러올 수 없습니다</p>
                  </div>
                `;

                target.parentNode?.replaceChild(errorDiv, target);
              }}
            />
          );
        },
      }}
    >
      {cleanContent}
    </ReactMarkdown>
    
    {/* 다운로드 섹션 렌더링 */}
    {downloadInfo.length > 0 && (
      <div className="mt-4 pt-3 border-t border-gray-200 dark:border-gray-700">
        <div className="space-y-2">
          {downloadInfo.map((item, index) => {
            const apiServerUrl = getApiServerUrl();
            const downloadUrl = `${apiServerUrl}/rag/download?filename=${encodeURIComponent(item.filename)}`;
            
            return (
              <div key={index} className="flex items-center justify-between p-2 bg-gray-50 dark:bg-gray-800 rounded border border-gray-200 dark:border-gray-600">
                <div className="flex items-center space-x-2">
                  <svg className="w-4 h-4 text-blue-600 dark:text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                  </svg>
                  <div>
                    <div className="text-xs font-medium text-gray-900 dark:text-gray-100">
                      {item.filename}
                    </div>
                    {item.pages && item.pages.length > 0 && (
                      <div className="text-xs text-gray-500 dark:text-gray-400">
                        페이지: {item.pages.join(', ')}
                      </div>
                    )}
                  </div>
                </div>
                <button
                  onClick={async () => {
                    try {
                      console.log('[다운로드] 시작:', item.filename);
                      
                      const response = await fetch(downloadUrl, {
                        method: 'GET',
                        headers: {
                          'X-API-Key': localStorage.getItem('apiKey') || 'airun_1_3d85009c98964579c622eded69997b16',
                          'Authorization': `Bearer ${localStorage.getItem('userToken') || ''}`
                        }
                      });
                      
                      if (!response.ok) {
                        const errorData = await response.json().catch(() => ({ error: { message: '파일을 찾을 수 없습니다.' } }));
                        throw new Error(errorData.error?.message || '파일을 찾을 수 없습니다.');
                      }
                      
                      const blob = await response.blob();
                      const url = window.URL.createObjectURL(blob);
                      const a = document.createElement('a');
                      a.href = url;
                      a.download = item.filename;
                      document.body.appendChild(a);
                      a.click();
                      window.URL.revokeObjectURL(url);
                      document.body.removeChild(a);
                      
                      console.log('[다운로드] 완료:', item.filename);
                    } catch (error) {
                      console.error('다운로드 오류:', error);
                      alert('다운로드에 실패했습니다: ' + (error instanceof Error ? error.message : String(error)));
                    }
                  }}
                  className="inline-flex items-center px-2 py-1 text-xs font-medium text-white bg-blue-600 hover:bg-blue-700 rounded transition-colors duration-200"
                  title={`${item.filename} 다운로드`}
                >
                  <svg className="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                  </svg>
                  다운로드
                </button>
              </div>
            );
          })}
        </div>
      </div>
    )}
  </div>
  );
}; 