'use client';

import { useState, useEffect } from 'react';
import { CheckCircle, Paperclip, X, File, Copy, Check } from 'lucide-react';
import Modal from './Modal';

interface FormField {
  id: string;
  label: string;
  type: 'text' | 'email' | 'number' | 'date' | 'textarea' | 'select' | 'checkbox';
  required: boolean;
  options?: string[];
  group?: string;
  placeholder?: string;
}

interface AIForm {
  id: string;
  title: string;
  description: string;
  form_schema: FormField[];
}

interface FormSubmissionModalProps {
  isOpen: boolean;
  onClose: () => void;
  shareLink: string;
  showShareLink?: boolean; // 공유 링크 섹션 표시 여부 (기본값: true)
}

export default function FormSubmissionModal({ isOpen, onClose, shareLink, showShareLink = true }: FormSubmissionModalProps) {
  const [form, setForm] = useState<AIForm | null>(null);
  const [formData, setFormData] = useState<Record<string, any>>({});
  const [submittedBy, setSubmittedBy] = useState('');
  const [attachments, setAttachments] = useState<File[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [isSubmitted, setIsSubmitted] = useState(false);
  const [errors, setErrors] = useState<Record<string, string>>({});
  const [isCopied, setIsCopied] = useState(false);

  useEffect(() => {
    if (isOpen && shareLink) {
      loadForm();
    }
  }, [isOpen, shareLink]);

  const loadForm = async () => {
    setIsLoading(true);
    try {
      const response = await fetch(`/api/formbuilder/form/${shareLink}`);
      if (response.ok) {
        const data = await response.json();
        setForm(data.data);
      } else {
        alert('폼을 찾을 수 없습니다.');
        onClose();
      }
    } catch (error) {
      console.error('Failed to load form:', error);
      alert('폼을 불러오는 중 오류가 발생했습니다.');
      onClose();
    } finally {
      setIsLoading(false);
    }
  };

  const handleInputChange = (fieldId: string, value: any) => {
    setFormData({ ...formData, [fieldId]: value });
    if (errors[fieldId]) {
      const newErrors = { ...errors };
      delete newErrors[fieldId];
      setErrors(newErrors);
    }
  };

  const validate = () => {
    const newErrors: Record<string, string> = {};

    if (!submittedBy.trim()) {
      newErrors.submittedBy = '이름/이메일을 입력해주세요';
    }

    form?.form_schema.forEach((field) => {
      if (field.required && !formData[field.id]) {
        newErrors[field.id] = `${field.label}은(는) 필수 항목입니다`;
      }
    });

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!validate()) {
      return;
    }

    setIsSubmitting(true);

    try {
      // FormData를 사용하여 파일과 데이터를 함께 전송
      const formDataToSend = new FormData();
      formDataToSend.append('response_data', JSON.stringify(formData));
      formDataToSend.append('submitted_by', submittedBy);

      // 첨부파일 추가
      attachments.forEach((file, index) => {
        formDataToSend.append(`attachment_${index}`, file);
      });

      const response = await fetch(`/api/formbuilder/form/${shareLink}/submit`, {
        method: 'POST',
        body: formDataToSend, // Content-Type은 자동으로 설정됨
      });

      if (response.ok) {
        setIsSubmitted(true);
        // 3초 후 모달 닫기
        setTimeout(() => {
          handleClose();
        }, 3000);
      } else {
        throw new Error('Submission failed');
      }
    } catch (error) {
      console.error('Form submission error:', error);
      alert('제출 중 오류가 발생했습니다. 다시 시도해주세요.');
    } finally {
      setIsSubmitting(false);
    }
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files) {
      const newFiles = Array.from(e.target.files);
      setAttachments([...attachments, ...newFiles]);
    }
  };

  const removeAttachment = (index: number) => {
    setAttachments(attachments.filter((_, i) => i !== index));
  };

  const handleClose = () => {
    setForm(null);
    setFormData({});
    setSubmittedBy('');
    setAttachments([]);
    setIsSubmitted(false);
    setErrors({});
    setIsCopied(false);
    onClose();
  };

  const copyShareLink = async () => {
    const fullUrl = `${window.location.origin}/form/${shareLink}`;
    try {
      await navigator.clipboard.writeText(fullUrl);
      setIsCopied(true);
      setTimeout(() => setIsCopied(false), 2000);
    } catch (err) {
      console.error('Failed to copy:', err);
    }
  };

  const renderField = (field: FormField) => {
    const commonClasses = "w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-800 focus:ring-2 focus:ring-blue-500 focus:border-transparent";
    const errorClass = errors[field.id] ? "border-red-500" : "";

    switch (field.type) {
      case 'textarea':
        return (
          <textarea
            value={formData[field.id] || ''}
            onChange={(e) => handleInputChange(field.id, e.target.value)}
            rows={4}
            className={`${commonClasses} ${errorClass}`}
            placeholder={field.placeholder}
            required={field.required}
          />
        );
      case 'select':
        return (
          <select
            value={formData[field.id] || ''}
            onChange={(e) => handleInputChange(field.id, e.target.value)}
            className={`${commonClasses} ${errorClass}`}
            required={field.required}
          >
            <option value="">선택하세요</option>
            {field.options?.map((option, index) => (
              <option key={index} value={option}>{option}</option>
            ))}
          </select>
        );
      case 'checkbox':
        return (
          <input
            type="checkbox"
            checked={formData[field.id] || false}
            onChange={(e) => handleInputChange(field.id, e.target.checked)}
            className="w-5 h-5 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
          />
        );
      default:
        return (
          <input
            type={field.type}
            value={formData[field.id] || ''}
            onChange={(e) => handleInputChange(field.id, e.target.value)}
            className={`${commonClasses} ${errorClass}`}
            placeholder={field.placeholder}
            required={field.required}
          />
        );
    }
  };

  return (
    <Modal isOpen={isOpen} onClose={handleClose} size="lg">
      {isLoading ? (
        <div className="flex items-center justify-center py-20">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
        </div>
      ) : isSubmitted ? (
        <div className="py-12 text-center">
          <div className="bg-green-100 dark:bg-green-900/20 rounded-full p-6 w-24 h-24 mx-auto mb-6 flex items-center justify-center">
            <CheckCircle className="w-12 h-12 text-green-600" />
          </div>
          <h2 className="text-2xl font-bold mb-2 text-gray-900 dark:text-white">제출 완료!</h2>
          <p className="text-gray-600 dark:text-gray-400 mb-6">폼이 성공적으로 제출되었습니다.</p>
          <button
            onClick={handleClose}
            className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            확인
          </button>
        </div>
      ) : form ? (
        <>
          {/* 공유 링크 섹션 - showShareLink prop이 true일 때만 표시 */}
          {showShareLink && (
            <div className="mb-6 p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-200 dark:border-blue-800">
              <label className="block text-xs font-medium text-gray-600 dark:text-gray-400 mb-2">
                폼 작성 링크
              </label>
              <div className="flex items-center gap-2">
                <input
                  type="text"
                  value={`${typeof window !== 'undefined' ? window.location.origin : ''}/form/${shareLink}`}
                  readOnly
                  className="flex-1 px-3 py-2 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded text-sm text-gray-700 dark:text-gray-300 font-mono"
                />
                <button
                  onClick={copyShareLink}
                  className="px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700 transition-colors flex items-center gap-2 flex-shrink-0"
                >
                  {isCopied ? (
                    <>
                      <Check className="w-4 h-4" />
                      <span>복사됨</span>
                    </>
                  ) : (
                    <>
                      <Copy className="w-4 h-4" />
                      <span>복사</span>
                    </>
                  )}
                </button>
              </div>
              <p className="mt-2 text-xs text-gray-600 dark:text-gray-400">
                이 링크를 공유하여 다른 사람들이 폼을 작성할 수 있습니다.
              </p>
            </div>
          )}

          <h1 className="text-3xl font-bold mb-2 text-gray-900 dark:text-white">{form.title}</h1>
          {form.description && (
            <p className="text-gray-600 dark:text-gray-400 mb-8">{form.description}</p>
          )}

          <form onSubmit={handleSubmit} className="space-y-6">
            {/* 제출자 정보 */}
            <div>
              <label className="block text-sm font-medium mb-2 text-gray-900 dark:text-white">
                이름 또는 이메일 <span className="text-red-500">*</span>
              </label>
              <input
                type="text"
                value={submittedBy}
                onChange={(e) => {
                  setSubmittedBy(e.target.value);
                  if (errors.submittedBy) {
                    const newErrors = { ...errors };
                    delete newErrors.submittedBy;
                    setErrors(newErrors);
                  }
                }}
                placeholder="홍길동 또는 hong@example.com"
                className={`w-full px-4 py-2 border ${errors.submittedBy ? 'border-red-500' : 'border-gray-300 dark:border-gray-600'} rounded-lg bg-white dark:bg-gray-800 focus:ring-2 focus:ring-blue-500 focus:border-transparent`}
                required
              />
              {errors.submittedBy && (
                <p className="mt-1 text-sm text-red-500">{errors.submittedBy}</p>
              )}
            </div>

            <div className="border-t border-gray-200 dark:border-gray-700 my-6"></div>

            {/* 폼 필드들 (그룹별) */}
            {(() => {
              // 필드를 그룹별로 묶기
              const fieldsByGroup: { [key: string]: FormField[] } = {};
              form.form_schema.forEach(field => {
                const groupName = field.group || '';
                if (!fieldsByGroup[groupName]) {
                  fieldsByGroup[groupName] = [];
                }
                fieldsByGroup[groupName].push(field);
              });

              // 그룹별로 렌더링
              return Object.entries(fieldsByGroup).map(([groupName, groupFields]) => (
                <div key={groupName || 'no-group'} className="space-y-4">
                  {groupName && (
                    <div className="font-semibold text-lg text-blue-600 dark:text-blue-400 bg-blue-50 dark:bg-blue-900/20 px-4 py-3 rounded-lg border-l-4 border-blue-600 mt-6">
                      📋 {groupName}
                    </div>
                  )}
                  <div className="space-y-4">
                    {groupFields.map((field) => (
                      <div key={field.id}>
                        <label className="block text-sm font-medium mb-2 text-gray-900 dark:text-white">
                          {field.label}
                          {field.required && <span className="text-red-500 ml-1">*</span>}
                        </label>
                        {renderField(field)}
                        {errors[field.id] && (
                          <p className="mt-1 text-sm text-red-500">{errors[field.id]}</p>
                        )}
                      </div>
                    ))}
                  </div>
                </div>
              ));
            })()}

            {/* 첨부파일 섹션 */}
            <div className="mt-8 pt-6 border-t border-gray-200 dark:border-gray-700">
              <div className="font-semibold text-lg text-gray-700 dark:text-gray-300 mb-4 flex items-center gap-2">
                <Paperclip className="w-5 h-5" />
                첨부파일
              </div>

              {/* 파일 선택 버튼 */}
              <div className="mb-4">
                <label className="inline-flex items-center px-4 py-2 bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 rounded-lg border border-gray-300 dark:border-gray-600 hover:bg-gray-200 dark:hover:bg-gray-600 cursor-pointer transition-colors">
                  <Paperclip className="w-4 h-4 mr-2" />
                  파일 선택
                  <input
                    type="file"
                    multiple
                    onChange={handleFileChange}
                    className="hidden"
                  />
                </label>
                <p className="text-xs text-gray-500 dark:text-gray-400 mt-2">
                  여러 파일을 선택할 수 있습니다
                </p>
              </div>

              {/* 선택된 파일 목록 */}
              {attachments.length > 0 && (
                <div className="space-y-2">
                  {attachments.map((file, index) => (
                    <div
                      key={index}
                      className="flex items-center justify-between p-3 bg-gray-50 dark:bg-gray-900 border border-gray-200 dark:border-gray-700 rounded-lg"
                    >
                      <div className="flex items-center gap-2 flex-1 min-w-0">
                        <File className="w-4 h-4 text-gray-500 flex-shrink-0" />
                        <div className="min-w-0 flex-1">
                          <p className="text-sm font-medium text-gray-700 dark:text-gray-300 truncate">
                            {file.name}
                          </p>
                          <p className="text-xs text-gray-500 dark:text-gray-400">
                            {(file.size / 1024).toFixed(2)} KB
                          </p>
                        </div>
                      </div>
                      <button
                        type="button"
                        onClick={() => removeAttachment(index)}
                        className="p-1 text-red-600 hover:bg-red-50 dark:hover:bg-red-900/20 rounded transition-colors flex-shrink-0"
                      >
                        <X className="w-4 h-4" />
                      </button>
                    </div>
                  ))}
                </div>
              )}
            </div>

            <div className="pt-6 flex gap-3">
              <button
                type="button"
                onClick={handleClose}
                className="flex-1 px-6 py-3 bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-gray-200 rounded-lg hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors font-medium"
              >
                취소
              </button>
              <button
                type="submit"
                disabled={isSubmitting}
                className="flex-1 px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed font-medium"
              >
                {isSubmitting ? '제출 중...' : '제출하기'}
              </button>
            </div>
          </form>
        </>
      ) : (
        <div className="py-12 text-center">
          <h2 className="text-2xl font-bold mb-2 text-gray-900 dark:text-white">폼을 찾을 수 없습니다</h2>
          <p className="text-gray-600 dark:text-gray-400">유효하지 않은 링크입니다.</p>
        </div>
      )}
    </Modal>
  );
}
