'use client';

import React, { useState, useEffect } from 'react';
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Trash2, Plus, Eye, EyeOff } from 'lucide-react';

// 간단한 UI 컴포넌트들
const Label = ({ htmlFor, children, ...props }: { htmlFor?: string; children: React.ReactNode; [key: string]: any }) => (
  <label htmlFor={htmlFor} className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1" {...props}>
    {children}
  </label>
);

const Textarea = ({ className = '', ...props }: { className?: string; [key: string]: any }) => (
  <textarea 
    className={`w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white ${className}`} 
    {...props} 
  />
);

const Select = ({ value, onValueChange, children }: { value: string; onValueChange: (value: string) => void; children: React.ReactNode }) => (
  <select 
    value={value} 
    onChange={(e) => onValueChange(e.target.value)}
    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
  >
    {children}
  </select>
);

const SelectTrigger = ({ children }: { children: React.ReactNode }) => <>{children}</>;
const SelectValue = () => null;
const SelectContent = ({ children }: { children: React.ReactNode }) => <>{children}</>;
const SelectItem = ({ value, children }: { value: string; children: React.ReactNode }) => (
  <option value={value}>{children}</option>
);

const Badge = ({ variant, className = '', children }: { variant?: string; className?: string; children: React.ReactNode }) => {
  const baseClasses = "inline-flex items-center px-2 py-1 rounded-full text-xs font-medium";
  const variantClasses = {
    secondary: "bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-200",
    destructive: "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200",
    outline: "border border-gray-300 text-gray-700 dark:border-gray-600 dark:text-gray-300"
  };
  
  return (
    <span className={`${baseClasses} ${variantClasses[variant as keyof typeof variantClasses] || variantClasses.secondary} ${className}`}>
      {children}
    </span>
  );
};

interface TableColumn {
  name: string;
  type: 'text' | 'number' | 'percentage' | 'date' | 'category' | 'boolean';
  required: boolean;
  options?: string[];
  validation?: {
    min?: number;
    max?: number;
    max_length?: number;
    pattern?: string;
  };
  unit?: string;
  description?: string;
}

interface TableSchema {
  title: string;
  description?: string;
  headers: TableColumn[];
  validation_rules?: {
    min_rows?: number;
    max_rows?: number;
    sum_validation?: {
      column: string;
      expected_sum?: number;
    };
    required_values?: {
      column: string;
      values: string[];
    };
    required_categories?: string[];
    alert_thresholds?: Record<string, number>;
  };
}

interface TableSchemaEditorProps {
  schema?: TableSchema;
  onSchemaChange?: (schema: TableSchema) => void;
  onClose?: () => void;
  isVisible?: boolean;
}

const TableSchemaEditor: React.FC<TableSchemaEditorProps> = ({
  schema,
  onSchemaChange,
  onClose,
  isVisible = true
}) => {
  const [currentSchema, setCurrentSchema] = useState<TableSchema>(
    schema || {
      title: '',
      description: '',
      headers: [],
      validation_rules: {}
    }
  );

  const [showPreview, setShowPreview] = useState(false);

  useEffect(() => {
    if (schema) {
      setCurrentSchema(schema);
    }
  }, [schema]);

  const handleSchemaUpdate = (updatedSchema: TableSchema) => {
    setCurrentSchema(updatedSchema);
    if (onSchemaChange) {
      onSchemaChange(updatedSchema);
    }
  };

  const addColumn = () => {
    const newColumn: TableColumn = {
      name: '',
      type: 'text',
      required: true,
      description: ''
    };
    
    const updatedSchema = {
      ...currentSchema,
      headers: [...currentSchema.headers, newColumn]
    };
    
    handleSchemaUpdate(updatedSchema);
  };

  const removeColumn = (index: number) => {
    const updatedSchema = {
      ...currentSchema,
      headers: currentSchema.headers.filter((_, i) => i !== index)
    };
    
    handleSchemaUpdate(updatedSchema);
  };

  const updateColumn = (index: number, field: keyof TableColumn, value: any) => {
    const updatedHeaders = [...currentSchema.headers];
    
    if (field === 'options' && typeof value === 'string') {
      updatedHeaders[index] = {
        ...updatedHeaders[index],
        [field]: value.split(',').map(s => s.trim()).filter(s => s)
      };
    } else {
      updatedHeaders[index] = {
        ...updatedHeaders[index],
        [field]: value
      };
    }
    
    const updatedSchema = {
      ...currentSchema,
      headers: updatedHeaders
    };
    
    handleSchemaUpdate(updatedSchema);
  };

  const updateValidationRule = (field: string, value: any) => {
    const updatedSchema = {
      ...currentSchema,
      validation_rules: {
        ...currentSchema.validation_rules,
        [field]: value
      }
    };
    
    handleSchemaUpdate(updatedSchema);
  };

  const generateSampleData = () => {
    const sampleRows = [];
    const minRows = Math.max(currentSchema.validation_rules?.min_rows || 2, 2);
    
    for (let i = 0; i < minRows; i++) {
      const row = currentSchema.headers.map((header) => {
        switch (header.type) {
          case 'text':
            return `샘플 ${header.name} ${i + 1}`;
          case 'number':
            return `${Math.floor(Math.random() * 100)}${header.unit ? ' ' + header.unit : ''}`;
          case 'percentage':
            return `${Math.floor(Math.random() * 100)}%`;
          case 'date':
            return new Date().toLocaleDateString();
          case 'category':
            return header.options && header.options.length > 0 
              ? header.options[Math.floor(Math.random() * header.options.length)]
              : '카테고리';
          case 'boolean':
            return Math.random() > 0.5 ? '예' : '아니오';
          default:
            return `값 ${i + 1}`;
        }
      });
      sampleRows.push(row);
    }
    
    return sampleRows;
  };

  if (!isVisible) {
    return null;
  }

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
      <div className="bg-white dark:bg-gray-800 rounded-lg shadow-xl max-w-6xl w-full max-h-[90vh] overflow-y-auto">
        <div className="p-6">
          <div className="flex justify-between items-center mb-6">
            <h2 className="text-2xl font-bold text-gray-900 dark:text-white">
              테이블 스키마 에디터
            </h2>
            <div className="flex space-x-2">
              <Button
                variant="outline"
                size="sm"
                onClick={() => setShowPreview(!showPreview)}
                className="flex items-center space-x-2"
              >
                {showPreview ? <EyeOff className="w-4 h-4" /> : <Eye className="w-4 h-4" />}
                <span>{showPreview ? '미리보기 숨기기' : '미리보기'}</span>
              </Button>
              {onClose && (
                <Button variant="outline" onClick={onClose}>
                  닫기
                </Button>
              )}
            </div>
          </div>

          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {/* 스키마 편집 영역 */}
            <div className="space-y-6">
              {/* 기본 정보 */}
              <Card>
                <CardHeader>
                  <CardTitle>기본 정보</CardTitle>
                  <CardDescription>테이블의 기본 정보를 설정합니다.</CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <Label htmlFor="title">테이블 제목</Label>
                    <Input
                      id="title"
                      value={currentSchema.title}
                      onChange={(e) => handleSchemaUpdate({...currentSchema, title: e.target.value})}
                      placeholder="예: 주간 완료 업무 현황"
                    />
                  </div>
                  <div>
                    <Label htmlFor="description">설명</Label>
                    <Textarea
                      id="description"
                      value={currentSchema.description || ''}
                      onChange={(e) => handleSchemaUpdate({...currentSchema, description: e.target.value})}
                      placeholder="테이블에 대한 상세 설명을 입력하세요"
                      rows={2}
                    />
                  </div>
                </CardContent>
              </Card>

              {/* 컬럼 정의 */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex justify-between items-center">
                    <span>컬럼 정의</span>
                    <Button onClick={addColumn} size="sm">
                      <Plus className="w-4 h-4 mr-2" />
                      컬럼 추가
                    </Button>
                  </CardTitle>
                  <CardDescription>테이블의 각 컬럼을 정의합니다.</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    {currentSchema.headers.map((header, index) => (
                      <div key={index} className="p-4 border rounded-lg space-y-3">
                        <div className="flex justify-between items-center">
                          <span className="font-medium">컬럼 {index + 1}</span>
                          <Button
                            variant="destructive"
                            size="sm"
                            onClick={() => removeColumn(index)}
                          >
                            <Trash2 className="w-4 h-4" />
                          </Button>
                        </div>
                        
                        <div className="grid grid-cols-2 gap-3">
                          <div>
                            <Label>컬럼명</Label>
                            <Input
                              value={header.name}
                              onChange={(e) => updateColumn(index, 'name', e.target.value)}
                              placeholder="예: 업무명"
                            />
                          </div>
                          <div>
                            <Label>데이터 타입</Label>
                                                         <Select
                               value={header.type}
                               onValueChange={(value: string) => updateColumn(index, 'type', value)}
                             >
                              <SelectTrigger>
                                <SelectValue />
                              </SelectTrigger>
                              <SelectContent>
                                <SelectItem value="text">텍스트</SelectItem>
                                <SelectItem value="number">숫자</SelectItem>
                                <SelectItem value="percentage">백분율</SelectItem>
                                <SelectItem value="date">날짜</SelectItem>
                                <SelectItem value="category">카테고리</SelectItem>
                                <SelectItem value="boolean">예/아니오</SelectItem>
                              </SelectContent>
                            </Select>
                          </div>
                        </div>

                        <div>
                          <Label>설명</Label>
                          <Input
                            value={header.description || ''}
                            onChange={(e) => updateColumn(index, 'description', e.target.value)}
                            placeholder="컬럼에 대한 설명"
                          />
                        </div>

                        {header.type === 'category' && (
                          <div>
                            <Label>선택지 (쉼표로 구분)</Label>
                            <Input
                              value={header.options?.join(', ') || ''}
                              onChange={(e) => updateColumn(index, 'options', e.target.value)}
                              placeholder="예: 높음, 보통, 낮음"
                            />
                          </div>
                        )}

                        {(header.type === 'number' || header.type === 'percentage') && (
                          <div>
                            <Label>단위</Label>
                            <Input
                              value={header.unit || ''}
                              onChange={(e) => updateColumn(index, 'unit', e.target.value)}
                              placeholder="예: 시간, 개, %"
                            />
                          </div>
                        )}

                                                 <div className="flex items-center space-x-2">
                           <input
                             type="checkbox"
                             id={`required-${index}`}
                             checked={header.required}
                             onChange={(e: React.ChangeEvent<HTMLInputElement>) => updateColumn(index, 'required', e.target.checked)}
                           />
                           <Label htmlFor={`required-${index}`}>필수 필드</Label>
                         </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>

              {/* 검증 규칙 */}
              <Card>
                <CardHeader>
                  <CardTitle>검증 규칙</CardTitle>
                  <CardDescription>테이블 데이터의 검증 규칙을 설정합니다.</CardDescription>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <Label>최소 행 수</Label>
                      <Input
                        type="number"
                        value={currentSchema.validation_rules?.min_rows || ''}
                        onChange={(e) => updateValidationRule('min_rows', parseInt(e.target.value) || undefined)}
                        placeholder="예: 3"
                      />
                    </div>
                    <div>
                      <Label>최대 행 수</Label>
                      <Input
                        type="number"
                        value={currentSchema.validation_rules?.max_rows || ''}
                        onChange={(e) => updateValidationRule('max_rows', parseInt(e.target.value) || undefined)}
                        placeholder="예: 15"
                      />
                    </div>
                  </div>
                  
                  <div>
                    <Label>필수 카테고리 (쉼표로 구분)</Label>
                    <Input
                      value={currentSchema.validation_rules?.required_categories?.join(', ') || ''}
                      onChange={(e) => updateValidationRule('required_categories', 
                        e.target.value.split(',').map(s => s.trim()).filter(s => s)
                      )}
                      placeholder="예: 높음, 1순위"
                    />
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* 미리보기 영역 */}
            {showPreview && (
              <div className="space-y-6">
                <Card>
                  <CardHeader>
                    <CardTitle>스키마 미리보기</CardTitle>
                    <CardDescription>정의된 스키마의 미리보기입니다.</CardDescription>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-4">
                      <div>
                        <h4 className="font-semibold mb-2">{currentSchema.title || '제목 없음'}</h4>
                        {currentSchema.description && (
                          <p className="text-sm text-gray-600 dark:text-gray-400 mb-4">
                            {currentSchema.description}
                          </p>
                        )}
                      </div>

                      {currentSchema.headers.length > 0 && (
                        <div className="overflow-x-auto">
                          <table className="min-w-full border-collapse border border-gray-300 dark:border-gray-600">
                            <thead>
                              <tr className="bg-gray-100 dark:bg-gray-700">
                                {currentSchema.headers.map((header, index) => (
                                  <th key={index} className="border border-gray-300 dark:border-gray-600 px-4 py-2 text-left">
                                    <div className="flex flex-col space-y-1">
                                      <span className="font-medium">{header.name}</span>
                                      <div className="flex flex-wrap gap-1">
                                        <Badge variant="secondary" className="text-xs">
                                          {header.type}
                                        </Badge>
                                        {header.required && (
                                          <Badge variant="destructive" className="text-xs">
                                            필수
                                          </Badge>
                                        )}
                                        {header.unit && (
                                          <Badge variant="outline" className="text-xs">
                                            {header.unit}
                                          </Badge>
                                        )}
                                      </div>
                                      {header.description && (
                                        <span className="text-xs text-gray-500">
                                          {header.description}
                                        </span>
                                      )}
                                    </div>
                                  </th>
                                ))}
                              </tr>
                            </thead>
                            <tbody>
                              {generateSampleData().map((row, rowIndex) => (
                                <tr key={rowIndex} className="hover:bg-gray-50 dark:hover:bg-gray-800">
                                  {row.map((cell, cellIndex) => (
                                    <td key={cellIndex} className="border border-gray-300 dark:border-gray-600 px-4 py-2 text-sm">
                                      {cell}
                                    </td>
                                  ))}
                                </tr>
                              ))}
                            </tbody>
                          </table>
                        </div>
                      )}

                      {currentSchema.validation_rules && (
                        <div className="mt-4 p-3 bg-gray-50 dark:bg-gray-700 rounded">
                          <h5 className="font-medium mb-2">검증 규칙</h5>
                          <ul className="text-sm text-gray-600 dark:text-gray-400 space-y-1">
                            {currentSchema.validation_rules.min_rows && (
                              <li>• 최소 {currentSchema.validation_rules.min_rows}개 행</li>
                            )}
                            {currentSchema.validation_rules.max_rows && (
                              <li>• 최대 {currentSchema.validation_rules.max_rows}개 행</li>
                            )}
                            {currentSchema.validation_rules.required_categories && 
                             currentSchema.validation_rules.required_categories.length > 0 && (
                              <li>• 필수 카테고리: {currentSchema.validation_rules.required_categories.join(', ')}</li>
                            )}
                          </ul>
                        </div>
                      )}
                    </div>
                  </CardContent>
                </Card>
              </div>
            )}
          </div>

          <div className="mt-6 flex justify-end space-x-3">
            <Button
              variant="outline"
              onClick={() => {
                const jsonSchema = JSON.stringify(currentSchema, null, 2);
                navigator.clipboard.writeText(jsonSchema);
                alert('스키마가 클립보드에 복사되었습니다.');
              }}
            >
              JSON 복사
            </Button>
            <Button onClick={() => onSchemaChange?.(currentSchema)}>
              적용
            </Button>
          </div>
        </div>
      </div>
    </div>
  );
};

export default TableSchemaEditor; 