'use client';

import { ReactNode, useEffect, useState } from 'react';

interface ThemeProviderProps {
  children: ReactNode;
}

export default function ThemeProvider({ children }: ThemeProviderProps) {
  const [isDarkMode, setIsDarkMode] = useState(false);

  useEffect(() => {
    // 테마 상태 초기화
    const savedTheme = localStorage.getItem('theme');
    const prefersDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
    
    if (savedTheme === 'dark' || (!savedTheme && prefersDark)) {
      document.documentElement.classList.add('dark');
      setIsDarkMode(true);
    } else {
      document.documentElement.classList.remove('dark');
      setIsDarkMode(false);
    }

    // 테마 변경 이벤트 리스너
    const handleThemeChange = (e: CustomEvent) => {
      setIsDarkMode(e.detail.theme === 'dark');
      
      // body와 html 요소에 직접 스타일 적용
      const htmlEl = document.documentElement;
      
      if (e.detail.theme === 'dark') {
        htmlEl.classList.add('dark');
        updateRootStyles(true);
      } else {
        htmlEl.classList.remove('dark');
        updateRootStyles(false);
      }
    };
    
    // 초기 스타일 업데이트
    updateRootStyles(isDarkMode);

    window.addEventListener('themeChange', handleThemeChange as EventListener);

    return () => {
      window.removeEventListener('themeChange', handleThemeChange as EventListener);
    };
  }, [isDarkMode]);
  
  // 루트 레벨 스타일 업데이트 함수
  const updateRootStyles = (isDark: boolean) => {
    const bodyEl = document.body;
    const htmlEl = document.documentElement;
    
    if (isDark) {
      bodyEl.style.backgroundColor = 'var(--body-bg)';
      bodyEl.style.color = 'var(--text-primary)';
      htmlEl.style.backgroundColor = 'var(--body-bg)';
    } else {
      bodyEl.style.backgroundColor = 'var(--body-bg)';
      bodyEl.style.color = 'var(--text-primary)';
      htmlEl.style.backgroundColor = 'var(--body-bg)';
    }
  };

  return (
    <div className={isDarkMode ? 'dark' : ''}>
      {children}
    </div>
  );
} 