'use client';

import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Badge } from '@/components/ui/badge';
import { Trash2, Download, Upload, Brain, Zap, HardDrive, Calendar } from 'lucide-react';

interface Model {
  id: string;
  name: string;
  type: 'full' | 'lora';
  baseModel: string;
  size: string;
  accuracy: number;
  createdAt: string;
  status: 'active' | 'inactive' | 'training';
  downloadUrl?: string;
}

export default function LLMModelsTab() {
  const [models, setModels] = useState<Model[]>([]);
  const [loading, setLoading] = useState(false);
  const [showUploadModal, setShowUploadModal] = useState(false);

  useEffect(() => {
    loadModels();
  }, []);

  const loadModels = async () => {
    try {
      setLoading(true);
      const response = await fetch('/api/llm/models');
      const data = await response.json();
      if (data.success) {
        setModels(data.models || []);
      }
    } catch (error) {
      console.error('모델 로드 실패:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleDeleteModel = async (id: string) => {
    if (!confirm('정말로 이 모델을 삭제하시겠습니까?')) return;

    try {
      const response = await fetch(`/api/llm/models/${id}`, {
        method: 'DELETE'
      });
      const data = await response.json();
      if (data.success) {
        setModels(models.filter(m => m.id !== id));
      }
    } catch (error) {
      console.error('모델 삭제 실패:', error);
    }
  };

  const handleDownloadModel = async (model: Model) => {
    if (!model.downloadUrl) return;

    try {
      const response = await fetch(model.downloadUrl);
      const blob = await response.blob();
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `${model.name}.zip`;
      document.body.appendChild(a);
      a.click();
      window.URL.revokeObjectURL(url);
      document.body.removeChild(a);
    } catch (error) {
      console.error('모델 다운로드 실패:', error);
    }
  };

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <h2 className="text-2xl font-bold">LLM 모델 관리</h2>
        <div className="flex gap-2">
          <Button onClick={() => setShowUploadModal(true)} variant="outline" className="flex items-center gap-2">
            <Upload className="h-4 w-4" />
            모델 업로드
          </Button>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium">전체 모델</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{models.length}</div>
            <p className="text-xs text-muted-foreground">생성된 모델 수</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium">활성 모델</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {models.filter(m => m.status === 'active').length}
            </div>
            <p className="text-xs text-muted-foreground">사용 가능한 모델</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium">LoRA 모델</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {models.filter(m => m.type === 'lora').length}
            </div>
            <p className="text-xs text-muted-foreground">효율적 학습 모델</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium">풀 튜닝</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {models.filter(m => m.type === 'full').length}
            </div>
            <p className="text-xs text-muted-foreground">완전 학습 모델</p>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>모델 목록</CardTitle>
        </CardHeader>
        <CardContent>
          {loading ? (
            <div className="text-center py-4">모델을 불러오는 중...</div>
          ) : models.length === 0 ? (
            <div className="text-center py-8 text-gray-500">
              <Brain className="h-12 w-12 mx-auto mb-4 opacity-50" />
              <p>생성된 모델이 없습니다.</p>
              <p className="text-sm">먼저 모델 학습을 시작하세요.</p>
            </div>
          ) : (
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
              {models.map((model) => (
                <Card key={model.id} className="border">
                  <CardHeader className="pb-3">
                    <div className="flex justify-between items-start">
                      <div>
                        <CardTitle className="text-lg">{model.name}</CardTitle>
                        <p className="text-sm text-gray-500">{model.baseModel}</p>
                      </div>
                      <div className="flex gap-1">
                        <Badge variant={model.type === 'lora' ? 'secondary' : 'default'}>
                          {model.type === 'lora' ? (
                            <><Zap className="h-3 w-3 mr-1" />LoRA</>
                          ) : (
                            <><Brain className="h-3 w-3 mr-1" />Full</>
                          )}
                        </Badge>
                        <Badge variant={
                          model.status === 'active' ? 'default' :
                          model.status === 'training' ? 'secondary' : 'outline'
                        }>
                          {model.status === 'active' ? '활성' :
                           model.status === 'training' ? '학습중' : '비활성'}
                        </Badge>
                      </div>
                    </div>
                  </CardHeader>
                  
                  <CardContent className="pt-0">
                    <div className="grid grid-cols-2 gap-4 text-sm mb-4">
                      <div className="flex items-center gap-2">
                        <HardDrive className="h-4 w-4 text-gray-400" />
                        <span>크기: {model.size}</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <Calendar className="h-4 w-4 text-gray-400" />
                        <span>{model.createdAt}</span>
                      </div>
                    </div>

                    <div className="mb-4">
                      <div className="flex justify-between text-sm mb-1">
                        <span>정확도</span>
                        <span>{model.accuracy}%</span>
                      </div>
                      <div className="w-full bg-gray-200 rounded-full h-2">
                        <div
                          className={`h-2 rounded-full ${
                            model.accuracy >= 90 ? 'bg-green-500' :
                            model.accuracy >= 70 ? 'bg-yellow-500' : 'bg-red-500'
                          }`}
                          style={{ width: `${model.accuracy}%` }}
                        />
                      </div>
                    </div>

                    <div className="flex justify-end gap-2">
                      {model.downloadUrl && (
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => handleDownloadModel(model)}
                          className="flex items-center gap-1"
                        >
                          <Download className="h-4 w-4" />
                          다운로드
                        </Button>
                      )}
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => handleDeleteModel(model.id)}
                        className="text-red-600 hover:text-red-800"
                      >
                        <Trash2 className="h-4 w-4" />
                        삭제
                      </Button>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          )}
        </CardContent>
      </Card>

      <Dialog open={showUploadModal} onOpenChange={setShowUploadModal}>
        <DialogContent className="max-w-lg">
          <DialogHeader>
            <DialogTitle>모델 업로드</DialogTitle>
          </DialogHeader>
          <ModelUploadForm 
            onClose={() => setShowUploadModal(false)} 
            onSuccess={() => {
              loadModels();
              setShowUploadModal(false);
            }} 
          />
        </DialogContent>
      </Dialog>
    </div>
  );
}

interface ModelUploadFormProps {
  onClose: () => void;
  onSuccess: () => void;
}

function ModelUploadForm({ onClose, onSuccess }: ModelUploadFormProps) {
  const [formData, setFormData] = useState({
    name: '',
    type: 'lora' as 'full' | 'lora',
    baseModel: '',
    file: null as File | null
  });
  const [loading, setLoading] = useState(false);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!formData.file) return;

    setLoading(true);
    try {
      const uploadFormData = new FormData();
      uploadFormData.append('file', formData.file);
      uploadFormData.append('name', formData.name);
      uploadFormData.append('type', formData.type);
      uploadFormData.append('baseModel', formData.baseModel);

      const response = await fetch('/api/llm/models', {
        method: 'POST',
        body: uploadFormData
      });

      const data = await response.json();
      if (data.success) {
        onSuccess();
      }
    } catch (error) {
      console.error('모델 업로드 실패:', error);
    } finally {
      setLoading(false);
    }
  };

  return (
    <form onSubmit={handleSubmit} className="space-y-4">
      <div>
        <Label htmlFor="name">모델 이름</Label>
        <Input
          id="name"
          value={formData.name}
          onChange={(e) => setFormData(prev => ({ ...prev, name: e.target.value }))}
          placeholder="my-custom-model"
          required
        />
      </div>

      <div>
        <Label htmlFor="baseModel">베이스 모델</Label>
        <Input
          id="baseModel"
          value={formData.baseModel}
          onChange={(e) => setFormData(prev => ({ ...prev, baseModel: e.target.value }))}
          placeholder="microsoft/DialoGPT-small"
          required
        />
      </div>

      <div>
        <Label htmlFor="file">모델 파일</Label>
        <Input
          id="file"
          type="file"
          accept=".zip,.tar,.tar.gz"
          onChange={(e) => setFormData(prev => ({ ...prev, file: e.target.files?.[0] || null }))}
          required
        />
        <p className="text-sm text-gray-500 mt-1">
          ZIP, TAR 또는 TAR.GZ 형식으로 압축된 모델 파일을 업로드하세요.
        </p>
      </div>

      <div className="flex justify-end gap-2 pt-4">
        <Button type="button" variant="outline" onClick={onClose}>취소</Button>
        <Button type="submit" disabled={loading || !formData.file}>
          {loading ? '업로드 중...' : '업로드'}
        </Button>
      </div>
    </form>
  );
}