'use client';

import { useState, useEffect } from 'react';
import apiClient from '@/lib/apiClient';

interface Agent {
  id: string;
  name: string;
  description: string;
  status: 'active' | 'inactive' | 'running' | 'error';
  lastRun?: string;
  config?: any;
}

interface AgentExecution {
  id: string;
  agentId: string;
  status: 'pending' | 'running' | 'completed' | 'failed';
  startTime: string;
  endTime?: string;
  result?: any;
  error?: string;
}

export default function AutomationContent() {
  const [agents, setAgents] = useState<Agent[]>([]);
  const [executions, setExecutions] = useState<AgentExecution[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [selectedAgent, setSelectedAgent] = useState<Agent | null>(null);
  const [isExecuting, setIsExecuting] = useState<string | null>(null);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    loadAgents();
    loadExecutions();
  }, []);

  const loadAgents = async () => {
    try {
      const response = await apiClient.get('/agents');

      if (response.success && response.data) {
        setAgents(response.data);
      } else {
        console.error('에이전트 로드 실패:', response.error);
      }
    } catch (error) {
      console.error('에이전트 로드 중 오류:', error);
    }
  };

  const loadExecutions = async () => {
    try {
      const response = await apiClient.get('/agents/executions');

      if (response.success && response.data) {
        setExecutions(response.data);
      } else {
        console.error('실행 기록 로드 실패:', response.error);
      }
    } catch (error) {
      console.error('실행 기록 로드 중 오류:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const executeAgent = async (agentId: string, config?: any) => {
    try {
      setIsExecuting(agentId);
      setError(null);

      const response = await apiClient.post('/agents/execute', {
        agent_id: agentId,
        config: config || {}
      });

      if (response.success && response.data?.jobId) {
        await pollExecutionStatus(response.data.jobId, agentId);
      } else {
        throw new Error(response.error?.message || '에이전트 실행 요청 실패');
      }
    } catch (err) {
      setError(err instanceof Error ? err.message : '에이전트 실행 중 오류가 발생했습니다.');
    } finally {
      setIsExecuting(null);
    }
  };

  const pollExecutionStatus = async (jobId: string, agentId: string) => {
    const maxAttempts = 60;
    let attempts = 0;

    const checkStatus = async (): Promise<void> => {
      try {
        const response = await apiClient.get(`/agents/status/${jobId}`);

        if (response.success && response.data) {
          const { state, result } = response.data;

          if (state === 'completed') {
            await loadExecutions();
            await loadAgents();
            return;
          } else if (state === 'failed') {
            throw new Error('에이전트 실행 실패');
          } else if (attempts < maxAttempts) {
            attempts++;
            setTimeout(checkStatus, 2000);
          } else {
            throw new Error('실행 시간 초과');
          }
        }
      } catch (error) {
        console.error('상태 확인 실패:', error);
        setError('에이전트 실행 상태 확인 중 오류가 발생했습니다.');
      }
    };

    await checkStatus();
  };

  const stopAgent = async (agentId: string) => {
    try {
      const response = await apiClient.post(`/agents/${agentId}/stop`);

      if (response.success) {
        await loadAgents();
      } else {
        setError(response.error?.message || '에이전트 중지 실패');
      }
    } catch (error) {
      console.error('에이전트 중지 중 오류:', error);
      setError('에이전트 중지 중 오류가 발생했습니다.');
    }
  };

  const getStatusBadge = (status: string) => {
    const baseClasses = 'px-2 py-1 rounded-full text-xs font-medium';
    switch (status) {
      case 'active':
        return `${baseClasses} bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200`;
      case 'running':
        return `${baseClasses} bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200`;
      case 'error':
      case 'failed':
        return `${baseClasses} bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200`;
      case 'completed':
        return `${baseClasses} bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200`;
      case 'pending':
        return `${baseClasses} bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200`;
      default:
        return `${baseClasses} bg-gray-100 text-gray-800 dark:bg-gray-900 dark:text-gray-200`;
    }
  };

  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    return date.toLocaleString('ko-KR');
  };

  if (isLoading) {
    return (
      <div className="flex justify-center items-center h-64">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500"></div>
      </div>
    );
  }

  return (
    <div className="max-w-6xl mx-auto">
      <h1 className="text-2xl font-bold mb-6 text-gray-800 dark:text-gray-200">자동화</h1>

      {error && (
        <div className="mb-6 p-4 bg-red-100 dark:bg-red-900 text-red-700 dark:text-red-200 rounded-lg">
          {error}
          <button onClick={() => setError(null)} className="ml-4 text-sm underline">닫기</button>
        </div>
      )}

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* 에이전트 목록 */}
        <div className="bg-white dark:bg-gray-800 rounded-lg shadow-md p-6">
          <div className="flex justify-between items-center mb-4">
            <h2 className="text-xl font-semibold text-gray-800 dark:text-gray-200">에이전트 목록</h2>
            <button
              onClick={loadAgents}
              className="px-3 py-1 bg-gray-500 text-white rounded hover:bg-gray-600"
            >
              새로고침
            </button>
          </div>

          {agents.length === 0 ? (
            <div className="text-center py-8 text-gray-500">등록된 에이전트가 없습니다.</div>
          ) : (
            <div className="space-y-4">
              {agents.map((agent) => (
                <div
                  key={agent.id}
                  className="border border-gray-200 dark:border-gray-700 rounded-lg p-4 hover:shadow-md transition-shadow"
                >
                  <div className="flex justify-between items-start mb-2">
                    <div>
                      <h3 className="font-medium text-lg text-gray-800 dark:text-gray-200">{agent.name}</h3>
                      <p className="text-sm text-gray-600 dark:text-gray-400">{agent.description}</p>
                    </div>
                    <span className={getStatusBadge(agent.status)}>{agent.status}</span>
                  </div>

                  {agent.lastRun && (
                    <p className="text-xs text-gray-500 mb-3">마지막 실행: {formatDate(agent.lastRun)}</p>
                  )}

                  <div className="flex space-x-2">
                    <button
                      onClick={() => executeAgent(agent.id)}
                      disabled={isExecuting === agent.id || agent.status === 'running'}
                      className="px-3 py-1 bg-blue-500 text-white rounded hover:bg-blue-600 disabled:opacity-50 disabled:cursor-not-allowed"
                    >
                      {isExecuting === agent.id ? '실행 중...' : '실행'}
                    </button>

                    {agent.status === 'running' && (
                      <button
                        onClick={() => stopAgent(agent.id)}
                        className="px-3 py-1 bg-red-500 text-white rounded hover:bg-red-600"
                      >
                        중지
                      </button>
                    )}

                    <button
                      onClick={() => setSelectedAgent(agent)}
                      className="px-3 py-1 bg-gray-500 text-white rounded hover:bg-gray-600"
                    >
                      상세
                    </button>
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>

        {/* 실행 기록 */}
        <div className="bg-white dark:bg-gray-800 rounded-lg shadow-md p-6">
          <div className="flex justify-between items-center mb-4">
            <h2 className="text-xl font-semibold text-gray-800 dark:text-gray-200">실행 기록</h2>
            <button
              onClick={loadExecutions}
              className="px-3 py-1 bg-gray-500 text-white rounded hover:bg-gray-600"
            >
              새로고침
            </button>
          </div>

          {executions.length === 0 ? (
            <div className="text-center py-8 text-gray-500">실행 기록이 없습니다.</div>
          ) : (
            <div className="space-y-4 max-h-96 overflow-y-auto">
              {executions.slice().reverse().map((execution) => {
                const agent = agents.find(a => a.id === execution.agentId);
                return (
                  <div key={execution.id} className="border border-gray-200 dark:border-gray-700 rounded-lg p-3">
                    <div className="flex justify-between items-start mb-2">
                      <div>
                        <h4 className="font-medium text-gray-800 dark:text-gray-200">{agent?.name || execution.agentId}</h4>
                        <p className="text-sm text-gray-600 dark:text-gray-400">시작: {formatDate(execution.startTime)}</p>
                        {execution.endTime && (
                          <p className="text-sm text-gray-600 dark:text-gray-400">종료: {formatDate(execution.endTime)}</p>
                        )}
                      </div>
                      <span className={getStatusBadge(execution.status)}>{execution.status}</span>
                    </div>

                    {execution.error && (
                      <div className="mt-2 p-2 bg-red-50 dark:bg-red-900 rounded text-sm text-red-700 dark:text-red-200">
                        {execution.error}
                      </div>
                    )}

                    {execution.result && (
                      <div className="mt-2 p-2 bg-green-50 dark:bg-green-900 rounded text-sm">
                        <pre className="whitespace-pre-wrap text-gray-800 dark:text-gray-200">
                          {typeof execution.result === 'string' ? execution.result : JSON.stringify(execution.result, null, 2)}
                        </pre>
                      </div>
                    )}
                  </div>
                );
              })}
            </div>
          )}
        </div>
      </div>

      {/* 에이전트 상세 모달 */}
      {selectedAgent && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white dark:bg-gray-800 rounded-lg p-6 max-w-2xl w-full mx-4 max-h-96 overflow-y-auto">
            <div className="flex justify-between items-center mb-4">
              <h3 className="text-xl font-semibold text-gray-800 dark:text-gray-200">{selectedAgent.name}</h3>
              <button onClick={() => setSelectedAgent(null)} className="text-gray-500 hover:text-gray-700">✕</button>
            </div>

            <div className="space-y-4">
              <div>
                <h4 className="font-medium mb-2 text-gray-800 dark:text-gray-200">설명</h4>
                <p className="text-gray-600 dark:text-gray-400">{selectedAgent.description}</p>
              </div>

              <div>
                <h4 className="font-medium mb-2 text-gray-800 dark:text-gray-200">상태</h4>
                <span className={getStatusBadge(selectedAgent.status)}>{selectedAgent.status}</span>
              </div>

              {selectedAgent.lastRun && (
                <div>
                  <h4 className="font-medium mb-2 text-gray-800 dark:text-gray-200">마지막 실행</h4>
                  <p className="text-gray-600 dark:text-gray-400">{formatDate(selectedAgent.lastRun)}</p>
                </div>
              )}

              {selectedAgent.config && (
                <div>
                  <h4 className="font-medium mb-2 text-gray-800 dark:text-gray-200">설정</h4>
                  <pre className="bg-gray-100 dark:bg-gray-900 p-3 rounded text-sm overflow-x-auto text-gray-800 dark:text-gray-200">
                    {JSON.stringify(selectedAgent.config, null, 2)}
                  </pre>
                </div>
              )}

              <div className="flex space-x-2 pt-4">
                <button
                  onClick={() => { executeAgent(selectedAgent.id); setSelectedAgent(null); }}
                  disabled={isExecuting === selectedAgent.id || selectedAgent.status === 'running'}
                  className="px-4 py-2 bg-blue-500 text-white rounded hover:bg-blue-600 disabled:opacity-50"
                >
                  실행
                </button>
                <button onClick={() => setSelectedAgent(null)} className="px-4 py-2 bg-gray-500 text-white rounded hover:bg-gray-600">
                  닫기
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
