'use client';

// FormBuilder 페이지의 핵심 기능을 재사용
// 기존 페이지 로직을 그대로 활용하여 에이전트 페이지에서 렌더링

import { useState, useEffect, useRef } from 'react';
import { Upload, FileText, Image, List, Eye, Plus, Trash2, Edit2, ExternalLink, Copy, Check } from 'lucide-react';
import { useRouter } from 'next/navigation';
import * as pdfjsLib from 'pdfjs-dist';
import ReactMarkdown from 'react-markdown';
import remarkGfm from 'remark-gfm';
import { html as beautifyHtml } from 'js-beautify';
import FormSubmissionModal from '@/components/FormSubmissionModal';
import ResponsesViewModal from '@/components/ResponsesViewModal';

interface FormField {
  id: string;
  label: string;
  type: 'text' | 'email' | 'number' | 'date' | 'textarea' | 'select' | 'checkbox';
  required: boolean;
  options?: string[];
  group?: string;
  placeholder?: string;
}

interface AIForm {
  id: string;
  title: string;
  description: string;
  document_filename: string;
  share_link: string;
  view_link: string;
  response_count: number;
  status: string;
  created_at: string;
}

interface UploadedDocument {
  filename: string;
  preview?: string;
  type: 'pdf' | 'image';
}

type ViewMode = 'upload' | 'document-preview' | 'analyzing' | 'html-preview' | 'form-editor' | 'final';

export default function FormBuilderContent() {
  const router = useRouter();
  const [viewMode, setViewMode] = useState<ViewMode>('upload');
  const [uploadedDocument, setUploadedDocument] = useState<UploadedDocument | null>(null);
  const [uploadedFile, setUploadedFile] = useState<File | null>(null);
  const [pdfPreviewImages, setPdfPreviewImages] = useState<string[]>([]);
  const [analyzedContent, setAnalyzedContent] = useState<string>('');
  const [analyzedHtml, setAnalyzedHtml] = useState<string>('');
  const [analyzedMarkdown, setAnalyzedMarkdown] = useState<string>('');
  const [analyzedStructuredJson, setAnalyzedStructuredJson] = useState<any>(null);
  const [analyzedMetadata, setAnalyzedMetadata] = useState<any>(null);
  const [formFields, setFormFields] = useState<FormField[]>([]);
  const [formTitle, setFormTitle] = useState('');
  const [formDescription, setFormDescription] = useState('');
  const [savedForms, setSavedForms] = useState<AIForm[]>([]);
  const [isLoading, setIsLoading] = useState(false);
  const [analysisProgress, setAnalysisProgress] = useState(0);
  const [user, setUser] = useState<{ id: string; username: string } | null>(null);
  const [isPdfLoading, setIsPdfLoading] = useState(false);

  // 페이지 선택 관련 state
  const [totalPages, setTotalPages] = useState(0);
  const [selectedPages, setSelectedPages] = useState<number[]>([]);
  const [pageSelectionMode, setPageSelectionMode] = useState<'all' | 'range'>('all');
  const [pageRangeInput, setPageRangeInput] = useState('');

  // 모달 상태
  const [isFormSubmissionModalOpen, setIsFormSubmissionModalOpen] = useState(false);
  const [isResponsesViewModalOpen, setIsResponsesViewModalOpen] = useState(false);
  const [selectedFormShareLink, setSelectedFormShareLink] = useState('');
  const [selectedFormViewLink, setSelectedFormViewLink] = useState('');
  const [copiedLinkId, setCopiedLinkId] = useState<string | null>(null);

  // 탭 상태
  const [activePreviewTab, setActivePreviewTab] = useState<1 | 2>(1);
  const [activeContentTab, setActiveContentTab] = useState<'html' | 'markdown' | 'json'>('html');
  const [activeSourceTab, setActiveSourceTab] = useState<'html' | 'markdown' | 'json'>('html');

  // 미리보기 페이지네이션
  const [currentPreviewPage, setCurrentPreviewPage] = useState(1);
  const PAGES_PER_VIEW = 5;

  // PDF.js 워커 설정
  useEffect(() => {
    pdfjsLib.GlobalWorkerOptions.workerSrc = '/pdf-worker/pdf.worker.min.js';
  }, []);

  useEffect(() => {
    const userId = localStorage.getItem('userId');
    const username = localStorage.getItem('username');
    if (userId && username) {
      setUser({ id: userId, username });
      loadSavedForms(username);
    }
  }, []);

  const loadSavedForms = async (username: string) => {
    try {
      const response = await fetch(`/api/formbuilder/forms?username=${username}`);
      const result = await response.json();
      if (result.success) {
        setSavedForms(result.data);
      }
    } catch (error) {
      console.error('폼 목록 로드 실패:', error);
    }
  };

  const handleFileUpload = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;

    setIsLoading(true);
    setUploadedFile(file);

    const isPdf = file.type === 'application/pdf';
    const isImage = file.type.startsWith('image/');

    if (isPdf) {
      setUploadedDocument({
        filename: file.name,
        type: 'pdf'
      });

      try {
        setIsPdfLoading(true);
        const arrayBuffer = await file.arrayBuffer();
        const pdf = await pdfjsLib.getDocument({ data: arrayBuffer }).promise;
        setTotalPages(pdf.numPages);
        setSelectedPages(Array.from({ length: pdf.numPages }, (_, i) => i + 1));

        const images: string[] = [];
        for (let i = 1; i <= Math.min(pdf.numPages, 20); i++) {
          const page = await pdf.getPage(i);
          const scale = 1.5;
          const viewport = page.getViewport({ scale });
          const canvas = document.createElement('canvas');
          const context = canvas.getContext('2d');
          canvas.height = viewport.height;
          canvas.width = viewport.width;

          await page.render({
            canvasContext: context!,
            viewport: viewport
          }).promise;

          images.push(canvas.toDataURL('image/png'));
        }
        setPdfPreviewImages(images);
        setIsPdfLoading(false);
      } catch (error) {
        console.error('PDF 미리보기 로드 실패:', error);
        setIsPdfLoading(false);
      }

      setViewMode('document-preview');
    } else if (isImage) {
      const reader = new FileReader();
      reader.onload = (event) => {
        setUploadedDocument({
          filename: file.name,
          preview: event.target?.result as string,
          type: 'image'
        });
        setViewMode('document-preview');
      };
      reader.readAsDataURL(file);
    }

    setIsLoading(false);
  };

  const parsePageRange = (input: string): number[] => {
    const pages: Set<number> = new Set();
    const parts = input.split(',').map(s => s.trim());

    for (const part of parts) {
      if (part.includes('-')) {
        const [start, end] = part.split('-').map(n => parseInt(n.trim()));
        if (!isNaN(start) && !isNaN(end)) {
          for (let i = Math.max(1, start); i <= Math.min(totalPages, end); i++) {
            pages.add(i);
          }
        }
      } else {
        const pageNum = parseInt(part);
        if (!isNaN(pageNum) && pageNum >= 1 && pageNum <= totalPages) {
          pages.add(pageNum);
        }
      }
    }

    return Array.from(pages).sort((a, b) => a - b);
  };

  const analyzeDocument = async () => {
    if (!uploadedFile) return;

    setViewMode('analyzing');
    setIsLoading(true);
    setAnalysisProgress(0);

    const formData = new FormData();
    formData.append('file', uploadedFile);
    formData.append('username', user?.username || 'anonymous');

    if (uploadedDocument?.type === 'pdf' && selectedPages.length > 0 && selectedPages.length < totalPages) {
      formData.append('pages', JSON.stringify(selectedPages));
    }

    try {
      const progressInterval = setInterval(() => {
        setAnalysisProgress(prev => {
          if (prev >= 90) {
            clearInterval(progressInterval);
            return 90;
          }
          return prev + 10;
        });
      }, 500);

      const response = await fetch('/api/formbuilder/analyze', {
        method: 'POST',
        body: formData,
      });

      clearInterval(progressInterval);
      setAnalysisProgress(100);

      const result = await response.json();

      if (result.success) {
        const contentHtml = result.data.html || result.data.content || '';
        const contentMarkdown = result.data.markdown || '';
        const structuredJson = result.data.structuredJson || null;
        const metadata = result.data.metadata || null;

        setAnalyzedContent(contentHtml);
        setAnalyzedHtml(contentHtml);
        setAnalyzedMarkdown(contentMarkdown);
        setAnalyzedStructuredJson(structuredJson);
        setAnalyzedMetadata(metadata);

        if (result.data.fields && Array.isArray(result.data.fields)) {
          setFormFields(result.data.fields.map((field: any, index: number) => ({
            id: `field-${index}`,
            label: field.label || field.name,
            type: field.type || 'text',
            required: field.required ?? false,
            options: field.options,
            group: field.group,
            placeholder: field.placeholder
          })));
        }

        if (result.data.formTitle) {
          setFormTitle(result.data.formTitle);
        }
        if (result.data.formDescription) {
          setFormDescription(result.data.formDescription);
        }

        setViewMode('html-preview');
      } else {
        console.error('문서 분석 실패:', result.error);
        setViewMode('document-preview');
      }
    } catch (error) {
      console.error('문서 분석 오류:', error);
      setViewMode('document-preview');
    } finally {
      setIsLoading(false);
    }
  };

  const generateFormFields = async () => {
    if (!analyzedContent) return;

    setIsLoading(true);

    try {
      const response = await fetch('/api/formbuilder/generate-fields', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          content: analyzedContent,
          html: analyzedHtml,
          markdown: analyzedMarkdown,
          structuredJson: analyzedStructuredJson,
          metadata: analyzedMetadata,
          username: user?.username || 'anonymous'
        }),
      });

      const result = await response.json();

      if (result.success && result.data.fields) {
        setFormFields(result.data.fields.map((field: any, index: number) => ({
          id: `field-${index}`,
          label: field.label || field.name,
          type: field.type || 'text',
          required: field.required ?? false,
          options: field.options,
          group: field.group,
          placeholder: field.placeholder
        })));

        if (result.data.title) setFormTitle(result.data.title);
        if (result.data.description) setFormDescription(result.data.description);

        setViewMode('form-editor');
      }
    } catch (error) {
      console.error('폼 필드 생성 오류:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const saveForm = async () => {
    if (!formTitle || formFields.length === 0) return;

    setIsLoading(true);

    try {
      const response = await fetch('/api/formbuilder/save', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          title: formTitle,
          description: formDescription,
          fields: formFields,
          documentFilename: uploadedDocument?.filename || '',
          username: user?.username || 'anonymous'
        }),
      });

      const result = await response.json();

      if (result.success) {
        setViewMode('final');
        if (user?.username) {
          loadSavedForms(user.username);
        }
      }
    } catch (error) {
      console.error('폼 저장 오류:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const resetBuilder = () => {
    setViewMode('upload');
    setUploadedDocument(null);
    setUploadedFile(null);
    setPdfPreviewImages([]);
    setAnalyzedContent('');
    setAnalyzedHtml('');
    setAnalyzedMarkdown('');
    setAnalyzedStructuredJson(null);
    setAnalyzedMetadata(null);
    setFormFields([]);
    setFormTitle('');
    setFormDescription('');
    setTotalPages(0);
    setSelectedPages([]);
    setPageSelectionMode('all');
    setPageRangeInput('');
    setCurrentPreviewPage(1);
  };

  const groupFields = () => {
    const groups: { [key: string]: FormField[] } = {};
    formFields.forEach(field => {
      const groupName = field.group || '기타';
      if (!groups[groupName]) groups[groupName] = [];
      groups[groupName].push(field);
    });
    return groups;
  };

  const copyToClipboard = async (text: string, id: string) => {
    try {
      await navigator.clipboard.writeText(text);
      setCopiedLinkId(id);
      setTimeout(() => setCopiedLinkId(null), 2000);
    } catch (err) {
      console.error('클립보드 복사 실패:', err);
    }
  };

  const getFullUrl = (path: string) => {
    if (typeof window !== 'undefined') {
      return `${window.location.origin}${path}`;
    }
    return path;
  };

  // 렌더링
  return (
    <div className="min-h-screen" style={{ backgroundColor: 'var(--body-bg)', color: 'var(--text-primary)' }}>
      <div className="max-w-7xl mx-auto">
        {/* 저장된 폼 목록 */}
        {savedForms.length > 0 && viewMode === 'upload' && (
          <div className="mb-8">
            <h2 className="text-xl font-semibold mb-4">저장된 폼</h2>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              {savedForms.map((form) => (
                <div
                  key={form.id}
                  className="p-4 rounded-lg border border-gray-200 dark:border-gray-700 hover:border-blue-500 transition-colors"
                  style={{ backgroundColor: 'var(--card-bg)' }}
                >
                  <h3 className="font-medium mb-2">{form.title}</h3>
                  <p className="text-sm text-gray-500 dark:text-gray-400 mb-3">
                    {form.description || '설명 없음'}
                  </p>
                  <div className="flex items-center justify-between text-xs text-gray-400">
                    <span>응답: {form.response_count}건</span>
                    <span>{new Date(form.created_at).toLocaleDateString()}</span>
                  </div>
                  <div className="flex gap-2 mt-3">
                    <button
                      onClick={() => {
                        setSelectedFormShareLink(form.share_link);
                        setIsFormSubmissionModalOpen(true);
                      }}
                      className="flex-1 px-3 py-1.5 text-sm bg-blue-500 text-white rounded hover:bg-blue-600"
                    >
                      <Eye className="w-4 h-4 inline mr-1" /> 미리보기
                    </button>
                    <button
                      onClick={() => {
                        setSelectedFormViewLink(form.view_link);
                        setIsResponsesViewModalOpen(true);
                      }}
                      className="flex-1 px-3 py-1.5 text-sm bg-green-500 text-white rounded hover:bg-green-600"
                    >
                      <List className="w-4 h-4 inline mr-1" /> 응답보기
                    </button>
                  </div>
                  <div className="mt-2">
                    <button
                      onClick={() => copyToClipboard(getFullUrl(`/form/${form.share_link}`), form.id)}
                      className="w-full px-3 py-1.5 text-sm border border-gray-300 dark:border-gray-600 rounded hover:bg-gray-100 dark:hover:bg-gray-700 flex items-center justify-center gap-1"
                    >
                      {copiedLinkId === form.id ? (
                        <>
                          <Check className="w-4 h-4 text-green-500" /> 복사됨
                        </>
                      ) : (
                        <>
                          <Copy className="w-4 h-4" /> 링크 복사
                        </>
                      )}
                    </button>
                  </div>
                </div>
              ))}
            </div>
          </div>
        )}

        {/* 업로드 화면 */}
        {viewMode === 'upload' && (
          <div className="text-center py-12">
            <div className="max-w-xl mx-auto">
              <h1 className="text-2xl font-bold mb-4">AI 폼빌더</h1>
              <p className="text-gray-600 dark:text-gray-400 mb-8">
                문서를 업로드하면 AI가 자동으로 양식을 생성합니다.
              </p>
              <label className="block w-full p-8 border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-lg cursor-pointer hover:border-blue-500 transition-colors">
                <Upload className="w-12 h-12 mx-auto mb-4 text-gray-400" />
                <span className="text-lg font-medium">파일을 선택하거나 드래그하세요</span>
                <p className="text-sm text-gray-500 mt-2">PDF, 이미지 파일 지원</p>
                <input
                  type="file"
                  accept=".pdf,image/*"
                  onChange={handleFileUpload}
                  className="hidden"
                />
              </label>
            </div>
          </div>
        )}

        {/* 문서 미리보기 */}
        {viewMode === 'document-preview' && uploadedDocument && (
          <div>
            <div className="flex justify-between items-center mb-4">
              <h2 className="text-xl font-semibold">{uploadedDocument.filename}</h2>
              <div className="flex gap-2">
                <button
                  onClick={resetBuilder}
                  className="px-4 py-2 border border-gray-300 dark:border-gray-600 rounded hover:bg-gray-100 dark:hover:bg-gray-700"
                >
                  다시 선택
                </button>
                <button
                  onClick={analyzeDocument}
                  disabled={isLoading}
                  className="px-4 py-2 bg-blue-500 text-white rounded hover:bg-blue-600 disabled:opacity-50"
                >
                  {isLoading ? '분석 중...' : '문서 분석'}
                </button>
              </div>
            </div>

            {uploadedDocument.type === 'pdf' && pdfPreviewImages.length > 0 && (
              <div>
                {/* 페이지 선택 UI */}
                <div className="mb-4 p-4 rounded-lg border border-gray-200 dark:border-gray-700" style={{ backgroundColor: 'var(--card-bg)' }}>
                  <div className="flex items-center gap-4 mb-3">
                    <span className="font-medium">페이지 선택:</span>
                    <label className="flex items-center gap-2">
                      <input
                        type="radio"
                        checked={pageSelectionMode === 'all'}
                        onChange={() => {
                          setPageSelectionMode('all');
                          setSelectedPages(Array.from({ length: totalPages }, (_, i) => i + 1));
                        }}
                      />
                      전체 ({totalPages}페이지)
                    </label>
                    <label className="flex items-center gap-2">
                      <input
                        type="radio"
                        checked={pageSelectionMode === 'range'}
                        onChange={() => setPageSelectionMode('range')}
                      />
                      범위 지정
                    </label>
                  </div>
                  {pageSelectionMode === 'range' && (
                    <div className="flex items-center gap-2">
                      <input
                        type="text"
                        value={pageRangeInput}
                        onChange={(e) => {
                          setPageRangeInput(e.target.value);
                          setSelectedPages(parsePageRange(e.target.value));
                        }}
                        placeholder="예: 1-3, 5, 7-10"
                        className="px-3 py-2 border border-gray-300 dark:border-gray-600 rounded w-48"
                        style={{ backgroundColor: 'var(--input-bg)', color: 'var(--text-primary)' }}
                      />
                      <span className="text-sm text-gray-500">
                        선택: {selectedPages.length}페이지
                      </span>
                    </div>
                  )}
                </div>

                {/* PDF 미리보기 */}
                <div className="grid grid-cols-5 gap-4">
                  {pdfPreviewImages.slice((currentPreviewPage - 1) * PAGES_PER_VIEW, currentPreviewPage * PAGES_PER_VIEW).map((img, index) => {
                    const actualPage = (currentPreviewPage - 1) * PAGES_PER_VIEW + index + 1;
                    return (
                      <div
                        key={actualPage}
                        className={`relative cursor-pointer rounded overflow-hidden border-2 transition-all ${
                          selectedPages.includes(actualPage)
                            ? 'border-blue-500 ring-2 ring-blue-200'
                            : 'border-gray-200 dark:border-gray-700'
                        }`}
                        onClick={() => {
                          if (pageSelectionMode === 'range') {
                            setSelectedPages(prev =>
                              prev.includes(actualPage)
                                ? prev.filter(p => p !== actualPage)
                                : [...prev, actualPage].sort((a, b) => a - b)
                            );
                          }
                        }}
                      >
                        <img src={img} alt={`Page ${actualPage}`} className="w-full" />
                        <div className="absolute bottom-0 left-0 right-0 bg-black/50 text-white text-center text-sm py-1">
                          {actualPage}
                        </div>
                      </div>
                    );
                  })}
                </div>

                {/* 페이지네이션 */}
                {Math.ceil(pdfPreviewImages.length / PAGES_PER_VIEW) > 1 && (
                  <div className="flex justify-center gap-2 mt-4">
                    <button
                      onClick={() => setCurrentPreviewPage(prev => Math.max(1, prev - 1))}
                      disabled={currentPreviewPage === 1}
                      className="px-3 py-1 border rounded disabled:opacity-50"
                    >
                      이전
                    </button>
                    <span className="px-3 py-1">
                      {currentPreviewPage} / {Math.ceil(pdfPreviewImages.length / PAGES_PER_VIEW)}
                    </span>
                    <button
                      onClick={() => setCurrentPreviewPage(prev => Math.min(Math.ceil(pdfPreviewImages.length / PAGES_PER_VIEW), prev + 1))}
                      disabled={currentPreviewPage === Math.ceil(pdfPreviewImages.length / PAGES_PER_VIEW)}
                      className="px-3 py-1 border rounded disabled:opacity-50"
                    >
                      다음
                    </button>
                  </div>
                )}
              </div>
            )}

            {uploadedDocument.type === 'image' && uploadedDocument.preview && (
              <div className="flex justify-center">
                <img src={uploadedDocument.preview} alt="Preview" className="max-w-full max-h-[600px] rounded" />
              </div>
            )}
          </div>
        )}

        {/* 분석 중 */}
        {viewMode === 'analyzing' && (
          <div className="text-center py-16">
            <div className="animate-spin rounded-full h-16 w-16 border-b-2 border-blue-600 mx-auto mb-4"></div>
            <h2 className="text-xl font-semibold mb-2">문서 분석 중...</h2>
            <div className="w-64 mx-auto bg-gray-200 dark:bg-gray-700 rounded-full h-2 mt-4">
              <div
                className="bg-blue-600 h-2 rounded-full transition-all duration-500"
                style={{ width: `${analysisProgress}%` }}
              ></div>
            </div>
            <p className="text-gray-500 mt-2">{analysisProgress}%</p>
          </div>
        )}

        {/* HTML 미리보기 */}
        {viewMode === 'html-preview' && (
          <div>
            <div className="flex justify-between items-center mb-4">
              <h2 className="text-xl font-semibold">분석 결과</h2>
              <div className="flex gap-2">
                <button
                  onClick={resetBuilder}
                  className="px-4 py-2 border border-gray-300 dark:border-gray-600 rounded hover:bg-gray-100 dark:hover:bg-gray-700"
                >
                  다시 시작
                </button>
                <button
                  onClick={generateFormFields}
                  disabled={isLoading}
                  className="px-4 py-2 bg-blue-500 text-white rounded hover:bg-blue-600 disabled:opacity-50"
                >
                  {isLoading ? '생성 중...' : '폼 생성'}
                </button>
              </div>
            </div>

            {/* 탭 */}
            <div className="flex gap-2 mb-4">
              <button
                onClick={() => setActiveContentTab('html')}
                className={`px-4 py-2 rounded ${activeContentTab === 'html' ? 'bg-blue-500 text-white' : 'bg-gray-200 dark:bg-gray-700'}`}
              >
                HTML
              </button>
              <button
                onClick={() => setActiveContentTab('markdown')}
                className={`px-4 py-2 rounded ${activeContentTab === 'markdown' ? 'bg-blue-500 text-white' : 'bg-gray-200 dark:bg-gray-700'}`}
              >
                Markdown
              </button>
              <button
                onClick={() => setActiveContentTab('json')}
                className={`px-4 py-2 rounded ${activeContentTab === 'json' ? 'bg-blue-500 text-white' : 'bg-gray-200 dark:bg-gray-700'}`}
              >
                JSON
              </button>
            </div>

            <div className="p-4 rounded-lg border border-gray-200 dark:border-gray-700 max-h-[500px] overflow-auto" style={{ backgroundColor: 'var(--card-bg)' }}>
              {activeContentTab === 'html' && (
                <div dangerouslySetInnerHTML={{ __html: analyzedHtml }} />
              )}
              {activeContentTab === 'markdown' && (
                <ReactMarkdown remarkPlugins={[remarkGfm]} className="prose dark:prose-invert max-w-none">
                  {analyzedMarkdown || '마크다운 내용이 없습니다.'}
                </ReactMarkdown>
              )}
              {activeContentTab === 'json' && (
                <pre className="text-sm overflow-x-auto">
                  {analyzedStructuredJson ? JSON.stringify(analyzedStructuredJson, null, 2) : 'JSON 데이터가 없습니다.'}
                </pre>
              )}
            </div>
          </div>
        )}

        {/* 폼 에디터 */}
        {viewMode === 'form-editor' && (
          <div>
            <div className="flex justify-between items-center mb-4">
              <h2 className="text-xl font-semibold">폼 편집</h2>
              <div className="flex gap-2">
                <button
                  onClick={resetBuilder}
                  className="px-4 py-2 border border-gray-300 dark:border-gray-600 rounded hover:bg-gray-100 dark:hover:bg-gray-700"
                >
                  다시 시작
                </button>
                <button
                  onClick={saveForm}
                  disabled={isLoading || !formTitle}
                  className="px-4 py-2 bg-green-500 text-white rounded hover:bg-green-600 disabled:opacity-50"
                >
                  {isLoading ? '저장 중...' : '폼 저장'}
                </button>
              </div>
            </div>

            <div className="grid grid-cols-2 gap-6">
              {/* 폼 설정 */}
              <div className="space-y-4">
                <div>
                  <label className="block text-sm font-medium mb-1">폼 제목 *</label>
                  <input
                    type="text"
                    value={formTitle}
                    onChange={(e) => setFormTitle(e.target.value)}
                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded"
                    style={{ backgroundColor: 'var(--input-bg)', color: 'var(--text-primary)' }}
                    placeholder="폼 제목을 입력하세요"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium mb-1">폼 설명</label>
                  <textarea
                    value={formDescription}
                    onChange={(e) => setFormDescription(e.target.value)}
                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded h-24"
                    style={{ backgroundColor: 'var(--input-bg)', color: 'var(--text-primary)' }}
                    placeholder="폼에 대한 설명을 입력하세요"
                  />
                </div>

                {/* 필드 목록 */}
                <div>
                  <h3 className="font-medium mb-2">필드 목록 ({formFields.length}개)</h3>
                  <div className="space-y-2 max-h-[400px] overflow-auto">
                    {Object.entries(groupFields()).map(([groupName, fields]) => (
                      <div key={groupName} className="mb-4">
                        <h4 className="text-sm font-medium text-gray-600 dark:text-gray-400 mb-2">{groupName}</h4>
                        {fields.map((field) => (
                          <div
                            key={field.id}
                            className="flex items-center justify-between p-3 rounded border border-gray-200 dark:border-gray-700 mb-2"
                            style={{ backgroundColor: 'var(--card-bg)' }}
                          >
                            <div>
                              <span className="font-medium">{field.label}</span>
                              <span className="text-xs text-gray-500 ml-2">({field.type})</span>
                              {field.required && <span className="text-red-500 ml-1">*</span>}
                            </div>
                            <button
                              onClick={() => setFormFields(prev => prev.filter(f => f.id !== field.id))}
                              className="p-1 text-red-500 hover:bg-red-50 dark:hover:bg-red-900/20 rounded"
                            >
                              <Trash2 className="w-4 h-4" />
                            </button>
                          </div>
                        ))}
                      </div>
                    ))}
                  </div>
                </div>
              </div>

              {/* 미리보기 */}
              <div className="p-4 rounded-lg border border-gray-200 dark:border-gray-700" style={{ backgroundColor: 'var(--card-bg)' }}>
                <h3 className="font-medium mb-4">미리보기</h3>
                <div className="space-y-4">
                  <h4 className="text-lg font-semibold">{formTitle || '제목 없음'}</h4>
                  {formDescription && <p className="text-gray-600 dark:text-gray-400">{formDescription}</p>}
                  {formFields.slice(0, 5).map((field) => (
                    <div key={field.id}>
                      <label className="block text-sm font-medium mb-1">
                        {field.label}
                        {field.required && <span className="text-red-500 ml-1">*</span>}
                      </label>
                      {field.type === 'textarea' ? (
                        <textarea
                          className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded h-20"
                          style={{ backgroundColor: 'var(--input-bg)' }}
                          placeholder={field.placeholder}
                          disabled
                        />
                      ) : field.type === 'select' ? (
                        <select
                          className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded"
                          style={{ backgroundColor: 'var(--input-bg)' }}
                          disabled
                        >
                          <option>선택하세요</option>
                          {field.options?.map((opt, i) => (
                            <option key={i}>{opt}</option>
                          ))}
                        </select>
                      ) : field.type === 'checkbox' ? (
                        <input type="checkbox" disabled />
                      ) : (
                        <input
                          type={field.type}
                          className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded"
                          style={{ backgroundColor: 'var(--input-bg)' }}
                          placeholder={field.placeholder}
                          disabled
                        />
                      )}
                    </div>
                  ))}
                  {formFields.length > 5 && (
                    <p className="text-gray-500 text-center">... 외 {formFields.length - 5}개 필드</p>
                  )}
                </div>
              </div>
            </div>
          </div>
        )}

        {/* 완료 화면 */}
        {viewMode === 'final' && (
          <div className="text-center py-16">
            <div className="w-16 h-16 bg-green-100 dark:bg-green-900/30 rounded-full flex items-center justify-center mx-auto mb-4">
              <Check className="w-8 h-8 text-green-600" />
            </div>
            <h2 className="text-2xl font-semibold mb-2">폼이 생성되었습니다!</h2>
            <p className="text-gray-600 dark:text-gray-400 mb-6">이제 링크를 공유하여 응답을 수집할 수 있습니다.</p>
            <button
              onClick={resetBuilder}
              className="px-6 py-3 bg-blue-500 text-white rounded-lg hover:bg-blue-600"
            >
              새 폼 만들기
            </button>
          </div>
        )}
      </div>

      {/* 모달 */}
      {isFormSubmissionModalOpen && (
        <FormSubmissionModal
          shareLink={selectedFormShareLink}
          onClose={() => setIsFormSubmissionModalOpen(false)}
        />
      )}
      {isResponsesViewModalOpen && (
        <ResponsesViewModal
          viewLink={selectedFormViewLink}
          onClose={() => setIsResponsesViewModalOpen(false)}
        />
      )}
    </div>
  );
}
