'use client';

import React, { useState, useEffect } from 'react';
import { formatLocalDate } from '@/lib/dateUtils';
import {
  Calendar,
  Search,
  FileText,
  Sparkles,
  ClipboardList,
  AlertCircle,
  CheckCircle2,
  Loader2,
  ChevronDown,
  ChevronUp,
  User,
  Building,
  Download,
  RefreshCw,
  File,
  Clock,
  Edit3,
  MessageSquare,
  Send
} from 'lucide-react';

// 문서 정보 타입
interface DocumentInfo {
  fileName: string;
  source: string;
  modificationDate: string;
  chunkCount: number;
  content: string;
  embeddingStatus: string;
  selected: boolean;
}

// 생성된 보고서 타입
interface GeneratedReport {
  summary: string;
  completedTasks: Array<{
    taskName: string;
    result: string;
    timeSpent?: string;
  }>;
  ongoingTasks: Array<{
    taskName: string;
    progress: string;
    nextPlan: string;
  }>;
  issues: Array<{
    category: string;
    content: string;
    solution: string;
    status: string;
  }>;
  nextPeriodPlans: Array<{
    taskName: string;
    priority: string;
    targetResult: string;
  }>;
}

// 보고서 데이터 타입
interface ReportData {
  reportType: 'weekly' | 'monthly';
  reportPeriodStart: string;
  reportPeriodEnd: string;
  department: string;
  author: string;
  summary: string;
  completedTasks: Array<{
    id: string;
    taskName: string;
    priority: string;
    progress: number;
    timeSpent?: number;
    result?: string;
  }>;
  ongoingTasks: Array<{
    id: string;
    taskName: string;
    priority: string;
    progress: number;
    dueDate?: string;
    nextWeekPlan?: string;
  }>;
  issues: Array<{
    id: string;
    category: string;
    content: string;
    impact: string;
    solution: string;
    status: string;
  }>;
  nextPeriodPlans: Array<{
    id: string;
    taskName: string;
    priority: string;
    estimatedTime?: number;
    targetResult?: string;
  }>;
  aiSummary?: string;
}

// 유틸리티 함수
const generateId = () => Math.random().toString(36).substr(2, 9);

const getDefaultDateRange = (type: 'weekly' | 'monthly'): { start: string; end: string } => {
  const now = new Date();
  if (type === 'weekly') {
    const dayOfWeek = now.getDay();
    const mondayOffset = dayOfWeek === 0 ? -6 : 1 - dayOfWeek;
    const monday = new Date(now);
    monday.setDate(now.getDate() + mondayOffset);
    const friday = new Date(monday);
    friday.setDate(monday.getDate() + 4);
    return {
      start: monday.toISOString().split('T')[0],
      end: friday.toISOString().split('T')[0]
    };
  } else {
    const firstDay = new Date(now.getFullYear(), now.getMonth(), 1);
    const lastDay = new Date(now.getFullYear(), now.getMonth() + 1, 0);
    return {
      start: firstDay.toISOString().split('T')[0],
      end: lastDay.toISOString().split('T')[0]
    };
  }
};

const formatDate = (dateStr: string): string => {
  if (!dateStr) return '-';
  try {
    // Use dateUtils for proper timezone conversion
    return formatLocalDate(dateStr);
  } catch (error) {
    // Fallback to simple formatting if dateUtils fails
    const date = new Date(dateStr);
    if (isNaN(date.getTime())) return dateStr;
    return `${date.getFullYear()}.${String(date.getMonth() + 1).padStart(2, '0')}.${String(date.getDate()).padStart(2, '0')}`;
  }
};

// 워크플로우 단계
type WorkflowStep = 'search' | 'generate' | 'edit' | 'download';

// 메인 컴포넌트
export default function WeeklyReportContent() {
  // 상태 관리
  const [currentStep, setCurrentStep] = useState<WorkflowStep>('search');
  const [reportType, setReportType] = useState<'weekly' | 'monthly'>('weekly');
  const [dateRange, setDateRange] = useState(getDefaultDateRange('weekly'));
  const [department, setDepartment] = useState('');
  const [author, setAuthor] = useState('');

  // 문서 검색 관련 상태
  const [documents, setDocuments] = useState<DocumentInfo[]>([]);
  const [isSearching, setIsSearching] = useState(false);
  const [searchError, setSearchError] = useState<string | null>(null);

  // 보고서 생성 관련 상태
  const [generatedReport, setGeneratedReport] = useState<GeneratedReport | null>(null);
  const [isGenerating, setIsGenerating] = useState(false);
  const [generateError, setGenerateError] = useState<string | null>(null);

  // 최종 보고서 데이터
  const [reportData, setReportData] = useState<ReportData | null>(null);

  // 다운로드 관련 상태
  const [isDownloading, setIsDownloading] = useState(false);

  // AI 수정 요청 관련 상태
  const [modificationPrompt, setModificationPrompt] = useState('');
  const [isModifying, setIsModifying] = useState(false);

  // 알림 상태
  const [success, setSuccess] = useState<string | null>(null);

  // 보고서 유형 변경 시 날짜 범위 업데이트
  useEffect(() => {
    setDateRange(getDefaultDateRange(reportType));
  }, [reportType]);

  // Debug: Log reportData changes
  useEffect(() => {
    console.log('[WeeklyReport] reportData changed:', reportData);
    console.log('[WeeklyReport] reportData?.aiSummary:', reportData?.aiSummary);
    console.log('[WeeklyReport] currentStep:', currentStep);
  }, [reportData, currentStep]);

  // 문서 검색
  const searchDocuments = async () => {
    setIsSearching(true);
    setSearchError(null);
    setDocuments([]);

    try {
      const apiKey = localStorage.getItem('apiKey');
      const token = localStorage.getItem('userToken');
      // RAG 문서는 username으로 저장되므로 username 사용
      const userId = localStorage.getItem('username') || 'admin';

      const response = await fetch('/api/weekly-report/search-docs', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          ...(apiKey && { 'X-API-Key': apiKey }),
          ...(token && { 'Authorization': `Bearer ${token}` })
        },
        body: JSON.stringify({
          startDate: dateRange.start,
          endDate: dateRange.end,
          userId
        })
      });

      const data = await response.json();

      if (!response.ok || !data.success) {
        throw new Error(data.error?.message || '문서 검색 실패');
      }

      const docs = (data.data.documents || []).map((doc: Omit<DocumentInfo, 'selected'>) => ({
        ...doc,
        selected: true
      }));

      setDocuments(docs);

      if (docs.length === 0) {
        setSearchError('해당 기간에 업로드된 문서가 없습니다.');
      }
    } catch (err) {
      setSearchError(err instanceof Error ? err.message : '문서 검색 중 오류가 발생했습니다.');
    } finally {
      setIsSearching(false);
    }
  };

  // 문서 선택 토글
  const toggleDocumentSelection = (index: number) => {
    setDocuments(prev => prev.map((doc, i) =>
      i === index ? { ...doc, selected: !doc.selected } : doc
    ));
  };

  // 전체 선택/해제
  const toggleAllDocuments = () => {
    const allSelected = documents.every(doc => doc.selected);
    setDocuments(prev => prev.map(doc => ({ ...doc, selected: !allSelected })));
  };

  // AI 보고서 생성
  const generateReport = async () => {
    const selectedDocs = documents.filter(doc => doc.selected);
    if (selectedDocs.length === 0) {
      setGenerateError('최소 1개 이상의 문서를 선택해주세요.');
      return;
    }

    setIsGenerating(true);
    setGenerateError(null);

    try {
      const apiKey = localStorage.getItem('apiKey');
      const token = localStorage.getItem('userToken');

      const response = await fetch('/api/weekly-report/generate', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          ...(apiKey && { 'X-API-Key': apiKey }),
          ...(token && { 'Authorization': `Bearer ${token}` })
        },
        body: JSON.stringify({
          reportType,
          startDate: dateRange.start,
          endDate: dateRange.end,
          documents: selectedDocs.map(doc => ({
            fileName: doc.fileName,
            content: doc.content,
            modificationDate: doc.modificationDate
          })),
          department,
          author
        })
      });

      const data = await response.json();

      if (!response.ok || !data.success) {
        throw new Error(data.error?.message || 'AI 보고서 생성 실패');
      }

      console.log('[WeeklyReport] API response:', JSON.stringify(data, null, 2));
      console.log('[WeeklyReport] Report object:', data.data?.report);
      console.log('[WeeklyReport] Summary:', data.data?.report?.summary);

      const report = data.data?.report;

      // 보고서 내용 검증 - 요약이 비어있으면 에러
      if (!report || !report.summary || report.summary.trim().length === 0) {
        console.error('[WeeklyReport] Empty summary received');
        throw new Error('AI가 보고서 내용을 생성하지 못했습니다. 다시 시도해 주세요.');
      }

      setGeneratedReport(report);
      convertToReportData(report);
      setCurrentStep('edit');
      setSuccess('AI 보고서가 생성되었습니다. 내용을 확인하고 수정할 수 있습니다.');
      setTimeout(() => setSuccess(null), 3000);
    } catch (err) {
      setGenerateError(err instanceof Error ? err.message : '보고서 생성 중 오류가 발생했습니다.');
    } finally {
      setIsGenerating(false);
    }
  };

  // GeneratedReport를 ReportData로 변환
  const convertToReportData = (report: GeneratedReport) => {
    console.log('[WeeklyReport] convertToReportData called with:', report);
    console.log('[WeeklyReport] report.summary:', report.summary);

    const newReportData = {
      reportType,
      reportPeriodStart: dateRange.start,
      reportPeriodEnd: dateRange.end,
      department,
      author,
      summary: report.summary,
      completedTasks: report.completedTasks.map(task => ({
        id: generateId(),
        taskName: task.taskName,
        priority: '보통',
        progress: 100,
        timeSpent: task.timeSpent ? parseInt(task.timeSpent) || 0 : 0,
        result: task.result
      })),
      ongoingTasks: report.ongoingTasks.map(task => ({
        id: generateId(),
        taskName: task.taskName,
        priority: '보통',
        progress: parseInt(task.progress) || 50,
        nextWeekPlan: task.nextPlan
      })),
      issues: report.issues.map(issue => ({
        id: generateId(),
        category: issue.category,
        content: issue.content,
        impact: '보통',
        solution: issue.solution,
        status: issue.status
      })),
      nextPeriodPlans: report.nextPeriodPlans.map(plan => ({
        id: generateId(),
        taskName: plan.taskName,
        priority: plan.priority,
        targetResult: plan.targetResult
      })),
      aiSummary: report.summary
    };

    console.log('[WeeklyReport] Setting reportData with aiSummary:', newReportData.aiSummary);
    setReportData(newReportData);
  };

  // AI 수정 요청
  const modifyReportWithAI = async () => {
    if (!reportData || !modificationPrompt.trim()) {
      setGenerateError('수정 요청 내용을 입력해주세요.');
      return;
    }

    setIsModifying(true);
    setGenerateError(null);

    try {
      const apiKey = localStorage.getItem('apiKey');
      const token = localStorage.getItem('userToken');

      // 현재 보고서 내용을 텍스트로 변환
      const currentReportText = `
현재 보고서 내용:

## AI 요약
${reportData.aiSummary || reportData.summary}

## 완료 업무
${reportData.completedTasks.map(t => `- ${t.taskName}: ${t.result || ''}`).join('\n')}

## 진행중 업무
${reportData.ongoingTasks.map(t => `- ${t.taskName} (${t.progress}%): ${t.nextWeekPlan || ''}`).join('\n')}

## 이슈
${reportData.issues.map(i => `- [${i.category}] ${i.content}: ${i.solution} (${i.status})`).join('\n')}

## 차주 계획
${reportData.nextPeriodPlans.map(p => `- ${p.taskName} (${p.priority}): ${p.targetResult || ''}`).join('\n')}
`;

      const periodLabel = reportType === 'weekly' ? '주간' : '월간';
      const nextPeriodLabel = reportType === 'weekly' ? '차주' : '차월';

      const prompt = `다음은 현재 작성된 ${periodLabel} 업무보고서입니다.

${currentReportText}

사용자의 수정 요청:
"${modificationPrompt}"

위 수정 요청에 따라 보고서를 수정해주세요.
반드시 아래 JSON 형식만 출력하고, 다른 설명이나 텍스트는 포함하지 마세요.

**중요: 행정·공공 R&D 보고서 문체 유지**
- 모든 문장은 "~이다.", "~함.", "~됨.", "~있음." 등의 종결 어미로 작성할 것
- "~습니다", "~ㅂ니다" 등의 경어체는 사용하지 말 것
- 간결하고 단정적인 표현을 사용할 것

{
  "summary": "수정된 전체 업무 요약 (행정·공공 R&D 보고서 문체)",
  "completedTasks": [
    {
      "taskName": "완료된 업무 제목",
      "result": "업무 성과 및 결과",
      "timeSpent": "소요 시간"
    }
  ],
  "ongoingTasks": [
    {
      "taskName": "진행중인 업무 제목",
      "progress": "진행률",
      "nextPlan": "${nextPeriodLabel} 계획"
    }
  ],
  "issues": [
    {
      "category": "이슈 분류",
      "content": "이슈 내용",
      "solution": "해결 방안",
      "status": "상태"
    }
  ],
  "nextPeriodPlans": [
    {
      "taskName": "${nextPeriodLabel} 계획 업무",
      "priority": "높음/보통/낮음",
      "targetResult": "목표 성과"
    }
  ]
}`;

      const response = await fetch('/api/weekly-report/generate', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          ...(apiKey && { 'X-API-Key': apiKey }),
          ...(token && { 'Authorization': `Bearer ${token}` })
        },
        body: JSON.stringify({
          reportType,
          startDate: dateRange.start,
          endDate: dateRange.end,
          documents: [{
            fileName: '현재_보고서_수정요청',
            content: prompt,
            modificationDate: new Date().toISOString()
          }],
          department,
          author
        })
      });

      const data = await response.json();

      if (!response.ok || !data.success) {
        throw new Error(data.error?.message || 'AI 수정 실패');
      }

      const modifiedReport = data.data?.report;

      if (!modifiedReport || !modifiedReport.summary) {
        throw new Error('AI가 수정된 보고서를 생성하지 못했습니다.');
      }

      // 수정된 보고서로 업데이트
      setGeneratedReport(modifiedReport);
      convertToReportData(modifiedReport);
      setModificationPrompt('');
      setSuccess('보고서가 수정되었습니다.');
      setTimeout(() => setSuccess(null), 3000);
    } catch (err) {
      setGenerateError(err instanceof Error ? err.message : '보고서 수정 중 오류가 발생했습니다.');
    } finally {
      setIsModifying(false);
    }
  };

  // 보고서 다운로드
  const downloadReport = async (format: 'pdf' | 'docx' | 'hwp') => {
    if (!reportData) return;

    setIsDownloading(true);

    try {
      const apiKey = localStorage.getItem('apiKey');
      const token = localStorage.getItem('userToken');

      const response = await fetch('/api/weekly-report/download', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          ...(apiKey && { 'X-API-Key': apiKey }),
          ...(token && { 'Authorization': `Bearer ${token}` })
        },
        body: JSON.stringify({ ...reportData, format })
      });

      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error?.message || '다운로드 실패');
      }

      const blob = await response.blob();
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `${reportType === 'weekly' ? '주간' : '월간'}업무보고서_${dateRange.start}_${dateRange.end}.${format}`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      window.URL.revokeObjectURL(url);

      setSuccess('보고서가 다운로드되었습니다.');
      setTimeout(() => setSuccess(null), 3000);
    } catch (err) {
      setGenerateError(err instanceof Error ? err.message : '다운로드 중 오류가 발생했습니다.');
    } finally {
      setIsDownloading(false);
    }
  };

  // 처음부터 다시 시작
  const resetWorkflow = () => {
    setCurrentStep('search');
    setDocuments([]);
    setGeneratedReport(null);
    setReportData(null);
    setSearchError(null);
    setGenerateError(null);
  };

  return (
    <div className="max-w-6xl mx-auto space-y-6">
      {/* 헤더 */}
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-3">
          <ClipboardList className="w-8 h-8 text-purple-600 dark:text-purple-400" />
          <div>
            <h1 className="text-2xl font-bold text-gray-800 dark:text-gray-100">
              주간/월간 업무보고서
            </h1>
            <p className="text-sm text-gray-600 dark:text-gray-400">
              RAG 문서 기반 AI 자동 보고서 생성
            </p>
          </div>
        </div>

        {currentStep !== 'search' && (
          <button
            onClick={resetWorkflow}
            className="px-4 py-2 text-gray-600 dark:text-gray-400 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-lg flex items-center gap-2 transition-colors"
          >
            <RefreshCw className="w-4 h-4" />
            처음부터 다시
          </button>
        )}
      </div>

      {/* 진행 단계 표시 */}
      <div className="flex items-center justify-center gap-4 py-4">
        {(['search', 'generate', 'edit', 'download'] as WorkflowStep[]).map((step, index) => (
          <React.Fragment key={step}>
            <div className={`flex items-center gap-2 ${currentStep === step ? 'text-purple-600 dark:text-purple-400' : 'text-gray-400'}`}>
              <div className={`w-8 h-8 rounded-full flex items-center justify-center text-sm font-medium ${
                currentStep === step
                  ? 'bg-purple-600 text-white'
                  : index < ['search', 'generate', 'edit', 'download'].indexOf(currentStep)
                    ? 'bg-green-500 text-white'
                    : 'bg-gray-200 dark:bg-gray-700'
              }`}>
                {index < ['search', 'generate', 'edit', 'download'].indexOf(currentStep) ? '✓' : index + 1}
              </div>
              <span className="text-sm font-medium hidden sm:inline">
                {step === 'search' && '문서 검색'}
                {step === 'generate' && '보고서 생성'}
                {step === 'edit' && '내용 편집'}
                {step === 'download' && '다운로드'}
              </span>
            </div>
            {index < 3 && <div className="w-8 h-0.5 bg-gray-200 dark:bg-gray-700" />}
          </React.Fragment>
        ))}
      </div>

      {/* 알림 메시지 */}
      {(searchError || generateError) && (
        <div className="p-4 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg flex items-center gap-3">
          <AlertCircle className="w-5 h-5 text-red-600 dark:text-red-400 flex-shrink-0" />
          <span className="text-red-700 dark:text-red-300">{searchError || generateError}</span>
        </div>
      )}

      {success && (
        <div className="p-4 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg flex items-center gap-3">
          <CheckCircle2 className="w-5 h-5 text-green-600 dark:text-green-400 flex-shrink-0" />
          <span className="text-green-700 dark:text-green-300">{success}</span>
        </div>
      )}

      {/* Step 1: 문서 검색 */}
      {currentStep === 'search' && (
        <div className="space-y-6">
          {/* 기본 정보 입력 */}
          <div className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-6 space-y-4">
            <h2 className="text-lg font-semibold text-gray-800 dark:text-gray-200 flex items-center gap-2">
              <Calendar className="w-5 h-5 text-purple-600" />
              보고서 기간 설정
            </h2>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                  보고서 유형
                </label>
                <select
                  value={reportType}
                  onChange={(e) => setReportType(e.target.value as 'weekly' | 'monthly')}
                  className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                >
                  <option value="weekly">주간보고서</option>
                  <option value="monthly">월간보고서</option>
                </select>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                  시작 날짜
                </label>
                <input
                  type="date"
                  value={dateRange.start}
                  onChange={(e) => setDateRange(prev => ({ ...prev, start: e.target.value }))}
                  className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                  종료 날짜
                </label>
                <input
                  type="date"
                  value={dateRange.end}
                  onChange={(e) => setDateRange(prev => ({ ...prev, end: e.target.value }))}
                  className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                />
              </div>

              <div className="flex items-end">
                <button
                  onClick={searchDocuments}
                  disabled={isSearching}
                  className="w-full px-4 py-2 bg-purple-600 hover:bg-purple-700 disabled:bg-purple-400 text-white rounded-lg transition-colors flex items-center justify-center gap-2"
                >
                  {isSearching ? (
                    <>
                      <Loader2 className="w-4 h-4 animate-spin" />
                      검색 중...
                    </>
                  ) : (
                    <>
                      <Search className="w-4 h-4" />
                      문서 검색
                    </>
                  )}
                </button>
              </div>
            </div>

  
  
          </div>

          {/* 검색된 문서 목록 */}
          {documents.length > 0 && (
            <div className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 overflow-hidden">
              <div className="p-4 border-b border-gray-200 dark:border-gray-700 flex items-center justify-between">
                <div className="flex items-center gap-2">
                  <FileText className="w-5 h-5 text-purple-600" />
                  <span className="font-medium text-gray-800 dark:text-gray-200">
                    검색된 문서 ({documents.filter(d => d.selected).length}/{documents.length}개 선택됨)
                  </span>
                </div>
                <button
                  onClick={toggleAllDocuments}
                  className="text-sm text-purple-600 hover:text-purple-700 dark:text-purple-400"
                >
                  {documents.every(d => d.selected) ? '전체 해제' : '전체 선택'}
                </button>
              </div>

              <div className="divide-y divide-gray-200 dark:divide-gray-700 max-h-96 overflow-y-auto">
                {documents.map((doc, index) => (
                  <div
                    key={index}
                    className={`p-4 hover:bg-gray-50 dark:hover:bg-gray-700/50 transition-colors ${
                      doc.selected ? '' : 'opacity-50'
                    }`}
                  >
                    <div className="flex items-start gap-3">
                      <input
                        type="checkbox"
                        checked={doc.selected}
                        onChange={() => toggleDocumentSelection(index)}
                        className="mt-1 w-4 h-4 text-purple-600 rounded focus:ring-purple-500"
                      />
                      <div className="flex-1 min-w-0">
                        <div className="flex items-center gap-2">
                          <File className="w-4 h-4 text-gray-400 flex-shrink-0" />
                          <span className="font-medium text-gray-800 dark:text-gray-200 truncate">
                            {doc.fileName}
                          </span>
                        </div>
                        <div className="flex items-center gap-4 mt-1 text-sm text-gray-500 dark:text-gray-400">
                          <span className="flex items-center gap-1">
                            <Clock className="w-3 h-3" />
                            {formatDate(doc.modificationDate)}
                          </span>
                          {doc.chunkCount > 0 && (
                            <span>{doc.chunkCount} 청크</span>
                          )}
                        </div>
                        {doc.content && (
                          <p className="mt-2 text-sm text-gray-600 dark:text-gray-400 line-clamp-2">
                            {doc.content.substring(0, 200)}...
                          </p>
                        )}
                      </div>
                    </div>
                  </div>
                ))}
              </div>

              {/* 부서 및 작성자 정보 입력 */}
              <div className="p-4 bg-gray-50 dark:bg-gray-700/30 border-t border-gray-200 dark:border-gray-700">
                <h3 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-3">보고서 정보</h3>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                      부서 (선택)
                    </label>
                    <div className="relative">
                      <Building className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-gray-400" />
                      <input
                        type="text"
                        value={department}
                        onChange={(e) => setDepartment(e.target.value)}
                        placeholder="예: 개발팀"
                        className="w-full pl-10 pr-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                      />
                    </div>
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                      작성자 (선택)
                    </label>
                    <div className="relative">
                      <User className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-gray-400" />
                      <input
                        type="text"
                        value={author}
                        onChange={(e) => setAuthor(e.target.value)}
                        placeholder="이름을 입력하세요"
                        className="w-full pl-10 pr-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                      />
                    </div>
                  </div>
                </div>
              </div>

              <div className="p-4 border-t border-gray-200 dark:border-gray-700 flex justify-end">
                <button
                  onClick={generateReport}
                  disabled={isGenerating || documents.filter(d => d.selected).length === 0}
                  className="px-6 py-2 bg-purple-600 hover:bg-purple-700 disabled:bg-purple-400 text-white rounded-lg transition-colors flex items-center gap-2"
                >
                  {isGenerating ? (
                    <>
                      <Loader2 className="w-4 h-4 animate-spin" />
                      보고서 생성 중...
                    </>
                  ) : (
                    <>
                      <Sparkles className="w-4 h-4" />
                      AI 보고서 생성
                    </>
                  )}
                </button>
              </div>
            </div>
          )}
        </div>
      )}

      {/* Step 2-3: 보고서 편집 */}
      {(currentStep === 'edit' || currentStep === 'download') && reportData && (
        <div className="space-y-6">
          {/* AI 요약 */}
          <div className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-6">
            <div className="flex items-center gap-2 mb-4">
              <Sparkles className="w-5 h-5 text-purple-600" />
              <h2 className="text-lg font-semibold text-gray-800 dark:text-gray-200">AI 요약</h2>
            </div>
            <textarea
              value={reportData.aiSummary || reportData.summary}
              onChange={(e) => setReportData(prev => prev ? { ...prev, summary: e.target.value, aiSummary: e.target.value } : null)}
              rows={4}
              className="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-purple-500 focus:border-transparent resize-none"
            />
          </div>

          {/* AI 수정 요청 */}
          <div className="bg-gradient-to-r from-purple-50 to-indigo-50 dark:from-purple-900/20 dark:to-indigo-900/20 rounded-lg border border-purple-200 dark:border-purple-700 p-6">
            <div className="flex items-center gap-2 mb-4">
              <MessageSquare className="w-5 h-5 text-purple-600" />
              <h2 className="text-lg font-semibold text-gray-800 dark:text-gray-200">AI에게 수정 요청</h2>
            </div>
            <p className="text-sm text-gray-600 dark:text-gray-400 mb-3">
              보고서 내용을 수정하고 싶으시면 아래에 요청사항을 입력하세요. AI가 보고서를 수정합니다.
            </p>
            <div className="flex gap-2">
              <textarea
                value={modificationPrompt}
                onChange={(e) => setModificationPrompt(e.target.value)}
                placeholder="예: 요약을 더 간결하게 만들어주세요 / 진행중 업무에 마케팅 분석 항목을 추가해주세요 / 이슈 해결방안을 더 구체적으로 작성해주세요"
                rows={2}
                disabled={isModifying}
                className="flex-1 px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-purple-500 focus:border-transparent resize-none disabled:opacity-50"
              />
              <button
                onClick={modifyReportWithAI}
                disabled={isModifying || !modificationPrompt.trim()}
                className="px-4 py-3 bg-purple-600 hover:bg-purple-700 disabled:bg-purple-400 text-white rounded-lg transition-colors flex items-center gap-2 self-end"
              >
                {isModifying ? (
                  <>
                    <Loader2 className="w-4 h-4 animate-spin" />
                    수정 중...
                  </>
                ) : (
                  <>
                    <Send className="w-4 h-4" />
                    수정 요청
                  </>
                )}
              </button>
            </div>
          </div>

          {/* 완료 업무 */}
          <ReportSection
            title="완료 업무"
            icon={<CheckCircle2 className="w-5 h-5 text-green-600" />}
            items={reportData.completedTasks}
            renderItem={(task, index) => (
              <div key={task.id} className="p-3 bg-gray-50 dark:bg-gray-700/50 rounded-lg">
                <input
                  type="text"
                  value={task.taskName}
                  onChange={(e) => {
                    setReportData(prev => {
                      if (!prev) return null;
                      const tasks = [...prev.completedTasks];
                      tasks[index] = { ...tasks[index], taskName: e.target.value };
                      return { ...prev, completedTasks: tasks };
                    });
                  }}
                  className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-purple-500 focus:border-transparent mb-2"
                  placeholder="업무명"
                />
                <input
                  type="text"
                  value={task.result || ''}
                  onChange={(e) => {
                    setReportData(prev => {
                      if (!prev) return null;
                      const tasks = [...prev.completedTasks];
                      tasks[index] = { ...tasks[index], result: e.target.value };
                      return { ...prev, completedTasks: tasks };
                    });
                  }}
                  className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                  placeholder="성과/결과"
                />
              </div>
            )}
          />

          {/* 진행중 업무 */}
          <ReportSection
            title="진행중 업무"
            icon={<Clock className="w-5 h-5 text-blue-600" />}
            items={reportData.ongoingTasks}
            renderItem={(task, index) => (
              <div key={task.id} className="p-3 bg-gray-50 dark:bg-gray-700/50 rounded-lg">
                <input
                  type="text"
                  value={task.taskName}
                  onChange={(e) => {
                    setReportData(prev => {
                      if (!prev) return null;
                      const tasks = [...prev.ongoingTasks];
                      tasks[index] = { ...tasks[index], taskName: e.target.value };
                      return { ...prev, ongoingTasks: tasks };
                    });
                  }}
                  className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-purple-500 focus:border-transparent mb-2"
                  placeholder="업무명"
                />
                <div className="grid grid-cols-2 gap-2">
                  <input
                    type="number"
                    value={task.progress}
                    onChange={(e) => {
                      setReportData(prev => {
                        if (!prev) return null;
                        const tasks = [...prev.ongoingTasks];
                        tasks[index] = { ...tasks[index], progress: parseInt(e.target.value) || 0 };
                        return { ...prev, ongoingTasks: tasks };
                      });
                    }}
                    className="px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                    placeholder="진행률 %"
                    min="0"
                    max="99"
                  />
                  <input
                    type="text"
                    value={task.nextWeekPlan || ''}
                    onChange={(e) => {
                      setReportData(prev => {
                        if (!prev) return null;
                        const tasks = [...prev.ongoingTasks];
                        tasks[index] = { ...tasks[index], nextWeekPlan: e.target.value };
                        return { ...prev, ongoingTasks: tasks };
                      });
                    }}
                    className="px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                    placeholder="차주 계획"
                  />
                </div>
              </div>
            )}
          />

          {/* 이슈 */}
          <ReportSection
            title="이슈 및 건의사항"
            icon={<AlertCircle className="w-5 h-5 text-red-600" />}
            items={reportData.issues}
            renderItem={(issue, index) => (
              <div key={issue.id} className="p-3 bg-gray-50 dark:bg-gray-700/50 rounded-lg space-y-2">
                <div className="grid grid-cols-2 gap-2">
                  <input
                    type="text"
                    value={issue.category}
                    onChange={(e) => {
                      setReportData(prev => {
                        if (!prev) return null;
                        const issues = [...prev.issues];
                        issues[index] = { ...issues[index], category: e.target.value };
                        return { ...prev, issues };
                      });
                    }}
                    className="px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                    placeholder="분류"
                  />
                  <input
                    type="text"
                    value={issue.status}
                    onChange={(e) => {
                      setReportData(prev => {
                        if (!prev) return null;
                        const issues = [...prev.issues];
                        issues[index] = { ...issues[index], status: e.target.value };
                        return { ...prev, issues };
                      });
                    }}
                    className="px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                    placeholder="상태"
                  />
                </div>
                <input
                  type="text"
                  value={issue.content}
                  onChange={(e) => {
                    setReportData(prev => {
                      if (!prev) return null;
                      const issues = [...prev.issues];
                      issues[index] = { ...issues[index], content: e.target.value };
                      return { ...prev, issues };
                    });
                  }}
                  className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                  placeholder="이슈 내용"
                />
                <input
                  type="text"
                  value={issue.solution}
                  onChange={(e) => {
                    setReportData(prev => {
                      if (!prev) return null;
                      const issues = [...prev.issues];
                      issues[index] = { ...issues[index], solution: e.target.value };
                      return { ...prev, issues };
                    });
                  }}
                  className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                  placeholder="대응 방안"
                />
              </div>
            )}
          />

          {/* 차주/차월 계획 */}
          <ReportSection
            title={reportType === 'weekly' ? '차주 업무 계획' : '차월 업무 계획'}
            icon={<Calendar className="w-5 h-5 text-purple-600" />}
            items={reportData.nextPeriodPlans}
            renderItem={(plan, index) => (
              <div key={plan.id} className="p-3 bg-gray-50 dark:bg-gray-700/50 rounded-lg">
                <input
                  type="text"
                  value={plan.taskName}
                  onChange={(e) => {
                    setReportData(prev => {
                      if (!prev) return null;
                      const plans = [...prev.nextPeriodPlans];
                      plans[index] = { ...plans[index], taskName: e.target.value };
                      return { ...prev, nextPeriodPlans: plans };
                    });
                  }}
                  className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-purple-500 focus:border-transparent mb-2"
                  placeholder="계획 업무"
                />
                <div className="grid grid-cols-2 gap-2">
                  <select
                    value={plan.priority}
                    onChange={(e) => {
                      setReportData(prev => {
                        if (!prev) return null;
                        const plans = [...prev.nextPeriodPlans];
                        plans[index] = { ...plans[index], priority: e.target.value };
                        return { ...prev, nextPeriodPlans: plans };
                      });
                    }}
                    className="px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                  >
                    <option value="높음">높음</option>
                    <option value="보통">보통</option>
                    <option value="낮음">낮음</option>
                  </select>
                  <input
                    type="text"
                    value={plan.targetResult || ''}
                    onChange={(e) => {
                      setReportData(prev => {
                        if (!prev) return null;
                        const plans = [...prev.nextPeriodPlans];
                        plans[index] = { ...plans[index], targetResult: e.target.value };
                        return { ...prev, nextPeriodPlans: plans };
                      });
                    }}
                    className="px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                    placeholder="목표 성과"
                  />
                </div>
              </div>
            )}
          />

          {/* 다운로드 버튼 */}
          <div className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-6">
            <div className="flex flex-wrap items-center justify-center gap-4">
              <button
                onClick={() => downloadReport('pdf')}
                disabled={isDownloading}
                className="px-6 py-3 bg-red-600 hover:bg-red-700 disabled:bg-red-400 text-white rounded-lg transition-colors flex items-center gap-2"
              >
                {isDownloading ? (
                  <Loader2 className="w-5 h-5 animate-spin" />
                ) : (
                  <Download className="w-5 h-5" />
                )}
                PDF 다운로드
              </button>

              <button
                onClick={() => downloadReport('docx')}
                disabled={isDownloading}
                className="px-6 py-3 bg-blue-600 hover:bg-blue-700 disabled:bg-blue-400 text-white rounded-lg transition-colors flex items-center gap-2"
              >
                {isDownloading ? (
                  <Loader2 className="w-5 h-5 animate-spin" />
                ) : (
                  <Download className="w-5 h-5" />
                )}
                DOCX 다운로드
              </button>

              <button
                onClick={() => downloadReport('hwp')}
                disabled={isDownloading}
                className="px-6 py-3 bg-green-600 hover:bg-green-700 disabled:bg-green-400 text-white rounded-lg transition-colors flex items-center gap-2"
              >
                {isDownloading ? (
                  <Loader2 className="w-5 h-5 animate-spin" />
                ) : (
                  <Download className="w-5 h-5" />
                )}
                HWP 다운로드
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}

// 보고서 섹션 컴포넌트
function ReportSection<T>({
  title,
  icon,
  items,
  renderItem
}: {
  title: string;
  icon: React.ReactNode;
  items: T[];
  renderItem: (item: T, index: number) => React.ReactNode;
}) {
  const [isExpanded, setIsExpanded] = useState(true);

  if (items.length === 0) return null;

  return (
    <div className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 overflow-hidden">
      <button
        onClick={() => setIsExpanded(!isExpanded)}
        className="w-full flex items-center justify-between p-4 hover:bg-gray-50 dark:hover:bg-gray-700/50 transition-colors"
      >
        <div className="flex items-center gap-2">
          {icon}
          <span className="font-medium text-gray-800 dark:text-gray-200">{title}</span>
          <span className="px-2 py-0.5 text-xs bg-purple-100 dark:bg-purple-900 text-purple-700 dark:text-purple-300 rounded-full">
            {items.length}개
          </span>
        </div>
        {isExpanded ? (
          <ChevronUp className="w-5 h-5 text-gray-400" />
        ) : (
          <ChevronDown className="w-5 h-5 text-gray-400" />
        )}
      </button>

      {isExpanded && (
        <div className="p-4 space-y-3 border-t border-gray-200 dark:border-gray-700">
          {items.map((item, index) => renderItem(item, index))}
        </div>
      )}
    </div>
  );
}
