'use client';

import { useState, useEffect } from 'react';
import type { WorkflowData, ExecutionResult } from '@/types/flowai';

interface SidebarProps {
  isCollapsed: boolean;
  workflows: WorkflowData[];
  recentWorkflows: WorkflowData[];
  executionHistory: ExecutionResult[];
  onWorkflowSelect: (workflowId: string) => void;
  onNewWorkflow: () => void;
  onShowTemplates: () => void;
  onShowWorkflowList: () => void;
}

const templates = [
  {
    id: 'text-generation',
    name: '텍스트 생성',
    description: '텍스트 입력 → AI 처리 → 텍스트 출력',
    icon: '📝',
    nodes: ['input', 'process', 'output'],
  },
  {
    id: 'data-analysis',
    name: '데이터 분석',
    description: '데이터 입력 → 분석 → 결과 출력',
    icon: '📊',
    nodes: ['input', 'process', 'condition', 'output'],
  },
  {
    id: 'batch-processing',
    name: '일괄 처리',
    description: '데이터 입력 → 반복 처리 → 결과 출력',
    icon: '🔄',
    nodes: ['input', 'loop', 'process', 'output'],
  },
  {
    id: 'decision-flow',
    name: '의사결정 플로우',
    description: '입력 → 조건 분석 → 분기 처리 → 출력',
    icon: '🤔',
    nodes: ['input', 'process', 'condition', 'output'],
  },
];

export default function Sidebar({
  isCollapsed,
  workflows,
  recentWorkflows,
  executionHistory,
  onWorkflowSelect,
  onNewWorkflow,
  onShowTemplates,
  onShowWorkflowList,
}: SidebarProps) {
  const [deleteModalOpen, setDeleteModalOpen] = useState(false);
  const [workflowToDelete, setWorkflowToDelete] = useState<{id: string, name: string} | null>(null);
  
  // 워크플로우 삭제 함수
  const deleteWorkflow = async (workflowId: string, workflowName: string, e: React.MouseEvent) => {
    e.stopPropagation();
    setWorkflowToDelete({ id: workflowId, name: workflowName });
    setDeleteModalOpen(true);
  };

  const confirmDelete = async () => {
    if (!workflowToDelete) return;

    try {
      const response = await fetch(`/api/flowai/workflows/${workflowToDelete.id}`, {
        method: 'DELETE',
        credentials: 'include',
      });

      if (!response.ok) {
        throw new Error('워크플로우 삭제에 실패했습니다');
      }

      // 성공 시 부모 컴포넌트에서 목록을 새로고침하도록 알림
      setDeleteModalOpen(false);
      setWorkflowToDelete(null);
      window.location.reload();
    } catch (error) {
      console.error('워크플로우 삭제 오류:', error);
      // 에러도 모달로 처리하거나 toast로 처리할 수 있음
      alert(error instanceof Error ? error.message : '워크플로우 삭제 중 오류가 발생했습니다');
      setDeleteModalOpen(false);
      setWorkflowToDelete(null);
    }
  };

  // 워크플로우 복제 함수
  const duplicateWorkflow = async (workflow: WorkflowData, e: React.MouseEvent) => {
    e.stopPropagation();
    
    try {
      const duplicatedWorkflow: WorkflowData = {
        ...workflow,
        id: '',
        name: `${workflow.name} (복사본)`,
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString(),
      };

      const response = await fetch('/api/flowai/workflows', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        credentials: 'include',
        body: JSON.stringify(duplicatedWorkflow),
      });

      if (!response.ok) {
        throw new Error('워크플로우 복제에 실패했습니다');
      }

      // 성공 시 부모 컴포넌트에서 목록을 새로고침하도록 알림
      window.location.reload();
    } catch (error) {
      console.error('워크플로우 복제 오류:', error);
      alert(error instanceof Error ? error.message : '워크플로우 복제 중 오류가 발생했습니다');
    }
  };
  const [activeTab, setActiveTab] = useState<'workflows' | 'templates' | 'history'>('workflows');
  const [searchQuery, setSearchQuery] = useState('');

  const filteredWorkflows = workflows.filter(workflow =>
    workflow.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
    workflow.description?.toLowerCase().includes(searchQuery.toLowerCase())
  );

  const applyTemplate = (template: typeof templates[0]) => {
    const event = new CustomEvent('applyWorkflowTemplate', {
      detail: {
        name: template.name,
        description: template.description,
        nodes: template.nodes.map((type, index) => ({
          type,
          position: { x: 200 + index * 200, y: 200 },
          config: {},
        })),
      },
    });
    window.dispatchEvent(event);
  };

  if (isCollapsed) {
    return (
      <div className="w-16 border-r flex flex-col items-center py-4 space-y-4" style={{ backgroundColor: 'var(--sidebar-bg)', borderColor: 'var(--sidebar-border)' }}>
        {/* 축소된 탭 아이콘들 */}
        <button
          onClick={() => setActiveTab('workflows')}
          className={`p-3 rounded-lg transition-colors ${
            activeTab === 'workflows'
              ? 'bg-blue-100 dark:bg-blue-600 text-blue-700 dark:text-white'
              : 'text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700'
          }`}
          title="워크플로우"
        >
          <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
          </svg>
        </button>
        
        <button
          onClick={() => setActiveTab('templates')}
          className={`p-3 rounded-lg transition-colors ${
            activeTab === 'templates'
              ? 'bg-blue-100 dark:bg-blue-600 text-blue-700 dark:text-white'
              : 'text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700'
          }`}
          title="템플릿"
        >
          <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 5a1 1 0 011-1h14a1 1 0 011 1v2a1 1 0 01-1 1H5a1 1 0 01-1-1V5zM4 13a1 1 0 011-1h6a1 1 0 011 1v6a1 1 0 01-1 1H5a1 1 0 01-1-1v-6zM16 13a1 1 0 011-1h2a1 1 0 011 1v6a1 1 0 01-1 1h-2a1 1 0 01-1-1v-6z" />
          </svg>
        </button>
        
        <button
          onClick={() => setActiveTab('history')}
          className={`p-3 rounded-lg transition-colors ${
            activeTab === 'history'
              ? 'bg-blue-100 dark:bg-blue-600 text-blue-700 dark:text-white'
              : 'text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700'
          }`}
          title="실행 이력"
        >
          <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
          </svg>
        </button>
      </div>
    );
  }

  return (
    <div className="w-80 border-r flex flex-col" style={{ backgroundColor: 'var(--sidebar-bg)', borderColor: 'var(--sidebar-border)' }}>
      {/* 탭 헤더 */}
      <div className="flex border-b border-gray-200 dark:border-gray-700">
        <button
          onClick={() => setActiveTab('workflows')}
          className={`flex-1 px-4 py-3 text-sm font-medium transition-colors ${
            activeTab === 'workflows'
              ? 'text-blue-600 dark:text-blue-400 border-b-2 border-blue-600 dark:border-blue-400'
              : 'text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-200'
          }`}
        >
          워크플로우
        </button>
        <button
          onClick={() => setActiveTab('templates')}
          className={`flex-1 px-4 py-3 text-sm font-medium transition-colors ${
            activeTab === 'templates'
              ? 'text-blue-600 dark:text-blue-400 border-b-2 border-blue-600 dark:border-blue-400'
              : 'text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-200'
          }`}
        >
          템플릿
        </button>
        <button
          onClick={() => setActiveTab('history')}
          className={`flex-1 px-4 py-3 text-sm font-medium transition-colors ${
            activeTab === 'history'
              ? 'text-blue-600 dark:text-blue-400 border-b-2 border-blue-600 dark:border-blue-400'
              : 'text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-200'
          }`}
        >
          실행 이력
        </button>
      </div>

      {/* 탭 컨텐츠 */}
      <div className="flex-1 overflow-y-auto">
        {activeTab === 'workflows' && (
          <div className="p-4">
            {/* 새 워크플로우 생성 버튼 */}
            <button
              onClick={onNewWorkflow}
              className="w-full mb-4 px-4 py-2.5 bg-blue-600 dark:bg-blue-600 text-white font-medium rounded-xl hover:bg-blue-700 dark:hover:bg-blue-700 transition-all duration-150 flex items-center justify-center shadow-sm hover:shadow-md hover:scale-[1.02] active:scale-[0.98]"
            >
              <span>새 워크플로우 생성</span>
            </button>

            {/* 검색 */}
            <div className="mb-4">
              <div className="relative">
                <input
                  type="text"
                  placeholder="워크플로우 검색..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="w-full pl-10 pr-4 py-2 text-sm bg-gray-50 dark:bg-gray-700 border border-gray-200 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
                <svg
                  className="absolute left-3 top-2.5 w-4 h-4 text-gray-400"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
                </svg>
              </div>
            </div>

            {/* 최근 워크플로우 */}
            {recentWorkflows.length > 0 && (
              <div className="mb-6">
                <h3 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-3">최근 사용</h3>
                <div className="space-y-2">
                  {recentWorkflows.slice(0, 3).map((workflow) => (
                    <button
                      key={workflow.id}
                      onClick={() => onWorkflowSelect(workflow.id)}
                      className="w-full text-left p-3 bg-gray-50 dark:bg-gray-700 hover:bg-gray-100 dark:hover:bg-gray-600 rounded-lg transition-colors"
                    >
                      <div className="font-medium text-sm text-gray-800 dark:text-gray-200 truncate">
                        {workflow.name}
                      </div>
                      <div className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                        {new Date(workflow.updatedAt).toLocaleDateString()}
                      </div>
                    </button>
                  ))}
                </div>
              </div>
            )}

            {/* 모든 워크플로우 */}
            <div>
              <div className="flex items-center justify-between mb-3">
                <h3 className="text-sm font-medium text-gray-700 dark:text-gray-300">
                  모든 워크플로우 ({filteredWorkflows.length})
                </h3>
              </div>
              
              {filteredWorkflows.length === 0 ? (
                <div className="text-center py-8 text-gray-500 dark:text-gray-400">
                  <div className="text-2xl mb-2">📝</div>
                  <p className="text-sm">워크플로우가 없습니다</p>
                  <p className="text-xs mt-1">새 워크플로우를 만들어보세요</p>
                </div>
              ) : (
                <div className="space-y-2 max-h-[500px] overflow-y-auto">
                  {filteredWorkflows.map((workflow) => (
                    <div
                      key={workflow.id}
                      className="border border-gray-200 dark:border-gray-600 hover:bg-gray-50 dark:hover:bg-gray-700 rounded-lg transition-colors"
                    >
                      <div
                        onClick={() => onWorkflowSelect(workflow.id)}
                        className="w-full text-left p-3 cursor-pointer"
                      >
                        <div className="font-medium text-sm text-gray-800 dark:text-gray-200 truncate">
                          {workflow.name}
                        </div>
                        {workflow.description && (
                          <div className="text-xs text-gray-500 dark:text-gray-400 mt-1 line-clamp-2">
                            {workflow.description}
                          </div>
                        )}
                        <div className="flex items-center justify-between text-xs mt-2">
                          <div className="text-gray-400 dark:text-gray-500">
                            {workflow.nodes.length}개 노드 • {new Date(workflow.updatedAt).toLocaleDateString()}
                          </div>
                          <div className="flex space-x-1" onClick={(e) => e.stopPropagation()}>
                            <button
                              onClick={(e) => {
                                e.stopPropagation();
                                duplicateWorkflow(workflow, e);
                              }}
                              className="p-1.5 bg-blue-100 dark:bg-blue-900/40 text-blue-600 dark:text-blue-300 rounded hover:bg-blue-200 dark:hover:bg-blue-800/60 transition-colors"
                              title="복제"
                            >
                              📄
                            </button>
                            <button
                              onClick={(e) => {
                                e.stopPropagation();
                                deleteWorkflow(workflow.id, workflow.name, e);
                              }}
                              className="p-1.5 bg-red-100 dark:bg-red-900/40 text-red-600 dark:text-red-300 rounded hover:bg-red-200 dark:hover:bg-red-800/60 transition-colors"
                              title="삭제"
                            >
                              🗑️
                            </button>
                          </div>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </div>
          </div>
        )}

        {activeTab === 'templates' && (
          <div className="p-4">
            <p className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-4">워크플로우 템플릿</p>
            <div className="space-y-3">
              {templates.map((template) => (
                <button
                  key={template.id}
                  onClick={() => applyTemplate(template)}
                  className="w-full text-left p-4 border border-gray-200 dark:border-gray-600 hover:bg-gray-50 dark:hover:bg-gray-700 rounded-lg transition-colors group"
                >
                  <div className="flex items-start space-x-3">
                    <div className="text-2xl">{template.icon}</div>
                    <div className="flex-1 min-w-0">
                      <div className="font-medium text-sm text-gray-800 dark:text-gray-200">
                        {template.name}
                      </div>
                      <div className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                        {template.description}
                      </div>
                      <div className="flex items-center space-x-1 mt-2">
                        {template.nodes.map((nodeType, index) => (
                          <div
                            key={index}
                            className="text-xs bg-gray-100 dark:bg-gray-600 text-gray-600 dark:text-gray-300 px-2 py-1 rounded"
                          >
                            {nodeType === 'input' && '입력'}
                            {nodeType === 'process' && '처리'}
                            {nodeType === 'output' && '출력'}
                            {nodeType === 'condition' && '조건'}
                            {nodeType === 'loop' && '반복'}
                          </div>
                        ))}
                      </div>
                    </div>
                  </div>
                </button>
              ))}
            </div>
          </div>
        )}

        {activeTab === 'history' && (
          <div className="p-4">
            <p className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-4">실행 이력</p>
            {executionHistory.length === 0 ? (
              <div className="text-center py-8 text-gray-500 dark:text-gray-400">
                <div className="text-2xl mb-2">⏱️</div>
                <p className="text-sm">실행 이력이 없습니다</p>
                <p className="text-xs mt-1">워크플로우를 실행해보세요</p>
              </div>
            ) : (
              <div className="space-y-3">
                {executionHistory.slice(0, 10).map((execution, index) => (
                  <div
                    key={`${execution.nodeId}-${index}`}
                    className="p-3 border border-gray-200 dark:border-gray-600 rounded-lg"
                  >
                    <div className="flex items-center justify-between mb-2">
                      <div className="text-sm font-medium text-gray-800 dark:text-gray-200">
                        노드 실행
                      </div>
                      <div className={`text-xs px-2 py-1 rounded-full ${
                        execution.status === 'success'
                          ? 'bg-green-100 dark:bg-green-900 text-green-800 dark:text-green-200'
                          : execution.status === 'error'
                          ? 'bg-red-100 dark:bg-red-900 text-red-800 dark:text-red-200'
                          : 'bg-yellow-100 dark:bg-yellow-900 text-yellow-800 dark:text-yellow-200'
                      }`}>
                        {execution.status === 'success' && '성공'}
                        {execution.status === 'error' && '실패'}
                        {execution.status === 'running' && '실행중'}
                      </div>
                    </div>
                    {execution.startTime && (
                      <div className="text-xs text-gray-500 dark:text-gray-400">
                        {new Date(execution.startTime).toLocaleString()}
                      </div>
                    )}
                    {execution.error && (
                      <div className="text-xs text-red-600 dark:text-red-400 mt-1">
                        {execution.error}
                      </div>
                    )}
                  </div>
                ))}
              </div>
            )}
          </div>
        )}
      </div>

      {/* 삭제 확인 모달 */}
      {deleteModalOpen && workflowToDelete && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white dark:bg-gray-800 rounded-xl shadow-2xl p-6 w-full max-w-md mx-4">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                워크플로우 삭제
              </h3>
              <button
                onClick={() => {
                  setDeleteModalOpen(false);
                  setWorkflowToDelete(null);
                }}
                className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
              >
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                </svg>
              </button>
            </div>

            <div className="mb-6">
              <p className="text-gray-700 dark:text-gray-300">
                <span className="font-medium">"{workflowToDelete.name}"</span> 워크플로우를 삭제하시겠습니까?
              </p>
              <p className="text-sm text-gray-500 dark:text-gray-400 mt-2">
                이 작업은 되돌릴 수 없습니다.
              </p>
            </div>

            <div className="flex space-x-3">
              <button
                onClick={() => {
                  setDeleteModalOpen(false);
                  setWorkflowToDelete(null);
                }}
                className="flex-1 px-4 py-2.5 text-sm font-medium rounded-xl border border-gray-200 dark:border-gray-700 bg-transparent text-gray-600 dark:text-gray-400 hover:bg-gray-50 dark:hover:bg-gray-800 transition-colors"
              >
                취소
              </button>
              <button
                onClick={confirmDelete}
                className="flex-1 px-4 py-2.5 text-sm font-medium rounded-xl bg-red-600 hover:bg-red-700 text-white transition-colors"
              >
                삭제
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}