"use client";

import React, { useState, useEffect } from "react";
import { XMarkIcon } from "@heroicons/react/24/solid";

interface ChunkEditModalProps {
  isOpen: boolean;
  onClose: () => void;
  chunk: any | null;
  onSave: (chunkId: string, content: string, method: "direct" | "chandra") => Promise<void>;
}

export default function ChunkEditModal({ isOpen, onClose, chunk, onSave }: ChunkEditModalProps) {
  const [editMode, setEditMode] = useState<"direct" | "chandra">("direct");
  const [editedContent, setEditedContent] = useState("");
  const [isSaving, setIsSaving] = useState(false);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    if (chunk && isOpen) {
      setEditedContent(chunk.content || "");
      setEditMode("direct");
      setError(null);
    }
  }, [chunk, isOpen]);

  if (!isOpen || !chunk) return null;

  const handleSave = async () => {
    if (!chunk.id) {
      setError("청크 ID가 없습니다.");
      return;
    }

    setIsSaving(true);
    setError(null);

    try {
      await onSave(chunk.id, editedContent, editMode);
      onClose();
    } catch (err: any) {
      setError(err.message || "저장 중 오류가 발생했습니다.");
    } finally {
      setIsSaving(false);
    }
  };

  const pageNumber = chunk.page_number || chunk.metadata?.page_number;

  return (
    <div className="fixed inset-0 z-50 flex items-center justify-center bg-black bg-opacity-50" onClick={onClose}>
      <div
        className="bg-white dark:bg-gray-800 rounded-lg shadow-xl max-w-4xl w-full mx-4 max-h-[90vh] overflow-hidden flex flex-col"
        onClick={(e) => e.stopPropagation()}
      >
        {/* Header */}
        <div className="flex items-center justify-between p-4 border-b border-gray-200 dark:border-gray-700">
          <h3 className="text-lg font-semibold text-gray-900 dark:text-gray-100">청크 내용 편집</h3>
          <button
            onClick={onClose}
            className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300 transition-colors"
            aria-label="닫기"
          >
            <XMarkIcon className="w-6 h-6" />
          </button>
        </div>

        {/* Content */}
        <div className="flex-1 overflow-y-auto p-4 space-y-4">
          {/* Chunk Info */}
          <div className="bg-gray-50 dark:bg-gray-900 rounded-lg p-3 text-sm">
            <div className="grid grid-cols-2 gap-2">
              <div>
                <span className="text-gray-500 dark:text-gray-400">청크 ID:</span>
                <span className="ml-2 text-gray-900 dark:text-gray-100 font-mono text-xs">{chunk.id}</span>
              </div>
              {pageNumber && (
                <div>
                  <span className="text-gray-500 dark:text-gray-400">페이지:</span>
                  <span className="ml-2 text-gray-900 dark:text-gray-100">{pageNumber}</span>
                </div>
              )}
            </div>
          </div>

          {/* Edit Mode Selection */}
          <div className="space-y-2">
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">편집 방법 선택</label>
            <div className="flex gap-4">
              <label className="flex items-center cursor-pointer">
                <input
                  type="radio"
                  name="editMode"
                  value="direct"
                  checked={editMode === "direct"}
                  onChange={(e) => setEditMode("direct")}
                  className="mr-2"
                />
                <span className="text-sm text-gray-700 dark:text-gray-300">직접 편집</span>
              </label>
              <label className="flex items-center cursor-pointer">
                <input
                  type="radio"
                  name="editMode"
                  value="chandra"
                  checked={editMode === "chandra"}
                  onChange={(e) => setEditMode("chandra")}
                  className="mr-2"
                  disabled={!pageNumber}
                />
                <span className={`text-sm ${!pageNumber ? "text-gray-400" : "text-gray-700 dark:text-gray-300"}`}>
                  Intelligence 분석 {!pageNumber && "(페이지 정보 필요)"}
                </span>
              </label>
            </div>
          </div>

          {/* Edit Content Area */}
          {editMode === "direct" ? (
            <div className="space-y-2">
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">청크 내용</label>
              <textarea
                value={editedContent}
                onChange={(e) => setEditedContent(e.target.value)}
                className="w-full h-96 p-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-900 text-gray-900 dark:text-gray-100 font-mono text-sm focus:ring-2 focus:ring-blue-500 focus:border-transparent resize-vertical"
                placeholder="청크 내용을 입력하세요..."
              />
              <div className="text-xs text-gray-500 dark:text-gray-400">
                {editedContent.length} 자 | 라인: {editedContent.split("\n").length}
              </div>
            </div>
          ) : (
            <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
              <div className="flex items-start gap-3">
                <div className="flex-shrink-0 text-2xl">🔄</div>
                <div className="flex-1 space-y-2">
                  <h4 className="font-semibold text-blue-900 dark:text-blue-100">Intelligence 분석 모드</h4>
                  <p className="text-sm text-blue-700 dark:text-blue-300">
                    이 옵션을 선택하면 Intelligence 분석을 사용하여 <strong>페이지 {pageNumber}</strong>를 다시 분석하고, 현재
                    청크의 내용을 새로운 분석 결과로 대체합니다.
                  </p>
                  <ul className="text-xs text-blue-600 dark:text-blue-400 list-disc list-inside space-y-1">
                    <li>선택한 페이지의 내용이 다시 분석되어 추출됩니다</li>
                    <li>처리에 몇 초 정도 소요될 수 있습니다</li>
                  </ul>
                </div>
              </div>
            </div>
          )}

          {/* Error Message */}
          {error && (
            <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg p-3 text-sm text-red-700 dark:text-red-300">
              {error}
            </div>
          )}
        </div>

        {/* Footer */}
        <div className="flex items-center justify-end gap-3 p-4 border-t border-gray-200 dark:border-gray-700">
          <button
            onClick={onClose}
            disabled={isSaving}
            className="px-4 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-600 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
          >
            취소
          </button>
          <button
            onClick={handleSave}
            disabled={isSaving || (editMode === "direct" && !editedContent.trim())}
            className="px-4 py-2 text-sm font-medium text-white bg-blue-600 rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors flex items-center gap-2"
          >
            {isSaving && (
              <svg className="animate-spin h-4 w-4" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                <path
                  className="opacity-75"
                  fill="currentColor"
                  d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                ></path>
              </svg>
            )}
            {isSaving ? "저장 중..." : editMode === "direct" ? "저장" : "Intelligence 분석 및 저장"}
          </button>
        </div>
      </div>
    </div>
  );
}
