/**
 * 날짜/시간 관련 유틸리티 함수들
 */

/**
 * ISO 문자열을 현지 타임존으로 변환
 * @param isoString - ISO 형식의 날짜 문자열 (UTC 또는 타임존 정보 포함)
 * @returns 현지 타임존으로 변환된 날짜 문자열
 */
export function convertToLocaleDateTime(isoString: string): string {
  if (!isoString) return isoString;

  try {
    // ISO 문자열을 Date 객체로 변환
    const date = new Date(isoString);

    // 유효한한 날짜인지 확인
    if (isNaN(date.getTime())) return isoString;

    // 현지 시간으로 포맷팅
    return date.toLocaleString('ko-KR', {
      year: 'numeric',
      month: '2-digit',
      day: '2-digit',
      hour: '2-digit',
      minute: '2-digit',
      second: '2-digit',
      timeZone: 'Asia/Seoul'
    });
  } catch (error) {
    console.error('날짜 변환 오류:', error);
    return isoString;
  }
}

/**
 * ISO 문자열을 한국 표준 형식으로 변환
 * @param isoString - ISO 형식의 날짜 문자열
 * @returns YYYY-MM-DD HH:mm:ss 형식의 한국 시간
 */
export function convertToKST(isoString: string): string {
  if (!isoString) return isoString;

  try {
    const date = new Date(isoString);

    if (isNaN(date.getTime())) return isoString;

    // 한국 시간 (UTC+9)으로 변환
    const kstDate = new Date(date.getTime() + (9 * 60 * 60 * 1000));

    const pad = (num: number) => num.toString().padStart(2, '0');

    const year = kstDate.getFullYear();
    const month = pad(kstDate.getMonth() + 1);
    const day = pad(kstDate.getDate());
    const hours = pad(kstDate.getHours());
    const minutes = pad(kstDate.getMinutes());
    const seconds = pad(kstDate.getSeconds());

    return `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
  } catch (error) {
    console.error('KST 변환 오류:', error);
    return isoString;
  }
}

/**
 * 현지 타임존 기준으로 날짜 포맷팅
 * @param dateString - 날짜 문자열
 * @returns 포맷팅된 날짜 문자열
 */
export function formatLocalDate(dateString: string): string {
  if (!dateString) return '';

  try {
    const date = new Date(dateString);

    if (isNaN(date.getTime())) return dateString;

    return date.toLocaleString('ko-KR', {
      year: 'numeric',
      month: 'long',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
      second: '2-digit',
      timeZone: 'Asia/Seoul'
    });
  } catch (error) {
    console.error('날짜 포맷팅 오류:', error);
    return dateString;
  }
}

/**
 * 상대적인 시간을 사람이 읽기 쉽게 표시
 * @param date - Date 객체
 * @returns "방금", "N분 전", "N시간 전", "N일 전" 등
 */
export function formatRelativeTime(date: Date): string {
  const now = new Date();
  const diffInSeconds = Math.floor((now.getTime() - date.getTime()) / 1000);

  if (diffInSeconds < 60) {
    return '방금';
  } else if (diffInSeconds < 3600) {
    const minutes = Math.floor(diffInSeconds / 60);
    return `${minutes}분 전`;
  } else if (diffInSeconds < 86400) {
    const hours = Math.floor(diffInSeconds / 3600);
    return `${hours}시간 전`;
  } else if (diffInSeconds < 2592000) {
    const days = Math.floor(diffInSeconds / 86400);
    return `${days}일 전`;
  } else if (diffInSeconds < 7776000) {
    const weeks = Math.floor(diffInSeconds / 604800);
    return `${weeks}주 전`;
  } else {
    const months = Math.floor(diffInSeconds / 2592000);
    return `${months}개월 전`;
  }
}

/**
 * 날짜 범위 검증
 * @param startDate - 시작 날짜
 * @param endDate - 종료 날짜
 * @returns 유효한 범위인지 여부
 */
export function isValidDateRange(startDate: string, endDate: string): boolean {
  if (!startDate || !endDate) return false;

  try {
    const start = new Date(startDate);
    const end = new Date(endDate);

    if (isNaN(start.getTime())) return false;
    if (isNaN(end.getTime())) return false;

    return start <= end;
  } catch (error) {
    return false;
  }
}